/*
 * Copyright (C) 2006 Sjoerd Simons <sjoerd@luon.net>
 * Copyright (C) 2003, 2004 Laurent Sansonetti <lrz@gnome.org>
 *                     2004 Erwan Loisant <erwan@loisant.org>
 *
 * This file is part of Ruby/GStreamer.
 *
 * Ruby/GStreamer is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * Ruby/GStreamer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Ruby/GStreamer; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "rbgst.h"

/* Class: Gst::Bin
 * Base container element.
 */

/*
 * Class method: new(name=nil)
 * name: a name for the bin.
 *
 * Constructs a new Gst::Bin object.
 *
 * If element name is ommited (or nil), then the bin will receive a guaranteed
 * unique name, consisting of the "bin" string and a number.
 * If name is given, it will be given the name supplied.
 *
 * Returns: a newly allocated Gst::Bin object.
 */
static VALUE
rb_gst_bin_new (int argc, VALUE * argv, VALUE self)
{
    GstElement *bin;
    VALUE name;

    rb_scan_args (argc, argv, "01", &name);

    bin = gst_bin_new (NIL_P (name) ? NULL : RVAL2CSTR (name));
    if (bin != NULL)
        RBGST_INITIALIZE (self, bin);
    return Qnil;
}

/*
 * Method: add(*elements)
 * elements: a list of Gst::Element objects.
 *
 * Adds one or more Gst::Element objects to the bin.
 *
 * Returns: Nil 
 */
static VALUE
rb_gst_bin_add (int argc, VALUE * argv, VALUE self)
{
    int i;
    VALUE klass = GTYPE2CLASS(GST_TYPE_ELEMENT);
    GstBin* bin = RGST_BIN(self);

    for (i = 0; i < argc; i++) {
       if (!rb_obj_is_kind_of(argv[i], klass)) {
	 rb_raise(rb_eTypeError, "Gst::Element expected");
       }
       /* The bin takes over ownership, so add an extra ref for the ruby holder
        * */
       gst_object_ref(RVAL2GOBJ(argv[i]));
       gst_bin_add (bin, RGST_ELEMENT (argv[i]));
       G_CHILD_ADD(self, argv[i]);
    }
    return Qnil;
}

/*
 * Method: remove(*elements)
 * elements: a list of Gst::Element objects.
 *
 * Removes one or more Gst::Element objects from the bin, unparenting 
 * as well.
 *
 * Returns: Nil
 */
static VALUE
rb_gst_bin_remove (int argc, VALUE * argv, VALUE self)
{
    int i;

    for (i = 0; i < argc; i++) {
        gst_bin_remove (RGST_BIN (self), RGST_ELEMENT (argv[i]));
        G_CHILD_REMOVE(self, argv[i]);
    }
    return Qnil;
}

/*
 * Method: remove_all
 *
 * Removes all Gst::Element objects in the bin.
 *
 * Returns: an empty array.
 */
/* Fixme
static VALUE
rb_gst_bin_remove_all (VALUE self)
{
    VALUE arr;
    int i;

    arr = rb_gst_bin_get_elements (self);
    for (i = 0; i < RARRAY (arr)->len; i++) {
        VALUE element = rb_ary_entry (arr, i);
        rb_gst_bin_remove (1, &element, self);
    }
    return rb_ary_clear (arr);
}
*/

/*
 * Method: each_element { |element| ... }
 *
 * Calls the block for each element in the bin, passing a reference to
 * the Gst::Element as parameter.
 *
 * Returns: always  nil.
 */
static VALUE
rb_gst_bin_each_element (VALUE self)
{
  GstIterator* it = gst_bin_iterate_elements(RGST_BIN(self));
  GstElement *elem;
  gboolean done = FALSE;

  while (!done) {
    switch (gst_iterator_next (it, (gpointer)&elem)) {
      case GST_ITERATOR_OK:
        rb_yield(RGST_ELEMENT_NEW(elem));
        gst_object_unref (elem);
        break;
      case GST_ITERATOR_RESYNC:
        gst_iterator_resync (it);
        break;
      case GST_ITERATOR_ERROR:
      case GST_ITERATOR_DONE:
        done = TRUE;
        break;
    }
  }
  gst_iterator_free (it);
  
  return Qnil;
}

/*
 * Method: get_by_name(name)
 * name: a name.
 *
 * Gets the element with the given name from the bin, as a reference to 
 * a Gst::Element object.
 *
 * Returns: a Gst::Element reference, or nil if the bin does not contain
 * an element with the given name.
 */
static VALUE
rb_gst_bin_get_by_name (VALUE self, VALUE name)
{
    GstElement *element = gst_bin_get_by_name (RGST_BIN (self),
                                               RVAL2CSTR (name));

    return element != NULL ? RGST_ELEMENT_NEW (element)
        : Qnil;
}

/*
 * Method: get_by_name_recurse_up(name)
 * name: a name.
 *
 * Gets the element with the given name from the bin, as a reference to 
 * a Gst::Element object. If the element is not found, a recursion is 
 * performed on the parent bin.
 *
 * Returns: a Gst::Element reference, or nil if no element with the
 * given name is found.
 */
static VALUE
rb_gst_bin_get_by_name_recurse_up (VALUE self, VALUE name)
{
    GstElement *element = gst_bin_get_by_name_recurse_up (RGST_BIN (self),
                                                          RVAL2CSTR (name));

    return element != NULL ? RGST_ELEMENT_NEW (element)
        : Qnil;
}

/*
 * Method: get_by_interface(interface)
 * interface: an interface (Ruby class).
 *
 * Looks for the first element inside the bin that implements the 
 * given interface. If such an element is found, it returns the element.
 * If you want all elements that implement the interface, use 
 * Gst::Bin#get_all_by_interface. The method recurses bins inside bins.
 *
 * Returns: An element inside the bin implementing the interface, as a
 * Gst::Element object.
 */
static VALUE
rb_gst_bin_get_by_if (VALUE self, VALUE klass)
{
    return RGST_ELEMENT_NEW (gst_bin_get_by_interface (RGST_BIN (self),
                                                       CLASS2GTYPE (klass)));
}

void
Init_gst_bin (void)
{
    VALUE c = G_DEF_CLASS (GST_TYPE_BIN, "Bin", mGst);

    rb_define_method (c, "initialize", rb_gst_bin_new, -1);

    rb_define_method (c, "add", rb_gst_bin_add, -1);
    rb_define_method (c, "remove", rb_gst_bin_remove, -1);

    rb_define_method (c, "each_element", rb_gst_bin_each_element, 0);

    rb_define_method (c, "get_by_name", rb_gst_bin_get_by_name, 1);
    rb_define_method (c, "get_by_name_recurse_up",
                      rb_gst_bin_get_by_name_recurse_up, 1);
    rb_define_method (c, "get_by_interface", rb_gst_bin_get_by_if, 1);
    //rb_define_method (c, "get_all_by_interface", rb_gst_bin_get_all_by_if, 1);
    //rb_define_method (c, "each_by_interface", rb_gst_bin_each_by_if, 1);

    G_DEF_CLASS (GST_TYPE_BIN_FLAGS, "Flags", c);
    G_DEF_CONSTANTS (c, GST_TYPE_BIN_FLAGS, "GST_BIN_");
}
