/*
    Bear Engine - Level editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/item_field_edit.hpp
 * \brief A frame to edit the fields of an item.
 * \author Julien Jorge
 */
#ifndef __BF_ITEM_FIELD_EDIT_HPP__
#define __BF_ITEM_FIELD_EDIT_HPP__

#include <wx/panel.h>
#include <wx/listctrl.h>

#include "bf/item_instance.hpp"
#include "bf/type_field.hpp"

#include <vector>

namespace bf
{
  /**
   * \brief A frame to edit the fields of an item.
   * \author Julien Jorge
   */
  class item_field_edit:
    public wxListView
  {
  public:
    /** \brief The proxy will apply the modifications to the items for us. */
    struct proxy
    {
      /**
       * \brief Get the item identifiers valid for being given to a field of an
       *        item.
       * \param id (out) The identifiers.
       * \param f The type of the field currently edited.
       */
      virtual void get_item_identifiers
      ( wxArrayString& id, const type_field& f ) { };

      /**
       * \brief Remove the value of a field.
       * \param item The item of which a field is cleared.
       * \param name The name of the field to clear.
       */
      virtual void
      delete_field( item_instance& item, const std::string& name ) { };

      /* simple types */

      /**
       * \brief Set the value of a field.
       * \param item The item in which the field is changed.
       * \param name The name of the field.
       * \param v The new value.
       * @{
       */
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const integer_type& v ) { };

      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const u_integer_type& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const real_type& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const bool_type& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const string_type& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const sprite& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const animation_file_type& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const item_reference_type& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const font_file_type& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const sample_file_type& v ) { };

      /* lists types */
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const std::list<integer_type>& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const std::list<u_integer_type>& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const std::list<real_type>& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const std::list<bool_type>& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const std::list<string_type>& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const std::list<sprite>& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const std::list<animation_file_type>& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const std::list<item_reference_type>& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const std::list<font_file_type>& v ) { };
      virtual void
      set_field_value( item_instance& item, const std::string& name,
                       const std::list<sample_file_type>& v ) { };
      /** @} */
    }; // proxy

  public:
    item_field_edit( proxy& p, wxWindow* parent, wxWindowID id = wxID_ANY );

    void set_item( item_instance* item );

    bool has_item() const;
    item_instance& get_item();
    const item_instance& get_item() const;

    bool get_field_name( unsigned int i, std::string& name ) const;

  private:
    void enumerate_properties();
    void find_hierachy
    ( std::list<std::string>& hierarchy, const std::string& class_name ) const;
    void get_fields_of
    ( std::vector<std::string>& fields, const item_class& item ) const;
    void show_fields( const std::vector<std::string>& fields );

    void update_values();
    void set_required_color( unsigned int i, bool b );
    void set_default_value_color( unsigned int i, bool b );

    wxString convert_value_to_text( const type_field& f ) const;

    void adjust_last_column_size();

    void delete_selected_field();
    void create_field_editor( const std::string& name );
    void show_string_property_dialog( const type_field& f );
    void show_item_reference_property_dialog( const type_field& f );

    template<typename Type>
    void
    show_simple_property_dialog( const type_field& f, const wxString& type );

    template<typename Control>
    void show_property_dialog( const type_field& f, const wxString& type );

    template<typename Control, typename Type>
    void edit_field( const type_field& f, const wxString& type );

    template<typename Type>
    void edit_item_reference_field
    ( const type_field& f, const wxArrayString& values );

    template<typename DialogType>
    void show_dialog( const std::string& field_name, DialogType& dlg );

    template<typename Type>
    wxString convert_value_to_text( const std::string& field_name ) const;

    void on_size(wxSizeEvent& event);
    void on_column_begin_drag(wxListEvent& event);
    void on_column_end_drag(wxListEvent& event);
    void on_item_activated(wxListEvent& event);
    void on_key_up(wxKeyEvent& event);

  private:
    /** \brief This object will do the changes on the item. */
    proxy& m_proxy;

    /** \brief The item of which we modify the fields. */
    item_instance* m_item;

    /** \brief The name of the class of the current item. */
    std::string m_item_class_name;

    /** \brief The prefixes of the not displayed fields. */
    std::set<std::string> m_hidden;

    /** \brief The backround color of the prefix of the fields. */
    static const wxColour s_field_prefix_colour;

    /** \brief The text used for prefix for fields without prefix. */
    static const std::string s_no_prefix;

  }; // class item_field_edit
} // namespace bf

#include "bf/impl/item_field_edit.tpp"

#endif // __BF_ITEM_FIELD_EDIT_HPP__
