/*
    Bear Engine - Animation editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/animation_editor.cpp
 * \brief Implementation of the bf::animation_editor class.
 * \author Julien Jorge.
 */
#include "bf/animation_editor.hpp"

#include "bf/config_frame.hpp"
#include "bf/image_pool.hpp"
#include "bf/main_frame.hpp"
#include "bf/path_configuration.hpp"

#include <wx/tooltip.h>
#include <claw/logger.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bf::animation_editor::animation_editor()
  : m_locale( wxLocale::GetSystemLanguage() )
{
  claw::logger.set( new claw::console_logger() );
  claw::logger.set_level( claw::log_verbose );

  m_locale.AddCatalog( wxT("bear-factory") );
} // animation_editor::animation_editor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bf::animation_editor::~animation_editor()
{
  // frames are deleted by wxWidgets

  claw::logger.clear();
} // animation_editor::~animation_editor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called when the application is initializing.
 */
bool bf::animation_editor::OnInit()
{
  wxToolTip::Enable(true);

  init();

  main_frame* frame = NULL;

  if (argc > 1)
    for (int i=1; i<argc; ++i)
      {
        frame = new main_frame();
        frame->load_animation( argv[i] );
        frame->Show();
      }
  else
    {
      frame = new main_frame();
      frame->SetSize( m_config.main_rect );
      frame->Show();
    }

  return true;
} // animation_editor::OnInit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Configure the program.
 */
void bf::animation_editor::configure()
{
  config_frame dlg(NULL);

  if ( dlg.ShowModal() == wxID_OK )
    update_image_pool();
} // animation_editor::configure()

/*----------------------------------------------------------------------------*/
/**
 * \brief Update the image pool.
 */
void bf::animation_editor::update_image_pool() const
{
  std::list<std::string>::const_iterator it;

  for ( it=path_configuration::get_instance().data_path.begin();
        it!=path_configuration::get_instance().data_path.end(); ++it )
    image_pool::get_instance().scan_directory(*it);
} // animation_editor::update_image_pool()

/*----------------------------------------------------------------------------*/
/**
 * \brief Save the position and the size of the main frame.
 */
void bf::animation_editor::set_main_rect( const wxRect& r )
{
  m_config.main_rect = r;
  m_config.save();
} // animation_editor::set_main_rect()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load and apply the configuration.
 */
void bf::animation_editor::init()
{
  m_config.load();

  if ( path_configuration::get_instance().data_path.empty() )
    {
      config_frame dlg(NULL);
      dlg.ShowModal();
    }

  update_image_pool();
} // animation_editor::init()
