/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file item_handle.cpp
 * \brief Implementation of the bear::universe::item_handle class.
 * \author Julien Jorge
 */
#include "universe/item_handle.hpp"

#include "universe/physical_item.hpp"
#include <cstdlib>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
template<typename ItemType>
bear::universe::base_item_handle<ItemType>::base_item_handle()
  : m_item(NULL)
{

} // base_item_handle::base_item_handle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param item The item to handle.
 */
template<typename ItemType>
bear::universe::base_item_handle<ItemType>::base_item_handle( ItemType* item )
  : m_item(item)
{
  if ( m_item )
    m_item->add_handle( this );
} // base_item_handle::base_item_handle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param item The item to handle.
 */
template<typename ItemType>
bear::universe::base_item_handle<ItemType>::base_item_handle( ItemType& item )
  : m_item(&item)
{
  m_item->add_handle( this );
} // base_item_handle::base_item_handle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that The handle from which we copy.
 */
template<typename ItemType>
bear::universe::base_item_handle<ItemType>::base_item_handle
( const self_type& that )
  : m_item(that.m_item)
{
  if ( m_item )
    m_item->add_handle( this );
} // base_item_handle::base_item_handle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
template<typename ItemType>
bear::universe::base_item_handle<ItemType>::~base_item_handle()
{
  if ( m_item )
    m_item->remove_handle( this );
} // base_item_handle::base_item_handle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the pointer.
 */
template<typename ItemType>
ItemType* bear::universe::base_item_handle<ItemType>::get() const
{
  return m_item;
} // base_item_handle::get()

/*----------------------------------------------------------------------------*/
/**
 * \brief Dereference operator.
 */
template<typename ItemType>
ItemType& bear::universe::base_item_handle<ItemType>::operator*() const
{
  return *m_item;
} // base_item_handle::operator*()

/*----------------------------------------------------------------------------*/
/**
 * \brief Pointer-to-member operator.
 */
template<typename ItemType>
ItemType* bear::universe::base_item_handle<ItemType>::operator->() const
{
  return m_item;
} // base_item_handle::operator->()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assigment operator.
 * \param item The item to assign.
 */
template<typename ItemType>
bear::universe::base_item_handle<ItemType>&
bear::universe::base_item_handle<ItemType>::operator=( ItemType* item )
{
  if ( m_item != NULL )
    m_item->remove_handle( this );

  m_item = item;

  if ( m_item != NULL )
    m_item->add_handle( this );

  return *this;
} // base_item_handle::operator=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assigment operator.
 * \param that The item to assign.
 */
template<typename ItemType>
bear::universe::base_item_handle<ItemType>&
bear::universe::base_item_handle<ItemType>::operator=( const self_type& that )
{
  return *this = that.m_item;
} // base_item_handle::operator=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Equality.
 * \param item The pointer to compare to.
 */
template<typename ItemType>
bool bear::universe::base_item_handle<ItemType>::operator==
( const ItemType* item ) const
{
  return m_item == item;
} // base_item_handle::operator==()

/*----------------------------------------------------------------------------*/
/**
 * \brief Equality.
 * \param that The pointer to compare to.
 */
template<typename ItemType>
bool bear::universe::base_item_handle<ItemType>::operator==
( const self_type& that ) const
{
  return m_item == that.m_item;
} // base_item_handle::operator==()

/*----------------------------------------------------------------------------*/
/**
 * \brief Disequality.
 * \param item The pointer to compare to.
 */
template<typename ItemType>
bool bear::universe::base_item_handle<ItemType>::operator!=
( const ItemType* item ) const
{
  return m_item != item;
} // base_item_handle::operator!=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Disequality.
 * \param that The instance to compare to.
 */
template<typename ItemType>
bool bear::universe::base_item_handle<ItemType>::operator!=
( const self_type& that ) const
{
  return m_item != that.m_item;
} // base_item_handle::operator!=()

/*----------------------------------------------------------------------------*/
/**
 * \brief "Less than" operator.
 * \param that The pointer to compare to.
 */
template<typename ItemType>
bool bear::universe::base_item_handle<ItemType>::operator<
  ( const self_type& that ) const
{
  return m_item < that.m_item;
} // base_item_handle::operator<()
