#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Copyright(C) 2007-2008 INL
Written by Romain Bignon <romain AT inl.fr>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

$Id: nulog.py 12325 2008-01-16 18:00:56Z romain $
"""

from nevow import rend, loaders, tags#, static, context, flat
from nevow import inevow, guard
from nevow.i18n import _, I18NConfig, render as i18nrender
from nucentral.client import Service
from ConfigParser import SafeConfigParser
from twisted.python import util
from tools import args2url, getBacktrace
import tools
from sessions import getSession, createAuthResource
import SOAPpy
from cStringIO import StringIO
import ajax
import csv
import os
import time
from pages import BasePage, LoginPage, The404Page
from frags import PageContent
from zope.interface import implements

LOCALE_DIR = util.sibpath(__file__, 'po')
mypath = os.path.dirname(__file__) + os.sep
etcpath = '/etc/nulog/'
nulog_version = '2.0'
nulog_conf = 'nulog.conf'
nulog_defconf = 'default.nulog.conf'

def get_langs(ctx):

    # get user langs
    my_langs = ctx.locate(inevow.ILanguages)

    # available langs
    langs = ['fr', 'en']

    # Find what is my lang in array
    my_lang = None
    for l in my_langs:
        if l in langs:
            my_lang = l
            break

    try:
        langs.remove(my_lang)
    except:
        pass

    return langs

class SearchFragment(rend.Fragment):
    """ This is a rend.Fragment of a webpage, with its own XML template. """

    docFactory = loaders.xmlfile(mypath + 'xml/search.xml', ignoreDocType=True)

    render_i18n = i18nrender()

class EdenNavBar(rend.Fragment):

    docFactory = loaders.xmlfile(mypath + 'xml/edennavbar.xml', ignoreDocType=True)

    render_i18n = i18nrender()

    def render_langs(self, ctx, data):

        args = ctx.locate(inevow.IRequest).args
        args = dict((i, args[i][0]) for i in args)

        langs = get_langs(ctx)

        return ctx.tag.clear()[tags.li[
                        [tags.a(href=args2url(args, lang=language))[
                                tags.img(src=('img/%s.jpg' % language), alt=language), ' '
                            ] for language in langs]
                    ]]

class LeftMenu(rend.Fragment):
    """ Fragment for left menu """

    docFactory = loaders.xmlfile(mypath + 'xml/left.xml', ignoreDocType=True)

    render_i18n = i18nrender()

    def __init__(self, conf):
        self.conf = conf
        rend.Fragment.__init__(self)

    def render_nupik(self, ctx, data):
        return loaders.stan(tags.a(href=self.conf.get('Links', 'url'))[tags.img(src="img/nupik.png", alt="Nupik!")])

    def _float_render(self, value):
        """
            Default is %1.4f, and we use this callback
            to only show %1.2f.
        """

        if isinstance(value, tuple):
            return '%1.2f' % value[0][0]
        else:
            self.conf.install_error = value
            return '0'

    def render_average1(self, ctx, data):
        return Service.callService("nulog-core", "count", ('average1')).addCallback(self._float_render)

    def render_average5(self, ctx, data):
        return Service.callService("nulog-core", "count", ('average5')).addCallback(self._float_render)

    def render_average15(self, ctx, data):
        return Service.callService("nulog-core", "count", ('average15')).addCallback(self._float_render)

    def render_userconnected(self, ctx, data):
        return Service.callService("nulog-core", "count", ('conusers')).addCallback(self._must_int)

    def render_perfs(self, ctx, data):
        return Service.callService("nulog-core", "count", ('packets')).addCallback(self._must_int)

    def _must_int(self, result):

        try:
            return result[0]
        except:
            return 0

    def render_time(self, ctx, data):
        now = time.localtime()
        return time.strftime('%H:%M:%S', now)

    def render_version(self, ctx, data):
        return 'v%s' % nulog_version

    def render_login(self, ctx, data):
        session = getSession(ctx)
        if session.username:
            return ctx.tag.clear()[_('Logged in as %s') % session.username]
        elif hasattr(Service, 'standalone'):
            return _('Not logged in')
        else:
            return ctx.tag.clear()[tags.a(href='login')[_('Login')]]

    def render_reset(self, ctx, data):
        return ctx.tag.clear()[(tags.a(_class='cmd', href='cmd/session/reset')[_('Reset settings')])]

    def data_badhosts(self, ctx, data):
        return Service.callService("nulog-core", "table", 'BadHosts', {'limit': 5})

    def data_badusers(self, ctx, data):
        return Service.callService("nulog-core", "table", 'BadUsers', {'limit': 5})

class NulogPage(BasePage):
    """ Main page (there is only one page in nulog)
        Content is a fragment.
    """

    docFactory = loaders.xmlfile(mypath + 'xml/nulog.xml')

    addSlash = True

    def __init__(self, conf, langcfg, AvatarId=None, content=None):
        """ @param conf [SafeConfigParser] conf object
            @param content [string] this is the requested directory in URL
        """

        # frag/ is a FragmentPage().
        self.children['frag'] = ajax.FragmentPage(conf, langcfg)
        self.children['csv'] = csv.CSVPage(conf, langcfg)
        self.children['cmd'] = ajax.CommandPage(conf, langcfg)
        self.children['login'] = LoginPage(conf, langcfg)

        if content:
            self.addSlash = False

        BasePage.__init__(self, conf, langcfg, AvatarId, content)

    def locateChild(self, ctx, segments):
        ctx.remember(The404Page(self.conf, self.langconfig, self.AvatarId), inevow.ICanHandleNotFound)
        return rend.Page.locateChild(self, ctx, segments)

    def childFactory(self, ctx, childSegment):
        """ My child is me! """

        # We tell childSegment to know what fragment call
        return NulogPage(self.conf, self.langconfig, self.AvatarId, childSegment)

    def render_refresh(self, ctx, data):

        if not self.content and int(self.conf.get('Misc', 'refresh_delay')) > 0:
            return ctx.tag.clear()[tags.meta(**{'http-equiv': "refresh", 'content': self.conf.get('Misc', 'refresh_delay')})]

        return ''

    def render_edencss(self, ctx, data):
        if int(self.conf.get('Misc', 'edenwall_integration')) > 0:
            return loaders.xmlfile(mypath + 'xml/edencss.xml')

        return ''

    def render_edennavbar(self, ctx, data):
        if int(self.conf.get('Misc', 'edenwall_integration')) > 0:
            return ctx.tag[EdenNavBar()]
        return ''

    def render_title(self, ctx, data):

        return ctx.tag.clear()[tags.a(href=self.conf.get('Links', 'url'))[self.conf.get('Links', 'maintitle')]]

    def render_langs(self, ctx, data):

        if int(self.conf.get('Misc', 'edenwall_integration')):
            return ''

        args = ctx.locate(inevow.IRequest).args
        args = dict((i, args[i][0]) for i in args)

        langs = get_langs(ctx)

        return ctx.tag.clear()[
                        [tags.a(href=args2url(args, lang=language))[
                                tags.img(src=('img/%s.jpg' % language), alt=language), ' '
                            ] for language in langs]
                    ]

    def render_content(self, ctx, data):

        return ctx.tag[PageContent(self.conf, self.content)]

    def render_leftmenu(self, ctx, data):
        frag = LeftMenu(self.conf)
        return ctx.tag[frag]

    def render_search(self, ctx, data):
        frag = SearchFragment()
        return ctx.tag[frag]

def getFragment(name, args, urlbase, render='table', title='', switch=False):
    """ Exported function to get HTML code of a fragment.
        This is *NOT* used by AJAX!
    """

    ctx = context.WovenContext()
    frag = FragmentFactory(ctx, render, name, title, urlbase=urlbase, args=args, switch=switch)

    f = frag

    io = StringIO()

    ctx = context.PageContext(parent=ctx, tag=f)
    f.rememberStuff(ctx)
    doc = f.docFactory.load(ctx)
    ctx =  context.WovenContext(ctx, tags.invisible[doc])

    def finisher(result):
        return io.getvalue()

    # usefull if we don't make an asynchronious call.
    # Note: if there is any render function, it'll fail.
    #print list(flat.iterflatten(doc, ctx, io.write, finisher))

    return flat.flattenFactory(doc, ctx, io.write, finisher)

def getComponentName():
    """ Called by NuCentral at launch """

    return 'nulog'

def getServiceList():
    """ Called by NuCentral, I return my services """

    services = dict()
    services['fragment'] = getFragment
    return services

def getPageList():
    """ Called by NuCentral at launch """

    parser = SafeConfigParser()

    parser.read([etcpath + nulog_defconf, mypath + nulog_defconf, etcpath + nulog_conf, mypath + nulog_conf])

    global LOCALE_DIR
    if not os.path.exists(LOCALE_DIR):
        LOCALE_DIR = None # We look for global path

    nulog_page = createAuthResource(NulogPage, LoginPage, parser, I18NConfig(domain='nulog', localeDir=LOCALE_DIR))

    tools.DATETIME_FORMAT = parser.get('Misc', 'datetime')
    csv.MAX_RESULTS = int(parser.get('Misc', 'csv_max_results'))

    # To know what is the name of page, we get the last word in url (specified in configuration).
    # If there is a slash at end of url, we take the second last word (because last is '')
    return {(parser.get('Links', 'url').split('/')[-1] or parser.get('Links', 'url').split('/')[-2]): nulog_page}
