#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Copyright(C) 2007-2008 INL
Written by Romain Bignon <romain AT inl.fr>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

$Id: graph.py 12313 2008-01-16 16:15:38Z romain $
"""
import matplotlib
matplotlib.use('Agg')
from matplotlib.figure import Figure
from matplotlib.backends.backend_cairo import FigureCanvasCairo as FigureCanvas
from pylab import arange, array, setp
from nevow.i18n import _, render as i18nrender

from cStringIO import StringIO
import table
from nevow import rend, loaders, tags, static
import os
import random
from tools import Args, trans
import time

mypath = os.path.dirname(__file__) + os.sep

class Graph:

    dpi = 70

    def __init__(self, size):

        self.size = size
        self.__image = None

    def get_image(self):
        if not self.__image:
            raise Exception('Please call create_img() before!')
        return self.__image

    def build(self):
        """ Callback """
        pass

    def get_coords(self):

        raise NotImplemented

    def create_img(self, filename):
        fig = Figure()
        canvas = FigureCanvas(fig)

        self.build(fig)

        fig.set_frameon(False) # Background is hidden
        fig.set_size_inches(self.size[0], self.size[1]) # Set size

        imdata=StringIO()
        fig.savefig(imdata,format='png', dpi=self.dpi)
        self.__image = imdata.getvalue()

        self.height = fig.get_figheight()* self.dpi
        self.width  = fig.get_figwidth() * self.dpi

class Pie(Graph):

    def __init__(self, size):

        self.entries = []
        self.labels = []
        Graph.__init__(self, size)

    def add_entry(self, label, value):
        """ Add a part in the pie.
            @param label [string] label
            @param value [int] Absolute value.
        """

        self.labels += [str(label),]
        self.entries += [value,]

    def build(self, fig):
        # Set of beautiful colors.

        ax = fig.add_subplot(121)

        colours = ['#21FF85', '#B2B6E0', '#5289FF', '#A9482E', '#C3F4F8', '#E5D587',
                   '#E07191', '#74D2F1', '#5BC466', '#92E0DF', '#FFFFFF', '#AFFF54',
                   '#C09858', '#FFCB75', '#33ADFF', '#9E4570', '#9AE0A1', '#47BE4F',
                   '#CC0099', '#E0DD8D', '#FF8A2B', '#4B5DFF', '#6DF8BE', '#9C56FF',
                   '#BE7344', '#CCBE78', '#E0ACD0', '#FF37E1', '#45709E', '#676FFF',
                   '#4CAC84', '#35FF1A', '#806170', '#C3BF46', '#E0829A', '#E6CBB7']
        self.__pie = ax.pie(self.entries, labels=None, colors=colours, autopct=None, shadow=True)

        ax = fig.add_subplot(122)
        self.__legend = ax.legend(self.__pie, self.labels, shadow=True)
        ax.grid(False)
        ax.set_axis_off()

    def get_coords(self):
        # We calculate coords to know where are all labels.
        #return []
        coords = []
        for i in self.__legend.get_texts():

            left, bottom, width, height = i.get_window_extent().get_bounds()
            coords += [[left,
                        self.height - bottom - height,
                        left + width,
                        self.height - bottom]]

        return coords

class Histogram(Graph):

    """
    We can create an Histogram with this syntax:
       >>> test = Histogram((255,255), ["redteam", "greenteam", "blueteam", "purpleteam"])

    test is an histogram with size 255x255, and 4 plots "redteam", "greenteam", "blueteam", "purpleteam"
       >>> test.add_entry(1, (1,2,3,4))
       >>> test.add_entry(2, (11,22,33,44))
       >>> test.create_img("score.png")
    """

    def __init__(self, size, plot_labels):
        """
           @param size (tuple) image size
           @param plot_labels (list) list of plot labels
        """

        self.size = size
        self.__data = []

        for i in plot_labels:
            self.__data += [[]]
        self.__plots = ()

        self.__plot_labels = plot_labels
        self.__labels = []

    def add_entry(self, label, values):
        """
           @param label (string) prout
        """

        if len(values) != len(self.__data):
            raise Exception("Bad call (values = %d, __plot_labels = %d)" % (len(values), len(self.__plot_labels)))

        self.__labels += [str(label)]

        i = 0
        for data in self.__data:
            data += [values[i]]
            i += 1

    def build(self, fig):

        ax = fig.add_subplot(111)

        rows = len(self.__data)

        ind = arange(len(self.__labels))/2.0  # the x locations for the groups

        def hex2f(hex):

            return float(int(hex, 16))/255

        cellText = []
        self.__bars = ()
        width = 0.5     # the width of the bars
        yoff = array([0.0] * len(self.__labels)) # the bottom values for stacked bar chart
        for row in xrange(rows):
            colours = []

            # Calculate color of a each bar; We use data value to gradiant from blue to red.
            for i in self.__data[row]:
                n = float(i)/max(self.__data[row])
                colours += [(
                             (hex2f('c3') + n * (hex2f('e0')-hex2f('c3'))),
                             (hex2f('82') + (1-n) * (hex2f('f4')-hex2f('82'))),
                             (hex2f('9a') + (1-n) * (hex2f('f8')-hex2f('9a')))
                            )]

            self.__bars += (ax.bar(ind, self.__data[row], width, bottom=yoff, color=colours),)
            yoff = yoff + self.__data[row]
            cellText.append(['%d' % (x) for x in yoff]) # Labels

        ax.set_xticks(ind+width/2 )
        ax.set_xticklabels(self.__labels)

        # Make a rotation on absisce labels
        setp(ax.get_xticklabels(), 'rotation', 45,
             'horizontalalignment', 'right', fontsize=8)


    def get_coords(self):
        # Calculate coords of each bars.
        # TODO: It probably doesn't work with multiple barlevel.
        coords = []
        for bar in self.__bars:
            if not bar:
                continue
            x1 = []
            y1 = []
            x2 = []
            y2 = []
            # For each bars, we get values of rectangles
            for b in bar:
                x1 += [b.get_x(),]
                y1 += [b.get_y(),]
                x2 += [b.get_x() + b.width,]
                y2 += [b.get_y() + b.height,]

            # We get the top left points for each bars...
            x1s, y1s = bar[0].get_transform().seq_x_y(x1, y1)

            # ...and the bottom right points for each bars.
            x2s, y2s = bar[0].get_transform().seq_x_y(x2, y2)

            # Now for each bars, we can make a (x,y,w,h) data coords for HTML map area.
            for sx1, sy1, sx2, sy2, b in zip(x1s, y1s, x2s, y2s, bar):
                cb = b.get_clip_box()
                # Put self.height - sy1 on 4th element to select only bar.
                coords += [[sx1, self.height - sy2, sx2, self.height],]
        return coords

class GraphFragment(table.TableFragment):

    #docFactory = loaders.xmlfile(mypath + 'xml/table.xml', ignoreDocType=True)

    def __init__(self, name, function, title, urlbase, args, icon='', switch=True, frag=None, pie=False):
        """ Init.
            @param function [string] this is the name of the table (TODO: rename?)
            @param title [string] title of table.
            @param icon [string] if specified, a picture will be showed before title name. (file in img/)
            @param multitable [bool] if True, we will use multitable arguments formats (Tablename_arg)
            @param switch [bool] if this fragment can be showed with a table or an other type of graph
            @param pie [bool] if true, show a pie, in other case, show a histogram.
        """
        table.TableFragment.__init__(self, name, function, title, urlbase, args, icon, switch, frag)
        self.pie = pie
        self.total_time = time.time()

    def render_table(self, context, data):

        if not hasattr(self, "table"):
            if hasattr(self, "error"):
                return self.error
            else:
                return _('Unable to find this table')

        if not self.table.table:
            return tags.div(_class='noinfo')[tags.p[_('No data')]]

        return self.render_entry(context, data)

    def render_head(self, context, data):
        """ This method is overloaded to not display the table header (with column names) """

        return ''

    def _render_htmlarea(self, graph):
        """ Calculate HTML areas """

        render = []

        for coords, entry in zip(graph.get_coords(), self.table.table):

            if isinstance(entry[0], tuple):
                label = entry[0][0]
                value = entry[0][1]
            else:
                label = entry[0]
                value = entry[0]

            # Get the url for the first (0) cell.
            url = self._get_cell_link(self.table.columns[0], label, value)

            if not url:
                continue

            render += [tags.area(alt=str(entry[0]),
                                 shape="rect",
                                 coords="%d,%d,%d,%d" % (coords[0],coords[1],coords[2],coords[3]),
                                 href=url)]

        return tags._map(name=self.function)[render]

    def render_entry(self, context, data):
        """ Print an entry.
            @param data [list] this is the list of args on this line
        """

        self.graph_time = time.time()

        if not hasattr(self, "table"):
            return ''

        a = Args(self.table.args)
        table = self.table

        if self.pie:
            graph = Pie((5,2.5))
            nbpackets = 0
            for line in self.table.table:
                graph.add_entry(a.get_data(table.columns[0], line[0]),
                                line[1])
                nbpackets += line[1]

            # Add Other part
            if self.table.count and self.table.count > nbpackets:
                graph.add_entry(str(trans(context, _("Other"))), self.table.count - nbpackets)

        else:
            graph = Histogram((5,2.5), [''])
            for line in self.table.table:
                graph.add_entry(a.get_data(table.columns[0], line[0]), [line[1]])

        while True:
            filename = '%s_%06d.png' % (self.name, random.randint(1,999999))
            if not GraphImages.images.has_key(filename):
                break

        graph.create_img(filename)
        GraphImages.images[filename] = graph.get_image()

        # HTML map is calculed
        htmlmap = self._render_htmlarea(graph)

        return loaders.stan([htmlmap,
                             tags.img(alt='Graph', src='graphs/%s' % filename, usemap='#%s' % self.function,
                                         width=int(graph.width), height=int(graph.height)),
                            # Show perfs in a comment
                             tags.comment['Graph: %1.3fs, Sql: %1.3fs, Total: %1.3fs' % (time.time() - self.graph_time,
                                                 time.time() - self.total_time - (time.time() - self.graph_time),
                                                 time.time() - self.total_time)
                                ]
                            ])

class GraphImages(rend.Page):

    images = {}

    def childFactory(self, ctx, childSegment):
        if self.images.has_key(childSegment):
            # FIXME: some browsers will call two times this image!
            return static.Data(self.images.pop(childSegment), 'image/png')
        return rend.Page.childFactory(self, ctx, childSegment)
