# -*- coding: utf-8 -*-
#
#  Copyright (C) 2004-2009 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

# TODO:
# - きのことサーフェスの間に他のウインドウが入ることができてしまうのを直す.
# - NAYUKI/2.0
# - 透明度の設定

import os

if 'DISPLAY' in os.environ:
    import gtk
    import gobject
    import ninix.pix

import ninix.config
import ninix.seriko


class Menu:

    def __init__(self, kinoko, accelgroup):
        self.__kinoko = kinoko
        ui_info = '''
        <ui>
          <popup name='popup'>
            <menuitem action='Settings'/>
            <menu action='Skin'>
            </menu>
            <separator/>
            <menuitem action='Exit'/>
          </popup>
        </ui>
        '''
        self.__menu_list = {
            'settings': [('Settings', None, ('Settings...(_O)'), None,
                          '', self.__kinoko.edit_preferences),
                         '/ui/popup/Settings'],
            'skin':     [('Skin', None, _('Skin(_K)'), None),
                         None, '/ui/popup/Skin'],
            'exit':     [('Exit', None, _('Exit(_Q)'), None,
                          '', self.__kinoko.close),
                         '/ui/popup/Exit'],
            }
        self.__skin_list = None
        actions = gtk.ActionGroup('Actions')
        entry = []
        for value in self.__menu_list.itervalues():
            entry.append(value[0])
        actions.add_actions(tuple(entry))
        ui_manager = gtk.UIManager()
        ui_manager.insert_action_group(actions, 0)
        ui_manager.add_ui_from_string(ui_info)
        self.__popup_menu = ui_manager.get_widget('/ui/popup')
        for key in self.__menu_list:
            path = self.__menu_list[key][-1]
            self.__menu_list[key][1] = ui_manager.get_widget(path)

    def popup(self, button):
        skin_list = self.__kinoko.get_skin_list()
        self.__set_skin_menu(skin_list)
        self.__popup_menu.popup(
            None, None, None, button, gtk.get_current_event_time())

    def __set_skin_menu(self, list): ## FIXME
        key = 'skin'
        if list:
            menu = gtk.Menu()
            for skin in list:
                item = gtk.MenuItem(skin['title'])
                item.connect('activate', self.__kinoko.select_skin, (skin))
                menu.add(item)
                item.show()
            self.__menu_list[key][1].set_submenu(menu)
            menu.show()
            self.__menu_list[key][1].show()
        else:
            self.__menu_list[key][1].hide()

class Nayuki:

    def __init__(self, kinoko):
        self.kinoko = kinoko

class Kinoko:

    def __init__(self, skin_list):
        self.skin_list = skin_list
        self.skin = None

    def edit_preferences(self, action):
        pass

    def finalize(self):
        self.__running = 0
        self.target.delete_observer(self)
        if self.skin is not None:
            self.skin.destroy()

    def notify_observer(self, event, args): ## FIXME
        if self.skin is None:
            return
        if event in ['set position', 'set surface']:
            self.skin.set_position()
            self.skin.show()
        elif event == 'set scale': ## FIXME
            scale = self.target.prefs.get('surface_scale')
            self.skin.set_scale(scale)
        elif event == 'hide':
            if not self.target.surface_is_shown(0):
                self.skin.hide()
        elif event == 'iconified':
            self.skin.hide()
        elif event == 'deiconified':
            self.skin.show()
        elif event == 'finalize':
            self.finalize()
        elif event == 'move surface':
            side, xoffset, yoffset = args
            if side == 0:
                self.skin.set_position(xoffset, yoffset)
        elif event == 'raise':
            side = args
            if side == 0:
                self.skin.set_position() ## FIXME
        else:
            ##print 'OBSERVER(kinoko): ignore -', event
            pass

    def load(self, data, target):
        self.data = data
        self.target = target
        self.target.set_observer(self)
        self.accelgroup = gtk.AccelGroup()
        scale = self.target.prefs.get('surface_scale')
        self.skin = Skin(self.data, self.accelgroup, self, scale)
        if self.skin is None:
            return 0
        else:
            self.send_event('OnKinokoObjectCreate')
        self.__running = 1
        gobject.timeout_add(10, self.do_idle_tasks) # 10ms
        return 1

    def do_idle_tasks(self):
        if not self.__running:
            return False
        return True

    def close(self, action):
        self.finalize()
        self.send_event('OnKinokoObjectDestroy')

    def send_event(self, event):
        if event not in ['OnKinokoObjectCreate', 'OnKinokoObjectDestroy',
                         'OnKinokoObjectChanging', 'OnKinokoObjectChanged',
                         'OnKinokoObjectInstalled']:
                         ## 'OnBatteryLow', 'OnBatteryCritical',
                         ## 'OnSysResourceLow', 'OnSysResourceCritical'
            return
        args = (self.data['title'],
                self.data['ghost'],
                self.data['category'])
        self.target.notify_event(event, *args)

    def get_skin_list(self):
        return self.skin_list

    def select_skin(self, widget, args):
        self.send_event('OnKinokoObjectChanging')
        self.skin.destroy()
        self.data = args
        scale = self.target.prefs.get('surface_scale')
        self.skin = Skin(self.data, self.accelgroup, self, scale)
        if self.skin is None:
            return 0
        else:
            self.send_event('OnKinokoObjectChanged')
        return 1

class Skin:

    def __init__(self, data, accelgroup, kinoko, scale):
        self.frame_buffer = []
        self.data = data
        self.accelgroup = accelgroup
        self.kinoko = kinoko
        self.__menu = Menu(self.kinoko, self.accelgroup)
        self.__scale = scale
        self.__shown = 0
        self.surface_id = 0 # dummy
        self.window = gtk.Window()
        ##self.window.set_title(''.join(('surface.', name)))
        self.window.set_decorated(False)
        self.window.set_resizable(False)
        self.window.set_skip_taskbar_hint(True)
        self.window.connect('delete_event', self.delete)
        self.window.connect('button_press_event', self.button_press)
        self.window.connect('button_release_event', self.button_release)
        self.window.connect('motion_notify_event', self.motion_notify)
        self.window.connect('leave_notify_event', self.leave_notify)
        self.window.set_events(gtk.gdk.BUTTON_PRESS_MASK|
                               gtk.gdk.BUTTON_RELEASE_MASK|
                               gtk.gdk.POINTER_MOTION_MASK|
                               gtk.gdk.LEAVE_NOTIFY_MASK)
        self.window.realize()
        self.window.add_accel_group(self.accelgroup)
        if self.data['animation'] is not None:
            path = os.path.join(self.data['dir'], self.data['animation'])
            self.seriko = ninix.seriko.Controler(
                {'': ninix.seriko.get_actors(ninix.config.open(path))})
        else:
            base, ext = os.path.splitext(self.data['base'])
            path = os.path.join(self.data['dir'], ''.join((base, 'a.txt')))
            if os.path.exists(path):
                self.seriko = ninix.seriko.Controler(
                    {'': ninix.seriko.get_actors(ninix.config.open(path))})
            else:
                self.seriko = ninix.seriko.Controler({'': []})
        path = os.path.join(self.data['dir'], self.data['base'])
        try:
            self.pixbuf = ninix.pix.create_pixbuf_from_file(path)
            scale = self.__scale
            if scale != 100:
                w = max(1, self.pixbuf.get_width() * scale / 100)
                h = max(1, self.pixbuf.get_height() * scale / 100)
                surface_pixbuf = self.pixbuf.scale_simple(
                    w, h, gtk.gdk.INTERP_BILINEAR)
            else:
                surface_pixbuf = self.pixbuf.copy()
            image, mask = surface_pixbuf.render_pixmap_and_mask(255)
            del surface_pixbuf
        except: ## FIXME
            self.kinoko.close(None)
            return
        ## FIXME
        self.w, self.h = image.get_size()
        self.set_position()
        self.window.shape_combine_mask(mask, 0, 0)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        self.darea.connect('expose_event', self.redraw)
        self.darea.set_size_request(self.w, self.h)
        self.darea.show()
        self.window.add(self.darea)
        self.darea.realize()
        self.darea.window.set_back_pixmap(image, False)
        target_window = self.kinoko.target.surface.window[0].window # XXX
        if self.data['ontop']:
            self.window.set_transient_for(target_window)
        else:
            target_window.set_transient_for(self.window)
        self.show()
        self.seriko.reset(self, '') # XXX
        self.seriko.update(self)
        self.seriko.invoke_kinoko(self)

    def show(self):
        if not self.__shown:
            self.window.show()
            self.__shown = 1

    def hide(self):
        if self.__shown:
            self.window.hide()
            self.__shown = 0

    def prefetch(self, surface_ids):
        pass

    def append_actor(self, frame, actor):
        self.seriko.append_actor(frame, actor)

    def set_position(self, xoffset=0, yoffset=0):
        base_x, base_y = self.kinoko.target.get_kinoko_position(
            self.data['baseposition'])
        a, b = [(0.5, 1), (0.5, 0), (0, 0.5), (1, 0.5), (0, 1),
                (1, 1), (0, 0), (1, 0), (0.5, 0.5)][self.data['baseadjust']]
        offsetx = self.data['offsetx']
        offsety = self.data['offsety']
        scale = self.__scale
        if scale != 100:
            offsetx = offsetx * scale / 100
            offsety = offsety * scale / 100
        self.x = base_x - int(self.w * a) + offsetx + xoffset
        self.y = base_y - int(self.h * b) + offsety + yoffset
        self.window.move(self.x, self.y)

    def set_scale(self, scale):
        self.__scale = scale
        self.udpate()
        self.set_position()

    def get_surface(self): ## FIXME
        return None

    def redraw(self, widget, event): ## FIXME
        pass

    def get_pixbuf(self, surface_id):
        path = os.path.join(self.data['dir'], ''.join(('surface', str(surface_id), '.png')))
        if os.path.exists(path):
            pixbuf = ninix.pix.create_pixbuf_from_file(path)
        else:
            pixbuf = None
        return pixbuf

    def create_surface_pixbuf(self, surface_id=None):
        if surface_id is not None and surface_id != '':
            surface_pixbuf = self.get_pixbuf(surface_id)
        else:
            surface_pixbuf = self.pixbuf.copy()
        return surface_pixbuf

    def clear_frame_buffer(self):
        self.frame_buffer = []

    def update_frame_buffer(self, seriko, frame, move, dirty):
        if dirty:
            surface_pixbuf = self.create_surface_pixbuf(seriko.base_id)
            if surface_pixbuf is not None:
                # draw overlays
                for pixbuf_id, x, y in seriko.iter_overlays():
                    try:
                        pixbuf = self.get_pixbuf(pixbuf_id)
                        w = pixbuf.get_width()
                        h = pixbuf.get_height()
                    except:
                        continue
                    # overlay surface pixbuf
                    sw = surface_pixbuf.get_width()
                    sh = surface_pixbuf.get_height()
                    if x + w > sw:
                        w = sw - x
                    if y + h > sh:
                        h = sh - y
                    if x < 0:
                        dest_x = 0
                        w += x
                    else:
                        dest_x = x
                    if y < 0:
                        dest_y = 0
                        h += y
                    else:
                        dest_y = y
                    pixbuf.composite(surface_pixbuf, dest_x, dest_y,
                                     w, h, x, y, 1.0, 1.0,
                                     gtk.gdk.INTERP_BILINEAR, 255)
        else:
            surface_pixbuf = None
        self.frame_buffer.append((frame, move, surface_pixbuf, dirty))

    def update(self):
        if len(self.frame_buffer) > 0:
            frame, move, surface_pixbuf, dirty = self.frame_buffer.pop(0)
            if move is not None:
                self.move_surface(*move)
            if not dirty:
                return
        else:
            surface_pixbuf = self.create_surface_pixbuf()
        scale = self.__scale
        if scale != 100:
            w = max(1, surface_pixbuf.get_width() * scale / 100)
            h = max(1, surface_pixbuf.get_height() * scale / 100)
            surface_pixbuf = surface_pixbuf.scale_simple(
                w, h, gtk.gdk.INTERP_BILINEAR)
        image, mask = surface_pixbuf.render_pixmap_and_mask(255)
        del surface_pixbuf
        self.w, self.h = image.get_size()
        self.darea.window.set_back_pixmap(image, False)
        self.window.shape_combine_mask(mask, 0, 0)
        #print 'DRAW:', dir(self.darea)
        #self.darea.queue_draw_area(0, 0, w, h)
        self.darea.queue_draw()

    def terminate(self):
        self.seriko.terminate(self)

    def add_overlay(self, actor, surface_id, x, y):
        self.seriko.add_overlay(self, actor, surface_id, x, y)

    def remove_overlay(self, actor):
        self.seriko.remove_overlay(actor)

    def move_surface(self, xoffset, yoffset): ## FIXME
        self.window.move(self.x + xoffset, self.y + yoffset)

    def set_surface(self, surface_id, restart=1): ## FIXME
        self.pixbuf = self.get_pixbuf(surface_id)
        self.update()

    def invoke(self, actor_id, update=0):
        self.seriko.invoke(self, actor_id, update)

    def delete(self, widget, event):
        self.kinoko.close(None)

    def destroy(self): ## FIXME
        self.seriko.terminate(self)
        self.window.destroy()

    def button_press(self, widget, event): ## FIXME
        self.x_root = event.x_root
        self.y_root = event.y_root
        if event.type == gtk.gdk.BUTTON_PRESS:
            click = 1
        else:
            click = 2
        button = event.button
        if button == 3 and click == 1:
            self.__menu.popup(button)
        return True

    def button_release(self,  widget, event): ## FIXME
        pass

    def motion_notify(self,  widget, event): ## FIXME
        pass

    def leave_notify(self,  widget, event): ## FIXME
        pass
