/* Copyright (C) 2000-2009 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_IO_H
#include <io.h>
#endif
#include <errno.h>
#include <math.h>

#include "udm_common.h"
#include "udm_utils.h"
#include "udm_searchtool.h"
#include "udm_boolean.h"
#include "udm_vars.h"
#include "udm_conf.h"
#include "udm_log.h"


typedef struct
{
  udm_pos_t     pos;
  udm_wordnum_t order;
} UDM_CRD;



typedef struct
{
  size_t   acoords;
  size_t   ncoords;
  UDM_CRD  *Coords;
  udm_secno_t secno;
} UDM_CRDLIST;



/********** QSORT functions *******************************/

static int
cmp_pos(UDM_CRD *s1, UDM_CRD *s2)
{
  return (int) s1->pos - (int) s2->pos;
}


static int
cmp_score(UDM_URL_SCORE *c1, UDM_URL_SCORE *c2)
{
  return ((int4) c2->score) - ((int4) c1->score);
}


static int
cmp_score_then_url_id(UDM_URL_SCORE *c1, UDM_URL_SCORE *c2)
{
  int diff= ((int4) c2->score) - ((int4) c1->score);
  if (diff)
    return diff;
  return c1->url_id - c2->url_id;
}


/* Find topcount best results */

#define UDM_URL_SCORE_IS_SMALLER(x,y) \
 (x->score < y->score ||(x->score == y->score && x->url_id >= y->url_id))

void
UdmURLScoreListSortByScoreThenURLTop(UDM_URLSCORELIST *ScoreList,
                                     size_t topcount)
{
  UDM_URL_SCORE *First= ScoreList->Item;
  UDM_URL_SCORE *Last= ScoreList->Item + ScoreList->nitems;
  UDM_URL_SCORE *TopCount= ScoreList->Item + topcount;
  UDM_URL_SCORE *Curr;

  UDM_ASSERT(topcount < ScoreList->nitems);
/*
  for (Curr= First; Curr < Last; Curr++)
    fprintf(stderr, "%d:%d\n", Curr->score, Curr->url_id);
*/  
  UdmSort((void*) ScoreList->Item, topcount + 1,
          sizeof(UDM_URL_SCORE), (udm_qsort_cmp) cmp_score_then_url_id);

  for (Curr= TopCount; Curr < Last; Curr++)
  {
    UDM_URL_SCORE tmp;
    UDM_URL_SCORE *Left, *Right;
    if (UDM_URL_SCORE_IS_SMALLER(Curr, TopCount))
        continue;

    /*
      The Curr item is bigger than the TopCount item.
      Find its sorted place, and insert into the "topcount" list.
      Put the old TopCount item instead of the Curr item.
    */
    for(Left= First, Right= TopCount; Left < Right; )
    {
      UDM_URL_SCORE *Middle= Left + (Right - Left) / 2;
      if (UDM_URL_SCORE_IS_SMALLER(Curr, Middle))
        Left= Middle + 1;
      else
        Right= Middle;
    }
    tmp= *TopCount;
    memmove(Right + 1, Right, (TopCount-Right)*sizeof(UDM_URL_SCORE));
    *Right= *Curr;
    *Curr= tmp;
  }
}


void UdmURLScoreListSortByScore(UDM_URLSCORELIST *ScoreList)
{
  UdmSort((void*) ScoreList->Item, ScoreList->nitems,
          sizeof(UDM_URL_SCORE), (udm_qsort_cmp) cmp_score);
}


void UdmURLScoreListSortByScoreThenURL(UDM_URLSCORELIST *ScoreList)
{
  UdmSort((void*) ScoreList->Item, ScoreList->nitems,
          sizeof(UDM_URL_SCORE), (udm_qsort_cmp) cmp_score_then_url_id);
}


#define NUMWORD_FACTOR_SIZE 256
#define NUMWORD_FACTOR(nwf, n) (nwf[n >= NUMWORD_FACTOR_SIZE ? NUMWORD_FACTOR_SIZE - 1 : n])

typedef struct udm_score_param_st
{
  unsigned int *R;
  unsigned int *D;
  unsigned int *Dsum_add;
  unsigned int *RDsum_add;
  size_t D_size;
  size_t ncosine;
  size_t nsections;
  float Rsum_factor;
  size_t dst_offs;
  size_t nwf_offs;
  size_t nwf_num;
  unsigned int dst_weight;
  UDM_WIDEWORD *Word;
  size_t nuniq;
  size_t nwords;
  int search_mode;
  char *count;
  size_t count_size;
  UDM_STACKITEMLIST ItemList;
  char wf[256];
  char wf2[256];
  char nwf[256];
  float numword_factor[NUMWORD_FACTOR_SIZE];
  float numdistinctword_factor[NUMWORD_FACTOR_SIZE];
  float max_coord_factor;
  unsigned int MinCoordFactor;
  unsigned int NumDistinctWordFactor;
  int have_WordFormFactor;
  float WordFormFactor;
  float WordFormFactorReminder;
  int SaveSectionSize;
  float WordDensityFactor;
  float WordDensityFactorReminder;
  urlid_t debug_url_id;
} UDM_SCORE_PARAM;


typedef struct udm_score_parts_st
{
  size_t RDsum;
  size_t dstsum;
  size_t dstnum;
  size_t wfsum;
  float min_max_factor;
  float density_factor;
  float numword_factor;
  float wordform_factor;
} UDM_SCORE_PARTS;


static void
UdmDebugScore(char *str, size_t nbytes,
              UDM_SCORE_PARAM *score_param,
              UDM_SCORE_PARTS *score_parts,
              urlid_t url_id,
              uint4 score)
{
  size_t dstsum= score_parts->dstnum ?
                 score_parts->dstsum *
                 score_param->dst_weight / score_parts->dstnum / 255 :
                 0;

  udm_snprintf(str, nbytes,
               "url_id=%d RDsum=%d distance=%d (%d/%d) "
               "minmax=%.8f density=%.8f "
               "numword=%.8f wordform=%.8f score=%d",
               url_id, score_parts->RDsum, dstsum,
               score_parts->dstsum, score_parts->dstnum,
               score_parts->min_max_factor, score_parts->density_factor,
               score_parts->numword_factor, score_parts->wordform_factor,
               (int) score);
}


/*
  R[i] and D[i] are in the range 0..64.
  ns is between 1..256
*/


#define MAXCOORD_FACTOR(factor, max_coord) ((1-((float)factor*(float)(max_coord > 0xFFFF ? 0xFFFF :  max_coord))))
#define MINCOORD_FACTOR(factor, coord) ((float) 0x1000 / (float) (0x1000 + factor * ((coord > 0xFFFF ? 0xFFFF : coord) - 1)))


static inline float
UdmMinMaxCoordFactor(UDM_SCORE_PARAM *score_param,
                     UDM_CRD *B, UDM_CRD *E)
{
  float x;
  x= MAXCOORD_FACTOR(score_param->max_coord_factor, E->pos) *
     MINCOORD_FACTOR(score_param->MinCoordFactor, B->pos);
  /*
  fprintf(stderr, "[%d] secno=%d min=%d max=%d x=%.8f\n",
                  B->url_id, (int) B->secno, (int) B->pos, (int) E->pos, x);
  */
  return x;
}


static inline void
CalcAverageWordDistanceTwoCoords(size_t wf2_secno,
                                 UDM_CRD *phr, size_t num,
                                 size_t *psum, size_t *pnp)
{
  size_t ord0= phr[0].order;
  size_t ord1= phr[1].order;
  size_t pos0= phr[0].pos;
  size_t pos1= phr[1].pos;
  size_t res= (ord0 == ord1) ? 0 : pos1 > pos0 ? pos1 - pos0 : pos0 - pos1;
  (*pnp)++;
  (*psum)+= res > 0 ? (res - 1) * wf2_secno : 0;
}


/*
  All words must be from the same section.
*/
static void
CalcAverageWordDistance(size_t wf2_secno,
                        UDM_CRD *phr, size_t num, size_t nuniq,
                        size_t *psum, size_t *pnp)
{
  UDM_CRD *last, *last1, *phr0, *phr2;
  size_t sum, np;
  udm_wordnum_t prev_order;

  if (num < 2)
    return;
  
  if (num == 2)
  {
    CalcAverageWordDistanceTwoCoords(wf2_secno, phr, num, psum, pnp);
    return;
  }

  sum= np= 0;
  phr0= phr;
  phr2= phr + 2;
  last= phr + num;
  last1= last - 1;

  /* Add special case: "BEG w1 w2 w2" */
  if (phr[1].order == phr[2].order &&
      phr[1].order != phr[0].order)
  {
    uint4 diff= phr[1].pos - phr->pos;
    sum+= diff;
    np++;
  }
  
  prev_order= phr->order;
  for (phr++; phr < last1; phr++)
  {
    udm_wordnum_t current_order= phr->order;
    
    /*
      Detect these sequences:
        "w1 w1 W2 w2"
           "w1 W2 w3"
           "w1 W2 w1"
    */
    
    if (prev_order != current_order)
    {
      UDM_CRD *phr1= phr + 1;
      if (phr1->order != current_order)
      {
        uint4 diff1= phr->pos - phr[-1].pos;
        uint4 diff2= phr1->pos - phr->pos;
        if (prev_order == phr1->order)
        {
          /* "w1 W2 w1" - add min distance */
          sum+= (diff1 < diff2 ? diff1 : diff2);
          np++;
        }
        else
        {
          /* "w1 W2 w3" - add two distances */
          sum+= (diff1 + diff2);
          np+= 2;
          if (diff1 + diff2 <= 2)
          {
            UDM_CRD *phr3= phr1 + 1;
            /*
             We have three different words consequently.
             This is much better than just pairs.
            */
            np+=2;
            if (nuniq == 3)
              np+= 512; /* Exact phrase */
            if (phr3 < last &&
                phr3->order != phr1->order &&
                phr3->pos - phr1->pos <= 1)
            {
              UDM_CRD *phr4= phr3 + 1;
              /* Four different words consequently */
              np+= 2;
              if (nuniq == 4) /* Exact phrase - all four words from the query */
                np+= 512;
              
              if (phr4 < last &&
                  phr4->order != phr3->order &&
                  phr4->pos - phr3->pos <= 1)
              {
                /* Exact phrase - all five words consequently */
                np+= 512;
              }
            }
          }
        }
      }
      else
      {
        /* "w1 W2 w2" */
        if (phr >= phr2 &&
            phr[-2].order == prev_order)
        {
          /* w1 w1 W2 w2 */
          uint4 diff= phr->pos - phr[-1].pos;
          sum+= diff;
          np++;
          phr++;
          prev_order= phr->order;
          continue;
        }
      }
    }
    prev_order= current_order;
  }
  
  phr= last1;
  /* Add special case: "w1 w1 w2 END" */
  if (phr[-1].order == phr[-2].order &&
      phr[-1].order != phr->order)
  {
    uint4 diff= phr->pos - phr[-1].pos;
    sum+= diff;
    np++;
  }
  
  /* Reduce all diffs by 1. */
  if (sum > np)
    sum-= np;
  else
    sum= 0;

  *psum+= sum * wf2_secno;
  *pnp+= np;
}


#define OFFS_FROM_NUNIQ_SECNO_ORDER(n,s,o) ((n)*(s)+(o))

/*
static uint4
UdmCalcCosineWeightOneCoord(UDM_SCORE_PARAM *score_param,
                            UDM_CRD *Crd,
                            float nwords_factor)
{
  float res;
  size_t wrdsec= Crd->secno;
  size_t order= Crd->order;
  size_t offs= OFFS_FROM_NUNIQ_SECNO_ORDER(score_param->nuniq, wrdsec, order);
  nwords_factor*= MAXCOORD_FACTOR(score_param->max_coord_factor, Crd->pos) *
                  MINCOORD_FACTOR(score_param->MinCoordFactor, Crd->pos);
  res= score_param->Rsum_factor * nwords_factor *
       (float) (score_param->R[offs]) + 0.5;
  return res;
}
*/

static uint4
UdmCalcCosineWeightManyCoords(UDM_SCORE_PARAM *score_param,
                              UDM_SECTION *Section, size_t nsections,
                              char *wf2, float nwords_factor,
                              UDM_SCORE_PARTS *score_parts)
{
  float res;

  size_t Dsum=0, RDsum= 0, max_added_offs= 0;
  char *added= (char*) score_param->D;
  
  if (score_param->nwords == 1)
  {
    for( ; nsections; Section++, nsections--)
    {
      size_t offs= OFFS_FROM_NUNIQ_SECNO_ORDER(1, Section->secno, 1);
      
      if (!added[offs])
      {
        Dsum+= score_param->Dsum_add[offs];
        RDsum+= score_param->RDsum_add[offs];
        added[offs]= 1;
        if (offs > max_added_offs)
          max_added_offs= offs;
      }
    }
  }
  else
  {
    size_t score_param_nuniq= score_param->nuniq;

    for( ; nsections; Section++, nsections--)
    {
      size_t offs= OFFS_FROM_NUNIQ_SECNO_ORDER(score_param_nuniq, Section->secno, Section->order);
      
      if (!added[offs])
      {
        Dsum+= score_param->Dsum_add[offs];
        RDsum+= score_param->RDsum_add[offs];
        added[offs]= 1;
        if (offs > max_added_offs)
          max_added_offs= offs;
      }
    }
    
    if (score_parts->dstnum)
    {
      size_t dstnum= score_parts->dstnum;
      size_t dstsum= score_parts->dstsum * score_param->dst_weight / dstnum / 255;
      Dsum+= dstsum * dstsum;
      RDsum+= score_param->R[score_param->dst_offs] * dstsum;
    }
  }

  /* Clear for the next call */
  bzero(added, max_added_offs + 1);

  score_parts->RDsum= RDsum;
  
  res= score_param->Rsum_factor * nwords_factor * score_parts->min_max_factor *
        (float) RDsum / sqrt(Dsum) + 0.5;

  return (uint4) res;
}



static int UdmOriginWeight(int origin)
{
  switch(origin)
  {
    case UDM_WORD_ORIGIN_QUERY: return 3;
    case UDM_WORD_ORIGIN_SPELL: return 1;
  }
  return 0;
}


static void UdmWideWordListSetOriginWeight(UDM_WIDEWORDLIST *WWList)
{
  size_t i;
  for (i=0; i < WWList->nwords; i++)
    WWList->Word[i].weight= UdmOriginWeight(WWList->Word[i].origin);
}


#if 0
static void
UdmPrintCoords(UDM_CRD *Coords, size_t ncoords)
{
  size_t i;
  for (i=0 ; i < ncoords; i++)
    fprintf(stderr, "[%d]secno=%d pos=%d seclen=%d num=%d\n", i,
                    Coords[i].secno,
                    Coords[i].pos,
                    Coords[i].seclen,
                    Coords[i].num);
}
#endif


static int
CheckOnePhrase(UDM_WIDEWORD *Word,
               UDM_STACK_ITEM *CmdBeg, UDM_STACK_ITEM *CmdEnd,
               UDM_CRD *CoordBeg, UDM_CRD *CoordEnd,
               int cmd_arg, size_t seclen)
{
  UDM_STACK_ITEM *CmdCur;
  UDM_CRD *CoordFirst;
  
  for (CoordFirst= CoordBeg; CoordFirst < CoordEnd; CoordFirst++)
  {
    size_t delta;
    UDM_CRD *CoordPrev, *CoordCurr;

    /* Skip words with wrong order */
    if (CmdBeg->arg != CoordFirst->order)
      continue;

    CoordPrev= CoordFirst;
    CoordCurr= CoordPrev + 1;
    delta= 1;

    for (CmdCur= CmdBeg + 1; CmdCur < CmdEnd; CmdCur++)
    {
      if (CmdCur->cmd == UDM_STACK_STOP)
      {
        delta++;
        continue;
      }

      /* find coord for this word */
      while (CoordCurr < CoordEnd &&
             (CoordPrev->pos == CoordCurr->pos ||
              (CoordPrev->pos + delta == CoordCurr->pos &&
               CoordCurr->order != CmdCur->arg)))
        CoordCurr++;

      if (CoordCurr == CoordEnd ||
          CoordPrev->pos != CoordCurr->pos - delta ||
          CmdCur->arg != CoordCurr->order)
        break;

      delta= 1;
      CoordPrev= CoordCurr;
    }
    
    if (CmdCur == CmdEnd)
    {
      if (cmd_arg == '=' && CmdEnd - CmdBeg + 1 != seclen)
        continue;
      
      return 1; /* Phrase found */
    }
  }
  return 0;
}
                          
static inline void
CheckPhrase(UDM_WIDEWORD *Word,
            UDM_STACK_ITEM *query, size_t nitems,
            UDM_CRD *coords, size_t ncoords,
            char *count, size_t seclen)
{
  size_t q;
  UDM_CRD *CoordEnd= coords + ncoords;

  /* find opening phrase command */
  for (q= 0; q < nitems; q++)
  {
    size_t i;
    size_t start, end, arg;
    size_t rstart, rend;
    int cmd_arg;
    if (query[q].cmd != UDM_STACK_PHRASE) continue;
    cmd_arg= query[q].arg;
    
    /* find closing phrase command */
    start= q + 1;
    for (end= start; end < nitems && query[end].cmd != UDM_STACK_PHRASE; end++);
    q= end;
    arg= 0;

    /* skip trailing stopwords for now */
    /* TODO: we have to check document length (for phrases like "word1 stopword1") */
    for (rstart= start; rstart < end && query[rstart].cmd == UDM_STACK_STOP; rstart++);
    for (rend= end; rend > rstart && query[rend].cmd == UDM_STACK_STOP; rend--);

    /* if phrase contains stopwords only, we assume this document is found */
    if (rstart == rend) arg= 1;
    else
      arg= CheckOnePhrase(Word, &query[rstart], &query[rend],
                          coords, CoordEnd, cmd_arg, seclen);

    /*
      count[] was previously filled in the loop
      in UdmGroupByURL2. Set count to 0 for those
      words which must be in a phrase, but they
      are not in a phrase.
      TODO: Call CheckPhrase per-section.
    */
    for (i= rstart; i < rend; i++)
    {
      if (query[i].cmd == UDM_STACK_WORD)
        count[query[i].arg]|= arg;
    }
  }
}

/*
  TODO: implement tf*idf as follows:
  w_ik = tf_ik * idf_k

  where

  T_k = term k in document D_i
  tf_ik = frequency of term T_k in document D_i
  idf_k = inversed document frequency of term T_k in the collection C:

  idf_k = log( N / n_k)

  N = total number of documents in the collection C (i.e. in the result)
  n_k = the number of documents in C that contain T_k

  QQ: how to combine several words and sections?
  QQ: "number of documents in C" should probably mean
      "number of documents in the current search result" in our case.
*/



static
void UdmNumWordFactorInit(float k, float *numword_factor, size_t ncoords)
{
  size_t i;
  if (k > 1) k= 1;
  if (k < 0) k= 0;
  for (i= 0; i < NUMWORD_FACTOR_SIZE; i++)
  {
    numword_factor[i]= (1-k) + ((float) i / ncoords)*k;
  }
}


#define UDM_MIX_FACTOR(Factor, newfactor, Reminder, oldfactor) \
((Factor) * (newfactor) + (Reminder) * (oldfactor))


static inline void
UdmApplyNWF(UDM_SCORE_PARAM *param,
            UDM_SECTION *Section, size_t nsections, float *nword_factor)
{
  size_t i;
  unsigned char secno_min= 255;
  unsigned char secno_max= 0;
  char section_vector[256];
  UDM_SECTION *Tmp;
  
  bzero((void*) &section_vector, param->nwf_num);
  for (Tmp= Section, i= nsections; i > 0; i--, Tmp++)
  {
    unsigned char secno= Tmp->secno;
    section_vector[secno]= 1;
    if (secno_min > secno)
      secno_min= secno;
    if (secno_max < secno)
      secno_max= secno;
  }
  if (secno_min == secno_max && param->nwf[secno_min])
    *nword_factor*= (1 -  (float) param->nwf[secno_min] / 16);
}


static inline void
UdmApplyNumDistinctWordFactor(UDM_SCORE_PARAM *param, char *count,
                              float *nword_factor)
{
  size_t z, nuniq;
  for (nuniq=0, z= 0; z < param->nuniq; z++)
  {
    if (count[z])
      nuniq++;
  }
  if (nuniq < param->nuniq)
    *nword_factor*= param->numdistinctword_factor[nuniq];
}


static inline void
CalcWordFormFactor(UDM_SCORE_PARAM *param,
                   UDM_SECTION *S, size_t nsections,
                   UDM_SCORE_PARTS *score_parts)
{
  size_t form_phr_n, ncoords;
  static int spell_factor[5]= {0, 2, 1, 1, 1};

  for (form_phr_n= 0, ncoords= 0; nsections; S++, nsections--)
  {
    int origin= param->Word[S->wordnum].origin;
    UDM_ASSERT(origin < 6);
    form_phr_n+= S->ncoords * spell_factor[origin];
    ncoords+= S->ncoords;
  }
  score_parts->wordform_factor=
     param->WordFormFactor +
     param->WordFormFactorReminder * (float) form_phr_n / ncoords / 2;
}


static inline uint4
UdmCalcScore(UDM_SCORE_PARAM *param,
             UDM_SECTION *Section, size_t nsections,
             UDM_SCORE_PARTS *score_parts)
{
  uint4 score;
  float nword_factor= score_parts->numword_factor;
  

  if (param->have_WordFormFactor)
  {
    CalcWordFormFactor(param, Section, nsections, score_parts);
    nword_factor*= score_parts->wordform_factor;
  }

  if (score_parts->wfsum)
  {
    /* Apply word density */
    nword_factor= UDM_MIX_FACTOR(param->WordDensityFactor,
                                 score_parts->density_factor,
                                 param->WordDensityFactorReminder,
                                 nword_factor);
  }
  
  if (param->search_mode != UDM_MODE_ALL)
  {
    UdmApplyNumDistinctWordFactor(param, param->count, &nword_factor);
  }

  if (param->nwf_num)
  {
    UdmApplyNWF(param, Section, nsections, &nword_factor);
  }

#if 0
  {
    char str[255];
    UdmDebugScore(str, sizeof(str), param, score_parts, Section->url_id);
    fprintf(stderr, "DebugScore: %s\n", str);
  }
#endif

  score= /*(phr_n == 1) ?
          UdmCalcCosineWeightOneCoord(param, CrdFrom, nword_factor) :*/
          
          UdmCalcCosineWeightManyCoords(param, Section, nsections,
                                        param->wf2, nword_factor,
                                        score_parts);
  return score;
}


/*
  Check if the boolean expression is simple enough
  and can be optimized the same way as "m=all" can.
*/
static int
UdmOptimizeUsingAND(UDM_STACK_ITEM *Item, size_t nitems, int search_mode)
{
  if (search_mode == UDM_MODE_ALL)
    return 1;
  if (search_mode == UDM_MODE_BOOL)
  {
    UDM_STACK_ITEM *Last;
    for (Last= Item + nitems; Item < Last; Item++)
    {
      if (Item->cmd != UDM_STACK_PHRASE &&
          Item->cmd != UDM_STACK_AND &&
          Item->cmd != UDM_STACK_WORD)
        return 0;
    }
    return 1;
  }
  return 0;
}


#define UDM_SEC_TO_CRD(dst, ord, Crd, count) \
{ \
  (dst)->order= (ord); \
  (dst)->pos= (Crd)->pos; \
  (Crd)++; \
  (count)--; \
  (dst)++; \
}


static size_t
UdmMergeCoordsN2(UDM_CRD *dst, UDM_SECTION *S1, UDM_SECTION *S2)
{
  UDM_COORD2 *src1= S1->Coord;
  UDM_COORD2 *src2= S2->Coord;
  size_t n1= S1->ncoords;
  size_t n2= S2->ncoords;
  udm_wordnum_t order1= S1->order;
  udm_wordnum_t order2= S2->order;
  udm_wordnum_t order;
  size_t n, total= n1 + n2;
  UDM_COORD2 *src;

  if (!n1 || !n2)
    return 0;
  
  for ( ; ; )
  {
    if (src1->pos < src2->pos)
    {
      UDM_SEC_TO_CRD(dst, order1, src1, n1);
      /* Skip all coords from src1, except the last one */
      for (src= src1; n1 && src1->pos < src2->pos; n1--, src1++, total--);
      if (src < src1)
      {
        n1++;
        src1--;
        total++;
        UDM_SEC_TO_CRD(dst, order1, src1, n1);
      }
      if (!n1)
      {
        n= n2;
        src= src2;
        order= order2;
        goto single;
      }
    }
    else
    {
      UDM_SEC_TO_CRD(dst, order2, src2, n2);
      /* Skip all coords from src1, except the last one */
      for (src= src2; n2 && src2->pos < src1->pos; n2--, src2++, total--);
      if (src < src2)
      {
        n2++;
        src2--;
        total++;
        UDM_SEC_TO_CRD(dst, order2, src2, n2);
      }
      if (!n2)
      {
        n= n1;
        src= src1;
        order= order1;
        goto single;
      }
    }
  }
  
single:

/*
  for ( ; n ; )
  {
    UDM_SEC_TO_CRD(dst, S, src, n);
  }
  return total;
*/

  /*
    Skip middle coords - they affect neither score, nor phrase.
    Only the first and the last coords are of interest.
  */
  if (n)
  {
    /* Put first coord, for WordDistance */
    UDM_SEC_TO_CRD(dst, order, src, n);
    
    if (n)
    {
      /* Put last coord, for MinMaxFactor */
      src+= n - 1;
      UDM_SEC_TO_CRD(dst, order, src, n);
    }
    return total - n;
  }
  return total;
}


static size_t
UdmMergeCoordsUsingSort(UDM_CRD *dst, UDM_SECTION *S, size_t srecs)
{
  size_t num;
  UDM_CRD *dst0= dst;
  for (num= 0 ;srecs; srecs--, S++)
  {
    UDM_COORD2 *src= S->Coord;
    size_t ncoords= S->ncoords;
  
    for (num+= ncoords ; ncoords; )
    {
      UDM_SEC_TO_CRD(dst, S->order, src, ncoords);
    }
  }
  if (num > 1)
    UdmSort((void*) dst0, num, sizeof(UDM_CRD), (udm_qsort_cmp) cmp_pos);
  return num;
}


#define MAX_NUM_FOR_MERGE 10
static size_t
UdmMergeCoordsMany(UDM_CRD *dst, UDM_SECTION *S, size_t srecs)
{
  UDM_COORD2 *p[MAX_NUM_FOR_MERGE];
  UDM_COORD2 *e[MAX_NUM_FOR_MERGE];
  UDM_SECTION *s[MAX_NUM_FOR_MERGE];
  size_t num, list, nlists;

  if (srecs >= MAX_NUM_FOR_MERGE)
    return UdmMergeCoordsUsingSort(dst, S, srecs);

  for (num= 0, nlists=0, list= 0; list < srecs; list++)
  {
    size_t ncoords= S[list].ncoords;
    if (ncoords)
    {
      num+= ncoords;
      p[nlists]= S[list].Coord;
      e[nlists]= S[list].Coord + ncoords;
      s[nlists]= &S[list];
      nlists++;
    }
  }

  if (!nlists)
    return 0;

  if (nlists == 1)
    goto single;

  for ( ; ; )
  {
    size_t i, min= 0;
    urlid_t p_min_pos= p[0]->pos;
    for (i= 1; i < nlists; i++)
    {
      if (p[i]->pos < p_min_pos)
      {
        min= i;
        p_min_pos= p[i]->pos;
      }
    }
    
    /**dst++= *p[min]++;*/
    dst->order= s[min]->order;
    dst->pos= p[min]->pos; 
    dst++;
    p[min]++;
    
    if (p[min] == e[min])
    {
      nlists--;
      p[min]= p[nlists];
      e[min]= e[nlists];
      s[min]= s[nlists];
      if (nlists == 1)
        break;
    }
  }

single:
  {
    size_t n= *e - *p;
    for ( ; n ; )
    {
      UDM_SEC_TO_CRD(dst, (*s)->order, *p, n);
    }
  }
  return num;
}


static void
UdmGroupByURLLoop2(UDM_AGENT *A,
                   UDM_RESULT *Res,
                   UDM_SECTIONLIST *SectionList,
                   UDM_URLSCORELIST *ScoreList,
                   UDM_SCORE_PARAM *param)
{
  UDM_SECTION *From= SectionList->Section;
  UDM_SECTION *Last= SectionList->Section + SectionList->nsections;
  UDM_SECTION *End;
  UDM_CRDLIST CoordList;
  UDM_URL_SCORE *CrdTo= ScoreList->Item;
  urlid_t url_id= From->url_id;
  UDM_WIDEWORD *Res_WWList_Word= param->Word;
  char *count= param->count;
  char *wf2= param->wf2;
  size_t count_size= param->count_size;
  int search_mode= param->search_mode;
  int check_count_all= (search_mode == UDM_MODE_ALL && param->nuniq > 30);
  int optimize_using_all, optimize_using_minmax_pos2;
  urlid_t debug_url_id= param->debug_url_id;
  uint4 expected_and_word_mask= (1UL << param->nuniq) - 1;
  size_t num_sections_for_bzero= param->nsections;
  UDM_SCORE_PARTS score_parts;

  bzero((void*) &CoordList, sizeof(CoordList));
  bzero((void*) &score_parts, sizeof(score_parts));

  optimize_using_all= UdmOptimizeUsingAND(param->ItemList.items,
                                       param->ItemList.nitems,
                                       search_mode);
  if (search_mode != UDM_MODE_ALL && optimize_using_all)
    UdmLog(A, UDM_LOG_DEBUG, "Using 'all words' optimization");

  if ((optimize_using_minmax_pos2= optimize_using_all &&
                                   param->nuniq == 2 &&
                                   Res_WWList_Word[1].phrpos > 0))
    UdmLog(A, UDM_LOG_DEBUG, "Using 'MinMaxPos2' phrase optimization");
  
  {
    char *added= (char*) param->D;
    bzero((void*) added, param->ncosine);
  }
  
  for (; From < Last ; From= End)
  {
    UDM_CRD *Tmp;
    size_t nsections;
    uint4 word_mask= 0, wfsum;
    udm_secno_t prev_secno;
    uint4 seclen[256];
    uint4 seccnt[256];
    UDM_SECTION *UniqSec[256];
    UDM_CRDLIST MergedSections[256];
    size_t nuniqsections, usec;
    size_t max_secno= 0, ncoords_for_malloc= 0;
    
    score_parts.min_max_factor= 1;
    score_parts.density_factor= 0;

    url_id= From->url_id;
    
    UniqSec[0]= From;
    nuniqsections= 0;
    prev_secno= From->secno + 1;
    
    for (End= From, CoordList.ncoords= 0, nsections= 0;
         End < Last && End->url_id == url_id;
         End++, nsections++)
    {
      ncoords_for_malloc+= End->ncoords;
      word_mask|= (1 << End->order);
      if (prev_secno != End->secno)
      {
        UniqSec[nuniqsections++]= End;
        prev_secno= End->secno;
      }
    }
    
    /* Quickly skip documents not having all words when m=all */
    if (optimize_using_all &&
        (word_mask != expected_and_word_mask ||
         nsections < param->nuniq))
      continue;

    if (optimize_using_minmax_pos2 && nsections == 2)
    {
      if (From->minpos > From[1].maxpos + 1 ||
          From[1].minpos > From->maxpos + 1)
        continue; 
    }

    UniqSec[nuniqsections]= End;
    
    bzero((void*) count, count_size);
    
    if (ncoords_for_malloc >= CoordList.acoords)
    {
      size_t nbytes;
      CoordList.acoords= ncoords_for_malloc + 1024;
      nbytes= CoordList.acoords * sizeof(UDM_CRD);
      CoordList.Coords= (UDM_CRD*) UdmRealloc(CoordList.Coords, nbytes);
    }

    bzero(&seclen, sizeof(uint4)*num_sections_for_bzero);
    bzero(&seccnt, sizeof(uint4)*num_sections_for_bzero);
    num_sections_for_bzero= 0;

    for (Tmp= CoordList.Coords, usec= 0; usec < nuniqsections; usec++)
    {
      size_t num, srecs= UniqSec[usec+1] - UniqSec[usec];
      UDM_SECTION *S= UniqSec[usec];
      UDM_CRD *PrevTmp= Tmp;
      udm_secno_t secno= S->secno;

      if (max_secno < secno)
        max_secno= secno;
      num_sections_for_bzero= max_secno + 1;

      if (srecs == 1)
      {
        UDM_WIDEWORD *W1= &Res_WWList_Word[S[0].wordnum];
        if (W1->phrlen < 2)
          count[W1->order]= 1;
        if (S->seclen)
        {
          seclen[secno]= S->seclen;
          seccnt[secno]+= S->ncoords;
        }
        score_parts.min_max_factor*= 
           MAXCOORD_FACTOR(param->max_coord_factor, S->minpos) *
           MINCOORD_FACTOR(param->MinCoordFactor, S->maxpos);

        MergedSections[usec].Coords= PrevTmp;
        MergedSections[usec].ncoords= 0;
        MergedSections[usec].secno= secno;
        continue;
      }
      else if (srecs == 2)
      {
        UDM_WIDEWORD *W1= &Res_WWList_Word[S[0].wordnum];
        UDM_WIDEWORD *W2= &Res_WWList_Word[S[1].wordnum];
        UDM_SECTION *S1= S + 1;

        if (W1->phrlen < 2)
          count[W1->order]= 1;
        if (W2->phrlen < 2)
          count[W2->order]= 1;

        if (S->seclen)
        {
          seclen[secno]= S->seclen;
          seccnt[secno]+= S->ncoords + S1->ncoords;
        }

        Tmp+= (num= UdmMergeCoordsN2(Tmp, S, S1));
      }
      else
      {
        size_t rec;
        for (rec= 0 ; rec < srecs; rec++, S++)
        {
          UDM_WIDEWORD *W= &Res_WWList_Word[S->wordnum];
          /*
            If word is not in a phrase, mark it as found.
            Words in phrase will be marked as found later in CheckPhrase()
          */
          if (W->phrlen < 2)
            count[W->order]= 1;

          if (S->seclen)
          {
            seclen[S->secno]= S->seclen;
            seccnt[S->secno]+= S->ncoords;
          }
        }

        Tmp+= (num= UdmMergeCoordsMany(Tmp, UniqSec[usec], srecs));
      }
      
      CoordList.ncoords+= num;
      score_parts.min_max_factor*= UdmMinMaxCoordFactor(param, PrevTmp, PrevTmp + num - 1);
      MergedSections[usec].Coords= PrevTmp;
      MergedSections[usec].ncoords= num;
      MergedSections[usec].secno= secno;
      
      if (search_mode == UDM_MODE_BOOL && num > 1)
        CheckPhrase(param->Word,
                    param->ItemList.items, param->ItemList.nitems,
                    PrevTmp, num, count, S->seclen);
    }

    if (search_mode == UDM_MODE_BOOL)
    {
      if(!UdmCalcBoolItems(param->ItemList.items, param->ItemList.nitems, count))
        continue;
    }

    if (check_count_all)
    {
      size_t z;
      for (z = 0; z < param->nuniq; z++)
      {
        if (count[z] == 0) break;
      }
      if (z < param->nuniq && count[z] == 0)
        continue;
    }

    {
      uint4 i;
      for (wfsum=0, i=0; i < max_secno + 1; i++)
      {
        if (seclen[i])
        {
          float add= (float) wf2[i] * seccnt[i] / seclen[i];
          wfsum+= wf2[i];
          score_parts.density_factor+= add;
        }
      }
      if (wfsum)
        score_parts.density_factor= (score_parts.density_factor > wfsum) ?
                                     1 : score_parts.density_factor / wfsum;
      score_parts.wfsum= wfsum;
    }
    
    {
      size_t i, dstnum= 0, dstsum= 0;
      for (i= 0; i < usec; i++)
      {
        UDM_CRDLIST *L= &MergedSections[i];
        CalcAverageWordDistance(wf2[L->secno],
                                L->Coords, L->ncoords, param->nuniq,
                                &dstsum, &dstnum);
      }
      score_parts.dstsum= dstsum;
      score_parts.dstnum= dstnum;
    }

    score_parts.numword_factor= NUMWORD_FACTOR(param->numword_factor, ncoords_for_malloc);

    CrdTo->score= UdmCalcScore(param, From, nsections, &score_parts);

    if (debug_url_id == url_id)
    {
      char str[255];
      UdmDebugScore(str, sizeof(str), param, &score_parts, url_id, CrdTo->score);
      UdmVarListAddStr(&A->Conf->Vars, "DebugScore", str);
    }
    
    if (CrdTo->score)
    {
      CrdTo->url_id= url_id;
      CrdTo++;
    }
  }
  UDM_FREE(CoordList.Coords);
  ScoreList->nitems= CrdTo - ScoreList->Item;
}


static void
UdmGroupByURLLoop_OneWord_All2(UDM_RESULT *Res,
                               UDM_SECTIONLIST *SectionList,
                               UDM_URLSCORELIST *ScoreList,
                               UDM_SCORE_PARAM *score_param)
{
  UDM_URL_SCORE *CrdTo= ScoreList->Item;
  UDM_SECTION *From;
  UDM_SECTION *Last= SectionList->Section + SectionList->nsections;
  char *wf2= score_param->wf2;
  char *nwf= score_param->nwf;
  size_t nwf_num= score_param->nwf_num;
  float *numword_factor= score_param->numword_factor;
  float max_coord_factor= score_param->max_coord_factor;
  float Rsum_factor= score_param->Rsum_factor;
  unsigned int MinCoordFactor= score_param->MinCoordFactor;
  char *added= (char*) score_param->D;
  size_t ncosine= score_param->ncosine;
  size_t weight= score_param->Word[0].weight;
  int have_WordFormFactor= score_param->have_WordFormFactor &&
                           Res->WWList.nwords > 1;
  float WordFormFactor= score_param->WordFormFactor;
  float WordFormFactorReminder= score_param->WordFormFactorReminder;
  /*
  TODO:
  float numword_factor_1= NUMWORD_FACTOR(numword_factor, 1);
  float Rsum_factor_mul_nword_factor_1= Rsum_factor * numword_factor_1;
  */
  unsigned int *R= score_param->R;
  unsigned int phr_n2= 0;
  unsigned int form_phr_n= 0;
  
  bzero((void*) added, ncosine);
  
  for (From= SectionList->Section ; From < Last ; )
  {
    urlid_t From_url_id= From->url_id;
    uint4 min_pos= From->minpos;
    uint4 max_pos= From->maxpos;
    size_t ncoords= 0;
    size_t Dsum= 0;
    size_t RDsum= 0;
    uint4 wfsum= 0;
    float density_factor= 0;
    unsigned char prev_secno= From->secno;
    unsigned char max_secno= 0;
    unsigned char min_secno= 255;
    float min_max_factor= 1;
    
    /* TODO: one coord, for nsections==1 and ncoords=1 */
    
    for (; ;)
    {
      unsigned char secno= From->secno;
      size_t order= score_param->Word[From->wordnum].order;
      size_t offs= OFFS_FROM_NUNIQ_SECNO_ORDER(score_param->nuniq, secno, order);

      ncoords+= From->ncoords;
      
      if (prev_secno != secno)
      {
        min_max_factor*= MAXCOORD_FACTOR(max_coord_factor, max_pos) *
                         MINCOORD_FACTOR(MinCoordFactor, min_pos);
        min_pos= From->minpos;
        max_pos= From->maxpos;
        prev_secno= secno;
      }
      else
      {
        if (max_secno < secno)
          max_secno= secno;
        if (min_secno > secno)
          min_secno= secno;
        if (min_pos > From->minpos)
          min_pos= From->minpos;
        if (max_pos < From->maxpos)
          max_pos= From->maxpos;
      }
      
      if (From->seclen)
      {
        float add= (float) wf2[secno] * From->ncoords / From->seclen;
        wfsum+= wf2[secno];
        density_factor+= add;
      }
      
      if (!added[offs])
      {
        uint4 add= wf2[secno] + weight;
        Dsum+= add * add;
        RDsum+= add * R[offs];
        added[secno]= 1;
        if (max_secno < secno)
          max_secno= secno;
      }
      
      if (have_WordFormFactor)
      {
        static int spell_factor_for_one[5]= {0, 2, 1, 1, 1};
        size_t wordnum= From->wordnum;
        int origin= wordnum < Res->WWList.nwords ? 
                    Res->WWList.Word[wordnum].origin : 1;
        UDM_ASSERT(origin < 6);
        UDM_ASSERT(From->Coord != NULL);
        form_phr_n+= From->ncoords * spell_factor_for_one[origin];
        phr_n2+= From->ncoords * 2;
      }
      
      From++;
      
      if (From >= Last || From->url_id != From_url_id)
      {
        float nword_factor;
        min_max_factor*= MAXCOORD_FACTOR(max_coord_factor, max_pos) *
                         MINCOORD_FACTOR(MinCoordFactor, min_pos);

        density_factor= (density_factor > wfsum) ? 1 : density_factor / wfsum;
        nword_factor= NUMWORD_FACTOR(numword_factor, ncoords);

        /* Applying WordFormFactor*/
        if (have_WordFormFactor)
        {
          float k;
          k= WordFormFactor + WordFormFactorReminder * (float) form_phr_n / phr_n2;
          nword_factor*= k;
          phr_n2= 0;
          form_phr_n= 0;
        }
        
        if (wfsum)
          nword_factor= UDM_MIX_FACTOR(score_param->WordDensityFactor,
                                       density_factor,
                                       score_param->WordDensityFactorReminder,
                                       nword_factor);
        /* Applying NWF */
        if (nwf_num && min_secno == max_secno && nwf[min_secno])
          nword_factor*= (1 -  (float) nwf[min_secno] / 16);
        
        CrdTo->score= Rsum_factor * nword_factor * min_max_factor *
                      (float) RDsum / sqrt(Dsum) + 0.5;
        CrdTo->url_id= From_url_id;
        CrdTo++;
        
        bzero((void*)added, (size_t) max_secno * score_param->nuniq + 1);

        break;
      }
    }
  }
  ScoreList->nitems= CrdTo - ScoreList->Item;
}


static void
UdmScoreParamInit(UDM_SCORE_PARAM *prm,
                  UDM_AGENT *query,
                  UDM_DB *db,
                  UDM_RESULT *Res)
{
  size_t i;
  float DefWordDensityFactor;
  UDM_VARLIST *Vars= &query->Conf->Vars;
  bzero((void*) prm, sizeof(UDM_SCORE_PARAM));
  prm->nsections = UdmVarListFindInt(Vars, "NumSections", 256);
  prm->dst_offs= Res->WWList.nuniq * prm->nsections;
  prm->nwf_offs= Res->WWList.nuniq * prm->nsections + 1;
  prm->max_coord_factor= ((float)UdmVarListFindInt(Vars, "MaxCoordFactor", 255)) / 0xFFFFFF;
  prm->MinCoordFactor= UdmVarListFindInt(Vars, "MinCoordFactor", 0);
  prm->have_WordFormFactor= UdmVarListFindInt(Vars, "WordFormFactor", 255) != 255;
  prm->WordFormFactor= ((float)UdmVarListFindDouble(Vars, "WordFormFactor", 255)) / 255;
  prm->WordFormFactorReminder= 1 - prm->WordFormFactor;
  prm->SaveSectionSize= UdmVarListFindBool(Vars, "SaveSectionSize", 1);
  DefWordDensityFactor= prm->SaveSectionSize ? 25 : 0;
  prm->WordDensityFactor= ((float)UdmVarListFindDouble(Vars, "WordDensityFactor", DefWordDensityFactor)) / 256;
  prm->WordDensityFactorReminder= 1 - prm->WordDensityFactor;
  prm->dst_weight= (unsigned int) UdmVarListFindInt(Vars, "WordDistanceWeight", 255);
  UdmWeightFactorsInit2(prm->wf, Vars, &db->Vars, "wf");
  prm->nwf_num= UdmWeightFactorsInit2(prm->nwf, Vars, &db->Vars, "nwf");
  prm->debug_url_id= UdmVarListFindInt(Vars, "DebugURLID", 0);

  for (i= 0; i < 256; i++)
    prm->wf2[i]= prm->wf[i] << 2;

  prm->ncosine= Res->WWList.nuniq * prm->nsections + 1;
  prm->D_size= prm->ncosine * sizeof(unsigned int);
}


static void
UdmNumDistinctWordFactorInit(unsigned int numdistinctwordfactor,
                             float *numdistinctword_factor, size_t nuniq)
{
  size_t i;
  for (i= 0; i < nuniq; i++)
  {
    unsigned int k= numdistinctwordfactor;
    float x= (float) i / nuniq;
    numdistinctword_factor[i]= ((x * x * x * x * k) + (256-k)) / 256;
  }
}


/*
  Initializing ncoords and search_mode dependent member,
  and members requiring allocation
*/
static int
UdmScoreParamInitStep2(UDM_SCORE_PARAM *score_param,
                       UDM_AGENT *query, UDM_RESULT *Res,
                       size_t ncoords, int search_mode)
{
  size_t wordnum, secno, Rsum;
  UDM_WIDEWORD *Res_WWList_Word= Res->WWList.Word;

  float numwordfactor= ((float)UdmVarListFindDouble(&query->Conf->Vars,
                                                "NumWordFactor", 25.5)) / 255;
  score_param->NumDistinctWordFactor= (unsigned int) UdmVarListFindInt(
                                                      &query->Conf->Vars,
                                                     "NumDistinctWordFactor", 0);
  UdmNumWordFactorInit(numwordfactor, score_param->numword_factor, ncoords);
  UdmNumDistinctWordFactorInit(score_param->NumDistinctWordFactor,
                               score_param->numdistinctword_factor,
                               Res->WWList.nuniq);  
  score_param->count_size= Res->WWList.nuniq;
  score_param->count= (char*)UdmMalloc(score_param->count_size);
  score_param->R= (unsigned int*)UdmMalloc(score_param->D_size);
  score_param->D= (unsigned int*)UdmMalloc(score_param->D_size);
  score_param->Dsum_add= (unsigned int*)UdmMalloc(score_param->D_size);
  score_param->RDsum_add= (unsigned int*)UdmMalloc(score_param->D_size);
  if (!score_param->count || !score_param->R || !score_param->D ||
      !score_param->Dsum_add || !score_param->RDsum_add)
    return UDM_ERROR;

  score_param->Word= Res->WWList.Word;
  score_param->nuniq= Res->WWList.nuniq;
  score_param->nwords= Res->WWList.nwords;

  bzero((void*) score_param->R, score_param->D_size);
  
  for(Rsum=0, secno= 0; secno < score_param->nsections; secno++)
  {
    for (wordnum= 0; wordnum < score_param->nwords; wordnum++)
    {
      size_t offs, order;
      if (Res_WWList_Word[wordnum].origin != UDM_WORD_ORIGIN_QUERY)
        continue;

      order= Res_WWList_Word[wordnum].order;
      offs= OFFS_FROM_NUNIQ_SECNO_ORDER(score_param->nuniq, secno, order);
      score_param->R[offs] = score_param->wf2[secno] + Res_WWList_Word[wordnum].weight;
      Rsum+= score_param->R[offs] * score_param->R[offs];
      {
        size_t add= score_param->wf2[secno] + Res_WWList_Word[order].weight;
        size_t Dsum_add= add*add;
        size_t RDsum_add= add * score_param->R[offs];
        score_param->Dsum_add[offs]= Dsum_add;
        score_param->RDsum_add[offs]= RDsum_add;
      }
    }
  }
  
  Rsum+= score_param->R[score_param->dst_offs] * score_param->R[score_param->dst_offs];
  score_param->Rsum_factor= 100000 / sqrt(Rsum);
  
  if (Res->ItemList.ncmds > 0 || search_mode == UDM_MODE_BOOL)
  {
    if (UDM_OK != UdmStackItemListCopy(&score_param->ItemList,
                                       &Res->ItemList, search_mode))
      return UDM_ERROR;
    search_mode= UDM_MODE_BOOL;
  }
  score_param->search_mode= search_mode;

  return UDM_OK;
}


static void
UdmScoreParamFreeStep2(UDM_SCORE_PARAM *score_param)
{
  UdmStackItemListFree(&score_param->ItemList);
  UDM_FREE(score_param->D);
  UDM_FREE(score_param->R);
  UDM_FREE(score_param->Dsum_add);
  UDM_FREE(score_param->RDsum_add);
  UDM_FREE(score_param->count);
}


static void
UdmGroupByURLInternal2(UDM_AGENT *query,
                       UDM_RESULT *Res,
                       UDM_SECTIONLIST *SectionList,
                       UDM_URLSCORELIST *ScoreList,
                       UDM_SCORE_PARAM *score_param,
                       int search_mode)
{
  size_t i, ncoords;
  for (ncoords=0, i= 0; i < SectionList->nsections; i++)
    ncoords+= SectionList->Section[i].ncoords;

  if(!ncoords) return;

  if (UdmScoreParamInitStep2(score_param, query, Res, ncoords, search_mode))
    goto err;

  if (Res->WWList.nuniq == 1)
  {
    /* Doesn't need coords */
    UdmGroupByURLLoop_OneWord_All2(Res, SectionList, ScoreList, score_param);
  }
  else
  {
    /* Need coords */
    UdmGroupByURLLoop2(query, Res, SectionList, ScoreList, score_param);
  }

err:

  UdmScoreParamFreeStep2(score_param);
  return;
}


void UdmGroupByURL2(UDM_AGENT *query,
                   UDM_DB *db,
                   UDM_RESULT *Res,
                   UDM_SECTIONLIST *SectionList,
                   UDM_URLSCORELIST *ScoreList)
{
  UDM_SCORE_PARAM *prm;
  int search_mode= UdmSearchMode(UdmVarListFindStr(&query->Conf->Vars, "m", "all"));
  size_t threshold= UdmVarListFindInt(&query->Conf->Vars, "StrictModeThreshold", 0);
  size_t ncoords= (search_mode == UDM_MODE_ALL) && threshold ?
                  SectionList->nsections : 0;
  size_t nbytes;

  UdmWideWordListSetOriginWeight(&Res->WWList);
  
  if (!(prm= UdmMalloc(sizeof(UDM_SCORE_PARAM))))
    return;
  
  UdmScoreParamInit(prm, query, db, Res);


  nbytes= SectionList->nsections * sizeof(UDM_URL_SCORE);
  ScoreList->Item= (UDM_URL_SCORE*) UdmMalloc(nbytes);
  UdmGroupByURLInternal2(query, Res, SectionList, ScoreList, prm, search_mode);
  if (ncoords && (ScoreList->nitems < threshold))
  {
    size_t strict_mode_found= ScoreList->nitems;
    UdmLog(query, UDM_LOG_DEBUG,
          "Too few results: %d, Threshold: %d, group in ANY mode", 
          strict_mode_found, threshold);
    UdmGroupByURLInternal2(query, Res, SectionList, ScoreList, prm, UDM_MODE_ANY);
    if (ScoreList->nitems > strict_mode_found)
      UdmVarListReplaceInt(&query->Conf->Vars, "StrictModeFound", strict_mode_found);
  }
 
  UdmFree(prm);
}

/*************** UserScore and UserSiteScore functions ******************/

/*
  TODO:
  reuse the same function from
  urlidlist.c, sql.c, dbmode-blob.c, dbmode-multi, dbmode-rawblob, score.c
*/
static int
cmpaurls (urlid_t *s1, urlid_t *s2)
{
  if (*s1 > *s2) return(1);
  if (*s1 < *s2) return(-1);
  return(0);
}

static void
UdmUserScoreFindMinMax(UDM_URL_INT4_LIST *List, int *minval, int *maxval)
{
  size_t i;
  *minval= *maxval= 0;
  for (i= 0; i < List->nitems; i++)
  {
    UDM_URL_INT4 *Item= &List->Item[i];
    if (*minval > Item->param)
      *minval= Item->param;
    if (*maxval < Item->param)
      *maxval= Item->param;
  }
}


int
UdmUserScoreListApplyToURLScoreList(UDM_URLSCORELIST *List,
                                    UDM_URL_INT4_LIST *UserScoreList,
                                    int UserScoreFactor)
{
  size_t i;
  int4 minval= -1;
  int4 maxval= 1;
  UDM_URL_SCORE *Coords= List->Item;

  UdmUserScoreFindMinMax(UserScoreList, &minval, &maxval);
  
  for (i= 0; i < List->nitems; i++)
  {
    urlid_t url_id= Coords[i].url_id;
    uint4 coord= Coords[i].score;
    UDM_URL_INT4 *found;
    found= (UDM_URL_INT4*) UdmBSearch(&url_id,
                                      UserScoreList->Item,
                                      UserScoreList->nitems,
                                      sizeof(UDM_URL_INT4),
                                      (udm_qsort_cmp)cmpaurls);
    if (found)
    {
      if (found->param >= 0)
        coord= coord + 
              ((int4) (((float) (100000 - coord)) * found->param / maxval)) *
               UserScoreFactor / 255;
      else
        coord= coord -
               ((int4) (((float) coord) * found->param / minval)) *
               UserScoreFactor / 255;
    }
    Coords[i].score= coord;
  }
  return UDM_OK;
}


int
UdmUserScoreListApplyToURLDataList(UDM_URLDATALIST *List,
                                   UDM_URL_INT4_LIST *UserScoreList,
                                   int UserScoreFactor)
{
  size_t i;
  int4 minval= -1;
  int4 maxval= 1;
  UDM_URLDATA *Coords= List->Item;

  UdmUserScoreFindMinMax(UserScoreList, &minval, &maxval);

  for (i= 0; i < List->nitems; i++)
  {
    urlid_t url_id= Coords[i].url_id;
    uint4 score= Coords[i].score;
    UDM_URL_INT4 *found;
    found= (UDM_URL_INT4*) UdmBSearch(&url_id,
                                      UserScoreList->Item,
                                      UserScoreList->nitems,
                                      sizeof(UDM_URL_INT4),
                                      (udm_qsort_cmp)cmpaurls);
    if (found)
    {
      if (found->param >= 0)
        score= score + 
              ((int4) (((float) (100000 - score)) * found->param / maxval)) *
               UserScoreFactor / 255;
      else
        score= score -
               ((int4) (((float) score) * found->param / minval)) *
               UserScoreFactor / 255;
    }
    Coords[i].score= score;
  }
  return UDM_OK;
}


/************** DateFactor and RelevancyFactor **********/

int
UdmURLDataListApplyRelevancyFactors(UDM_AGENT *Agent,
                                    UDM_URLDATALIST *DataList,
                                    int RelevancyFactor,
                                    int DateFactor)
{
  int i, sum;
  time_t current_time;
  unsigned long ticks;
  UdmLog(Agent, UDM_LOG_DEBUG, "Start applying relevancy factors");
  ticks= UdmStartTimer();
  if (!(current_time= UdmVarListFindInt(&Agent->Conf->Vars, "CurrentTime", 0)))
    time(&current_time);
  sum= RelevancyFactor + DateFactor;
  sum= sum ? sum : 1;
  
  for (i= 0; i < DataList->nitems; i++)
  {
    time_t doc_time= DataList->Item[i].last_mod_time;
    uint4 *score= &DataList->Item[i].score;
    float rel= *score * RelevancyFactor;
    float dat= ((doc_time < current_time) ? 
                ((float) doc_time / current_time) :
                ((float) current_time / doc_time)) * DateFactor * 100000;
    /* 100000 = 100% * 1000 = scale in db.c */
    *score= (rel + dat) / sum;
  }
  ticks= UdmStartTimer() - ticks;
  UdmLog(Agent, UDM_LOG_DEBUG, "Stop applying relevancy factors\t\t%.2f",
         (float)ticks / 1000);
  return UDM_OK;
}
