/* Copyright (C) 2000-2009 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_WINSOCK_H
#include <winsock.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_ARPA_NAMESER_H
#include <arpa/nameser.h>
#endif
#ifdef HAVE_RESOLV_H
#include <resolv.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#include <assert.h>


#include "udm_common.h"
#include "udm_utils.h"

#include "udm_db.h"
#include "udm_db_int.h"
#include "udm_sqldbms.h"
#include "udm_url.h"
#include "udm_sdp.h"
#include "udm_vars.h"
#include "udm_mutex.h"
#include "udm_searchtool.h"
#include "udm_result.h"
#include "udm_log.h"
#include "udm_agent.h"
#include "udm_proto.h"
#include "udm_host.h"
#include "udm_hash.h"
#include "udm_doc.h"
#include "udm_services.h"
#include "udm_xmalloc.h"
#include "udm_searchcache.h"
#include "udm_store.h"
#include "udm_match.h"
#include "udm_word.h"
/******************** BlobCache stuff *******************/

UDM_BLOB_CACHE*
UdmBlobCacheInit(UDM_BLOB_CACHE *cache)
{
  if (!cache)
  {
    cache= UdmMalloc(sizeof(UDM_BLOB_CACHE));
    if (!cache)
      return NULL;
    cache->free= 1;
  }
  else
  {
    cache->free= 0;
  }
  cache->errors= 0;
  cache->nwords= 0;
  cache->awords= 0;
  cache->words= NULL;

  return cache;
}


void
UdmBlobCacheFree(UDM_BLOB_CACHE *cache)
{
  size_t i;
  for (i= 0; i < cache->nwords; i++)
  {
    if (cache->words[i].freeme)
      UDM_FREE(cache->words[i].word);
  }

  UDM_FREE(cache->words);
  cache->errors= 0;
  cache->nwords= 0;
  cache->awords= 0;
  cache->words= NULL;

  if (cache->free)
    UdmFree(cache);
}


static inline int
UdmBlobCacheCheckValue(urlid_t url_id, unsigned char secno,
                       const char *word, size_t nintags,
                       const char *intag, size_t intaglen)
{
  if (! url_id)
  {
    fprintf(stderr, "url_id variable empty\n");
    return 1;
  }
  if (! secno)
  {
    fprintf(stderr, "secno variable empty\n");
    return 1;
  }
  if (! word)
  {
    fprintf(stderr, "word variable empty\n");
    return 1;
  }
  if (! nintags)
  {
    fprintf(stderr, "nintags variable empty\n");
    return 1;
  }
  if (! intag)
  {
    fprintf(stderr, "intag variable empty\n");
    return 1;
  }
  return 0;
}


static inline int
UdmBlobCacheRealloc(UDM_BLOB_CACHE *cache)
{
  if (cache->nwords == cache->awords)
  {
    UDM_BLOB_CACHE_WORD *tmp;
    size_t nbytes= (cache->awords + 256) * sizeof(UDM_BLOB_CACHE_WORD);
    tmp= UdmRealloc(cache->words, nbytes);
    if (!tmp)
    {
      cache->errors++;
      if (cache->errors < 10 || (cache->errors % 2048) == 0)
      fprintf(stderr, "BlobCacheRealloc: failed %d times: %d bytes, %d records\n",
              cache->errors, nbytes, (cache->awords + 256));
      return 1;
    }
    cache->words = tmp;
    cache->awords += 256;
  }
  return 0;
}


/*
 Adding with allocating memory for word
*/
size_t
UdmBlobCacheAdd(UDM_BLOB_CACHE *cache, urlid_t url_id,
                unsigned char secno, const char *word,
                size_t nintags, const char *intag, size_t intaglen)
{
  size_t word_len;
  UDM_BLOB_CACHE_WORD *W;
  
  if (UdmBlobCacheCheckValue(url_id, secno, word, nintags, intag, intaglen) ||
      UdmBlobCacheRealloc(cache))
    return 0;
  
  word_len= strlen(word);
  W= &cache->words[cache->nwords];
  W->secno= secno;
  W->url_id= url_id;
  W->nintags= nintags;
  W->ntaglen= intaglen;
  W->word= UdmMalloc(word_len + intaglen + 2);
  W->intags= W->word + word_len + 1;
  memcpy(W->word, word, word_len + 1);
  memcpy(W->intags, intag, intaglen);
  W->intags[intaglen]= '\0';
  W->freeme= 1;

  cache->nwords++;

  return(1);
}


/*
  Adding word without allocation (taking over from the caller function)
*/
size_t
UdmBlobCacheAdd2(UDM_BLOB_CACHE *cache, urlid_t url_id,
                 unsigned char secno, char *word,
                 size_t nintags, char *intag, size_t intaglen)
{
  UDM_BLOB_CACHE_WORD *W;
  
  if (UdmBlobCacheCheckValue(url_id, secno, word, nintags, intag, intaglen) ||
      UdmBlobCacheRealloc(cache))
    return 0;
  
  W= &cache->words[cache->nwords];
  W->secno= secno;
  W->url_id= url_id;
  W->nintags= nintags;
  W->ntaglen= intaglen;
  W->word= word;
  W->intags= intag;
  W->freeme= 0;

  cache->nwords++;

  return(1);
}


static int
bccmpwrd(UDM_BLOB_CACHE_WORD *s1, UDM_BLOB_CACHE_WORD *s2)
{
  int _ = strcmp(s1->word, s2->word);
  if (! _) _ = s1->secno - s2->secno;
  if (! _)
  {
    if (s1->url_id > s2->url_id) _ = 1;
    else if (s1->url_id < s2->url_id) _ = -1;
    else _ = 0;
  }
  return(_);
}


static void
UdmBlobCacheSort(UDM_BLOB_CACHE *cache)
{
  if (cache->nwords)
    UdmSort(cache->words, cache->nwords, sizeof(UDM_BLOB_CACHE_WORD), (udm_qsort_cmp)bccmpwrd);
}


/*******************************************/

int
UdmBlobGetTable(UDM_DB *db)
{
  UDM_SQLRES SQLRes;
  int rc;
  const char *val;

  return(1);

  rc = UdmSQLQuery(db, &SQLRes, "SELECT n FROM bdictsw");
  if (rc != UDM_OK) return(1);

  if (! UdmSQLNumRows(&SQLRes) || ! (val = UdmSQLValue(&SQLRes, 0, 0))) rc = 2;
  else if (*val != '1') rc = 3;
  else rc = 4;

  UdmSQLFree(&SQLRes);
  return(rc);
}


const char*
UdmBlobGetRTable (UDM_DB *db)
{
  if (db->DBType == UDM_DB_MYSQL)
    return "bdict";
  if (UdmBlobGetTable(db) == 3) return("bdict00");
  return("bdict");
}


int
UdmBlobGetWTable(UDM_DB *db, const char **name)
{
  int rc;
  *name= "bdict";
  if (db->DBType == UDM_DB_MYSQL)
  {
    if ((UDM_OK != (rc= UdmSQLDropTableIfExists(db, "bdict_tmp"))) ||
        (UDM_OK != (rc= UdmSQLQuery(db, NULL, "CREATE TABLE bdict_tmp MAX_ROWS=300000000 AVG_ROW_LENGTH=512 SELECT * FROM bdict LIMIT 0"))) ||
        (UDM_OK != (rc= UdmSQLQuery(db, NULL, "ALTER TABLE bdict_tmp ADD KEY (word)"))))
      return rc;
    *name= "bdict_tmp";
  }
  else if (db->DBType == UDM_DB_DB2)
  {
    /* Note, DB2 'CREATE TABLE LIKE' does not copy indexes */
    if (UDM_OK != (rc= UdmSQLDropTableIfExists(db, "bdict_tmp")) ||
        UDM_OK != (rc= UdmSQLQuery(db, NULL, "CREATE TABLE bdict_tmp LIKE bdict")))
      return rc;
    *name= "bdict_tmp";
  }
  if (UdmBlobGetTable(db) == 4)
    *name= "bdict00";
  return UDM_OK;
}


int
UdmBlobSetTable(UDM_DB *db)
{
  char qbuf[64];
  int rc, t, n;

  if (db->DBType == UDM_DB_MYSQL)
  {
    if (UDM_OK == (rc= UdmSQLDropTableIfExists(db, "bdict")))
      rc= UdmSQLQuery(db, NULL, "ALTER TABLE bdict_tmp RENAME bdict");
    return rc;
  }
  else if (db->DBType == UDM_DB_DB2)
  {
    if (UDM_OK != (rc= UdmSQLDropTableIfExists(db, "bdict")) ||
        UDM_OK != (rc= UdmSQLQuery(db, NULL, "RENAME TABLE bdict_tmp TO bdict")) ||
        UDM_OK != (rc= UdmSQLQuery(db, NULL, "CREATE INDEX bdict_word ON bdict (word)")))
      return rc;
    return UDM_OK;
  }
  
  t= UdmBlobGetTable(db);
  if (t == 1) return(UDM_OK);
  else if (t == 4) n = 0;
  else n = 1;

  rc = UdmSQLQuery(db, NULL, "DELETE FROM bdictsw");
  if (rc != UDM_OK) return(UDM_OK);
  udm_snprintf(qbuf, sizeof(qbuf), "INSERT INTO bdictsw VALUES(%d)", n);
  rc = UdmSQLQuery(db, NULL, qbuf);
  if (rc != UDM_OK) return(UDM_OK);
  return(UDM_OK);
}


/*******************************************/

int
UdmRewriteURL(UDM_AGENT *Indexer)
{
#ifdef HAVE_SQL
  size_t i;
  udm_timer_t ticks;

  UdmLog(Indexer,UDM_LOG_ERROR,"Rewriting URL data");
  ticks=UdmStartTimer();

  for (i = 0; i < Indexer->Conf->dbl.nitems; i++)
  {
    UDM_DB *db= &Indexer->Conf->dbl.db[i];
    int rc, use_deflate, tr= (db->flags & UDM_SQL_HAVE_TRANSACT) ? 1 : 0;
    if (!UdmDBIsActive(Indexer, i))
      continue;
    UDM_GETLOCK(Indexer, UDM_LOCK_DB);
    use_deflate= UdmVarListFindBool(&db->Vars, "deflate", 0);
    
    if ((tr && UDM_OK != (rc= UdmSQLBegin(db))) ||
        UDM_OK != (rc= UdmBlobWriteURL(Indexer, db, "bdict", use_deflate)) ||
        (tr && UDM_OK != (rc= UdmSQLCommit(db))))
      return rc;
    UDM_RELEASELOCK(Indexer, UDM_LOCK_DB);
    if (rc != UDM_OK)
    {
      UdmLog(Indexer,UDM_LOG_ERROR,"%s",db->errstr); 
      return rc;
    }
  }

  ticks=UdmStartTimer()-ticks;
  UdmLog(Indexer,UDM_LOG_ERROR,"Converting to blob finished\t%.2f",(float)ticks/1000);
#endif
  return UDM_OK;
}


int
UdmRewriteLimits(UDM_AGENT *Indexer)
{
#ifdef HAVE_SQL
  size_t i;
  udm_timer_t ticks;

  UdmLog(Indexer,UDM_LOG_ERROR,"Rewritting limits");
  ticks=UdmStartTimer();

  for (i = 0; i < Indexer->Conf->dbl.nitems; i++)
  {
    int rc;
    UDM_DB *db = &Indexer->Conf->dbl.db[i];
    if (!UdmDBIsActive(Indexer, i))
      continue;
    UDM_GETLOCK(Indexer, UDM_LOCK_DB);
    rc= UdmBlobWriteLimits(Indexer, db, "bdict",
    UdmVarListFindBool(&db->Vars, "deflate", 0));
    UDM_RELEASELOCK(Indexer, UDM_LOCK_DB);
    if (rc != UDM_OK)
    {
      UdmLog(Indexer,UDM_LOG_ERROR,"%s",db->errstr); 
      return rc;
    }
  }

  ticks=UdmStartTimer()-ticks;
  UdmLog(Indexer,UDM_LOG_ERROR,"Rewritting limits\t%.2f",(float)ticks/1000);
#endif
  return UDM_OK;
}

#define COORD_MAX_LEN 4


/*********** Coord tools *******************/

/*
  Scan one coord.
  Return the number of bytes scanned.
*/
static inline size_t
udm_coord_get(size_t *pwc, const unsigned char *s, const unsigned char *e)
{
  unsigned char c;
  
  if (s >= e)
    return 0;
  
  c= s[0];
  if (c < 0x80) 
  {
    *pwc = c;
    return 1;
  } 
  else if (c < 0xc2) 
    return UDM_ERROR;
  else if (c < 0xe0) 
  {
    if (s+2 > e) /* We need 2 characters */ 
      return 0;
    
    if (!((s[1] ^ 0x80) < 0x40))
      return 0;
    
    *pwc = ((size_t) (c & 0x1f) << 6) | (size_t) (s[1] ^ 0x80);
    return 2;
  } 
  else if (c < 0xf0) 
  {
    if (s+3 > e) /* We need 3 characters */
      return 0;
    
    if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40 && (c >= 0xe1 || s[1] >= 0xa0)))
      return 0;
    
    *pwc = ((size_t) (c & 0x0f) << 12)   | 
           ((size_t) (s[1] ^ 0x80) << 6) | 
            (size_t) (s[2] ^ 0x80);
    
    return 3;
  } 
  else if (c < 0xf8)
  {
    if (s + 4 > e) /* Need 4 bytes */
      return 0;
    if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40 && (s[3] ^ 0x80) < 0x40 && (c >= 0xf1 || s[1] >= 0x90)))
      return 0;
    
    *pwc= ((size_t) (c & 0x07) << 18)    |
          ((size_t) (s[1] ^ 0x80) << 12) |
          ((size_t) (s[2] ^ 0x80) << 6)  |
          (size_t)  (s[3] ^ 0x80);
    return 4;
  }
  return 0;
}


/*
  A faster version, without range checking
  The incoming string must have enough space
  to scan full miltibyte sequence.
*/
static inline size_t
udm_coord_get_quick(size_t *pwc, const unsigned char *s)
{
  unsigned char c;
  
  c= s[0];
  if (c < 0x80) 
  {
    *pwc = c;
    return 1;
  } 
  else if (c < 0xc2) 
    return UDM_ERROR; /* The caller will skip one byte */
  else if (c < 0xe0) 
  {
    size_t s1;
    if (!((s1= (s[1] ^ 0x80)) < 0x40))
      return 0;
    *pwc= (((size_t) (c & 0x1f)) << 6) | s1;
    return 2;
  } 
  else if (c < 0xf0) 
  {
    if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40 && (c >= 0xe1 || s[1] >= 0xa0)))
      return 0;
    
    *pwc = ((size_t) (c & 0x0f) << 12)   | 
           ((size_t) (s[1] ^ 0x80) << 6) | 
            (size_t) (s[2] ^ 0x80);
    
    return 3;
  } 
  else if (c < 0xf8)
  {
    if (!((s[1] ^ 0x80) < 0x40 && (s[2] ^ 0x80) < 0x40 && (s[3] ^ 0x80) < 0x40 && (c >= 0xf1 || s[1] >= 0x90)))
      return 0;
    
    *pwc= ((size_t) (c & 0x07) << 18)    |
          ((size_t) (s[1] ^ 0x80) << 12) |
          ((size_t) (s[2] ^ 0x80) << 6)  |
          (size_t)  (s[3] ^ 0x80);
    return 4;
  }

  return 0;
}


/*
  Skip the given amount of coords, accumulating sum.
*/

static inline const unsigned char *
udm_coord_sum(size_t *sump,
              const unsigned char *s,
              const unsigned char *e,
              size_t n)
{
  size_t sum= 0, nbytes, tmp;
  
  /* String must have at least "n" bytes */
  if (s + (n * 4) > e)
    goto mb;
  
  /* Quickly sum leading 7bit numbers */
  for (; *s < 128 && n; n--)
    sum+= *s++;

  for ( ; n; n--)
  {
    if (!(nbytes= udm_coord_get_quick(&tmp, s)))
      return e;
    s+= nbytes;
    sum+= tmp;
  }
  *sump= sum;
  return s;
  
mb:
  for ( ; n; n--)
  {
    if (!(nbytes= udm_coord_get(&tmp, s, e)))
      return e;
    s+= nbytes;
    sum+= tmp;
  }
  *sump= sum;
  return s;
}


/*
  Skip the given anount of coords in the given string
  s - beginning of the string
  e - end of the string (last byte + 1)
  n - how many characters to skip
  RETURNS
    position after n characters, or end of the string (if shorter)
*/
static inline const unsigned char *
udm_coord_skip(const unsigned char *s, const unsigned char *e, size_t n)
{
#if defined(__i386__) && !defined(_WIN64)
  {
    const unsigned char *e4= e - 4;
    for ( ; n > 4 && s < e4; n-= 4, s+= 4)
    {
      if (*((const uint4*)s) & 0x80808080)
        break;
    }
  }
#endif

  for ( ; n && s < e; n--)
  {
    if (*s++ < 0x80)
    {
      /* Single byte character, nothing to do */
    }
    else
    {
      unsigned char c= s[-1];
      if (c < 0xc2) /* Error */
        return e;
      else if (c < 0xe0) /* Two-byte sequence */
      {
        s++;
      }
      else if (c < 0xf0) /* Three-byte sequence */
      {
        s+= 2;
      }
      else if (c < 0xf8) /* Four-byte sequence */
      {
        s+= 3;
      }
      else
      {
        /* TODO: longer sequences */
        return e;
      }
      if (s > e)
        return e;
    }
  }
  return s;
}


static inline size_t
udm_coord_len(const char *str)
{
  size_t len, nbytes, lintag, crd;
  const char *s, *e;

  if (!str)
    return 0;
  lintag= strlen(str);
  for (len= 0, s= str, e = str + lintag; e > s; s += nbytes, len++)
  {
    if (!(nbytes= udm_coord_get(&crd,
                                (const unsigned char *)s,
                                (const unsigned char *)e)))
      break;
  }
  return len;
}


static inline size_t
udm_coord_put(size_t wc, unsigned char *r, unsigned char *e)
{
  int count;
  
  if (r >= e)
    return 0;
  
  if (wc < 0x80) 
    count= 1;
  else if (wc < 0x800) 
    count= 2;
  else if (wc < 0x10000) 
    count= 3;
  else if (wc < 0x200000)
    count= 4;
  else return 0;
  
  /* 
    e is a character after the string r, not the last character of it.
    Because of it (r+count > e), not (r+count-1 >e )
   */
  if (r + count > e) 
    return 0;
  
  switch (count)
  { 
    /* Fall through all cases */
    case 4: r[3]= (unsigned char) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0x10000;
    case 3: r[2]= (unsigned char) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0x800;
    case 2: r[1]= (unsigned char) (0x80 | (wc & 0x3f)); wc = wc >> 6; wc |= 0xc0;
    case 1: r[0]= (unsigned char) wc;
  }
  return count;
}


static size_t
UdmBlobCachePackOneWordForBdict(UDM_BLOB_CACHE *cache,
                                const size_t w,
                                UDM_DSTR *buf)
{
  UDM_BLOB_CACHE_WORD *word= &cache->words[w];
  size_t w1;
  unsigned char ubuf[COORD_MAX_LEN];
  unsigned char *ubufend= ubuf + sizeof(ubuf);

  for (w1= w; w1 < cache->nwords; w1++)
  {
    UDM_BLOB_CACHE_WORD *word1= &cache->words[w1];
    char utmp[4];
    size_t nbytes;

    if (word->secno != word1->secno ||
        strcmp(word->word, word1->word))
      break;

    udm_put_int4(word1->url_id, utmp);
    if (!(nbytes= udm_coord_put(word1->nintags, ubuf, ubufend)))
      continue;
    if (!UdmDSTRAppend(buf, utmp, sizeof(urlid_t)) ||
        !UdmDSTRAppend(buf, (char *)ubuf, nbytes) ||
        !UdmDSTRAppend(buf, word1->intags, word1->ntaglen))
    {
      fprintf(stderr, "BlobCachePackOneWord: DSTRAppend() failed: "
                      "word='%s' secno=%d len=%d",
                       word1->word, word1->secno, word1->ntaglen);
    }
  }
  return w1;
}


static int
UdmBlobEncodeBdictiRecord(UDM_DB *db,
                          UDM_DOCUMENT *Doc,
                          UDM_WORDLIST *WordList,
                          UDM_DSTR *dbuf, size_t *chunks,
                          size_t save_section_size)
{
  size_t i, j;
  UdmDSTRInit(dbuf, 1024*64);
  for (j= 0, i= 0; i < 32; i++)
  {
    UDM_WORDLIST *WL= &WordList[i];
    UDM_WORD *WLWord= WL->Word;
    int prev_pos= 0;
    unsigned char prev_secno= 0;
    const char *prev_word= "";
    chunks[i]= dbuf->size_data;
    
    for (j= 0; j < WL->nwords; )
    {
      unsigned char buf[COORD_MAX_LEN];
      size_t nbytes;
      UDM_WORD *W= &WLWord[j];
      int pos= W->pos;
      unsigned char secno= W->secno;
      if (strcmp(W->word, prev_word))
      {
        if (*prev_word)
          UdmDSTRAppend(dbuf, "\0\0", 2);
        UdmDSTRAppendSTR(dbuf, W->word);
        prev_secno= 0;
        prev_pos= 0;
        prev_word= W->word;
      }
      if (secno != prev_secno)
      {
        UdmDSTRAppend(dbuf, "", 1);
        UdmDSTRAppend(dbuf, (char*) &secno, 1);
        prev_pos= 0;
        prev_secno= 0;
      }
      nbytes= udm_coord_put((int) pos - prev_pos, buf, buf + sizeof(buf));
      UdmDSTRAppend(dbuf, (char *)buf, nbytes);
      prev_pos= pos;
      prev_secno= secno;
      j++;
      
      /* Detect end of section, and put section length */
      if (save_section_size &&
          (j >= WL->nwords ||
           strcmp(W->word, W[1].word) ||
           W->secno != W[1].secno))
      {
        int seclen= (Doc->CrossWords.wordpos[prev_secno] + 1) - prev_pos;
        nbytes= udm_coord_put(seclen, buf, buf + sizeof(buf));
        UdmDSTRAppend(dbuf, (char*) buf, nbytes);
      }
    }
  }
  chunks[32]= dbuf->size_data;
  return UDM_OK;
}



/* TODO: reuse the same function from urlidlist.c, sql.c, dbmode-blob.c */
static int
cmpaurls (urlid_t *s1, urlid_t *s2)
{
  if (*s1 > *s2) return(1);
  if (*s1 < *s2) return(-1);
  return(0);
}


static int
UdmSectionListBlobCoordsUnpack(UDM_SECTIONLIST *SectionList,
                               UDM_URLID_LIST *urls,
                               UDM_SECTION *SectionTemplate,
                               const unsigned char *s,
                               size_t length,
                               int save_section_size,
                               int need_coords)
{
  size_t ncoords= 0, nurls= urls->nurls;
  const unsigned char *e= s + length;
  const unsigned char *last_urlid_start= e - sizeof(urlid_t) - 1;
  UDM_COORD2 C, *Coord= need_coords ? SectionList->Coord : NULL;
  unsigned char secno= SectionTemplate->secno;
  unsigned char wordnum= SectionTemplate->wordnum;
  unsigned char order= SectionTemplate->order;  
  UDM_SECTION *Section= SectionList->Section;

  /*
    A chunk consists of:
    - sizeof(urlid_t)
    - at least one byte for length
  */
  for ( ; s < last_urlid_start; )
  {
    int active= 1;
    size_t nrecs;

    Section->Coord= Coord;
    Section->secno= secno;
    Section->url_id= (urlid_t)udm_get_int4(s);
    Section->wordnum= wordnum;
    Section->order= order;
    s+= 4;

    if (nurls)
    {
      void *found= UdmBSearch(&Section->url_id, urls->urls, urls->nurls,
                              sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
      if (found && urls->exclude)
        active= 0;
      if (!found && !urls->exclude)
        active= 0;
    }

    /* Get number of coords */
    if (*s < 128)
    {
      nrecs= *s++;
    }
    else
    {
      size_t nbytes= udm_coord_get(&nrecs, s, e);
      if (!nbytes) break;
      s+= nbytes;
    }

    if (!nrecs) /* extra safety */
      break;

    if (!active)
    {
      s= udm_coord_skip(s, e, nrecs);
      continue;
    }

    ncoords+= nrecs;

    if (save_section_size && nrecs > 1)
      ncoords--;
      
    /* Get first coord and put into S->minpos */
    if (*s < 128)
    {
      C.pos= *s++;
    }
    else
    {
      size_t crd;
      size_t nbytes= udm_coord_get(&crd, s, e);
      if (!nbytes) break;
      s+= nbytes;
      C.pos= crd;
    }

    Section->minpos= C.pos;
      
    /*
      If no coords anymore.
      Maybe the "section length" record didn't fit
      into "64K words per section" limit.
      Add section with seclen=0.
      QQ: shouldn't seclen be set to pos instead?
    */
    if (!--nrecs)
    {
      Section->seclen= 0;
      Section->ncoords= 1;
      Section->maxpos= C.pos;
      if (Coord)
        *Coord++= C;
      Section++;
      continue;
    }

    if (!Coord) /* Does not need coords, e.g. one word search */
    {
      size_t crd, nbytes;
      if (save_section_size)
      {
        if (nrecs > 1)
        {
          s= udm_coord_sum(&crd, s, e, nrecs - 1); /* Sum middle coords */
          C.pos+= crd;
          Section->maxpos= C.pos;
        }
        else
          Section->maxpos= C.pos; /* One coord, minpos=maxpos */

        if (!(nbytes= udm_coord_get(&crd, s, e))) /* Get seclen */
          break;
        s+= nbytes;
        C.pos+= crd;
        Section->seclen= C.pos;
        Section->ncoords= nrecs;
      }
      else
      {
        s= udm_coord_sum(&crd, s, e, nrecs);      /* Sum middle coords  */
        C.pos+= crd;
        Section->maxpos= C.pos;
        Section->seclen= 0;
        Section->ncoords= nrecs + 1;
      }
      Section++;
      continue;
    }

    *Coord++= C; /* Add first coord */

    /* Get all other coords */
    if (s + nrecs * COORD_MAX_LEN < e)
    {
      /* Not the last chunk: unpack without range check */
      for ( ; nrecs > 0 ; nrecs--)
      {
        if (*s < 128)
        {
          C.pos+= *s++;
        }
        else
        {
          size_t crd, nbytes;
          nbytes= udm_coord_get_quick(&crd, s);
          if (!nbytes) break;
          s+= nbytes;
          C.pos+= crd;
        }
        *Coord++= C;
      }
    }
    else
    {
      /* Possibly last chunk: unpack with range check */
      for ( ; nrecs > 0 ; nrecs--)
      {
        if (s < e && *s < 128)
        {
          C.pos+= *s++;
        }
        else
        {
          size_t crd, nbytes;
          nbytes= udm_coord_get(&crd, s, e);
          if (!nbytes) break;
          s+= nbytes;
          C.pos+= crd;
        }
        *Coord++= C;
      }
    }

    /* Set section length */
    nrecs= Coord - Section->Coord;
    if (save_section_size)
    {
      /*
        We need to check whether Coord > Coord0 in the above
        condition: URL could be skipped because of limit
      */
      Section->seclen= ((--Coord)->pos);
      Section->ncoords= nrecs - 1;
      Section->maxpos= Coord[-1].pos;;
    }
    else
    {
      Section->seclen= 0;
      Section->ncoords= nrecs;
      Section->maxpos= C.pos;
    }
    Section++;
  }

  SectionList->ncoords= ncoords;
  SectionList->nsections= Section - SectionList->Section;

  return UDM_OK;
}


static int
UdmStoreWordBlobPg(UDM_DB *db, urlid_t url_id,
                   UDM_DSTR *dbuf, size_t *chunks)
{
  int rc= UDM_OK;
  size_t i;
  UDM_DSTR qbuf;
  UdmDSTRInit(&qbuf, dbuf->size_data * 5 + 256);
  UdmDSTRAppendf(&qbuf, "INSERT INTO bdicti VALUES(%d,1", url_id);
  for (i= 0; i < 32; i++)
  {
    size_t srclen= chunks[i + 1] - chunks[i];

    /* Add 'E' prefix for modern Pg versions */
    if (db->version >= 80101)
      UdmDSTRAppend(&qbuf, ",E'", 3);
    else
      UdmDSTRAppend(&qbuf, ",'", 2);

    if (srclen)
    {
      size_t slen;
      slen= UdmSQLBinEscStr(db, qbuf.data + qbuf.size_data, dbuf->data + chunks[i], srclen);
      qbuf.size_data+= slen;
    }
    UdmDSTRAppend(&qbuf, "'", 1);
  }
  UdmDSTRAppend(&qbuf, ")", 1);
  rc= UdmSQLQuery(db, NULL, qbuf.data);
  UdmDSTRFree(&qbuf);
  return rc;
}


static int
UdmStoreWordBlobUsingHex(UDM_DB *db, urlid_t url_id,
                         UDM_DSTR *dbuf, size_t *chunks)
{
  size_t i;
  int rc;
  UDM_DSTR qbuf;
  const char *prefix= ",0x"; /* 0xAABBCC syntax by default */
  const char *suffix= "";
  size_t prefix_length= 3;
  size_t suffix_length= 0;

  if (db->flags & UDM_SQL_HAVE_STDHEX) /* X'AABBCC' syntax */
  {
    prefix= ",X'";
    suffix= "'";
    suffix_length= 1;
  }
  
  UdmDSTRInit(&qbuf, dbuf->size_data * 2 + 256);
  UdmDSTRAppendf(&qbuf, "INSERT INTO bdicti VALUES(%d,1", url_id);
  for (i= 0; i < 32; i++)
  {
    size_t length= chunks[i + 1] - chunks[i];
    char *chunk= dbuf->data + chunks[i];
    if (length)
    {
      UdmDSTRAppend(&qbuf, prefix, prefix_length);
      UdmDSTRAppendHex(&qbuf, chunk, length);
      if (suffix_length)
        UdmDSTRAppend(&qbuf, suffix, suffix_length);
    }
    else
      UdmDSTRAppend(&qbuf, ",''", 3);
  }
  UdmDSTRAppend(&qbuf, ")", 1);
  rc= UdmSQLQuery(db, NULL, qbuf.data);
  UdmDSTRFree(&qbuf);
  return rc;
}


static int
UdmStoreWordBlobUsingBind(UDM_DB *db, urlid_t url_id,
                          UDM_DSTR *dbuf, size_t *chunks)
{
  int rc= UDM_OK;
  size_t i;
  char qbuf[512];

  udm_snprintf(qbuf, sizeof(qbuf),
               "INSERT INTO bdicti VALUES(%d,1,"
               "%s,%s,%s,%s,%s,%s,%s,%s,"
               "%s,%s,%s,%s,%s,%s,%s,%s,"
               "%s,%s,%s,%s,%s,%s,%s,%s,"
               "%s,%s,%s,%s,%s,%s,%s,%s)",
               url_id,
               UdmSQLParamPlaceHolder(db, 1),
               UdmSQLParamPlaceHolder(db, 2),
               UdmSQLParamPlaceHolder(db, 3),
               UdmSQLParamPlaceHolder(db, 4),
               UdmSQLParamPlaceHolder(db, 5),
               UdmSQLParamPlaceHolder(db, 6),
               UdmSQLParamPlaceHolder(db, 7),
               UdmSQLParamPlaceHolder(db, 8),
               UdmSQLParamPlaceHolder(db, 9),
               UdmSQLParamPlaceHolder(db, 10),
               UdmSQLParamPlaceHolder(db, 11),
               UdmSQLParamPlaceHolder(db, 12),
               UdmSQLParamPlaceHolder(db, 13),
               UdmSQLParamPlaceHolder(db, 14),
               UdmSQLParamPlaceHolder(db, 15),
               UdmSQLParamPlaceHolder(db, 16),
               UdmSQLParamPlaceHolder(db, 17),
               UdmSQLParamPlaceHolder(db, 18),
               UdmSQLParamPlaceHolder(db, 19),
               UdmSQLParamPlaceHolder(db, 20),
               UdmSQLParamPlaceHolder(db, 21),
               UdmSQLParamPlaceHolder(db, 22),
               UdmSQLParamPlaceHolder(db, 23),
               UdmSQLParamPlaceHolder(db, 24),
               UdmSQLParamPlaceHolder(db, 25),
               UdmSQLParamPlaceHolder(db, 26),
               UdmSQLParamPlaceHolder(db, 27),
               UdmSQLParamPlaceHolder(db, 28),
               UdmSQLParamPlaceHolder(db, 29),
               UdmSQLParamPlaceHolder(db, 30),
               UdmSQLParamPlaceHolder(db, 31),
               UdmSQLParamPlaceHolder(db, 32));
  
  if (UDM_OK != (rc= UdmSQLPrepare(db, qbuf)))
    goto ret;

  for (i= 0; i < 32; i++)
  {
    int length= chunks[i + 1] - chunks[i];
    char *data= dbuf->data + chunks[i];
    if (length == 0 &&
        db->DBDriver == UDM_DB_ODBC &&
        db->DBType == UDM_DB_ORACLE8)
      length= UDM_SQL_NULL_DATA;
    if (UDM_OK != (rc= UdmSQLBindParameter(db, i + 1, data, length, UDM_SQLTYPE_LONGVARBINARY)))
      goto ret;
  }
  if(UDM_OK != (rc= UdmSQLExecute(db)) ||
     UDM_OK != (rc= UdmSQLStmtFree(db)))
    goto ret;

ret:
  return rc;
}


static int
swbcmp(UDM_WORD *w1, UDM_WORD *w2)
{
  register int _;
  /*
    We don't need to compare W->hash, because
    the words already distributed into different 32 lists,
    according to hash
    if ((_= (int) w1->hash - (int) w2->hash))
      return _;
  */
  if ((_= strcmp(w1->word, w2->word)))
    return _;
  if ((_= (int) w1->secno - (int) w2->secno))
    return _;
  return (int) w1->pos - (int) w2->pos;
}


/*   
   bdicti table has `state` field, that signals bdicti record state.
   There are three possible states:
   0 - deleted document (record is in bdicti and bdict)
   1 - new document (record is in bdicti only)
   2 - converted document (record is in bdicti and bdict)

   New document is added to a collection
   -------------------------------------
   New record is added to bdicti table with `state` set to 1.

   Convert to bdict (full)
   -----------------------
   Read records from bdicti with `state` > 0;
   Convert these records and write converted records to bdict;
   Set `state` to 2 for records with `state` = 1;
   Delete from bdicti records with `state` = 0;
   (afair no parallel bdicti updates assumed)

   Convert to bdict (partial)
   --------------------------
   Read records from bdicti with `state` set to either 0 (deleted) or 1 (new);
   Convert these records and merge converted records with those that are in bdict;
   Set `state` to 2 for records with `state` = 1;
   Delete from bdicti records with `state` = 0;
   (same as above - no parallel updates)

   Document is removed from collection
   -----------------------------------
   Either delete from bdicti document record with `state` = 1,
   Or set `state` to 0 if record `state` is 2.

   All documents are removed (truncation)
   --------------------------------------
   This is simple - truncate both bdict and bdicti tables.

   Document is updated
   -------------------
   See:
     Document is removed from collection;
     Document is added to a collection.

   Searching bdicti
   ----------------
   Only records with `state` 1 (new) or 2 (converted) matter.

   Searching bdicti + bdict
   ------------------------
   Only records with `state` 0 (deleted) or 1 (new) matter.
 */


int
UdmStoreWordsBlob(UDM_AGENT *Indexer, UDM_DOCUMENT *Doc, UDM_DB *db)
{
  size_t i, nwords, chunks[33];
  urlid_t url_id= UdmVarListFindInt(&Doc->Sections, "ID", 0);
  UDM_DSTR dbuf;
  int rc= UDM_OK;
  int save_section_size= UdmVarListFindInt(&Indexer->Conf->Vars, "SaveSectionSize", 1);
  UDM_WORDLIST WordList[32];

  if (UdmVarListFindInt(&Doc->Sections, "PrevStatus", 0))
    if (UDM_OK != UdmDeleteWordsFromURLBlob(Indexer, db, url_id))
      return UDM_ERROR;
  /*
   * Set hash to UdmStrHash32(word) & 31 for further sorting.
   * If this is stopword, set hash to 0xff (will be skipped while
   * inserting).
   */
  if (!(nwords= Doc->Words.nwords))
    return UDM_OK;
  bzero((void*) &WordList, sizeof(WordList));

  /* Count number of words in each WordList */
  for (i= 0; i < nwords; i++)
  {
    UDM_WORD *W= &Doc->Words.Word[i];
    W->hash= W->secno ? (UdmStrHash32(W->word) & 0x1f) : 0xff;
    if (W->hash < 32)
      WordList[W->hash].nwords++;
  }
  
  /* Allocate memory for each WordList */
  for (i= 0; i < 32; i++)
  {
    if (WordList[i].nwords)
    {
      WordList[i].Word= (UDM_WORD*) UdmMalloc(WordList[i].nwords * sizeof(UDM_WORD));
      WordList[i].nwords= 0;
    }
  }
  
  /* Copy distribute words into 32 lists */
  for (i= 0; i < nwords; i++)
  {
    UDM_WORD *W= &Doc->Words.Word[i];
    if (W->hash < 32)
    {
      UDM_WORDLIST *WL= &WordList[W->hash];
      assert(WL->Word != NULL);
      WL->Word[WL->nwords++]= *W;
    }
  }
  
  /*
    Sort 32 small lists,
    this is faster than sorting a single big word list.
  */
  for (i= 0; i < 32; i++)
  {
    UDM_WORDLIST *WL= &WordList[i];
    if (WL->nwords)
    {
      UdmSort(WL->Word, WL->nwords, sizeof(UDM_WORD), (udm_qsort_cmp) swbcmp);
    }
  }


  if (UDM_OK != (rc= UdmBlobEncodeBdictiRecord(db, Doc, WordList, &dbuf,
                                               chunks, save_section_size)))
    goto ret;


  if (db->flags & UDM_SQL_HAVE_BIND)
  {
    rc= UdmStoreWordBlobUsingBind(db, url_id, &dbuf, chunks);
  }
  else if (db->flags & UDM_SQL_HAVE_0xHEX ||
      db->flags & UDM_SQL_HAVE_STDHEX) /* For SQLite3 */
  {  
    rc= UdmStoreWordBlobUsingHex(db, url_id, &dbuf, chunks);
  }
  else if (db->DBType == UDM_DB_PGSQL)
  {
    rc= UdmStoreWordBlobPg(db, url_id, &dbuf, chunks);
  }


ret:
  /*
    Just free Word pointers
    Don't need UdmWordListFree here,
    because it also frees Word[x].word.
  */
  for (i= 0; i < 32; i++)
    UDM_FREE(WordList[i].Word);
  UdmDSTRFree(&dbuf);
  return rc;
}


int
UdmDeleteWordsFromURLBlob(UDM_AGENT *Indexer, UDM_DB *db, urlid_t url_id)
{
  char buf[64];
  /*
   * Remove document if it is not in searching table (state = 1).
   * Mark document for deletion if it is in searching table
   * (state = 2).
   */
  udm_snprintf(buf, sizeof(buf),
               "DELETE FROM bdicti WHERE state=1 AND url_id=%d", url_id);
  if (UDM_OK != UdmSQLQuery(db, NULL, buf))
    return UDM_ERROR;
  udm_snprintf(buf, sizeof(buf),
               "UPDATE bdicti SET state=0 WHERE state=2 AND url_id=%d", url_id);
  if (UDM_OK != UdmSQLQuery(db, NULL, buf))
    return UDM_ERROR;
  return UDM_OK;
}


static const char*
UdmBlobModeInflateOrSelf(UDM_AGENT *A,
                         UDM_DSTR *buf, const char *name,
                         const char *src, size_t *len)
{
  int use_zint4;
  int use_deflate;
  /* fprintf(stderr, "here, len=%d src=%p\n", *len, src); */
  if (!src || *len < 8 ||
      (unsigned char)src[0] != 0xFF ||
      (unsigned char)src[1] != 0xFF ||
      (unsigned char)src[2] != 0xFF ||
      (unsigned char)src[3] != 0xFF ||
      (src[4] != 1 && src[4] != 2 && src[4] != 3) ||
      src[5] != 0x00 ||
      src[6] != 0x00 ||
      src[7] != 0x00)
    return src;
  use_zint4= src[4] == 2 || src[4] == 3;
  use_deflate= src[4] == 1 || src[4] == 3;
  src+= 8;
  *len-= 8;
  if (name)
    UdmLog(A, UDM_LOG_DEBUG, "Unpacking '%s'", name);
  if (use_deflate)
  {
    udm_timer_t ticks= UdmStartTimer();
    size_t i, mul[4]= {10, 100, 1000, 10000};
    size_t len0= *len;
    UdmLog(A,UDM_LOG_DEBUG, "Deflate header detected");
#ifdef HAVE_ZLIB
    for (i= 0; i < 4; i++)
    {
      size_t reslen, nbytes= len[0] * mul[i];
      /* fprintf(stderr, "allocating %d bytes\n", nbytes); */
      UdmDSTRRealloc(buf, nbytes);
      reslen= UdmInflate(buf->data, buf->size_total, src, *len);
      /* fprintf(stderr, "reslen=%d site_total=%d\n", reslen, buf->size_total);*/
      if (reslen < buf->size_total)
      {
        src= buf->data;
        len[0]= reslen;
        UdmLog(A,UDM_LOG_DEBUG, "%d to %d bytes inflated", len0, reslen);
        break;
      }
    }
    ticks= UdmStartTimer() - ticks;
    UdmLog(A, UDM_LOG_DEBUG, "Inflating done: %.2f", (float) ticks/1000);
#endif
  }
  if (*len >= 5 && use_zint4)
  {
    udm_timer_t ticks= UdmStartTimer();
    char *zint4_buf= UdmMalloc(*len);
    UdmLog(A, UDM_LOG_DEBUG, "zint4 header detected (zint4ed data length: %d)", *len);
    if (! zint4_buf)
    {
      UdmLog(A, UDM_LOG_ERROR, "Malloc failed. Requested %u bytes", *len);
      return(NULL);
    }
    memcpy(zint4_buf, src, *len);
    if (buf->size_total < *len * 7 && UdmDSTRRealloc(buf, *len * 7) != UDM_OK)
    {
      UdmFree(zint4_buf);
      UdmLog(A, UDM_LOG_ERROR, "UdmDSTRRealloc failed. Requested %u bytes",
             *len * 7);
      return(NULL);
    }
    *len= udm_dezint4(zint4_buf, (int4 *)buf->data, *len) * 4;
    src= buf->data;
    UdmFree(zint4_buf);
    ticks= UdmStartTimer() - ticks;
    UdmLog(A, UDM_LOG_ERROR, "dezint4ed data length: %d", *len);
    UdmLog(A, UDM_LOG_ERROR, "dezint4 done: %.2f", (float) ticks/1000);
  }
  return src;
}


const char *
UdmBlobModeInflateOrAlloc(UDM_AGENT *A,
                          UDM_DSTR *buf, const char *name,
                          const char *src, size_t *len)
{
  const char *res= UdmBlobModeInflateOrSelf(A, buf, name, src, len);
  if (res == src)
  {
    UdmDSTRRealloc(buf, *len);
    memcpy(buf->data, src, *len);
    buf->size_data= *len;
    res= buf->data;
  }
  return res;
}


static int
UdmInflateBlobModeSQLRes(UDM_AGENT *A, UDM_SQLRES *src)
{
  UDM_DSTR ibuf;
  size_t row;
  UdmDSTRInit(&ibuf, 1024);
  for (row= 0; row < src->nRows; row++)
  {
    size_t len= UdmSQLLen(src, row, 1);
    const char *val= UdmSQLValue(src, row, 1);
    const char *iflt;
    iflt= UdmBlobModeInflateOrSelf(A, &ibuf, NULL, val, &len);
    if (iflt != val)
    {
      size_t offs= src->nCols*row + 1;
      UdmFree(src->Items[offs].val);
      src->Items[offs].val= UdmMalloc(len + 1);
      memcpy(src->Items[offs].val, iflt, len);
      src->Items[offs].len= len;
      src->Items[offs].val[len]= '\0';
    }
  }
  UdmDSTRFree(&ibuf);
  return UDM_OK;
}


static int
UdmAddCollationMatch(UDM_FINDWORD_ARGS *args, const char *word, size_t count)
{
  /*
    If match is not full, then we don't know whether
    the word is a substring or a collation match.
    Let's assume it is a substring, to avoid long 
    word lists in $(WE).
  */
  if (args->Word.match == UDM_MATCH_FULL)
  {
    UDM_WIDEWORD WW= args->WWList->Word[args->Word.order];
    WW.origin= UDM_WORD_ORIGIN_COLLATION;
    WW.count= count;
    UdmWideWordListAddLike(&args->CollationMatches, &WW, word);
  }
  return UDM_OK;
}


static int
UdmBlobAddCoords2(UDM_FINDWORD_ARGS *args,
                  UDM_SQLRES *SQLRes)
{
  size_t numrows= UdmSQLNumRows(SQLRes);
  size_t i;
  char *wf= args->wf;
  UDM_URLID_LIST *urls= &args->urls;
  int need_coords= (args->WWList->nwords > 1);
  int save_section_size= args->save_section_size;
  UDM_SECTION Section;
  
  bzero((void*) &Section, sizeof(Section));
  Section.wordnum= args->Word.order & 0xFF;
  Section.order= args->WWList->Word[Section.wordnum].order;

  for (i= 0; i < numrows; i++)
  {
    const unsigned char *s= (const unsigned char *)UdmSQLValue(SQLRes, i, 1);
    size_t length= UdmSQLLen(SQLRes, i, 1);
    unsigned char secno= UDM_ATOI(UdmSQLValue(SQLRes, i, 0));
    const char *cmatch= UdmSQLValue(SQLRes, i, 2);
    UDM_SECTIONLIST SectionList;

    if (!wf[secno])
      continue;

    Section.secno= secno;
        
    /*
      Shortest section is 6 bytes:
      - 4 bytes for URL id
      - 1 byte for "ncoords"
      - 1 byte for coord
    */

    UdmSectionListAlloc(&SectionList, length, length / 6);

    UdmSectionListBlobCoordsUnpack(&SectionList,
                                   urls, &Section,
                                   s, length,
                                   save_section_size,
                                   need_coords);

    if (SectionList.nsections && SectionList.ncoords)
    {
      UdmSectionListListAdd(&args->SectionListList, &SectionList);
      args->Word.count+= SectionList.ncoords;
      UdmAddCollationMatch(args, cmatch, SectionList.ncoords);
    }
    else
    {
      UdmSectionListFree(&SectionList);
    }
  }

  return UDM_OK;
}


int
UdmFindWordBlob(UDM_FINDWORD_ARGS *args)
{
  char qbuf[4096];
  char secno[32]= "";
  char special[32]= "";
  udm_timer_t ticks;
  UDM_SQLRES SQLRes;
  int rc;

  if (args->urls.empty)
  {
    UdmLog(args->Agent, UDM_LOG_DEBUG, "Not searching 'bdict': Base URL limit is empty");
    return UDM_OK;
  }
  
  ticks= UdmStartTimer();
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Start fetching");
  if (args->Word.secno)
    udm_snprintf(secno, sizeof(secno), " AND secno=%d", args->Word.secno);
  /*
    When performing substring or number search,
    don't include special data, like '#last_mod_time' or '#rec_id'
  */
  if (args->cmparg[0] != '=')
    udm_snprintf(special, sizeof(special), " AND word NOT LIKE '#%%'");
  udm_snprintf(qbuf, sizeof(qbuf), "SELECT secno,intag,word FROM bdict WHERE word%s%s%s", args->cmparg, secno, special);
  if(UDM_OK != (rc= UdmSQLQuery(args->db, &SQLRes, qbuf)))
    return rc;
  ticks= UdmStartTimer() - ticks;
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Stop fetching\t%.2f", (float) ticks / 1000);

  ticks= UdmStartTimer();
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Start UdmBlobAddCoords");
  UdmInflateBlobModeSQLRes(args->Agent, &SQLRes);
  UdmBlobAddCoords2(args, &SQLRes);
  ticks= UdmStartTimer()-ticks;
  UdmLog(args->Agent, UDM_LOG_DEBUG, "Stop UdmBlobAddCoords\t%.2f", (float)ticks / 1000);
  UdmSQLFree(&SQLRes);
  return(UDM_OK);
}


int
UdmFindWordBlobLiveUpdates(UDM_FINDWORD_ARGS *args)
{
  int rc;
  if ((!UDM_OK == (rc= UdmFindWordBlob(args))) ||
      (!UDM_OK == (rc= UdmFindWordRawBlob(args, UDM_RAWBLOB_DELTA))))
    goto ret;

ret:
  return rc;
}


int
UdmBlobReadTimestamp(UDM_AGENT *A, UDM_DB *db, int *ts, int def)
{
  int rc;
  char lname[]= "#ts";
  char qbuf[64];
  UDM_SQLRES SQLRes;

  udm_snprintf(qbuf, sizeof(qbuf), "SELECT intag FROM bdict WHERE word='%s'", lname);
  if (UDM_OK == (rc= UdmSQLQuery(db, &SQLRes, qbuf)) &&
      UdmSQLNumRows(&SQLRes) > 0)
    *ts= atoi(UdmSQLValue(&SQLRes, 0,0));
  else
    *ts= def;
  UdmSQLFree(&SQLRes);
  return rc;
}


static int
UdmBlobWriteWordPrepare(UDM_DB *db, const char *table)
{
  int rc;
  char qbuf[128];
  const char *int_cast= db->DBType == UDM_DB_PGSQL ? "::integer" : "";
  udm_snprintf(qbuf, sizeof(qbuf),
               "INSERT INTO %s (word, secno, intag) "
               "VALUES(%s, %s%s, %s)",
               table, 
               UdmSQLParamPlaceHolder(db, 1),
               UdmSQLParamPlaceHolder(db, 2),
               int_cast,
               UdmSQLParamPlaceHolder(db, 3));
  rc= UdmSQLPrepare(db, qbuf);
  return rc;
}


static int
UdmBlobWriteWord(UDM_DB *db, const char *table,
                 const char *word, size_t secno,
                 char *data, size_t len, UDM_DSTR *buf)
{
  int rc;
  int use_bind= db->flags & UDM_SQL_HAVE_BIND;
  
  if (use_bind)
  {
    if ((table && UDM_OK != (rc= UdmBlobWriteWordPrepare(db, table))) ||
        UDM_OK != (rc= UdmSQLBindParameter(db, 1, word, (int) strlen(word), UDM_SQLTYPE_VARCHAR)) ||
        UDM_OK != (rc= UdmSQLBindParameter(db, 2, &secno, (int) sizeof(secno), UDM_SQLTYPE_INT)) ||
        UDM_OK != (rc= UdmSQLBindParameter(db, 3, data, (int) len, UDM_SQLTYPE_LONGVARBINARY) ) ||
        UDM_OK != (rc= UdmSQLExecute(db)) ||
        (table && UDM_OK != (rc= UdmSQLStmtFree(db))))
      return rc;
  }
  else
  {
    size_t escape_factor= db->DBType == UDM_DB_PGSQL ? 5 : 2;
    const char *pf= db->DBType == UDM_DB_PGSQL ? "'" : "0x";
    const char *sf= db->DBType == UDM_DB_PGSQL ? "'" : "";
    const char *E= (db->DBDriver == UDM_DB_PGSQL && db->version >= 80101) ? "E" : "";
    char *s;
    size_t nbytes= 100 + len * escape_factor + 1;
    
    if (db->flags & UDM_SQL_HAVE_STDHEX) /* X'AABBCC' syntax */
    {
      pf= "X'";
      sf= "'";
    }

    UdmDSTRReset(buf);

    if (UdmDSTRAlloc(buf, nbytes))
    {
      fprintf(stderr, "BlobWriteWord: DSTRAlloc(%d) failed: word='%s' secno=%d len=%d",
              nbytes, word, secno, len);
      return UDM_OK; /* Skip this word - try to continue */
    }
    UdmDSTRAppendf(buf, "INSERT INTO %s VALUES('%s', %d, %s%s",
                   table, word, secno, E, pf);
    s= buf->data + buf->size_data;
    if (db->DBType == UDM_DB_PGSQL)
    {
      buf->size_data+= UdmSQLBinEscStr(db, s, (const char *) data, len);
    }
    else
    {
      buf->size_data+= UdmHexEncode(s, data, len);
    }
    UdmDSTRAppendf(buf, "%s)", sf);
    if (UDM_OK != (rc= UdmSQLQuery(db, NULL, buf->data)))
      return rc;
  }
  return UDM_OK;
}


static int
UdmBlobWriteWordCmpr(UDM_DB *db, const char *table,
                     const char *word, size_t secno,
                     char *data, size_t len,
                     UDM_DSTR *buf, UDM_DSTR *z,
                     int use_zint4)
{
#ifdef HAVE_ZLIB
  if (z && len > 256)
  {
    UdmDSTRReset(z);
    UdmDSTRRealloc(z, len + 8 + 1); /* 8 for two INTs */
    /* Append Format version */
    UdmDSTRAppendINT4(z, 0xFFFFFFFF);
    if (use_zint4)
    {
      UdmDSTRAppendINT4(z, 0x00000003);
      z->size_data+= UdmDeflate(z->data + z->size_data,
                                z->size_total - z->size_data, data + 8, len - 8);
    }
    else
    {
      UdmDSTRAppendINT4(z, 0x00000001);
      z->size_data+= UdmDeflate(z->data + z->size_data,
                                z->size_total - z->size_data, data, len);
    }
    if (z->size_data < len)
    {
      data= z->data;
      len= z->size_data;
    }
  }
#endif
  return UdmBlobWriteWord(db, table, word, secno, data, len, buf);
}


int
UdmBlobWriteTimestamp(UDM_AGENT *A, UDM_DB *db, const char *table)
{
  UDM_DSTR buf;
  int rc= UDM_OK;
  size_t size_data;
  char lname[64]= "#ts";
  char vname[64]= "#version";
  char data[64];
  char qbuf[64];

  UdmLog(A, UDM_LOG_DEBUG, "Writting '%s'", lname);
  UdmDSTRInit(&buf, 128);
  size_data= udm_snprintf(data, sizeof(data), "%d", (int) time(0));
  udm_snprintf(qbuf, sizeof(qbuf),
              "DELETE FROM %s WHERE word='%s'", table, lname);
  if (UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf)) ||
      UDM_OK != (rc= UdmBlobWriteWord(db, table, lname, 0,
                                       data, size_data, &buf)))
    goto ex;

  size_data= udm_snprintf(data, sizeof(data), "%d", UDM_VERSION_ID);
  udm_snprintf(qbuf, sizeof(qbuf),
               "DELETE FROM %s WHERE word='%s'", table, vname);
  if (UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf)) ||
      UDM_OK != (rc= UdmBlobWriteWord(db, table, vname, 0,
                                      data, size_data, &buf)))
    goto ex;
ex:
  UdmDSTRFree(&buf);
  return rc;
}


static int
UdmBlob2BlobConvertOneColumn(UDM_AGENT *A, UDM_DB *db, 
                             UDM_URLDATALIST *URLList,
                             UDM_SQLRES *SQLRes, size_t intag_column_number,
                             UDM_BLOB_CACHE *cache, udm_uint8 *nbytes)
{
  size_t rownum, nrows= UdmSQLNumRows(SQLRes);
  UDM_PSTR row[2];

  for (rownum= 0; rownum < nrows; rownum ++)
  {
    urlid_t url_id;
    size_t pos= 0;
    row[0].val= UdmSQLValue(SQLRes, rownum, 0);
    row[0].len= UdmSQLLen(SQLRes, rownum, 0);
    row[1].val= UdmSQLValue(SQLRes, rownum, intag_column_number);
    row[1].len= UdmSQLLen(SQLRes, rownum, intag_column_number);
    url_id= UDM_ATOI(row[0].val);
    
    if (!UdmURLDataListSearch(URLList, url_id))
      continue;
    
    while (pos < row[1].len)
    {
      char *word= &row[1].val[pos];
      udmhash32_t word_seed;
      while (pos < row[1].len && row[1].val[pos])
        pos++;
      if (++pos >= row[1].len)
        break;
      word_seed= UdmStrHash32(word) >> 8 & BLOB_CACHE_SIZE;
      while (pos < row[1].len)
      {
        unsigned char secno= (unsigned char)row[1].val[pos];
        char *intag= &row[1].val[++pos];
        size_t nintags, intaglen;
        while (pos < row[1].len && row[1].val[pos])
          pos++;
        nintags= udm_coord_len(intag);
        intaglen= row[1].val + pos - intag;
        (*nbytes)+= intaglen;
        UdmBlobCacheAdd2(&cache[word_seed],
                         url_id, secno, word, nintags, intag, intaglen);
        if (++pos >= row[1].len || ! row[1].val[pos])
        {
          pos++;
          break;
        }
      }
    }
  }
  return UDM_OK;
}




int
UdmBlobCacheWrite(UDM_DB *db,
                  UDM_BLOB_CACHE *cache,
                  const char *table, int use_deflate)
{
  size_t w;
  int rc= UDM_OK;
  UDM_DSTR buf, qbuf, zbuf;
  /* Whether to use a single Prepare for multiple Execute
    Ibase doesn't work for some reasons.
    Requires implicit transaction ID.
    Perhaps a bug in sql-ibase.c
  */
  int use_bind= (db->flags & UDM_SQL_HAVE_BIND) &&
                 (db->DBType != UDM_DB_IBASE);

  if (!cache->nwords)
    return(UDM_OK);

  UdmBlobCacheSort(cache);
  
  UdmDSTRInit(&buf, 8192);
  UdmDSTRInit(&qbuf, 8192);
  UdmDSTRInit(&zbuf, 8192);
  
  if (use_bind && UDM_OK != (rc= UdmBlobWriteWordPrepare(db, table)))
    goto end;
  
  for (w= 0; w < cache->nwords; w++)
  {
    UDM_BLOB_CACHE_WORD *word= &cache->words[w];
    size_t w1;
    
    w1= UdmBlobCachePackOneWordForBdict(cache, w, &buf);

    w= w1 - 1;
   
    if (UDM_OK != (rc= UdmBlobWriteWordCmpr(db,
                                            use_bind ? NULL : table,
                                            word->word, word->secno,
                                            buf.data, buf.size_data, &qbuf,
                                            use_deflate ? &zbuf : NULL, 0)))
      goto end;

    UdmDSTRReset(&buf); 
  }

  if (use_bind && UDM_OK != (rc= UdmSQLStmtFree(db)))
    goto end;


end:
  UdmDSTRFree(&zbuf);
  UdmDSTRFree(&qbuf);
  UdmDSTRFree(&buf);
  return rc;
}


static int
UdmBlob2BlobWriteCache(UDM_AGENT *Indexer, UDM_DB *db,
                       UDM_BLOB_CACHE *cache,
                       const char *wtable, int use_deflate, size_t *srows)
{
  size_t trows, i;
  int tr= (db->flags & UDM_SQL_HAVE_TRANSACT) ? 1 : 0;
  int rc= UDM_OK;

  if ((tr && UDM_OK != (rc= UdmSQLBegin(db))))
    goto ret;
  for (trows= 0, i= 0; i <= BLOB_CACHE_SIZE; i++)
  {
    if (cache[i].nwords)
    {
      (*srows)+= cache[i].nwords;
      trows+= cache[i].nwords;
      if (UDM_OK != (rc= UdmBlobCacheWrite(db, &cache[i], wtable, use_deflate)))
         goto ret;
    }
    UdmBlobCacheFree(&cache[i]);
    if (tr && trows > 16*1024)
    {
      if (UDM_OK != (rc= UdmSQLCommit(db)) ||
          UDM_OK != (rc= UdmSQLBegin(db)))
        goto ret;
      trows= 0;
    }
  }
  if (tr && UDM_OK != (rc= UdmSQLCommit(db)))
    goto ret;

ret:
  return rc;
}


static int
UdmBlob2BlobConsequent(UDM_AGENT *Indexer, UDM_DB *db,
                       UDM_URLDATALIST *URLList,
                       UDM_BLOB_CACHE *cache, const char *wtable,
                       int use_deflate)
{
  size_t t;
  int rc= UDM_OK;
  size_t srows= 0;
  udm_uint8 nbytes= 0;
  
  for (t= 0; t <= 0x1f; t++)
  {
    size_t nrows;
    UDM_SQLRES SQLRes;
    char buf[128];
    udm_timer_t ticks= UdmStartTimer();
    UdmLog(Indexer, UDM_LOG_DEBUG, "Loading intag%02X", t);
    udm_snprintf(buf, sizeof(buf), "SELECT url_id,intag%02X FROM bdicti WHERE state>0", t);
    if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, buf)))
      goto ret;
    nrows= UdmSQLNumRows(&SQLRes);
    UdmLog(Indexer, UDM_LOG_DEBUG, "Loading intag%02X done, %d rows, %.2f sec",
                                   t, nrows, UdmStopTimer(&ticks));
    
    ticks= UdmStartTimer();
    UdmLog(Indexer, UDM_LOG_ERROR, "Converting intag%02X", t);
    UdmBlob2BlobConvertOneColumn(Indexer, db, URLList, &SQLRes, 1, cache, &nbytes);
    UdmLog(Indexer, UDM_LOG_DEBUG,
           "Converting intag%02X done, %.2f sec", t, UdmStopTimer(&ticks));

    ticks= UdmStartTimer();
    UdmLog(Indexer, UDM_LOG_DEBUG, "Writting intag%02X", t);

    if (UDM_OK != (rc= UdmBlob2BlobWriteCache(Indexer, db,
                        cache, wtable, use_deflate, &srows)))
      goto ret;
    
    /* Can't free SQLRes earlier, "cache" has pointers to it */
    UdmSQLFree(&SQLRes);
    UdmLog(Indexer, UDM_LOG_DEBUG,
           "Writting intag%02X done, %.2f sec", t, UdmStopTimer(&ticks));
  }
  UdmLog(Indexer, UDM_LOG_ERROR, "Total converted: %d records, %llu bytes", srows, nbytes);
ret:
  return rc;
}


static int
UdmBlob2BlobParallel(UDM_AGENT *Indexer, UDM_DB *db,
                     UDM_URLDATALIST *URLList,
                     UDM_BLOB_CACHE *cache, const char *wtable,
                     int use_deflate, size_t step)
{
  int rc= UDM_OK;
  size_t srows= 0, id;
  udm_uint8 nbytes= 0;
  size_t min_id= URLList->Item[0].url_id;
  size_t max_id= URLList->Item[URLList->nitems - 1].url_id;

  for (id= min_id; id < max_id; id+= step)
  {
    size_t nrows, t;
    UDM_SQLRES SQLRes;
    char buf[512];
    udm_timer_t ticks= UdmStartTimer();
    UdmLog(Indexer, UDM_LOG_ERROR, "Loading documents %d-%d", id, id+step);
    udm_snprintf(buf, sizeof(buf),
                 "SELECT url_id,"
                         "intag00,intag01,intag02,intag03,"
                         "intag04,intag05,intag06,intag07,"
                         "intag08,intag09,intag0A,intag0B,"
                         "intag0C,intag0D,intag0E,intag0F,"
                         "intag10,intag11,intag12,intag13,"
                         "intag14,intag15,intag16,intag17,"
                         "intag18,intag19,intag1A,intag1B,"
                         "intag1C,intag1D,intag1E,intag1F "
                 "FROM bdicti WHERE state>0 AND url_id>=%d AND url_id<%d",
                 id, id + step);
    if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, buf)))
      goto ret;
    nrows= UdmSQLNumRows(&SQLRes);
    UdmLog(Indexer, UDM_LOG_DEBUG, "Loading documents done, %d rows, %.2f sec",
                                    nrows, UdmStopTimer(&ticks));
    
    for (t= 0; nrows > 0 && t <= 0x1F; t++)
    {
      ticks= UdmStartTimer();
      UdmLog(Indexer, UDM_LOG_ERROR, "Converting intag%02X", t);
      UdmBlob2BlobConvertOneColumn(Indexer, db, URLList, &SQLRes, t+1, cache, &nbytes);
      UdmLog(Indexer, UDM_LOG_DEBUG,
             "Converting intag%02X done, %.2f sec", t, UdmStopTimer(&ticks));

      ticks= UdmStartTimer();
      UdmLog(Indexer, UDM_LOG_DEBUG, "Writting intag%02X", t);

      if (UDM_OK != (rc= UdmBlob2BlobWriteCache(Indexer, db,
                          cache, wtable, use_deflate, &srows)))
        goto ret;

      UdmLog(Indexer, UDM_LOG_DEBUG,
             "Writting intag%02X done, %.2f sec", t, UdmStopTimer(&ticks));
    }
    /* Can't free SQLRes earlier, "cache" has pointers to it */
    UdmSQLFree(&SQLRes);
  }
  UdmLog(Indexer, UDM_LOG_ERROR, "Total converted: %d records, %llu bytes", srows, nbytes);
ret:
  return rc;
}


int
UdmTruncateDictBlob(UDM_AGENT *Indexer,UDM_DB *db)
{
  int rc;
  if(UDM_OK != (rc= UdmSQLTableTruncateOrDelete(db, "bdicti")) ||
     UDM_OK != (rc= UdmSQLTableTruncateOrDelete(db, "bdict")))
    return rc;
  return UDM_OK;
}


int
UdmBlob2BlobSQL(UDM_AGENT *Indexer, UDM_DB *db, UDM_URLDATALIST *URLList)
{
  size_t i, use_deflate= 0;
  int rc;
  char buf[128];
  UDM_BLOB_CACHE cache[BLOB_CACHE_SIZE + 1];
  const char *wtable;
  int tr= (db->flags & UDM_SQL_HAVE_TRANSACT) ? 1 : 0;
  int tr_truncate= tr && (db->DBType != UDM_DB_SYBASE);
  udm_timer_t ticks;
  int step= UdmVarListFindInt(&db->Vars, "step", 0);
  int disable_keys= UdmVarListFindBool(&db->Vars, "DisableKeys", 0);
  
#ifdef HAVE_ZLIB
  if (UdmVarListFindBool(&db->Vars, "deflate", UDM_DEFAULT_DEFLATE))
  {
    UdmLog(Indexer, UDM_LOG_DEBUG, "Using deflate");
    use_deflate= 1;
  }
  else
    UdmLog(Indexer, UDM_LOG_DEBUG, "Not using deflate");
#endif
  /* Get table to write to */
  if (UDM_OK != (rc= UdmBlobGetWTable(db, &wtable)))
    return rc;
  /* Lock tables for MySQL */
  if (db->DBType == UDM_DB_MYSQL)
  {
    if (db->version >= 40000 && disable_keys)
    {
      sprintf(buf, "ALTER TABLE %s DISABLE KEYS", wtable);
      if (UDM_OK != UdmSQLQuery(db, NULL, buf))
        goto ret;
    }
    udm_snprintf(buf, sizeof(buf), "LOCK TABLES bdicti READ, %s WRITE", wtable);
    if (UDM_OK != (rc= UdmSQLQuery(db, NULL, buf)))
      return rc;
  }

  /* Delete old words from bdict */
  if ((tr_truncate && UDM_OK != (rc= UdmSQLBegin(db))) ||
      UDM_OK != (rc= UdmSQLTableTruncateOrDelete(db, wtable)) ||
      (tr_truncate && UDM_OK != (rc= UdmSQLCommit(db))))
    goto ret;


  /* Convert words */
  /* Initialize blob cache */
  for (i= 0; i <= BLOB_CACHE_SIZE; i++)
    UdmBlobCacheInit(&cache[i]);
  
  
  if (UDM_OK != (rc= step ? UdmBlob2BlobParallel(Indexer, db, URLList,
                                                 cache, wtable,
                                                 use_deflate, step)
                          : UdmBlob2BlobConsequent(Indexer, db, URLList,
                                                   cache, wtable,
                                                   use_deflate)))
    goto ret;
  /* Free blob cache  */
  for (i= 0; i <= BLOB_CACHE_SIZE; i++)
    UdmBlobCacheFree(&cache[i]);


  if (db->DBType == UDM_DB_MYSQL)
  {
    if (UDM_OK != (rc= UdmSQLQuery(db, NULL, "UNLOCK TABLES")))
      return rc;
    udm_snprintf(buf, sizeof(buf), "LOCK TABLES bdicti WRITE, %s WRITE", wtable);
    if (UDM_OK != (rc= UdmSQLQuery(db, NULL, buf)))
      return rc;
  }
  
  /* Put timestamp */
  if ((tr && (UDM_OK != (rc= UdmSQLBegin(db)))) ||
      (UDM_OK != (rc= UdmBlobWriteTimestamp(Indexer, db, wtable))) ||
      (tr && (UDM_OK != (rc= UdmSQLCommit(db)))))
    goto ret;
  
  
  /* Clean bdicti */
  ticks= UdmStartTimer();
  UdmLog(Indexer, UDM_LOG_DEBUG, "Cleaning up table 'bdicti'");
  if ((tr && UDM_OK != (rc= UdmSQLBegin(db))) ||
      UDM_OK != (rc= UdmSQLQuery(db, NULL, "DELETE FROM bdicti WHERE state=0")) ||
      UDM_OK != (rc= UdmSQLQuery(db, NULL, "UPDATE bdicti SET state=2")) ||
      (tr && UDM_OK != (rc= UdmSQLCommit(db))))
    goto ret;
  UdmLog(Indexer, UDM_LOG_DEBUG,
         "Cleaning up table 'bdicti' done, %.2f sec", UdmStopTimer(&ticks));

  if (db->DBType == UDM_DB_MYSQL)
  {
    UdmSQLQuery(db, NULL, "UNLOCK TABLES");
    if (db->version >= 40000 && disable_keys)
    {
      ticks= UdmStartTimer();
      UdmLog(Indexer, UDM_LOG_EXTRA, "Enabling SQL indexes");
      sprintf(buf, "ALTER TABLE %s ENABLE KEYS", wtable);
      UdmSQLQuery(db, NULL, buf);
      UdmLog(Indexer, UDM_LOG_DEBUG,
             "Enabling SQL indexes done, %.2f sec", UdmStopTimer(&ticks));
    }
  }

  /* Convert URL */
  ticks= UdmStartTimer();
  UdmLog(Indexer, UDM_LOG_ERROR, "Converting url data");
  if ((tr && UDM_OK != (rc= UdmSQLBegin(db))) ||
      UDM_OK != (rc= UdmBlobWriteURL(Indexer, db, wtable, use_deflate)) ||
      (tr && UDM_OK != (rc= UdmSQLCommit(db))))
    return rc;
  UdmLog(Indexer, UDM_LOG_DEBUG,
         "Converting URL data done, %.2f sec", UdmStopTimer(&ticks));

  /* Switch to new blob table */
  UdmLog(Indexer, UDM_LOG_ERROR, "Switching to new blob table.");
  rc= UdmBlobSetTable(db);
  return rc;

ret:
  for (i= 0; i <= BLOB_CACHE_SIZE; i++)
    UdmBlobCacheFree(&cache[i]);
  if (db->DBType == UDM_DB_MYSQL)
    UdmSQLQuery(db, NULL, "UNLOCK TABLES");
  return rc;
}


int
UdmBlobWriteURL(UDM_AGENT *A, UDM_DB *db, const char *table, int use_deflate)
{
  const char *url;
  int use_zint4= UdmVarListFindBool(&db->Vars, "zint4", UDM_DEFAULT_ZINT4);
  UDM_DSTR buf;
  UDM_DSTR r, s, l, p, z, *pz= use_deflate ? &z : NULL;
  UDM_SQLRES SQLRes;
  int rc= UDM_OK;
  UDM_PSTR row[4];

  /* Need this if "indexer -Erewriteurl" */
  UdmSQLBuildWhereCondition(A->Conf, db);
  url= (db->from && db->from[0]) ? "url." : "";

  UdmDSTRInit(&buf, 8192);
  UdmDSTRInit(&r, 8192);
  UdmDSTRInit(&s, 8192);
  UdmDSTRInit(&l, 8192);
  UdmDSTRInit(&p, 8192);
  UdmDSTRInit(&z, 8192);

  UdmDSTRAppendf(&buf,
              "SELECT %srec_id, site_id, last_mod_time, pop_rank FROM url%s%s%s",
              url, db->from, db->where[0] ? " WHERE " : "", db->where);
  rc= UdmSQLExecDirect(db, &SQLRes, buf.data);
  UdmDSTRReset(&buf);
  if (rc != UDM_OK) goto ex;

  while (db->sql->SQLFetchRow(db, &SQLRes, row) == UDM_OK)
  {
    double pop_rank= UDM_ATOF(row[3].val);
    UdmDSTRAppendINT4(&r, UDM_ATOI(row[0].val));
    UdmDSTRAppendINT4(&s, UDM_ATOI(row[1].val));
    UdmDSTRAppendINT4(&l, UDM_ATOI(row[2].val));
    /* FIXME: little and big endian incompatibility: */
    UdmDSTRAppend(&p, (char *)&pop_rank, sizeof(pop_rank));
  }
  UdmSQLFree(&SQLRes);

  if (use_zint4)
  {
    size_t i, nrec_ids= r.size_data / 4;
    urlid_t *rec_id= (urlid_t *)r.data;
    char *zint4_buf= UdmMalloc(nrec_ids * 5 + 5);
    UDM_ZINT4_STATE zint4_state;
    if (! zint4_buf)
    {
      rc= UDM_ERROR;
      goto ex;
    }
    udm_zint4_init(&zint4_state, zint4_buf);
    for (i= 0; i < nrec_ids; i++)
      udm_zint4(&zint4_state, rec_id[i]);
    udm_zint4_finalize(&zint4_state);
    UdmDSTRReset(&r);
    UdmDSTRAppendINT4(&r, 0xFFFFFFFF);
    UdmDSTRAppendINT4(&r, 0x00000002);
    UdmDSTRAppend(&r, (char *)zint4_state.buf,
                  zint4_state.end - zint4_state.buf);
  }

  if (pz)
    UdmDSTRRealloc(pz, p.size_data + 8 + 1);

  UdmDSTRAppendf(&buf, "DELETE FROM %s WHERE word IN ('#rec_id', '#site_id', '#last_mod_time', '#pop_rank')", table);
  if (UDM_OK != (rc= UdmSQLQuery(db, NULL, buf.data)))
    goto ex;
  UdmDSTRReset(&buf);

  if (UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, "#rec_id", 0, r.data,
                                          r.size_data, &buf, pz, use_zint4)) ||
      UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, "#site_id", 0, s.data,
                                          s.size_data, &buf, pz, 0)) ||
      UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, "#last_mod_time", 0,
                                          l.data, l.size_data, &buf, pz, 0)) ||
      UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, "#pop_rank", 0, p.data,
                                          p.size_data, &buf, pz, 0)))
    goto ex;

  if (UDM_OK != (rc= UdmBlobWriteLimitsInternal(A, db, table, use_deflate)) ||
      UDM_OK != (rc= UdmBlobWriteTimestamp(A, db, table)))
    goto ex;

ex:
  UdmDSTRFree(&buf);
  UdmDSTRFree(&r);
  UdmDSTRFree(&s);
  UdmDSTRFree(&l);
  UdmDSTRFree(&p);
  UdmDSTRFree(&z);
  return rc;
}


/*
  Write limits, but don't COMMIT and don't write timestamp
*/
int
UdmBlobWriteLimitsInternal(UDM_AGENT *A, UDM_DB *db,
                           const char *table, int use_deflate)
{
  UDM_VARLIST *Vars= &A->Conf->Vars;
  UDM_VAR *Var;
  UDM_DSTR l, buf;
  int rc= UDM_OK;
  
  UdmDSTRInit(&l, 8192);
  UdmDSTRInit(&buf, 8192);
  for (Var= Vars->Var; Var < Vars->Var + Vars->nvars; Var++)
  {
    size_t i;
    char qbuf[128];
    char lname[64];
    UDM_URLID_LIST list;
    
    if (!strncasecmp(Var->name, "Limit.", 6))
      udm_snprintf(lname, sizeof(lname), "#limit#%s", Var->name + 6);
    else if (!strncasecmp(Var->name, "Order.", 6))
      udm_snprintf(lname, sizeof(lname), "#order#%s", Var->name + 6);
    else
      continue;
    UdmLog(A, UDM_LOG_DEBUG, "Writting '%s'", lname);

    if (UDM_OK != (rc= UdmLoadSlowLimit(db, &list, Var->val)))
      goto ret;
    
    if (!strncasecmp(Var->name, "Limit.", 6) && list.nurls > 1)
      UdmSort(list.urls, list.nurls, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
    
    UdmDSTRReset(&buf);
    UdmDSTRReset(&l);
    for (i= 0; i < list.nurls; i++)
    {
      UdmDSTRAppendINT4(&l, list.urls[i]);
    }

    udm_snprintf(qbuf, sizeof(qbuf), "DELETE FROM %s WHERE word=('%s')", table, lname);
    if(UDM_OK != (rc= UdmSQLQuery(db, NULL, qbuf)))
      goto ret;

    if (l.size_data)
    {
      if (UDM_OK != (rc= UdmBlobWriteWordCmpr(db, table, lname, 0, l.data,
                                              l.size_data, &buf, NULL, 0)))
        goto ret;
    }

    UDM_FREE(list.urls);
    UdmLog(A, UDM_LOG_DEBUG, "%d documents written to '%s'", list.nurls, lname);
  }
ret:
  UdmDSTRFree(&l);
  UdmDSTRFree(&buf);
  return rc;
}


int
UdmBlobLoadFastURLLimit(UDM_DB *db, const char *name, UDM_URLID_LIST *buf)
{
  int rc= UDM_OK;
  UDM_SQLRES SQLRes;
  char qbuf[256], ename[130], exclude;
  size_t nrows, nurls, i, row, namelen= strlen(name);
  
  if (namelen > 64)
    return UDM_OK;
  
  UdmSQLEscStrSimple(db, ename, name, namelen); /* Escape limit name */
  exclude= buf->exclude;
  bzero((void*)buf, sizeof(*buf));
  buf->exclude= exclude;
  udm_snprintf(qbuf, sizeof(qbuf), "SELECT intag FROM bdict WHERE word LIKE '#limit#%s'", ename);
  if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, qbuf)))
   goto ret;

  if (! (nrows= UdmSQLNumRows(&SQLRes)))
  {
    buf->empty= 1;
    goto ret;
  }
  nurls= 0;
  for (row= 0; row < nrows; row++)
    nurls+= UdmSQLLen(&SQLRes, row, 0) / 4;

  if (!(buf->urls= UdmMalloc(sizeof(urlid_t) * nurls)))
    goto ret;

  for (row= 0; row < nrows; row++)
  {
    const char *src= UdmSQLValue(&SQLRes, row, 0);
    nurls= UdmSQLLen(&SQLRes, row, 0) / 4;
    if (src && nurls)
      for (i = 0; i < nurls; i++, src+= 4)
        buf->urls[buf->nurls++]= (urlid_t) udm_get_int4(src);
  }
  if (nrows > 1)
    UdmSort(buf->urls, buf->nurls, sizeof(urlid_t), (udm_qsort_cmp)cmpaurls);
  
ret:
  UdmSQLFree(&SQLRes);
  return rc;
}


int
UdmBlobLoadFastOrder(UDM_DB *db, UDM_URL_INT4_LIST *List, const char *name)
{
  int rc= UDM_OK;
  UDM_SQLRES SQLRes;
  char qbuf[256], ename[130];
  size_t nrows, nurls, row, param, namelen= strlen(name);
  
  if (namelen > 64)
    return UDM_OK;
  
  UdmSQLEscStrSimple(db, ename, name, namelen); /* Escape order name */
  bzero((void*)List, sizeof(*List));
  udm_snprintf(qbuf, sizeof(qbuf), "SELECT intag FROM bdict WHERE word LIKE '#order#%s'", ename);
  if (UDM_OK != (rc= UdmSQLQuery(db, &SQLRes, qbuf)))
   goto ret;

  if (!(nrows= UdmSQLNumRows(&SQLRes)))
    goto ret;

  nurls= 0;
  for (row= 0; row < nrows; row++)
    nurls+= UdmSQLLen(&SQLRes, row, 0) / 4;

  if (!(List->Item= UdmMalloc(sizeof(UDM_URL_INT4) * nurls)))
    goto ret;

  for (param= 0x7FFFFFFF, row= 0; row < nrows; row++)
  {
    size_t i;
    const char *src= UdmSQLValue(&SQLRes, row, 0);
    nurls= UdmSQLLen(&SQLRes, row, 0) / 4;
    if (src && nurls)
    {
      for (i= 0; i < nurls; i++, src+= 4)
      {
        UDM_URL_INT4 *Item= &List->Item[List->nitems++];
        Item->url_id= (urlid_t) udm_get_int4(src);
        Item->param= --param;
      }
    }
  }
  if (List->nitems > 1)
    UdmSort(List->Item, List->nitems, sizeof(UDM_URL_INT4), (udm_qsort_cmp)cmpaurls);
ret:
  UdmSQLFree(&SQLRes);
  return rc;
}
