/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file ObsTable.cc
    \brief Implementation of the Template class LocalTable.
    
    Magics Team - ECMWF 2004
    
    Started: Mon 21-Jun-2004
    
    Changes:
    
*/
#include "ObsTable.h"
#include "Exception.h"
#include "expat.h"
#include <string>
#include "Layout.h"
#include "Transformation.h"
#include "Symbol.h"


using std::basic_string;

using namespace magics;


static void XMLCALL
startElement(void *userData, const char *name, const char **atts)
{
	ObsTable* table  = (ObsTable*) userData; 
        map<string, string> def;
        while (*atts) {
            def[*(atts)] = *(atts+1);
            atts+=2;
        }
        table->add(name, def);
	
}

static void XMLCALL
endElement(void *, const char *)
{
}

ObsTable* ObsTable::table_ = 0;


ObsTable::ObsTable() 
{
 
    string filename = getEnvVariable("MAGPLUS_HOME") + MAGPLUS_PATH_TO_SHARE_ + "obs.xml";
	char buf[BUFSIZ];
	XML_Parser parser = XML_ParserCreate(NULL);
	int done;
	XML_SetUserData(parser, this);
	XML_SetElementHandler(parser, startElement, endElement);

	Log::dev() << "Load observation templates --->" << filename << endl;
	FILE* in  = fopen(filename.c_str(), "r");

	if (!in) throw NoSuchFileException(filename);

	do
	{
		size_t len = fread(buf, 1, sizeof(buf), in);
		done = len < sizeof(buf);
		if (XML_Parse(parser, buf, len, done) == XML_STATUS_ERROR)
		{
			ostringstream s;
			s << "XmlException : " << XML_ErrorString(XML_GetErrorCode(parser))  << " at line  " <<  XML_GetCurrentLineNumber(parser)  << ends;
			cerr <<  s.str() << "\n";
			//throw MagicsException(s.str());
		}
	} while (!done);
	XML_ParserFree(parser);
	fclose(in);
}


ObsTable::~ObsTable() 
{}

/*!
 Class information are given to the output-stream.
*/	

    
void ObsTable::print(ostream& out)  const
{
	out << "ObsTable[";	
	for ( const_iterator item = begin(); item != end(); ++item) 
			    out << "\t" << item->first << "---->" << *(item->second) << "\n";
			out << "\n";
	out << "]";
}

void ObsTable::add(const string& tag, const map<string, string>& def) 
{
 	if ( tag == "observations" ) return;
 	if ( tag == "obs_template" ) {
 		current_ = new ObsTemplate(def);
 		insert(make_pair(def.find("type")->second, current_));
 	} 
 	else {
 		try {
 			ObsItem* obs = SimpleObjectMaker<ObsItem>::create(tag);
 			obs->set(def);
 			current_->push_back(obs);
 		}
 		catch (NoFactoryException&) {
 			Log::dev() << "can not find ObsItem for : " << tag << "\n";
 		}
 	}
}
   
const ObsTemplate& ObsTable::get(const string& type)
{
	const_iterator entry = find(type);
	if ( entry != end() ) { return *(entry->second); }
	throw exception();
}   

void ObsTemplate::set(double apart, const string& box) const
{
	working_.clear();
	apart_ = apart;
	box_   = box;
}

void ObsTemplate::operator()(CustomisedPoint& obs, BasicGraphicsObjectContainer& out) const 
{
		if ( empty() ) 
			return; // Nothing to display.
			
		// Create the Observation layout 
		const Transformation& transformation = out.transformation();
		
		if (!transformation.in(obs.longitude(), obs.latitude()) ) 
			return; // The obs is outside the drawing area
		
		ostringstream box; 
		box << "obs[" << obs.latitude() << "," << obs.longitude() << "]";
		PaperPoint pp = transformation(GeoPoint(obs.longitude(), obs.latitude()));
		
		
		for ( vector<PaperPoint>::const_iterator previous = working_.begin(); previous != working_.end(); ++previous) {
			//if (owner.layout().distance(box_, pp, *previous) < apart_ ) return;
		}
		
		working_.push_back(pp);
		
		ComplexSymbol* symbol = new ComplexSymbol(rows_, columns_);
		
		symbol->push_back(pp);
		
		out.push_back(symbol);

		for ( const_iterator item = begin(); item != end(); ++item) 
			    (*(*item))(obs, *symbol);
}
