/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*!
    \file OpenGLPickObjectManager.cc
    \brief Definition of OpenGLPickObjectManager
    \author Graphics Section, ECMWF

    Started: May 2004
*/


#include <OpenGLPickObjectManager.h>
#include <OpenGLPickObject.h>
#include <OpenGLNode.h>

#include "MtInputEvent.h"

OpenGLPickObjectManager::OpenGLPickObjectManager(OpenGLDriver *d) : driver_(d)
{
	pickMode_=PM_NONE;
	interactiveMode_=IAM_NONE;
		
} 



//============================================
//
//   INTERACTIVE MODE
// 
//============================================

void OpenGLPickObjectManager::setInteractiveMode(InteractiveMode new_mode)
{
	if(new_mode != IAM_PICK)
	{										
		if(interactiveMode_ == IAM_PICK)
		{
			leaveMode();
			return;		
		}
	}
	else
	{
		if(interactiveMode_ == IAM_PICK)
		{
			leaveMode();					
		}		
		else
		{		
			enterMode();		
		}
	}				
}	
	


void OpenGLPickObjectManager::select(const int x, const int y)
{	
	//Identify the selected pick object
	//currentPickObject_=identifyUserPickObject(x,y);
	
	currentPickObject_=identifyPickObject(x,y);
	
	if( currentPickObject_ == NULL) return;				
	
	currentPickObject_->driver(driver_);
	
	//modifyPickObject_select(x,y);
					
	glMatrixMode( GL_MODELVIEW );		     	         
	glPushMatrix();	     
	glLoadIdentity();  
				
	//Draw modify boxes around the selected object
	
	currentPickObject_->renderPickBox(true,true); 
		
	glPopMatrix();
        driver_->swapFb();
				
	//Save the actual framebuffer
	driver_->saveFbToBgImage(1);
		
	pickX_=x;
	pickY_=y;
		
	pickMode_=PM_REPLACE;				
}

void OpenGLPickObjectManager::unselect()
{
	//Unselect 
	pickMode_ = PM_NONE;
	currentPickObject_ = NULL;	
	
	//Map the saved bg to fb

	glMatrixMode( GL_MODELVIEW );		     	         
	glPushMatrix();	     
	glLoadIdentity();
	
	driver_->mapBgImageToFb(0);	
	
	//drawAllUserPickObjects();
	
	glPopMatrix();
		
	driver_->swapFb();
}

void OpenGLPickObjectManager::begin(const int x, const int y)
{	        	
        //Resize				
	if(currentPickObject_->checkPointInCornerBox(x,y) == true ||
	   currentPickObject_->checkPointInSideBox(x,y) ==true )
	{
		pickMode_=PM_RESIZE;    
	} 		  
	//Reshape				
	else if(currentPickObject_->checkPointInObjectBox(x,y))
	{
		pickMode_=PM_RESHAPE;    
	} 			
	//Replace
	else if(currentPickObject_->checkPointInBoundingBox(x,y))
	{
		 pickMode_=PM_REPLACE;
	}		
	else	
	{									
		//Unselect first
		unselect();
						
		//New selection					  
		select(x,y);
		return;
	}					
		
	pickX_=x;
	pickY_=y;			  		  								
}	

void OpenGLPickObjectManager::replace(const int x, const int y)
{								
	float x0,y0,x1,y1;
	currentPickObject_->getBoundingBoxCover(x0,y0,x1,y1,0,0,driver_->dimensionX(),driver_->dimensionY());
	
	//Replace object
	currentPickObject_->replaceObject(x-pickX_,y-pickY_);	
		
	//Map bg image #1 to the framebuffer
	driver_->mapBgImageToFb(1,x0,y0,x1,y1);	
	
	glMatrixMode( GL_MODELVIEW);		     	         
	glPushMatrix();	     
	glLoadIdentity();  	
		
	//Draw replaced bounding box and object outline
	currentPickObject_->renderTmpPickObject(true,true); 	
	
	glPopMatrix();
        driver_->swapFb();	
	
	pickX_=x;
	pickY_=y;
}	


void OpenGLPickObjectManager::resize(const int x, const int y)
{
	float x0,y0,x1,y1;
	currentPickObject_->getBoundingBoxCover(x0,y0,x1,y1,0,0,driver_->dimensionX(),driver_->dimensionY());
	
	//Resize object
	currentPickObject_->resizeObject(x-pickX_,y-pickY_);
					
	//Map bg image #1 to the framebuffer
	driver_->mapBgImageToFb(1,x0,y0,x1,y1);	
				
	glMatrixMode( GL_MODELVIEW );		     	         
	glPushMatrix();	     
	glLoadIdentity();  	
	
	//Draw resized bounding box and object outline
	currentPickObject_->renderTmpPickObject(true,true); 
				
	glPopMatrix();
        driver_->swapFb();	
	
	pickX_=x;
	pickY_=y;
}	
	
void OpenGLPickObjectManager::reshape(const int x, const int y)
{	
	return;
}	

void OpenGLPickObjectManager::end(const int x, const int y)
{
	//Check mode
	if(currentPickObject_->getModifyStatus() == false) return;
	
	currentPickObject_->finishModification();
		
	//Redraw all
	
	//Map the saved bg to fb
	driver_->mapBgImageToFb(0);
		
	glMatrixMode( GL_MODELVIEW );		     	         
	glPushMatrix();	     
	glLoadIdentity();  
	
	//drawAllUserPickObjects();	
		
	if(pickMode_ != PM_NONE)
	{
		//Draw pick boxes around the selected object
		currentPickObject_->renderPickBox(true,true);
	}
					
	glPopMatrix();
        driver_->swapFb();  
	
	//Save framebuffer	  
	driver_->saveFbToBgImage(1);
		
	//
	if(currentPickObject_->getType() == POT_MAGBOX)
	{
		OpenGLBox *mbox=currentPickObject_->getMagBox();
		OpenGLBox *bbox=currentPickObject_->getBoundingBox();
		OpenGLBox *pbox=currentPickObject_->getParentBox();
		
		float xx,yy;
		double cx0,cy0,cx1,cy1;
		xx=bbox->x();
		yy=bbox->y();
				
		//pbox->getPointCoordinates(xx,yy,cx0,cy0);
		mbox->x(cx0);
		mbox->y(cy0);
		
		Log::dev()<< "win: " << "(0)" << xx << " " << yy << endl;
		Log::dev()<< "mag: " << "(0) " << cx0 << " " << cy0 << endl;
		
		xx=bbox->x()+bbox->width();
		yy=bbox->y()+bbox->height();
		if(yy>=driver_->dimensionY()) yy=driver_->dimensionY()-1;
		if(yy> pbox->y()+pbox->height()) yy= pbox->y()+pbox->height()-5;
		
		
		//pbox->getPointCoordinates(xx,yy,cx1,cy1);
		mbox->width(cx1-cx0);
		mbox->height(cy1-cy0);
		
		Log::dev()<< "win: " << "(1) " << xx << " " << yy << endl;
		Log::dev()<< "mag: " << "(1) " << cx1 << " " << cy1 << endl;
		 											
		driver_->getObserver()->redraw();
		

		//currentPickObject_=driver_->tree()->node("root.root.Page0.TextNode0.drawing_area")->pickObject();
		
		currentPickObject_=0;

		currentPickObject_->driver(driver_);
		
		glMatrixMode( GL_MODELVIEW );		     	         
		glPushMatrix();	     
		glLoadIdentity();  
				
		//Draw modify boxes around the selected object
	
		currentPickObject_->renderPickBox(true,true); 
		
		glPopMatrix();
        	driver_->swapFb();
				
		//Save the actual framebuffer
		driver_->saveFbToBgImage(1);
		
	}
	
}	
void OpenGLPickObjectManager::enterMode()
{
	interactiveMode_ = IAM_PICK;
	pickMode_ = PM_NONE;
	//XtAddCallback(driver_->widget(),GLwNinputCallback, 
	//              &OpenGLPickObjectManager::inputCB, this);
}

void OpenGLPickObjectManager::leaveMode()
{	
	currentPickObject_=NULL;
	
	if(currentPickObject_ != NULL  && pickMode_ != PM_NONE)
	{
		//Map the saved bg to fb
		driver_->mapBgImageToFb(0);
		
		glMatrixMode( GL_MODELVIEW );		     	         
		glPushMatrix();	     
		glLoadIdentity();  
	
		//drawAllUserPickObjects();	
					
		glPopMatrix();
        	driver_->swapFb();  
	
		//Save framebuffer	  
		driver_->saveFbToBgImage(1);
	}
	
	currentPickObject_=NULL;
	interactiveMode_ = IAM_NONE;
	pickMode_ = PM_NONE;	 
}	



/*void OpenGLPickObjectManager::drawAllUserPickObjects()
{   						
	OpenGLPickObjectList::iterator it = userPickObjectList_.begin();
	while ( it != userPickObjectList_.end())
	{					 
		drawUserPickObject(*it,false); 
		it++; 
	}	
}*/



OpenGLPickObject* OpenGLPickObjectManager::identifyPickObject(const int x, const int y)
{			
	//Check user pickobjects	
	
	//Check opengl boxes
	
	//Search for editable boxes
	
	//Check page
	
/*	OpenGLPageNode* p;
	
	string page;
	
	list<OpenGLPageNode*>::iterator it=driver_->pageList_.begin();
	while(it != driver_->pageList_.end())
	{
		if((*it)->checkPointInBox(x,y))
		{
			page=(*it)->name();
			Log::dev()<< "Page: " << page << endl;
			p=*it;
			//return (*it)->pickObject();
			break;
		}
		it++;
	}
	
	list<OpenGLBoxNode*> glbList;
	return driver_->tree()->node("root.root.Page0.TextNode0.drawing_area")->pickObject();	 */

	return 0;			
}


void OpenGLPickObjectManager::event(MtInputEvent *event)
{
	//Check mode
	if(interactiveMode_ != IAM_PICK) return;
	
 	int x, y; 	
 	MtMouseEvent* mev;
        	
 	switch(event->type())
 	{

 	case Mt::MousePressEvent:
 		mev = (MtMouseEvent *) event;
		x = mev->x();
 		y = driver_->dimensionY()-mev->y();				
		
		if(mev->button() & Mt::LeftButton) // left  mouse button
 		{			
 			if(pickMode_ == PM_NONE)
			{			 			
				select(x, y);
			}
			else
			{
				begin(x, y);
			}					
			
 		}
		
		if(mev->button() & Mt::RightButton) //right  mouse button
 		{			
 			//instance->modifyPickObject_delete(x, y);						
 		}		
 		break;

 	case Mt::MouseMoveEvent: 		
		mev = (MtMouseEvent *) event;
		x = mev->x();
 		y = driver_->dimensionY()-mev->y();

		//On OpenSuse 10.3 for button1 motion event->x.motion.state has
		//a value of 272. That is why this hack was added.			
		if(mev->button() & Mt::LeftButton)
 		{			 			
 			if(pickMode_ == PM_REPLACE)
			{
				replace(x, y);									
			}
			else if(pickMode_ == PM_RESIZE)
			{
				resize(x, y);									
			}
			else if(pickMode_ == PM_RESHAPE)
			{
				reshape(x, y);									
			}
 		}		
 		break;
		
	case Mt::MouseReleaseEvent:
		mev = (MtMouseEvent *) event;
		x = mev->x();
 		y = driver_->dimensionY()-mev->y();
		
		if(mev->button() & Mt::LeftButton) // left  mouse button
 		{			 			
 			if(pickMode_ != PM_NONE)
			{
				end(x, y);			
			}
 		}
		break;
	}

}



