/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*!
    \file OpenGLPickObjectCreator.cc
    \brief Definition of OpenGLPickObjectCreator
    \author Graphics Section, ECMWF

    Started: May 2008
*/

#include <magics.h>
#include "Log.h"
#include "OpenGLPickObjectCreator.h"
#include "OpenGLPickObject.h"
#include "OpenGLNode.h"
#include "OpenGLLayoutNode.h"
#include "OpenGLCore.h"

#include "MtInputEvent.h"

using namespace magics;


OpenGLPickObjectCreator::OpenGLPickObjectCreator(OpenGLDriver *driver) : driver_(driver)
{
	parent_=0;	
	object_=0;
	
	pickMode_=PM_NONE;
	interactiveMode_=IAM_NONE;
		
	creationModes_.insert(IAM_ZOOM);
	creationModes_.insert(IAM_AREA);
	creationModes_.insert(IAM_POLYLINE);
	creationModes_.insert(IAM_POLYGON);
	creationModes_.insert(IAM_LINE);
	creationModes_.insert(IAM_POINT);
} 
	
OpenGLPickObjectCreator::~OpenGLPickObjectCreator()
{
	if(interactiveMode_ != IAM_NONE)		
	{
		leaveMode();
	}

	if (object_) delete object_;
}

void OpenGLPickObjectCreator::setInteractiveMode(InteractiveMode new_mode)
{
			
}	
	
void OpenGLPickObjectCreator::begin(const int x, const int y)
{	
	parent_=identifyParent(x,y);
	
	if(parent_==0) return;
		
	PickObjectType type;
	switch(interactiveMode_)
	{
		case IAM_ZOOM:
			type=POT_ZOOMRECT;
			break;
		case IAM_AREA:
			type=POT_AREA;
			break;
		case IAM_POLYLINE:
			type=POT_POLYLINE;
			break;
		case IAM_POLYGON:
			type=POT_POLYGON;
			break;
		case IAM_POINT:
			type=POT_POINT;
			break;
		case IAM_LINE:
			type=POT_LINE;
			break;	
		default:
			return;
	}	
			
	Log::debug() << "Create new pickObject \n [box = " << parent_->name() << 
	               " type= " << type << endl;
		
	//Create new pick object			
	
	//parent_->addUserObject(type);

	object_ = new OpenGLPickObject(parent_->window(),type);	
	//object_=parent_->lastUserObject();
	object_->driver(driver_);
	
	//Save the actual framebuffer
	
	
	//driver_->saveFbToBgImage(1);
		
	pickMode_= PM_DRAW;
	
	PaperPoint pp(x,y);
		
	switch(interactiveMode_)
	{
		case IAM_ZOOM:
		case IAM_AREA:	
			//Add first point 4 times			
			object_->addPoint(pp);
			object_->addPoint(pp);
			object_->addPoint(pp);
			object_->addPoint(pp);		
			break;		
		
		case IAM_LINE:
		case IAM_POLYLINE:
		case IAM_POLYGON:		
			//Add first point twice			
			object_->addPoint(pp);
			object_->addPoint(pp);		
			break;			
	
		case IAM_POINT:					
			object_->addPoint(pp);
			draw(x,y);
			end(x,y);
			break;

		default:
			break;		
	}		        		
}	

void OpenGLPickObjectCreator::addpoint(const int x, const int y)
{				
	PaperPoint pp(x,y);
		
        float x0,y0,x1,y1;
	object_->getBoundingBoxCover(x0,y0,x1,y1,0,0,driver_->dimensionX(),driver_->dimensionY());
	
	if(object_->addPoint(pp) == false) return;
	
	glMatrixMode(GL_MODELVIEW);		     	         
	glPushMatrix();	     
	glLoadIdentity();     	
		
	//Map bg image #1 to the framebuffer
	driver_->mapBgImageToFb(1,x0,y0,x1,y1);
		
	//Draw the whole object
	object_->renderObject(false);

  	glPopMatrix();
        driver_->swapFb();
}	
	      
void OpenGLPickObjectCreator::draw(const int x, const int y)
{
	float x0,y0,x1,y1;
	object_->getBoundingBoxCover(x0,y0,x1,y1,0,0,driver_->dimensionX(),driver_->dimensionY());
		
	//Set last point and check if it is inside the
	//parent coordinate system
	if(interactiveMode_ != IAM_POINT)
	{
		PaperPoint pp(x,y);		
		if(object_->setLastPoint(pp) == false)
		{
			return;
		}
	}
		
	glMatrixMode(GL_MODELVIEW);		     	         
	glPushMatrix();	     
	glLoadIdentity();     
	
	//Copy fb texture to back buffer 
	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		OpenGLCore::instance()->buildFb();
	}

	//Copy bg texture to back buffer
	else if(OpenGLCore::instance()->renderMode() == OpenGLCore::BackBufferMode)
	{
		OpenGLCore::instance()->bgToFb(x0,y0,x1,y1);	
	}

	//Draw the whole object to the backbuffer
	object_->renderObject(false);

  	glPopMatrix();
        driver_->swapFbWithCurrentContent();
}  


void OpenGLPickObjectCreator::end(const int x, const int y)
{
	pickMode_= PM_NONE;
	
	if(interactiveMode_ == IAM_ZOOM)
	{
	  //notify the 
	  SelectionObject so(parent_->layout());	 
	  //Box *pbox=parent_->box()->getBoundingBox();

	  //so.info(pbox->info());	  
	  //so.info(parent_->boxInfo());
	  	  
	  vector<PaperPoint> *vp=object_->getObject();

	  //Check rect size
	  float minsize=2; //in pixels
	  if(fabs((*vp)[0].x() - (*vp)[2].x()) < minsize && 
             fabs((*vp)[0].y() - (*vp)[2].y()) < minsize)
	  {
		delete object_;
		if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
		{
			OpenGLCore::instance()->buildFb();
		}
		//driver_->mapBgImageToFb(1);
		driver_->swapFb();
		return;
	  }	

	  int i;
	  const int nn=vp->size();
	  float xx,yy;
	  float cx,cy;	 
	  for(i=0; i< nn; i++)
	  {
		xx=(*vp)[i].x();
		yy=(*vp)[i].y();

		Log::dev() << "win: " << "(" << i << ") " << xx << " " << yy << endl;
		 		
		//parent_->box()->getPointCoordinates(xx,yy,cx,cy);
		if(parent_->winCoordToLayoutCoordByFitting(xx,yy,cx,cy) == false)
		{
			Log::error() << "Zoom cannot be performed!!! Zoom rectangle is outside the layout area!!" << endl;
			delete object_;
			if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
			{
				OpenGLCore::instance()->buildFb();
			}
			//driver_->mapBgImageToFb(1);
			driver_->swapFb();
			return;
		}		

		Log::dev() << "mag: " << "(" << i << ") " << cx << " " << cy << endl;
		
		PaperPoint pp(cx,cy);		
		so.push_back(pp);		
	  }	

	  //delete the local object
	  //parent_->deleteLastUserObject(); 	  

	  Log::debug() << "---> end: glLayoutStack size: " << 
				driver_->glLayoutStack_.size() << endl;
	  
	  //notify the observer
	  driver_->notifyObservers(&OpenGLDriverObserver::areaSelection,&so);	 

	  delete object_;
	}
	else
	{	
		object_->initBox();
				
		//Pass the object to the observer
		//getObserver->(pt);		
	}		
}		

void OpenGLPickObjectCreator::enterMode(InteractiveMode new_mode)
{	
	if(creationModes_.count(new_mode) != 0)
	{
	  	interactiveMode_ = new_mode;
	}	
}

void OpenGLPickObjectCreator::leaveMode()
{
	interactiveMode_=IAM_NONE;
	pickMode_=PM_NONE;							 
}


OpenGLLayoutNode* OpenGLPickObjectCreator::identifyParent(float x,float y)
{
	list<OpenGLNode*> gll_list;
	driver_->glTree()->findLayout(gll_list);

	for(list<OpenGLNode*>::iterator it=gll_list.begin(); it != gll_list.end(); it++)
	{
		OpenGLLayoutNode* gll = (OpenGLLayoutNode*) (*it);
		
		if(gll->layout().isZoomable() == true &&
		   gll->checkPointInWindow(x,y) == true)
		{
			
			return gll;
		}
	}
	
	return 0;

}


void OpenGLPickObjectCreator::event(MtInputEvent* event)
{	
 	int x, y; 	 	
        MtMouseEvent *mev;

 	switch(event->type())
 	{

 	case Mt::MousePressEvent:
 		mev = (MtMouseEvent *) event;
		x = mev->x();
 		y = mev->y();				
		
		if(mev->button() & Mt::LeftButton) // left  mouse button
 		{			 			
 			if(pickMode_ == PM_NONE)
			{
				//begin(x,y);						
			}
			else if(interactiveMode_ == IAM_POLYGON || 
			        interactiveMode_ == IAM_POLYLINE)
			{
				addpoint(x,y);
			}
			/*else
			{
				end(x,y);
			}*/
							
 		}
		else if (mev->button() & Mt::MidButton) // middle mouse button 
 		{
			if(interactiveMode_ == IAM_POLYGON || 
			   interactiveMode_ == IAM_POLYLINE)
			{ 
				end(x,y);	
			}		
 		} 			
 		break;

 	case Mt::MouseMoveEvent: 		
		mev = (MtMouseEvent *) event;
		x = mev->x();
 		y = mev->y();
			
		if(mev->button() & Mt::LeftButton)
		{
			if(pickMode_ == PM_NONE) 
			{
				begin(x,y);
			}
			else if(pickMode_== PM_DRAW)
			{
			
				if(interactiveMode_ == IAM_ZOOM ||
			  	   interactiveMode_ == IAM_AREA || 
			  	   interactiveMode_ == IAM_LINE)
				{				
 					draw(x, y);									
 				}
			}
		}
			
 		break;

	case Mt::MouseReleaseEvent:
		mev = (MtMouseEvent *) event;
		x = mev->x();
 		y = mev->y();				
		
		if(pickMode_ != PM_DRAW) return;
		
		if(mev->button() & Mt::LeftButton) // left  mouse button
 		{			
 			if(interactiveMode_ == IAM_ZOOM || 
			   interactiveMode_ == IAM_AREA || 
			   interactiveMode_ == IAM_LINE)
			{
				end(x, y);			
			}
 		}
		break;
	}

}

bool OpenGLPickObjectCreator::active()
{
	if (pickMode_ == PM_DRAW) 
	{
		//Log::debug() << "OpenGLPickObjectCreator active" << endl;
		return true;
	}	
	else return false;
}
