/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file Coordinate.h
    \brief Definition of the Template class Coordinate.
    
    Magics Team - ECMWF 2006
    
    Started: Thu 10-Aug-2006
    
    Changes:
    
*/

#ifndef Coordinate_H
#define Coordinate_H

#include "magics.h"
#include "Translator.h"
#include "Factory.h"

#include "XCoordinateAttributes.h"
#include "YCoordinateAttributes.h"
#include "XDateCoordinateAttributes.h"
#include "YDateCoordinateAttributes.h"

#include "XmlNode.h"

#include <limits>
namespace magics {

class XmlNode;

class Coordinate {

public:
	Coordinate();
	virtual ~Coordinate();
	
	virtual double min() { return 0; }
    virtual double max() { return 100; }
    virtual double minpc() { return 0; }
    virtual double maxpc() { return 100; }
    virtual string reference() { return ""; }
    
    virtual void min(double) { }
    virtual void max(double) { }
    
    virtual double operator()(double c) { return c; }
    
    virtual void set(const XmlNode&) {
        Log::dev() << "(const XmlNode&)---> to be checked!...\n";
    }
    virtual void set(const map<string, string>&) {
        Log::dev() << "(const map<string, string&)---> to be checked!...\n";
        
    }
    
    virtual bool accept(const string&) {
        Log::dev() << "(const map<string, string&)---> to be checked!...\n";
        return false;
    }
    
    virtual void toxml(ostream&, int) const {}   
    virtual bool automatic() { return false; }
    
    virtual double operator()(const string& val) const { return tonumber(val); }
    
    
    virtual void dataMin(double m) { 
    	if ( !automatic() ) return;
//    	Log::dev()<< "new min-->" << m << "< " << min() << "???" << endl;  
    	if (min() > m) min(m); 
    }
    virtual void dataMax(double m) { 
    	if (!automatic()) return;
//   	Log::dev()<< "new max-->" << m << endl; 
//    	Log::dev()<< "new max-->" << m << "> " << max() << "???" << endl;  
    	if (max() < m ) max(m); 
    }
    virtual void dataMin(double m, const string&) { if (min() > m) dataMin(m); }
    virtual void dataMax(double m, const string&) { if (max() < m)dataMax(m); }    

  
    
protected:
     //! Method to print string about this class on to a stream of type ostream (virtual).
	 virtual void print(ostream&) const; 

private:
    //! Copy constructor - No copy allowed
	Coordinate(const Coordinate&);
    //! Overloaded << operator to copy - No copy allowed
	Coordinate& operator=(const Coordinate&);

// -- Friends
    //! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const Coordinate& p)
		{ p.print(s); return s; }

};

class XCoordinate : public Coordinate
{
public:
	XCoordinate() {}
	virtual ~XCoordinate() {}
	virtual XCoordinate* clone() const {
        Log::dev() << "(const map<string, string&)---> to be checked!...\n";
        return new XCoordinate();
    }
    
    virtual void setMin(double) {}
    virtual void setMax(double) {}
    virtual void setAutomatic(bool) {}
    
};

class YCoordinate : public Coordinate
{
public:
	YCoordinate() {}
	virtual ~YCoordinate() {}
	virtual YCoordinate* clone() const {
        Log::dev() << "(const map<string, string&)---> to be checked!...\n";
        return new YCoordinate();
    }
    virtual void setMin(double) {}
    virtual void setMax(double) {}
    virtual void setAutomatic(bool) {}
};

class RegularCoordinate 
{
public:
	RegularCoordinate() {}
	virtual ~RegularCoordinate() {}
};

class XRegularCoordinate : public RegularCoordinate, public XCoordinateAttributes, public XCoordinate
{
public:
	XRegularCoordinate() { 
		if ( automatic_ ) {
			min_ = std::numeric_limits<double>::max();
			max_ = -min_;
		}
	}
	virtual ~XRegularCoordinate() {}
	
	virtual void set(const XmlNode& node) {
        XCoordinateAttributes::set(node);
        if ( automatic_ ) {
			min_ = std::numeric_limits<double>::max();
			max_ = -min_;
		}
    }
    virtual bool accept(const string& node) { return XCoordinateAttributes::accept(node); }
	double min() { return min_; }
    double max() { return max_; }
    double minpc() { return min_; }
    double maxpc() { return max_; }
    
    bool automatic() { return automatic_; }
    
  
    
    void min(double m) { min_ = m; }
    void max(double m) { max_ = m; }
    
    virtual XCoordinate* clone() const {
    	XRegularCoordinate* x = new XRegularCoordinate();
    	x->copy(*this);
        return x;
    }
    
    void setMin(double min) { XCoordinateAttributes::setMin(min); }
    void setMax(double max) { XCoordinateAttributes::setMax(max); }
    
  
    
protected:
	virtual void print(ostream& out) const  {
		XCoordinateAttributes::print(out);
	}
};

class YRegularCoordinate : public RegularCoordinate, public YCoordinateAttributes, public YCoordinate
{
public:
	YRegularCoordinate() {
			if ( automatic_ ) {
			min_ = std::numeric_limits<double>::max();
			max_ = -min_;
		}
	}
	virtual ~YRegularCoordinate() {}
	double min() { return min_; }
    double max() { return max_; }
    double minpc() { return min_; }
    double maxpc() { return max_; }
    
    void min(double m) { min_ = m; }
    void max(double m) { max_ = m; }
    
    virtual YCoordinate* clone() const {
    	YRegularCoordinate* y = new YRegularCoordinate();
    	y->copy(*this);
        return y;
    }
    void set(const XmlNode&  node ) { YCoordinateAttributes::set(node); }
    void set(const map<string, string>& map) { 
    	YCoordinateAttributes::set(map); 
    	if ( automatic_ ) {
			min_ = std::numeric_limits<double>::max();
			max_ = -min_;
		}
	}
    bool accept(const string& node) { return YCoordinateAttributes::accept(node); }

    void setMin(double min) { YCoordinateAttributes::setMin(min); }
    void setMax(double max) { YCoordinateAttributes::setMax(max); }
    bool automatic() { return automatic_; }
    
   
protected:
	virtual void print(ostream& out) const  {
		YCoordinateAttributes::print(out);
	}
};


class LogarithmicCoordinate : public Coordinate
{
public:
	LogarithmicCoordinate() {}
	virtual ~LogarithmicCoordinate() {}


};

class XLogarithmicCoordinate : public LogarithmicCoordinate, public XCoordinateAttributes, public XCoordinate
{
public:
	XLogarithmicCoordinate() {}
	virtual ~XLogarithmicCoordinate() {}
	double min() { return min_; }
    double max() { return max_; }
    double minpc() { return log10(min_); }
    double maxpc() { return log10(max_); }
    virtual void set(const XmlNode& node) {
	    if ( !magCompare(node.name(), "x_logarithmic") ) return; 
		XmlNode regular = node;
		regular.name("x_regular");
        XCoordinateAttributes::set(regular);
    }
    
    void min(double m) { min_ = m; }
    void max(double m) { max_ = m; }
    
    double operator()(double c ) { return log10(c); }
    
     virtual XCoordinate* clone() const {
    	XLogarithmicCoordinate* x = new XLogarithmicCoordinate();
    	x->copy(*this);
        Log::dev() << "(const map<string, string&)---> to be checked!...\n";
        return x;
    }
    
    void setMin(double min) { XCoordinateAttributes::setMin(min); }
    void setMax(double max) { XCoordinateAttributes::setMax(max); }
    bool automatic() { return automatic_; }

       
protected:
	virtual void print(ostream& out) const  {
		out << "XLogarithmicCoordinate[";
		XCoordinateAttributes::print(out);
		out << "]";
	}
};


class YLogarithmicCoordinate : public LogarithmicCoordinate, public YCoordinateAttributes, public YCoordinate
{
public:
	YLogarithmicCoordinate() {}
	virtual ~YLogarithmicCoordinate() {}
	virtual void set(const XmlNode& node) {
		if ( !magCompare(node.name(), "y_logarithmic") ) return; 
		XmlNode regular = node;
		regular.name("y_regular");
        YCoordinateAttributes::set(regular);
    }
	double min() { return min_; }
    double max() { return max_; }
    double minpc() { return log10(min_); }
    double maxpc() { return log10(max_); }
    
    void min(double m) { min_ = m; }
    void max(double m) { max_ = m; }
    bool automatic() { return automatic_; }
    
    void setMin(double min) { YCoordinateAttributes::setMin(min); }
    void setMax(double max) { YCoordinateAttributes::setMax(max); }
    double operator()(double c ) { return log10(c); }
    virtual YCoordinate* clone() const {
    	YLogarithmicCoordinate* y = new YLogarithmicCoordinate();
    	y->copy(*this);
        return y;
    }

    
   
    
protected:
	virtual void print(ostream& out) const  {
		out << "YLogarithmicCoordinate[";
		YCoordinateAttributes::print(out);
		out << "]";
	}
};


class DateCoordinate 
{
public:
	DateCoordinate() {}
	virtual ~DateCoordinate() {}
};

class XDateCoordinate : public DateCoordinate, public XDateCoordinateAttributes, public XCoordinate
{
public:
	XDateCoordinate() { anna_ = false; }
	virtual ~XDateCoordinate() {}
	
	double min() { return 0; }
    double max() { return *max_ - *min_; }
    	
    double minpc() { return anna_ ? min_anna_ : 0; }
    double maxpc() { return anna_ ? max_anna_ : *max_ - *min_; }
    
    void min(double min) { min_anna_ = min; anna_ = true; }
    void max(double max) { max_anna_ = max; anna_ = true;}
    
    string reference() { return *min_; }
    virtual XCoordinate* clone() const {
    	XDateCoordinate* x = new XDateCoordinate();
    	x->copy(*this);
        return x;
    }
    bool automatic() { return automatic_; }
    void set(const XmlNode& node) { XDateCoordinateAttributes::set(node); }
    void set(const map<string, string>& map) { XDateCoordinateAttributes::set(map); }
    bool accept(const string& xml) { return XDateCoordinateAttributes::accept(xml); }
    
    void setMin(double) { }
    void setMax(double) { }
    
    void setAutomatic(bool automatic) { XDateCoordinateAttributes::setAutomatic(automatic); }

    
    void setMin(DateTime* min) { XDateCoordinateAttributes::setMin(min); }
    void setMax(DateTime* max) { XDateCoordinateAttributes::setMax(max);}
    
    void dataMin(double min, const string& date) { 
    	if ( !automatic_ ) return;
    	DateTime* newmin = new DateTime(date);
    	
    	*newmin = *newmin + Second(min); 
    	
    	//Log::dev()<< "NEW MIN DATE-->" << *newmin << endl;
    	//if ( *newmin < XDateCoordinateAttributes::getMin() ) 
    		setMin(newmin);
    }
    void dataMax(double max, const string& date) { 
    	if ( !automatic_ ) return;
    	DateTime* newmax = new DateTime(date);
    	
    	*newmax = *newmax  + Second(max); ;
    
    	//Log::dev()<< "NEW MAX DATE-->" << *newmax << endl;
    	//if ( *newmax > XDateCoordinateAttributes::getMax() )     		
    		setMax(newmax);
    } 
     
    double operator()(double c ) { return c; }
    
    double operator()(const string& val) const  { 
    	DateTime date(val); 
    	return date - *min_; 
    }

    
	
	
protected:
	virtual void print(ostream& out) const  {
		XDateCoordinateAttributes::print(out);
	}
	// patch for anna! 
	
	double min_anna_;
	double max_anna_;
	bool anna_;
};


class YDateCoordinate : public DateCoordinate, public YDateCoordinateAttributes, public YCoordinate
{
public:
	YDateCoordinate() {}
	virtual ~YDateCoordinate() {}
	double min()   { return 0; }
    double max()   { return *max_ - *min_; }
    double minpc()     { return 0; }
    double maxpc()     { return *max_ - *min_;}
    string reference() { return *min_; }
    
     bool accept(const string& xml) { return YDateCoordinateAttributes::accept(xml); }

    void setAutomatic(bool automatic) { YDateCoordinateAttributes::setAutomatic(automatic); }
    
    virtual YCoordinate* clone() const {
    	YDateCoordinate* y = new YDateCoordinate();
    	y->copy(*this);
        return y;
    }
    bool automatic() { return automatic_; }
    
    void setMin(double) { }
    void setMax(double) { }
    
    void setMin(DateTime* min) { YDateCoordinateAttributes::setMin(min); }
    void setMax(DateTime* max) { YDateCoordinateAttributes::setMax(max);}
    
    double operator()(double c ) { return c; }
    double operator()(const string& val) const  { 
    	DateTime date(val); 
    	return date - *min_; 
    }
    void dataMin(double min, const string& date) { 
    	if ( !automatic_ ) return;
    	DateTime newmin = DateTime(date) + Second(min); ;
    	if ( newmin < YDateCoordinateAttributes::getMin() ) 
    		setMin(new DateTime(newmin));
    }
    void dataMax(double max, const string& date) { 
    	if ( !automatic_ ) return;
    	DateTime newmax = DateTime(date) + Second(max); 
    	if ( newmax > YDateCoordinateAttributes::getMax() ) 
    		setMax(new DateTime(newmax));
    } 

protected:
	virtual void print(ostream& out) const  {
		YDateCoordinateAttributes::print(out);
	}
};



template <>
class Translator<string, XCoordinate> { 
public:
	XCoordinate* operator()(const string& val )
	{
		return SimpleObjectMaker<XCoordinate>::create(val);
	}     

	XCoordinate* magics(const string& param)
	{
		XCoordinate* object;
		ParameterManager::update(param, object);
		return object;
	}
};
template <>
class Translator<string, YCoordinate> { 
public:
	YCoordinate* operator()(const string& val )
	{
		return SimpleObjectMaker<YCoordinate>::create(val);
	}     

	YCoordinate* magics(const string& param)
	{
		YCoordinate* object;
		ParameterManager::update(param, object);
		return object;
	}
};
} // namespace magics
#endif
