/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the license.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Proxy_p.h"

#include <QColor>

#include <GTLCore/Type.h>
#include <GTLCore/TypesManager.h>
#include <GTLCore/Value.h>
#include <OpenShiva/Kernel.h>

#include "ParametersWidgetBase_p.h"


using namespace QtShiva;

Proxy::Proxy( QObject* parent, const GTLCore::String& _name, ParametersWidgetBase::Parametrisation* _kernel, GTLCore::Metadata::ParameterEntry::WidgetType _type ) : QObject( parent ), m_name(_name), m_kernel( _kernel ), m_type(_type)
{
  
}

Proxy::~Proxy()
{
}

void Proxy::setIntValue( int v )
{
  m_kernel->setParameter( m_name, v );
}

void Proxy::setDoubleValue( double v )
{
  m_kernel->setParameter( m_name, (float)v );
}

#include <cstdlib>
void Proxy::setRgb( const QColor& c )
{
  std::vector< GTLCore::Value > cv;
  cv.push_back( c.red() / 255.0f );
  cv.push_back( c.green() / 255.0f );
  cv.push_back( c.blue() / 255.0f );
  m_kernel->setParameter( m_name, GTLCore::Value( cv, GTLCore::TypesManager::getVector( GTLCore::Type::Float, 3 ) ) );
}

void Proxy::setRgba( const QColor& c )
{
  std::vector< GTLCore::Value > cv;
  cv.push_back( c.red() / 255.0f );
  cv.push_back( c.green() / 255.0f );
  cv.push_back( c.blue() / 255.0f );
  cv.push_back( c.alpha() / 255.0f );
  m_kernel->setParameter( m_name, GTLCore::Value( cv, GTLCore::TypesManager::getVector( GTLCore::Type::Float, 4 ) ) );
}

void Proxy::setValue( const GTLCore::Value& value)
{
  switch( m_type )
  {
    case GTLCore::Metadata::ParameterEntry::IntegerWidget:
    {
      emit intValueChanged( value.asInt32() );
    }
      break;
    case GTLCore::Metadata::ParameterEntry::FloatWidget:
    {
      emit doubleValueChanged(value.asFloat());
    }
      break;
    case GTLCore::Metadata::ParameterEntry::CurveWidget:
      
      break;
    case GTLCore::Metadata::ParameterEntry::RgbColorWidget:
    {
      const std::vector< GTLCore::Value >* valArr = value.asArray();
      Q_ASSERT( valArr );
      Q_ASSERT( valArr->size() >= 3 );
      double r = (*valArr)[0].asFloat() * 255;
      double g = (*valArr)[1].asFloat() * 255;
      double b = (*valArr)[2].asFloat() * 255;
      emit( rgbValueChanged(QColor(r,g,b)) );
    }
      break;
    case GTLCore::Metadata::ParameterEntry::RgbaColorWidget:
    {
      const std::vector< GTLCore::Value >* valArr = value.asArray();
      Q_ASSERT( valArr );
      Q_ASSERT( valArr->size() >= 3 );
      double r = (*valArr)[0].asFloat() * 255;
      double g = (*valArr)[1].asFloat() * 255;
      double b = (*valArr)[2].asFloat() * 255;
      double a = (*valArr)[3].asFloat() * 255;
      emit( rgbaValueChanged(QColor(r,g,b,a)) );
    }
      break;
  }
}

#include "Proxy_p.moc"
