// -*- c++ -*-
/***************************************************************************
                           vbimanager.cpp
                           --------------
    begin                : Fri Oct 31 2003
    copyright            : (C) 2003 by Dirk Ziegelmeier
    email                : dziegel@gmx.de
 ***************************************************************************/

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <kdebug.h>

#include "qptrlist.h"

#include "vbimanager.h"
#include "kdetvvbiplugin.h"
#include "pluginfactory.h"

VbiManager::VbiManager(PluginFactory* pf)
    : QObject(),
      _pf(pf),
      _plugin(0L),
      _clients(0),
      _suspendCount(0)
{
}

VbiManager::~VbiManager()
{
    if (_plugin) {
        _plugin->destroy();
        _plugin = 0;
    }
}

void VbiManager::scanPlugins()
{
    if (_plugin) {
        _plugin->destroy();
        _plugin = 0L;
    }

    if (_clients == 0) {
        return;
    }

    QPtrList<PluginDesc>& vbiPlugins(_pf->vbiPlugins());
    for (PluginDesc* plug = vbiPlugins.first();
         plug;
         plug = vbiPlugins.next()) {
        if (plug->enabled) {
            _plugin = _pf->getVbiPlugin(plug, this);
            if(_plugin) {
                break;
            }
        }
    }
}

void VbiManager::addClient()
{
    _clients++;
    if (_clients == 1) {
        scanPlugins();
    }

    if(_plugin) {
        if(_suspendCount > 0) {
            _plugin->resume();
        } else {
            _plugin->suspend();
        }
    }
}

void VbiManager::removeClient()
{
    _clients--;
    Q_ASSERT(_clients >= 0);
    if (_clients == 0 && _plugin) {
        _plugin->destroy();
        _plugin = 0;
    }
}

void VbiManager::restartPlugin()
{
    if (_plugin) {
        _plugin->restart();
    }
}

bool VbiManager::decoding() const
{
    return _plugin ? _plugin->decoding() : false;
}

bool VbiManager::tuned() const
{
    return _plugin ? _plugin->tuned() : false;
}

void* VbiManager::internalData() const
{
    return _plugin ? _plugin->decoder() : 0L;
}

void VbiManager::customEvent(QCustomEvent* e)
{
    switch (e->type()) {
    case EventIdStationName:
        {
            EventStationName* netev = static_cast<EventStationName*>(e);
            emit networkId(netev->_name, netev->_id, netev->_call);
        }
        break;
    case EventIdCaption:
        emit caption(static_cast<EventCaption*>(e)->_pgno);
        break;
    case EventIdTtx:
        {
            EventTtx* ttxev = static_cast<EventTtx*>(e);
            emit ttxPage(ttxev->_pgno, ttxev->_subno, ttxev->_pno, ttxev->_roll, ttxev->_header, ttxev->_clock);
        }
        break;
    case EventIdAspect:
        {
            EventAspect* aspev = static_cast<EventAspect*>(e);
            emit aspect(aspev->_fline, aspev->_lline, aspev->_ratio, aspev->_mode, aspev->_subtitles);
        }
        break;
    case EventIdProgTitle:
        emit progTitle(static_cast<EventProgTitle*>(e)->_title);
        break;
    case EventIdRating:
        emit ratingChanged(static_cast<EventRating*>(e)->_rating);
        break;
    case EventIdRunning:
        emit running(static_cast<EventRunning*>(e)->_running);
        break;
    default:
        break;
    }
}

void VbiManager::suspend()
{
    _suspendCount--;
    if((_suspendCount == 0) && (_plugin)) {
        _plugin->suspend();
    }
}

void VbiManager::resume()
{
    _suspendCount++;
    if((_suspendCount == 1) && (_plugin)) {
        _plugin->resume();
    }
}


// --------------------------------------------------------------- Event implementation

EventStationName::EventStationName(const QString& name, int id, const QString& call)
    : QCustomEvent(EventIdStationName),
      _name(name),
      _id(id),
      _call(call)
{
}

EventStationName::~EventStationName()
{
}

EventCaption::EventCaption(int pgno)
    : QCustomEvent(EventIdCaption),
      _pgno(pgno)
{
}

EventCaption::~EventCaption()
{
}

EventTtx::EventTtx(int pgno, int subno, int pno,
                   bool roll, bool header, bool clock)
    : QCustomEvent(EventIdTtx),
      _pgno(pgno),
      _subno(subno),
      _pno(pno),
      _roll(roll),
      _header(header),
      _clock(clock)
{
}

EventTtx::~EventTtx()
{
}

EventAspect::EventAspect(int fline, int lline, double ratio, bool mode, int subtitles)
    : QCustomEvent(EventIdAspect),
      _fline(fline),
      _lline(lline),
      _ratio(ratio),
      _mode(mode),
      _subtitles(subtitles)
{
}

EventAspect::~EventAspect()
{
}

EventProgTitle::EventProgTitle(const QString& title)
    : QCustomEvent(EventIdProgTitle),
      _title(title)
{
}

EventProgTitle::~EventProgTitle()
{
}

EventRating::EventRating(const QString& rating)
    : QCustomEvent(EventIdRating),
      _rating(rating)
{
}

EventRating::~EventRating()
{
}

EventRunning::EventRunning(bool running)
    : QCustomEvent(EventIdRunning),
      _running(running)
{
}

EventRunning::~EventRunning()
{
}

#include "vbimanager.moc"

// vim: sw=4 ts=4 noet
