///////
   //    SchedulerEntry .cc
   //    Scheduler Entry Class definitions
   //
   //    Class for interfacing the Schedule table of the DB
   //
   //    Copyright (c) 1999-2004 Comune di Prato - Prato - Italy
   //    Some Portions Copyright (c) 2008 Devise.IT srl <http://www.devise.it/>
   //    Author: Gabriele Bartolini - Prato - Italy <angusgb@users.sourceforge.net>
   //
   //    For copyright details, see the file COPYING in your distribution
   //    or the GNU General Public License version 2 or later 
   //    <http://www.gnu.org/copyleft/gpl.html>
   //
   //    $Id: SchedulerEntry.cc,v 1.24 2008-11-16 18:28:52 angusgb Exp $
   //
   //    G.Bartolini
   //    started: 14.09.1999
///////

#ifdef HAVE_CONFIG_H
#include "htconfig.h"
#endif /* HAVE_CONFIG_H */

#include "_Server.h"
#include "SchedulerEntry.h"

   // Static variables initialization
      int SchedulerEntry::debug = 0;

///////
   //    Construction
///////

// Default constructor
SchedulerEntry::SchedulerEntry()
: 
IDSchedule(0),
IDServer(0),
server(0),
ScheduleUrl(),
Status(Url_Empty),
Domain(Url_Unknown),
IDReferer(0),
HopCount(0),
is_malformed(0)
{
}

// From a string containing an URL
SchedulerEntry::SchedulerEntry(const std::string &u)
: 
IDSchedule(0),
IDServer(0),
server(0),
ScheduleUrl(),
Status(Url_Empty),
Domain(Url_Unknown),
IDReferer(0),
HopCount(0),
is_malformed(0)
{
   SetNewUrl(u);
}

///////
   //    Destruction
///////

SchedulerEntry::~SchedulerEntry ()
{
}

///////
   //    Reset the schedule content
///////

void SchedulerEntry::Reset()
{
   ScheduleUrl.clear();
   IDSchedule=0;
   IDServer=0;
   IDReferer=0;
   HopCount=0;
   server=0;
   Status=Url_Empty;
   Domain=Url_Unknown;
   is_malformed=false;
}


///////
   //    Set the new url to the schedule
///////

void SchedulerEntry::SetNewUrl(const std::string &u)
{
   Reset();
   SetScheduleUrl(u);
}


///////
   //    Converts the Schedule_Status value into the corresponding
   //    string value
///////

void SchedulerEntry::RetrieveStatus(std::string &Status) const
{

   switch(GetStatus())
   {
      case (Url_Empty):
         Status.clear();
         break;
      case (Url_ToBeRetrieved):
         Status="ToBeRetrieved";
         break;
      case (Url_Retrieved):
         Status="Retrieved";
         break;
      case (Url_CheckIfExists):
         Status="CheckIfExists";
         break;
      case (Url_Checked):
         Status="Checked";
         break;
      case (Url_BadQueryString):
         Status="BadQueryString";
         break;
      case (Url_BadExtension):
         Status="BadExtension";
         break;
      case (Url_MaxHopCount):
         Status="MaxHopCount";
         break;
      case (Url_FileProtocol):
         Status="FileProtocol";
         break;
      case (Url_EMail):
         Status="EMail";
         break;
      case (Url_Javascript):
         Status="Javascript";
         break;
      case (Url_NotValidService):
         Status="NotValidService";
         break;
      case (Url_Malformed):
         Status="Malformed";
         break;
      case (Url_MaxUrlsCount):
         Status="MaxUrlsCount";
         break;
   }

}


///////
   //    Converts the Schedule_Domain value into the corresponding
   //    string value
///////

void SchedulerEntry::RetrieveDomain(std::string &Domain) const
{

   switch(GetDomain())
   {
      case (Url_Unknown):
         Domain.clear();
         break;
      case (Url_Internal):
         Domain="Internal";
         break;
      case (Url_External):
         Domain="External";
         break;
   }

}


///////
   //    Set the server ID, given a server
   //    if the server is null, ID is set to 0
///////

void SchedulerEntry::SetServer (_Server *s)
{
   server = s;
   if (s) IDServer = s->GetID();
   else IDServer = 0;
}


///////
   //    Converts the Status string value
   //    into the corresponding Schedule_Status value 
   //    Returns 0 if an error occurs, 1 if OK.
///////

int SchedulerEntry::SetStatus(const std::string &Status)
{

   if (!Status.length())
      SetStatus (Url_Empty);
   if (Status == "ToBeRetrieved")
      SetStatus (Url_ToBeRetrieved);
   else if (Status == ("Retrieved"))
      SetStatus (Url_Retrieved);
   else if (Status == ("CheckIfExists"))
      SetStatus (Url_CheckIfExists);
   else if (Status == ("Checked"))
      SetStatus (Url_Checked);
   else if (Status == ("BadQueryString"))
      SetStatus (Url_BadQueryString);
   else if (Status == ("BadExtension"))
      SetStatus (Url_BadExtension);
   else if (Status == ("MaxHopCount"))
      SetStatus (Url_MaxHopCount);
   else if (Status == ("FileProtocol"))
      SetStatus (Url_FileProtocol);
   else if (Status == ("EMail"))
      SetStatus (Url_EMail);
   else if (Status == ("Javascript"))
      SetStatus (Url_Javascript);
   else if (Status == ("NotValidService"))
      SetStatus (Url_NotValidService);
   else if (Status == ("Malformed"))
      SetStatus (Url_Malformed);
   else if (Status == ("MaxUrlsCount"))
      SetStatus (Url_MaxUrlsCount);
   else return 0;
   
   return 1;
}


///////
   //    Converts the Domain string value
   //    into the corresponding Schedule_Domain value 
   //    Returns 0 if an error occurs, 1 if OK.
///////

int SchedulerEntry::SetDomain(const std::string &Domain)
{

   if (!Domain.length())
      SetDomain (Url_Unknown);
   if (Domain == "Internal")
      SetDomain (Url_Internal);
   else if (Domain == "External")
      SetDomain (Url_External);
   else return 0;
   
   return 1;
}


///////
   //    Output SchedulerEntry object
///////
ostream& operator<<( ostream &output, const SchedulerEntry &s)
{
   output << s.ScheduleUrl;
   
   if (s.debug < 3) return output;
   
   // Only if debug level is greater than 2
   
   output << " (ID: " << s.IDSchedule << " / "
      << "IDServer: " << s.IDServer << " / ";

   std::string Status("");

   s.RetrieveStatus(Status);

   if (!Status.length())
      Status = "Not yet assigned";
      
   std::string Domain("");

   s.RetrieveDomain(Domain);

   if (!Domain.length())
      Domain = "Not yet assigned";
      
   output << (int) s.Status << " - " << Status << " / "
      << (int) s.Domain << " - " << Domain << " )";
   
   if (s.server)
      output << " Server: " << s.server->host() << ":" << s.server->port();
   
   return output;
}
