/*
 * GPG Keys 
 * ckeyinfo.cpp
 * (c) 2001 Peter Mathiasson <peter@mathiasson.nu>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation
 */

#include <qstatusbar.h>
#include <qlineedit.h>
#include <qlayout.h>
#include <qfont.h>
#include <qprocess.h>
#include <qlistview.h>
#include <qpopupmenu.h>
#include "cdlgserversearch.h"
#include "clistviewitem.h"
#include "cdlgkeyinfo.h"
#include "functions.h"
#include "config.h"

/* --------------------------------------------------------------------------------- */

CDlgKeyInfo::CDlgKeyInfo( QWidget *parent, const QString tmpKeyId, const QString tmpUserId ) :
CDlgKeyInfoUI( parent, 0, TRUE )
{

    userId = tmpUserId;
    keyId = tmpKeyId;
    
    /* Set Font */
    QFont fixedFont = readConfigFont( "fixedfont", FIX_FONT );
    setFont( readConfigFont( "stdfont", STD_FONT ) );
    lvSubKeys->setFont( fixedFont );
    lvUserID->setFont( fixedFont );

    /* Create Popup Menu */
    popupMenu = new QPopupMenu( this );
    popupMenu->insertItem( tr( "&Lookup Key" ), this, SLOT(lookupKey()), CTRL+Key_L );

    /* UserID ListView */
    connect( lvUserID, SIGNAL(contextMenuRequested(QListViewItem*,const QPoint&,int)),
             this, SLOT(contextMenuRequested(QListViewItem*,const QPoint&,int)) );

    /* Fetch key information */
    getKeyInfo();

}

/* --------------------------------------------------------------------------------- */

void CDlgKeyInfo::getKeyInfo()
{
    /* Clear ListViews */
    lvUserID->clear();
    lvSubKeys->clear();
    
    /* Set Caption */
    setCaption( userId + tr( " (Fetching Information...)" ) );

    /* Create Process */
    gpgProc = new QProcess( this );
    gpgProc->addArgument( readConfigString( "gpgpath", GPGPATH ) );
    gpgProc->addArgument( "--list-sigs" );
    gpgProc->addArgument( "--fingerprint" );
    gpgProc->addArgument( "--with-colon" );
    gpgProc->addArgument( "0x" + keyId );
    connect( gpgProc, SIGNAL(processExited()), this, SLOT(processDone()) );
    connect( gpgProc, SIGNAL(readyReadStdout()), this, SLOT(readFromStdout()) );
    
    /* Run GPG */
    if ( !gpgProc->start() ) {        
        return;
    }
}

/* --------------------------------------------------------------------------------- */

void CDlgKeyInfo::processDone()
{
    setCaption( userId );
}

/* --------------------------------------------------------------------------------- */

void CDlgKeyInfo::readFromStdout()
{
    QStringList entry;
    static bool publicSig;

    while ( gpgProc->canReadLineStdout() ) {
        
        /* Read/Split Line */
        entry = QStringList::split( ':', gpgProc->readLineStdout(), TRUE );

        /* Public Key */
        if ( entry[0] == "pub" ) {

            /* Set Information */
            publicSig = TRUE;
            editKeyID->setText( entry[4] );
            editTrust->setText( getTrust( entry[1] ) );
            editType->setText( entry[2] + "/" + getCipher( entry[3] ) );
            editDate->setText( entry[5] );
            editExpiration->setText( entry[6] );

            /* Create UserID */
            itemUid = new CListViewItem( lvUserID, CListViewItem::USERID );
            itemUid->setText( 0, fromUtf8( entry[9] ) );
            itemUid->setText( 1, getTrust( entry[1] ) );

        }

        /* User ID */
        else if ( entry[0] == "uid" ) {
            itemUid = new CListViewItem( lvUserID, CListViewItem::USERID );
            itemUid->setText( 0, fromUtf8( entry[9] ) );
            itemUid->setText( 1, getTrust( entry[1] ) );
        }

        /* Fingerprint */
        else if ( entry[0] == "fpr" ) {

            /* Set Fingerprint */
            editFingerprint->setText( entry[9] );

        }

        /* Sub Key */
        else if ( entry[0] == "sub" ) {

            /* Add ListView Item */
            QListViewItem *item = new QListViewItem( lvSubKeys );
            item->setText( 0, entry[4] );
            item->setText( 1, entry[2] + "/" + getCipher( entry[3] ) );
            item->setText( 2, entry[5] );
            item->setText( 3, entry[6] );
            publicSig = FALSE;

        }

        /* Signature */
        else if ( entry[0] == "sig" && publicSig ) {
            CListViewItem *item = new CListViewItem( itemUid, CListViewItem::SIGNATURE );
            item->setText( 0, fromUtf8( entry[9] ) );
            item->setText( 2, getCipher( entry[3] ) );
            item->setText( 3, entry[5] );            
            item->setText( 4, entry[4] );
        }
        
    }
}

/* --------------------------------------------------------------------------------- */

void CDlgKeyInfo::contextMenuRequested( QListViewItem *tmpItem, const QPoint &pos, int )
{
    /* Abort if no item selected */
    if ( !tmpItem )
        return;

    /* Display Menu */
    CListViewItem *item = (CListViewItem*)tmpItem;
    if ( item->type == CListViewItem::SIGNATURE )
        popupMenu->exec( pos );
}

/* --------------------------------------------------------------------------------- */

void CDlgKeyInfo::lookupKey()
{
    /* Get Item */
    CListViewItem *item = (CListViewItem*)lvUserID->currentItem();

    /* Check Item */
    if ( item->type != CListViewItem::SIGNATURE )
        return;

    /* Display Dialog */
    CDlgServerSearch dlgServerSearch( this, "0x" + item->text(4).right(8) );
    connect( &dlgServerSearch, SIGNAL(newPublic()), this, SLOT(reReadInfo()) );
    dlgServerSearch.exec();
}

/* --------------------------------------------------------------------------------- */

void CDlgKeyInfo::reReadInfo()
{
    getKeyInfo();
    emit newPublic();
}

/* --------------------------------------------------------------------------------- */
