/*
 * Portions of this file Copyright 1999-2005 University of Chicago
 * Portions of this file Copyright 1999-2005 The University of Southern California.
 *
 * This file or a portion of this file is licensed under the
 * terms of the Globus Toolkit Public License, found at
 * http://www.globus.org/toolkit/download/license.html.
 * If you redistribute this file, with or without
 * modifications, you must include this notice in the file.
 */

/******************************************************************************
globus_duct_control.c

Description:
    globus_duct control API

CVS Information:

    $Source: /home/globdev/CVS/globus-packages/duct/control/source/globus_duct_control.c,v $
    $Date: 2005/04/18 21:29:28 $
    $Revision: 1.4 $
    $Author: smartin $
******************************************************************************/

/******************************************************************************
                             Include header files
******************************************************************************/
#include "globus_common.h"    /* for globus_list_t */

#include <assert.h>
#include <stdio.h>

#include "globus_nexus.h"
#include "globus_duct_control.h"
#include "globus_duct_common.h"

#include <string.h>
#include "version.h"



/******************************************************************************
                               Type definitions
******************************************************************************/

typedef struct globus_duct_checkin_s {
  int                addr;
  int                min_data_protocol_version;
  int                max_data_protocol_version;
  int                min_config_protocol_version;
  int                max_config_protocol_version;
  globus_nexus_startpoint_t data_sp;
  globus_nexus_startpoint_t config_sp;
} globus_duct_checkin_t;


/******************************************************************************
                          Module specific prototypes
******************************************************************************/

static
void s_checkin_msg_handler (globus_nexus_endpoint_t * endpointp,
			    globus_nexus_buffer_t   * bufferp,
			    globus_bool_t       is_non_threaded_handler);

static
void s_abort_req_handler (globus_nexus_endpoint_t * endpointp,
			  globus_nexus_buffer_t   * bufferp,
			  globus_bool_t       is_non_threaded_handler);

/******************************************************************************
                       Define module specific variables
******************************************************************************/

globus_module_descriptor_t globus_duct_control_module = 
{
  "globus_duct_control",
  globus_duct_control_activate,
  globus_duct_control_deactivate,
  GLOBUS_NULL /* at exit */,
  GLOBUS_NULL, /* get_pointer_func */
  &local_version
};

static
globus_nexus_handler_t s_checkin_handlert[] =
{
  {
    NEXUS_HANDLER_TYPE_THREADED,
    (globus_nexus_handler_func_t) s_checkin_msg_handler
  },
  {
    NEXUS_HANDLER_TYPE_THREADED,
    (globus_nexus_handler_func_t) s_abort_req_handler
  }
};
/* KEEP THIS TABLE CONSISTENT WITH globus_duct_runtime.c :
 * CHECKIN_MSG_ID  = 0 
 * ABORT_REQ_ID    = 1 */

#define CHECKIN_HANDLERT_SIZE 2


/* KEEP THIS DEFINITION CONSISTENT WITH duct-runtime.c */

#define CONFIG_GROUP_MSG_ID 0
#define ABORT_MSG_ID        1
#define PROTO_NEGOTIATE_ID  2

/******************************************************************************
Function:	globus_duct_control_activate()
Description:
Parameters:
Returns:
******************************************************************************/
int 
globus_duct_control_activate (void)
{
  if ( globus_module_activate (GLOBUS_COMMON_MODULE) != GLOBUS_SUCCESS ) 
    goto activate_common_module_error;

  if ( globus_module_activate (GLOBUS_THREAD_MODULE) != GLOBUS_SUCCESS )
    goto activate_thread_module_error;

  if ( globus_module_activate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    goto activate_globus_nexus_module_error;

  return GLOBUS_SUCCESS;


activate_globus_nexus_module_error:
  globus_module_deactivate (GLOBUS_THREAD_MODULE);
  
activate_thread_module_error:
  globus_module_deactivate (GLOBUS_COMMON_MODULE);

activate_common_module_error:
  return GLOBUS_FAILURE;
}

/******************************************************************************
Function:	globus_duct_control_deactivate()
Description:
Parameters:
Returns:
******************************************************************************/
int
globus_duct_control_deactivate (void)
{
  int rc;

  rc = GLOBUS_SUCCESS;

  if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    rc = GLOBUS_FAILURE;

  if ( globus_module_deactivate (GLOBUS_THREAD_MODULE) != GLOBUS_SUCCESS )
    rc = GLOBUS_FAILURE;

  if ( globus_module_deactivate (GLOBUS_COMMON_MODULE) != GLOBUS_SUCCESS )
    rc = GLOBUS_FAILURE;

  return rc;
}

static
int s_checkin_approval (void               * void_controlp,
			char               * ignored_url,
			globus_nexus_startpoint_t * sp)
{
  int err;
  globus_duct_control_t * controlp;

  if ( ignored_url == NULL ) {
    /* suppress 'unused' warning from ignored parameter */
    ignored_url = NULL;
  }

  controlp = ((globus_duct_control_t *) void_controlp);

  err = globus_mutex_lock (&(controlp->mutex)); 
  assert (!err);

  err = globus_nexus_startpoint_copy (sp, &(controlp->checkin_port.sp));
  assert (!err);

  err = globus_mutex_unlock (&(controlp->mutex)); 
  assert (!err);

  return 0;
}


/******************************************************************************
Function:	globus_duct_control_init()
Description:
Parameters:
Returns:
******************************************************************************/
/* prepare to control a new group of size 'size'
 *   size > 0: coordinate exactly size members
 *   size = 0: size will be provided by subsequent set_groupsize operation
 *   size < 0: reserved for future use
 * report asynchronous group configuration by calling 'callback'
 */
int 
globus_duct_control_init (globus_duct_control_t             * controlp,
			  int                                 size,
			  globus_duct_configured_callback_t   callback,
			  void                              * callback_userdata)
{
  int err;

  if ( controlp==NULL ) return GLOBUS_DUCT_ERROR_INVALID_PARAMETER;

  err = globus_mutex_init (&(controlp->mutex), NULL); assert (!err);

  controlp->aborted = 0;
  controlp->next_free_addr = 1;
  controlp->checkins = NULL;
  controlp->size = size;
  controlp->callback = callback;
  controlp->callback_userdata = callback_userdata;

  err = globus_nexus_endpointattr_init (&(controlp->checkin_port.epattr));
  assert (!err);

  err = globus_nexus_endpointattr_set_handler_table (
					&(controlp->checkin_port.epattr),
					s_checkin_handlert,
					CHECKIN_HANDLERT_SIZE);
  assert (!err);

  err = globus_nexus_endpoint_init (&(controlp->checkin_port.ep),
				    &(controlp->checkin_port.epattr));
  assert (!err);

  globus_nexus_endpoint_set_user_pointer (&(controlp->checkin_port.ep),
					  (void *) controlp);
  
  err = globus_nexus_startpoint_bind (&(controlp->checkin_port.sp),
				      &(controlp->checkin_port.ep));
  assert (!err);

  {
    char *hostname;

    (controlp->checkin_port.portno) = 0;

    err = globus_nexus_allow_attach (&(controlp->checkin_port.portno),
				     &(hostname),
				     s_checkin_approval,
				     (void *) controlp);
    assert (!err);

    controlp->checkin_port.attach_url 
      = globus_malloc (sizeof (char)
		       * ( strlen (hostname)
			   + strlen ("x-nexus:///")
			   + 8 /* portno */
			   + 1 /* nul */ ));
    assert ( (controlp->checkin_port.attach_url) != NULL );

    err = globus_libc_sprintf ((controlp->checkin_port.attach_url),
			 "x-nexus://%s:%d/",
			 hostname,
			 (controlp->checkin_port.portno));
    assert (err>0);
  }
			    

  return GLOBUS_DUCT_SUCCESS;
}


static void 
s_configure_group (globus_list_t * checkins, globus_duct_control_t *controlp)
{
  int err;
  globus_list_t * config_target;
  globus_list_t * checkins_iter;
  int min_data_protocol_version;
  int max_data_protocol_version;
  int min_config_protocol_version;
  int max_config_protocol_version;

  /* compute protocol versioning */
  checkins_iter = checkins;
  min_data_protocol_version = -1;
  max_data_protocol_version = -1;
  min_config_protocol_version = GLOBUS_DUCT_CONFIG_PROTOCOL_MIN_VERSION;
  max_config_protocol_version = GLOBUS_DUCT_CONFIG_PROTOCOL_MAX_VERSION;
  while (! globus_list_empty (checkins_iter) ) {
    if ( (min_data_protocol_version < 0)
	 || (min_data_protocol_version 
	     < (((globus_duct_checkin_t *) 
		 globus_list_first (checkins_iter))
		->min_data_protocol_version)) ) {
      min_data_protocol_version = (((globus_duct_checkin_t *) 
				    globus_list_first (checkins_iter))
				   ->min_data_protocol_version);
    }
    if ( (max_data_protocol_version < 0)
	 || (max_data_protocol_version 
	     > (((globus_duct_checkin_t *) 
		 globus_list_first (checkins_iter))
		->max_data_protocol_version)) ) {
      max_data_protocol_version = (((globus_duct_checkin_t *) 
				    globus_list_first (checkins_iter))
				   ->max_data_protocol_version);
    }

    if ( (min_config_protocol_version < 0)
	 || (min_config_protocol_version 
	     < (((globus_duct_checkin_t *) 
		 globus_list_first (checkins_iter))
		->min_config_protocol_version)) ) {
      min_config_protocol_version = (((globus_duct_checkin_t *) 
				      globus_list_first (checkins_iter))
				     ->min_config_protocol_version);
    }
    if ( (max_config_protocol_version < 0)
	 || (max_config_protocol_version 
	     > (((globus_duct_checkin_t *) 
		 globus_list_first (checkins_iter))
		->max_config_protocol_version)) ) {
      max_config_protocol_version = (((globus_duct_checkin_t *) 
				      globus_list_first (checkins_iter))
				     ->max_config_protocol_version);
    }

    checkins_iter = globus_list_rest (checkins_iter);
  }

  if ( min_data_protocol_version > max_data_protocol_version ) {
    /* couldn't find a protocol version all nodes could use! */
    globus_duct_control_abort (controlp, 
			       GLOBUS_DUCT_ERROR_DATA_PROTOCOL_MISMATCH);
  }
  else if ( (min_config_protocol_version > max_config_protocol_version) ) {
    /* couldn't find a protocol version all nodes could use! */
    globus_duct_control_abort (controlp, 
			       GLOBUS_DUCT_ERROR_CONFIG_PROTOCOL_MISMATCH);
  }
  else {
    /* all checkins can use max_*_protocol_version */

    /* send a configuration message to each checked-in participant */
    config_target = checkins;
    
    while ( ! globus_list_empty (config_target) ) {
      globus_nexus_buffer_t buffer;

      err = globus_nexus_buffer_init (&buffer, 0, 0); assert (!err);

      /* include all remote participants in each config message */
      checkins_iter = checkins;

      err = nxbuff_put_int (&buffer, max_config_protocol_version);
      assert (!err);
      err = nxbuff_put_int (&buffer, max_data_protocol_version);
      assert (!err);

      /* target's local_address */
      err = nxbuff_put_int (&buffer, 
			    ((globus_duct_checkin_t *) 
			     globus_list_first (config_target))
			    ->addr);
      assert (!err);
	
      /* target's remote_count */
      err = nxbuff_put_int (&buffer, globus_list_size (checkins_iter) - 1);
      assert (!err);
    
      while ( ! globus_list_empty (checkins_iter) ) {
	if ( globus_list_first (checkins_iter) 
	     != globus_list_first (config_target) ) {
	  globus_nexus_startpoint_t sp_copy;
	  
	  err = nxbuff_put_int (&buffer, (((globus_duct_checkin_t *) 
					   globus_list_first (checkins_iter))
					  ->addr));
	  assert (!err);
	  
	  err = globus_nexus_startpoint_copy (&sp_copy,
					      &(((globus_duct_checkin_t *) 
						 globus_list_first (checkins_iter))
						->data_sp));
	  err = nxbuff_put_startpoint_transfer (&buffer, &sp_copy);
	  assert (!err);
	}
	
	checkins_iter = globus_list_rest (checkins_iter);
      }
    
      err = globus_nexus_send_rsr (&buffer, &(((globus_duct_checkin_t *) 
					       globus_list_first (config_target))
					      ->config_sp),
				   CONFIG_GROUP_MSG_ID,
				   NEXUS_TRUE /* destroy buffer */,
				   NEXUS_TRUE /* always safe */);

      if (err) {
	/* fatal error condition!  group configure failed so now attempt
	 * to propogate fatal error information for this DUCT.
	 * (other nodes may want to give up rather than hang)
	 */
	
	globus_duct_control_abort (controlp,
				   GLOBUS_DUCT_ERROR_SEND_FAILED);
      }
	
      globus_nexus_startpoint_flush (&(((globus_duct_checkin_t *) 
					globus_list_first (config_target))
				       ->config_sp));
      
      config_target = globus_list_rest (config_target);
    }    
  }
}


static void 
s_checkin_msg_handler (globus_nexus_endpoint_t * endpointp,
		       globus_nexus_buffer_t   * bufferp,
		       globus_bool_t       ignored_is_non_threaded_handler)
{
  int              err;
  int              checkin_protocol_version;
  globus_duct_checkin_t * checkin;
  globus_duct_control_t * controlp;

  if ( ignored_is_non_threaded_handler == GLOBUS_TRUE ) {
    /* suppress 'unused' warning from ignored parameter */
    ignored_is_non_threaded_handler = GLOBUS_TRUE;
  }

  checkin = globus_malloc (sizeof (globus_duct_checkin_t));
  assert (checkin!=NULL);

  controlp 
    = (globus_duct_control_t *) globus_nexus_endpoint_get_user_pointer (endpointp);
  assert (controlp!=NULL);

  err = nxbuff_get_int (bufferp, &(checkin_protocol_version)); assert (!err);
  err = nxbuff_get_startpoint (bufferp, &(checkin->config_sp)); assert (!err);
  err = nxbuff_get_int (bufferp, &(checkin->min_config_protocol_version)); 
  assert (!err);
  err = nxbuff_get_int (bufferp, &(checkin->max_config_protocol_version)); 
  assert (!err);

  if ( (checkin_protocol_version 
	< GLOBUS_DUCT_CHECKIN_PROTOCOL_MIN_VERSION)
       || (checkin_protocol_version 
	   > GLOBUS_DUCT_CHECKIN_PROTOCOL_MAX_VERSION) ) {
    /* tell the runtime node to redo checkin and tell it which
     * protocols we can support. */
    globus_nexus_buffer_t     buffer;

    err = globus_nexus_buffer_init (&buffer, 0, 0); assert (!err);

    err = nxbuff_put_int (&buffer, GLOBUS_DUCT_CHECKIN_PROTOCOL_MIN_VERSION);
    assert (!err);
    err = nxbuff_put_int (&buffer, GLOBUS_DUCT_CHECKIN_PROTOCOL_MAX_VERSION);
    assert (!err);

    err = globus_nexus_send_rsr (&buffer, &(checkin->config_sp),
				 PROTO_NEGOTIATE_ID,
				 NEXUS_TRUE /* destroy buffer */,
				 NEXUS_TRUE /* always safe */);

    globus_nexus_startpoint_flush (&(checkin->config_sp));
    globus_nexus_startpoint_destroy (&(checkin->config_sp));
    globus_free (checkin);
    checkin = NULL;
  }
  else {
    /* we can handle this checkin message */
    /* right now MAX_VERSION==MIN_VERSION, so none of the following
     * buffer unpacking code is conditionalized on protocol version.
     * this will change if revised protocols are introduced. */

    err = nxbuff_get_startpoint (bufferp, &(checkin->data_sp)); assert (!err);

    err = nxbuff_get_int (bufferp, &(checkin->min_data_protocol_version)); 
    assert (!err);
    err = nxbuff_get_int (bufferp, &(checkin->max_data_protocol_version)); 
    assert (!err);

    err = globus_mutex_lock (&(controlp->mutex)); assert (!err);
    
    checkin->addr = controlp->next_free_addr++;
    
    err = globus_list_insert (&(controlp->checkins), (void *) checkin);
    assert (!err);

    err = globus_mutex_unlock (&(controlp->mutex)); assert (!err);

    checkin = NULL;
  }
 
  err = globus_mutex_lock (&(controlp->mutex)); assert (!err);
  if ( controlp->aborted ) {
    /* use this opportunity to propogate abort signals */
    
    err = globus_mutex_unlock (&(controlp->mutex)); assert (!err);

    globus_duct_control_abort (controlp,
			       controlp->abort_reason);
  }
  else if ( (controlp->size > 0)
	    && (controlp->size == globus_list_size (controlp->checkins)) ) {
    /* all expected checkins have been made.. 
     * so send group configuration commands */
    globus_list_t *checkins;
    
    /* capture current consistent snapshot of list for
     * group configuration, _before_ releasing control lock 
     * subsequent checkins will be ignored by this group
     * configuration pass */
    checkins = globus_list_copy (controlp->checkins);
      
    err = globus_mutex_unlock (&(controlp->mutex)); assert (!err);
    
    s_configure_group (checkins, controlp);

    /* free copy of list */
    globus_list_free (checkins);
  }
  else {
    
    err = globus_mutex_unlock (&(controlp->mutex)); assert (!err);
  }

  globus_nexus_buffer_destroy (bufferp);
}

static void 
s_abort_req_handler (globus_nexus_endpoint_t * endpointp,
		     globus_nexus_buffer_t   * bufferp,
		     globus_bool_t       ignored_is_non_threaded_handler)
{
  int              err;
  int reason;
  globus_duct_control_t * controlp;

  if ( ignored_is_non_threaded_handler == GLOBUS_TRUE ) {
    /* suppress 'unused' warning from ignored parameter */
    ignored_is_non_threaded_handler = GLOBUS_TRUE;
  }

  assert (endpointp!=NULL); assert (bufferp!=NULL);

  controlp = ((globus_duct_control_t *) 
	      globus_nexus_endpoint_get_user_pointer (endpointp));
  assert (controlp!=NULL);

  err = nxbuff_get_int (bufferp, &(reason)); assert (!err);

  globus_duct_control_abort (controlp, reason);

  globus_nexus_buffer_destroy (bufferp);
}


/*
 * set size for an already created control
 *   size >= 0: coordinate exactly size members
 *   size < 0: reserved for future use
 */
int 
globus_duct_control_set_groupsize (globus_duct_control_t * controlp,
				   int                     size)
{
  int err;

  if (controlp==NULL) return GLOBUS_DUCT_ERROR_INVALID_PARAMETER;


  err = globus_mutex_lock (&(controlp->mutex)); assert (!err);


  assert (size >= 0);

  controlp->size = size;

  assert (! (controlp->size < globus_list_size (controlp->checkins)));

  if ( (controlp->size == globus_list_size (controlp->checkins)) ) {
    /* all expected checkins have been made.. 
     * so send group configuration commands */
    globus_list_t *checkins;

    /* capture current consistent snapshot of list for
     * group configuration, _before_ releasing control lock 
     * subsequent checkins will be ignored by this group
     * configuration pass */
    checkins = globus_list_copy (controlp->checkins);


    err = globus_mutex_unlock (&(controlp->mutex)); assert (!err);


    s_configure_group (checkins, controlp);

    /* free copy of list */
    globus_list_free (checkins);
  }  
  else {
    
    err = globus_mutex_unlock (&(controlp->mutex)); assert (!err);

  }

  return GLOBUS_DUCT_SUCCESS;
}


/*
 * create a linearized startpoint 'contact' for this control
 */
int
globus_duct_control_contact_lsp (globus_duct_control_t  * controlp,
				 char                  ** contact)
{
  int                  err;
  globus_nexus_startpoint_t   sp_copy;
  globus_byte_t        bbuff[GLOBUS_DUCT_MAX_MSG_LENGTH];
  globus_byte_t      * ptr;
  int                  len;

  len = 0;

  err = globus_nexus_startpoint_copy (&sp_copy, 
			       &(controlp->checkin_port.sp)); 
  assert (!err);

  ptr = bbuff;
  globus_nexus_stdio_lock ();
  sprintf ((char *)ptr, "%d", globus_nexus_user_format());
  globus_nexus_stdio_unlock ();
  while ( (*ptr)!='\0' ) ptr++; ptr++; /* d d ... d \0 MESG */
  globus_nexus_user_put_startpoint_transfer (&ptr, &sp_copy, 1);
  len = (int) (ptr - bbuff);
  assert (len<=GLOBUS_DUCT_MAX_MSG_LENGTH);

  /* now hex-encode the buffer and prepend 'LSP' */
  
  (*contact) = globus_malloc (sizeof(char) * (3 /* LSP */
					     + (2 * len) + 1 /* hex buff */
					     + 1 /* \0 */));
  assert ( (*contact)!=NULL );

  globus_nexus_stdio_lock ();
  err = sprintf ( (*contact), "LSP"); assert (err==3);
  globus_nexus_stdio_unlock ();
  _nx_hex_encode_byte_array (bbuff, len, (*contact)+3);
  
  /* contact has form:
   *   >L S P hd hd ... hd<
   * "LSP" prefix identifies this as linearized startpoint
   * hex digit substring should be hex_decoded to obtain:
   *   >d d ... d \0 user-sp<
   * "d d ... d" is the user-buffer format
   * user-sp is the startpoint
   */
  return GLOBUS_DUCT_SUCCESS;
}

  

/*
 * create an attachment contact for this control
 */
int 
globus_duct_control_contact_url (globus_duct_control_t  * controlp,
				 char                  ** contact)
{
  if ( (contact==NULL)
       || (controlp==NULL) ) return GLOBUS_DUCT_ERROR_INVALID_PARAMETER;

  (*contact) = globus_malloc (sizeof (char)
			      * ( strlen (controlp->checkin_port.attach_url)
				  + strlen ("URL")
				  + 1));
  assert ((*contact)!=NULL);

  sprintf ((*contact),
	   "URL%s",
	   (controlp->checkin_port.attach_url));

  return GLOBUS_DUCT_SUCCESS;
}


/* 
 * tell known group members there was an error
 */
void
globus_duct_control_abort (globus_duct_control_t * controlp,
			   int reason)
{
  int err;
  globus_list_t * config_target;

  err = globus_mutex_lock (&(controlp->mutex));
  assert (!err);

  controlp->aborted = 1;
  controlp->abort_reason = reason;

  /* send an abort message to each checked-in participant */
  /* control structure access is locked, but list itself is constant
   * and need not be locked */
  config_target = controlp->checkins;
    
  err = globus_mutex_unlock (&(controlp->mutex));
  assert (!err);

  while ( ! globus_list_empty (config_target) ) {
    globus_nexus_buffer_t buffer;

    err = globus_nexus_buffer_init (&buffer, 0, 0); assert (!err);

    err = nxbuff_put_int (&buffer, reason); assert (!err);

    err = globus_nexus_send_rsr (&buffer, &(((globus_duct_checkin_t *) 
					     globus_list_first (config_target))
					    ->config_sp),
				 ABORT_MSG_ID,
				 NEXUS_TRUE /* destroy buffer */,
				 NEXUS_TRUE /* always safe */);

    globus_nexus_startpoint_flush (&(((globus_duct_checkin_t *) 
				      globus_list_first (config_target))
				     ->config_sp));
    
    config_target = globus_list_rest (config_target);
  }    
}


/*
 * tear down a listening port
 */
void 
globus_duct_control_destroy (globus_duct_control_t * controlp)
{
  globus_nexus_disallow_attach (controlp->checkin_port.portno);
}


