/*
    espctag.c : espctag
    v0.3 - 2011-09-19
    
    Copyright (C) 2010,2011 Jérôme SONRIER <jsid@emor3j.fr.eu.org>

    This file is part of espctag.

    espctag is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    espctag is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with espctag.  If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdio.h>
#include <stdlib.h>
#include <error.h>
#include <argp.h>
#include <argz.h>
#include <spctag.h>

#include "constants.h"


/* This structure holds all that is necessary to print or set a tag */
typedef struct
{
	const char *label;		/* The tag label                                 */
	char* (*get_func)();		/* The function that return the tag value        */
	int (*set_func)(char*);		/* The function that set a new value for the tag */
	int enabled;			/* Non null if the tag must be print or set      */
	char *new_value;		/* The new value of the tag                      */
} tag_params;

/* This array holds all the tags */
static tag_params tags[] = {
	{ "Song title",        spctag_get_songtitle,       spctag_set_songtitle       },
	{ "Game title",        spctag_get_gametitle,       spctag_set_gametitle       },
	{ "Dumper name",       spctag_get_dumpername,      spctag_set_dumpername      },
	{ "Comments",          spctag_get_comments,        spctag_set_comments        },
	{ "Dump date",         spctag_get_dumpdate,        spctag_set_dumpdate        },
	{ "Length(s)",         spctag_get_length,          spctag_set_length          },
	{ "Fade length(ms)",   spctag_get_fadelength,      spctag_set_fadelength      },
	{ "Artist",            spctag_get_artist,          spctag_set_artist          },
	{ "Default channels",  spctag_get_defaultchannels, spctag_set_defaultchannels },
	{ "Emulator",          spctag_get_emulator,        spctag_set_emulator        }
};

/* espctag version */
const char *argp_program_version = "espctag-0.3";
/* Bugtracker URL */
const char *argp_program_bug_address = "https://sourceforge.net/tracker/?group_id=563796&atid=2286949";
/* Help string */
static char doc[] = "espctag is an ID666 tags editor."
		"\vYou must use at least --get or --set, but if no one is present, --get will be used.";
/* Synopsis */
static char args_doc[] = "FILE ...";

/* This structure holds all "global" options */
struct arguments
{
	int set;		/* Non null if we want to set tags                  */
	int get;		/* Non null if we only want to print tags           */
	int file_name;		/* Non null if file names must be print before tags */
	int field_name;		/* Non null if field names must print               */
	int type;		/* Non null if type (text or binary) must be print  */
	int all;		/* Non null if the user use the --all switch        */
	char *argz;		/* SPC file names                                   */
	size_t argz_len;	/* Length of file names                             */
};

/* The options we understand. */
static struct argp_option options[] = {
	{ 0, 0, 0, 0, "Operations (mutually exclusive) :", 1 },
	{ "set",  's', 0, 0, "Set tags"   },
	{ "get",  'g', 0, 0, "Print tags" },
	{ 0, 0, 0, 0, "Global options :", 5 },
	{ "file",  'f', 0, 0, "Print file names"        },
	{ "field", 'n', 0, 0, "Don't print field names" },
	{ "type",  't', 0, 0, "Print tags format"       },
	{ 0, 0, 0, 0, "Field selection :", 10 },
	{ "all",      'a', 0,             0,                   "Print all tags"                   },
	{ "song",     'S', "SONG_TITLE",  OPTION_ARG_OPTIONAL, "Print/Set song title"             },
	{ "game",     'G', "GAME_TITLE",  OPTION_ARG_OPTIONAL, "Print/Set game title"             },
	{ "dumper",   'N', "DUMPER_NAME", OPTION_ARG_OPTIONAL, "Print/Set dumper name"            },
	{ "comments", 'C', "COMMENTS",    OPTION_ARG_OPTIONAL, "Print/Set comments"               },
	{ "date",     'D', "MM/DD/YYYY",  OPTION_ARG_OPTIONAL, "Print/Set dump date"              },
	{ "length",   'L', "LENGTH",      OPTION_ARG_OPTIONAL, "Print/Set length"                 },
	{ "fade",     'F', "LENGTH",      OPTION_ARG_OPTIONAL, "Print/Set fade length"            },
	{ "artist",   'A', "ARTIST",      OPTION_ARG_OPTIONAL, "Print/Set artist"                 },
	{ "channels", 'M', "CHANNELS",    OPTION_ARG_OPTIONAL, "Print/Set default channels state" },
	{ "emulator", 'E', "EMULATOR",    OPTION_ARG_OPTIONAL, "Print/Set emulator"               },
	{ 0 }
};

/* Option parser */
static error_t parse_opt( int key, char *arg, struct argp_state *state )
{
	/* Get the input argument from argp_parse, which we
	know is a pointer to our arguments structure. */
	struct arguments *arguments = state->input;

	switch (key) {
		case 's':
			arguments->set = 1;
			break;
		case 'g':
			arguments->get = 1;
			break;
		case 'f':
			arguments->file_name = 1;
			break;
		case 'n':
			arguments->field_name = 0;
			break;
		case 't':
			arguments->type = 1;
			break;
		case 'a':
			arguments->all = 1;
			tags[I_SONG_TITLE].enabled = 1;
			tags[I_GAME_TITLE].enabled = 1;
			tags[I_DUMPER_NAME].enabled = 1;
			tags[I_COMMENTS].enabled = 1;
			tags[I_DUMP_DATE].enabled = 1;
			tags[I_LENGTH].enabled = 1;
			tags[I_FADE_LENGTH].enabled = 1;
			tags[I_ARTIST].enabled = 1;
			tags[I_CHANNELS].enabled = 1;
			tags[I_EMULATOR].enabled = 1;
			break;
		case 'S':
			tags[I_SONG_TITLE].enabled = 1;
			if ( arg )
				tags[I_SONG_TITLE].new_value = arg;
			else
				tags[I_SONG_TITLE].new_value = "";
			break;
		case 'G':
			tags[I_GAME_TITLE].enabled = 1;
			if ( arg )
				tags[I_GAME_TITLE].new_value = arg;
			else
				tags[I_GAME_TITLE].new_value = "";
			break;
		case 'N':
			tags[I_DUMPER_NAME].enabled = 1;
			if ( arg )
				tags[I_DUMPER_NAME].new_value = arg;
			else
				tags[I_DUMPER_NAME].new_value = "";
			break;
		case 'C':
			tags[I_COMMENTS].enabled = 1;
			if ( arg )
				tags[I_COMMENTS].new_value = arg;
			else
				tags[I_COMMENTS].new_value = "";
			break;
		case 'D':
			tags[I_DUMP_DATE].enabled = 1;
			if ( arg )
				tags[I_DUMP_DATE].new_value = arg;
			else
				tags[I_DUMP_DATE].new_value = "";
			break;
		case 'L':
			tags[I_LENGTH].enabled = 1;
			if ( arg )
				tags[I_LENGTH].new_value = arg;
			else
				tags[I_LENGTH].new_value = "";
			break;
		case 'F':
			tags[I_FADE_LENGTH].enabled = 1;
			if ( arg )
				tags[I_FADE_LENGTH].new_value = arg;
			else
				tags[I_FADE_LENGTH].new_value = "";
			break;
		case 'A':
			tags[I_ARTIST].enabled = 1;
			if ( arg )
				tags[I_ARTIST].new_value = arg;
			else
				tags[I_ARTIST].new_value = "";
			break;
		case 'M':
			tags[I_CHANNELS].enabled = 1;
			if ( arg )
				tags[I_CHANNELS].new_value = arg;
			else
				tags[I_CHANNELS].new_value = "";
			break;
		case 'E':
			tags[I_EMULATOR].enabled = 1;
			if ( arg )
				tags[I_EMULATOR].new_value = arg;
			else
				tags[I_EMULATOR].new_value = "";
			break;
		case ARGP_KEY_INIT:
			arguments->argz = 0;
			arguments->argz_len = 0;
			break;
		case ARGP_KEY_NO_ARGS:
			argp_usage (state);
			break;
		case ARGP_KEY_ARG:
			argz_add( &arguments->argz, &arguments->argz_len, arg );
			break;
		default:
			return ARGP_ERR_UNKNOWN;
	}

	return 0;
}

static struct argp argp = { options, parse_opt, args_doc, doc };


int main( int argc, char **argv )
{
	FILE *spc_file;			/* Our SPC file      */
	char msgerror[1024];		/* Error string      */
	char *file;			/* File name to open */
	struct arguments arguments;	/* Our arguments     */
	const char *prev = NULL;
	int i, ret;

	/* Default options values. */
	arguments.set        = 0;
	arguments.get        = 0;
	arguments.file_name  = 0;
	arguments.field_name = 1;
	arguments.type       = 0;
	arguments.all        = 0;

	/* Parse arguments */
	argp_parse( &argp, argc, argv, 0, 0, &arguments );

	/* Print an error message if the user use --get and --set at the same time */
	if ( arguments.set && arguments.get ) {
		fprintf( stderr, "You can not use get and set operations at the same time !\n" );
		exit( E_WRONG_ARG );
	}
	/* If no --set or --get are used, only print tags */
	if ( ! arguments.set && ! arguments.get )
		arguments.get = 1;

	/* Print an error message if the user use --all and --set at the same time */
	if ( arguments.all && arguments.set ) {
		fprintf( stderr, "You can not use --all and --set at the same time !\n" );
		exit( E_WRONG_ARG );
	}

	while( ( file = argz_next( arguments.argz, arguments.argz_len, prev ) ) ) {
		/* Print file name */
		if ( arguments.file_name )
			printf("File : %s\n", file );

		/* Open file */
		if ( ( spc_file = fopen( file, "r+" ) ) == NULL ) {
			sprintf( msgerror, "Unable to open file '%s'!", file );
			perror( msgerror );
			exit( E_OPEN_FILE );
		}

		/* Init spctag */
		if ( ( ret = spctag_init( spc_file ) ) < 0 ) {
			fprintf( stderr, "Can not init libspctag!\n" );
			fclose( spc_file );
			exit( ret );
		}

		/* Print tag type (text or binary) */
		if ( arguments.type ) {
			if ( spctag_txt_tag )
				printf( "Tags format : Text\n" );
			else
				printf( "Tags format : Binary\n" );
		}

		/* For every know tag ... */
		for ( i=0; i<sizeof(tags)/sizeof(tag_params); i++ ) {
			/* Do nothing if the tag is not selected */ 
			if ( ! tags[i].enabled )
				continue;

			/* Print or set tag */
			if ( arguments.set ) {
				tags[i].set_func( tags[i].new_value );
			} else {
				if ( arguments.field_name )
					printf( "%s : ", tags[i].label );

				printf( "%s\n", tags[i].get_func() );
			}
		}

		/* Save the file if something have been set */
		if ( arguments.set )
			spctag_save( spc_file );

		/* Close file */
		fclose( spc_file );

		prev = file;
	}

	/* Free libspctag and argz memory */
	spctag_free();
	free( arguments.argz );

	return( SUCCESS );
}
