/***************************************************************************
*   Copyright (C) 2005 by Adam Treat                                      *
*   treat@kde.org                                                         *
*                                                                         *
*   Copyright (C) 2004 by Scott Wheeler                                   *
*   wheeler@kde.org                                                       *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#ifndef DATATABLE_H
#define DATATABLE_H

#include <kdebug.h>
#include <klocale.h>
#include <kurldrag.h>
#include <klistview.h>
#include <kglobalsettings.h>

#include <qapplication.h>
#include <qcursor.h>
#include <qmap.h>
#include <qtabwidget.h>
#include <qdatatable.h>
#include <qvaluevector.h>
#include <qmutex.h>

#include "datatableview.h"
#include "datatableedit.h"
#include "datatableeditorfactory.h"
#include "datafield.h"
#include "datatablesearch.h"
#include "datatableinterface.h"

class KPopupMenu;
class KActionMenu;

class Project;
class DatabaseConnection;
class QSqlPropertyMap;
class QSqlForm;

typedef QValueList<DataField *> DataFieldList;
typedef QValueList<DataTableSearch> DataSearchList;

class DataTable : public QTabWidget, public DataTableInterface
{
    Q_OBJECT

public:

    enum RelationToParent { None = 0, OneToOne = 1, OneToMany = 2, ManyToOne = 3, ManyToMany = 4 };
    DataTable( Project *project );

    DataTable( Project *project, DataTable* parent );

    virtual ~DataTable();

    bool initialize();
    void initializeFields( bool refresh = false );
    void initializeEditorFields();
    QStringList visibleColumns();

    virtual QString name() const;
    void setName( const QString &n );

    QString alias() const;
    static QString sanitize( const QString &str );

    QString iconName() const;
    void setIconName( const QString &n );

    QString connection() const;
    void setConnection( const QString &name );

    QString tableName() const;
    void setTableName( const QString &name );

    int number() const;
    void setNumber( int number );

    QString parentName() const;
    /*void setParentName( const QString &name );*/ //this is a slot

    RelationToParent relationToParent() const;
    void setRelationToParent( RelationToParent relation );

    QString parentKey() const;
    void setParentKey( const QString &name );

    QString foreignKey() const;
    void setForeignKey( const QString &name );

    QString foreignValue() const;
    void setForeignValue( const QString &name );

    QString defaultFilter() const;
    void setDefaultFilter( const QString &filter );

    QString searchFilter() const;
    void setSearchFilter( const QString &filter );

    QStringList sort() const;
    void setSort( const QStringList &sort );

    bool updatesAllowed() const;
    void setUpdatesAllowed( bool allowed );

    QStringList inheritanceTree() const;
    void setInheritanceTree( const QStringList &inheritTree );

    DataRelation* dataRelation( const QString &key );

    void addDataRelation( const QString &name, const QString &targetTable,
                          const QString &targetKey, const QString &targetField,
                          const QString &targetConstraint,
                          const QVariant::Type targetType, int number,
                          int editorWidth );

    DataFieldList fieldList();

    DataField* dataField( const QString &key );

    DataField* dataField( int col );

    void addDataField( const QString &name, const QVariant::Type type,
                       int isRequired, bool calculated, bool isVirtual,
                       const QString &equation, const QString &label,
                       int number, int width,
                       bool foreign, bool primary, bool hidden,
                       bool resizable, bool sortable );

    void removeDataField( DataField *field );

    KActionMenu *columnVisibleAction() const;

    DataTableSearch search() const;
    void setSearch( const DataTableSearch &search );

    void tableRefresh( QDataTable::Refresh mode = QDataTable::RefreshData );

    bool isRootTable();

    DataTableView *dataTableView() const
    {
        return m_tableView;
    }

    DataTableEdit *dataTableEdit() const
    {
        return m_tableEdit;
    }

    DataTableEditorFactory *dataTableEditorFactory() const
    {
        return m_factory;
    }

    Project *project() const
    {
        return m_project;
    }

    QString uniqueDataFieldName( const QString &suggest = i18n( "DataField" ) );

    DataFieldList calculatedFields();
    DataRecordList recordsInTree();
    void calculateFieldsInTree();

public slots:
    void setParentName( const QString &name );
    void slotUpdate();
    void slotSelectFirstRow();
    void slotDetailFromMaster( QSqlRecord * record );
    void slotValueFromChild( const QString &key, const QVariant &value );
    void slotFilterToBuffer( QSqlRecord * record );
    int scrollTabLeft( int i = -1 );
    int scrollTabRight( int i = -1 );
    void firstRecord();
    void previousRecord();
    void nextRecord();
    void lastRecord();
    void commit();
    void insertRecord();
    void changeRecord();
    void associateRecord();
    void deleteRecord();
    void deleteAbortedInsert();
    void addVirtualField();
    void configureDataField( DataField *field );

signals:
    void requestParentSelect();
    void signalNameChanged( const QString &name );
    void myCurrentChanged( QSqlRecord * record );
    void childValueChanged( const QString &key, const QVariant &value );
    void tableRefreshed();

private:
    void setup();

private:
    DataTableView *m_tableView;
    DataTableEdit *m_tableEdit;
    Project *m_project;
    DataTableEditorFactory *m_factory;
    QSqlPropertyMap *m_propMap;

    DataTableSearch m_search;
    DataSearchList m_searchList;

    bool m_updatesAllowed;

    QString m_editText;

    QString m_connection;
    QString m_tableName;
    int m_number;

    QString m_parentName;
    RelationToParent m_relationToParent;
    QString m_parentKey;
    QString m_foreignKey;
    QString m_foreignValue;

    QString m_defaultFilter;
    QString m_searchFilter;
    QStringList m_sort;

    QString m_dataTableName;
    QString m_dataTableIconName;
    DataFieldList m_fieldList;
    QStringList m_inheritanceTree;

    KPopupMenu *m_headerMenu;
    KActionMenu *m_columnVisibleAction;

    static bool m_visibleChanged;
};

// qBubblesort algorithm
template <class InputIterator>
Q_INLINE_TEMPLATES void fieldSorter( InputIterator b, InputIterator e )
{
    // Goto last element;
    InputIterator last = e;
    --last;
    // only one element or no elements ?
    if ( last == b )
        return ;

    // So we have at least two elements in here
    while ( b != last )
    {
        bool swapped = false;
        InputIterator swap_pos = b;
        InputIterator x = e;
        InputIterator y = x;
        y--;
        do
        {
            --x;
            --y;
            if ( compareFields( *x, *y ) )
            {
                swapped = true;
                qSwap( *x, *y );
                swap_pos = y;
            }
        }
        while ( y != b );
        if ( !swapped )
            return ;
        b = swap_pos;
        b++;
    }
}

//Returns true if f1 is less than f2; otherwise returns false.
inline bool compareFields( DataField *f1, DataField *f2 )
{
    if ( f1->hidden() && !f2->hidden() )
        return true;

    if ( !f1->hidden() && f2->hidden() )
        return false;

    return f1->number() < f2->number();
}

bool processEvents();

#endif
