/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "codecs/JpegExportCodec.h"

extern "C" {
#include <codecs/iccjpeg.h>
}

#include <QFileInfo>
#include <qendian.h>
#include <QMutexLocker>
             
#include <KDebug>
#include <KMessageBox>

#include <libkexiv2/kexiv2.h>

#include "ColorManager.h"
#include "MessagesModel.h"
#include "PostProcessor.h"
#include "ProcessingOptions.h"

#include "ui_JpegOptions.h"

struct JpegExportCodec::Private {
  Ui::JpegOptions jpegOptions;
};

JpegExportCodec::JpegExportCodec() : ExportCodec("JPEG", i18n("Jpeg"), "jpeg"), d(new Private)
{
  QWidget* widget = new QWidget;
  d->jpegOptions.setupUi( widget );
  connect( d->jpegOptions.progressive, SIGNAL(clicked(bool)), SIGNAL(optionsChanged()));
  connect( d->jpegOptions.optimize, SIGNAL(clicked(bool)), SIGNAL(optionsChanged()));
  connect( d->jpegOptions.smoothLevel, SIGNAL(valueChanged(int)), SIGNAL(optionsChanged()));
  connect( d->jpegOptions.qualityLevel, SIGNAL(valueChanged(int)), SIGNAL(optionsChanged()));
  connect( d->jpegOptions.subsampling, SIGNAL(currentIndexChanged(int)), SIGNAL(optionsChanged()));
  setConfigurationWidget( widget );
}

JpegExportCodec::~JpegExportCodec()
{
  delete d;
}

void JpegExportCodec::fillProcessingOptions( ProcessingOptions* _processingOptions ) const
{
  _processingOptions->setOption( "JpegProgressive", d->jpegOptions.progressive->isChecked() );
  _processingOptions->setOption( "JpegOptimize", d->jpegOptions.optimize->isChecked() );
  _processingOptions->setOption( "JpegSmooth", d->jpegOptions.smoothLevel->value() );
  _processingOptions->setOption( "JpegQuality", d->jpegOptions.qualityLevel->value() );
  _processingOptions->setOption( "JpegSubsampling", d->jpegOptions.subsampling->currentIndex() );
}


void JpegExportCodec::setProcessingOptions( const ProcessingOptions& _processingOptions ) const
{
  d->jpegOptions.progressive->setChecked( _processingOptions.asBool( "JpegProgressive" ) );
  d->jpegOptions.optimize->setChecked( _processingOptions.asBool( "JpegOptimize" ) );
  d->jpegOptions.smoothLevel->setValue( _processingOptions.asInteger( "JpegSmooth" ) );
  d->jpegOptions.qualityLevel->setValue( _processingOptions.asInteger( "JpegQuality" ) );
  d->jpegOptions.subsampling->setCurrentIndex( _processingOptions.asInteger( "JpegSubsampling" ) );
  
}

#define TELL_ERROR( msg ) \
  MessagesModel::instance()->tellError( i18n("Jpeg codec"), msg ); \
  return false;

#define COND_TELL_ERROR( cond, msg ) \
  if( not (cond ) ) \
  { \
    TELL_ERROR( msg ); \
  }

#define ICC_MARKER  (JPEG_APP0 + 2) /* JPEG marker code for ICC */
#define ICC_OVERHEAD_LEN  14    /* size of non-profile data in APP2 */
#define MAX_BYTES_IN_MARKER  65533  /* maximum data len of a JPEG marker */
#define MAX_DATA_BYTES_IN_MARKER  (MAX_BYTES_IN_MARKER - ICC_OVERHEAD_LEN)

bool JpegExportCodec::writeFile( RawImageInfoSP rawImageInfo, const QString& _fileName, const QByteArray& _imageData, int width, int height, const ProcessingOptions& _processingOptions )
{
  FILE *fp = fopen(_fileName.toLocal8Bit(), "wb");
  COND_TELL_ERROR( fp, "Can't open file: " + _fileName);
  
  // Initialize structure
  struct jpeg_compress_struct cinfo;
  jpeg_create_compress(&cinfo);
  // Initialize error output
  struct jpeg_error_mgr jerr;
  cinfo.err = jpeg_std_error(&jerr);
  // Initialize output stream
  jpeg_stdio_dest(&cinfo, fp);
  
  cinfo.image_width = width;  // image width and height, in pixels
  cinfo.image_height = height;
  cinfo.input_components = 3;
  cinfo.in_color_space = JCS_RGB;
  
  // Set default compression parameters
  jpeg_set_defaults(&cinfo);
  // Customize them
  jpeg_set_quality(&cinfo,  _processingOptions.asInteger( "JpegQuality" ), true );
  
  if(_processingOptions.asBool( "JpegProgressive" ))
  {
      jpeg_simple_progression (&cinfo);
  }
  
  // Optimize ?
  cinfo.optimize_coding = _processingOptions.asBool( "JpegOptimize" );
  
  // Smoothing
  cinfo.smoothing_factor = _processingOptions.asInteger( "JpegSmooth" );
  
  // Subsampling
  switch(_processingOptions.asInteger( "JpegSubsampling" ))
  {
      default:
      case 0:
      {
          cinfo.comp_info[0].h_samp_factor = 2;
          cinfo.comp_info[0].v_samp_factor = 2;
          cinfo.comp_info[1].h_samp_factor = 1;
          cinfo.comp_info[1].v_samp_factor = 1;
          cinfo.comp_info[2].h_samp_factor = 1;
          cinfo.comp_info[2].v_samp_factor = 1;

      }
          break;
      case 1:
      {
          cinfo.comp_info[0].h_samp_factor = 2;
          cinfo.comp_info[0].v_samp_factor = 1;
          cinfo.comp_info[1].h_samp_factor = 1;
          cinfo.comp_info[1].v_samp_factor = 1;
          cinfo.comp_info[2].h_samp_factor = 1;
          cinfo.comp_info[2].v_samp_factor = 1;
      }
          break;
      case 2:
      {
          cinfo.comp_info[0].h_samp_factor = 1;
          cinfo.comp_info[0].v_samp_factor = 2;
          cinfo.comp_info[1].h_samp_factor = 1;
          cinfo.comp_info[1].v_samp_factor = 1;
          cinfo.comp_info[2].h_samp_factor = 1;
          cinfo.comp_info[2].v_samp_factor = 1;
      }
          break;
      case 3:
      {
          cinfo.comp_info[0].h_samp_factor = 1;
          cinfo.comp_info[0].v_samp_factor = 1;
          cinfo.comp_info[1].h_samp_factor = 1;
          cinfo.comp_info[1].v_samp_factor = 1;
          cinfo.comp_info[2].h_samp_factor = 1;
          cinfo.comp_info[2].v_samp_factor = 1;
      }
          break;
  }
  
  // Start compression
  jpeg_start_compress(&cinfo, true);
  
  // Save profile
  if( not _processingOptions.asBool("ConvertToSRGB") )
  {
    QByteArray profile = ColorManager::instance()->sRGBLinearProfile();
    write_icc_profile(& cinfo, (uchar*)profile.data(), profile.size());
  }
  
  // Save metadata  
  {
    QMutexLocker l( ExportCodec::exiv2Mutex() );

    KExiv2Iface::KExiv2 exiv2;
    if(exiv2.load( rawImageInfo->fileInfo().absoluteFilePath()))
    {
      exiv2.setImageOrientation( KExiv2Iface::KExiv2::ORIENTATION_NORMAL );
      // Save exif
      if(exiv2.hasExif()) {
        QByteArray exifArray = exiv2.getExif(true);
        if (exifArray.size() < MAX_DATA_BYTES_IN_MARKER)
        {
          jpeg_write_marker(&cinfo, JPEG_APP0 + 1, (const JOCTET*)exifArray.data(), exifArray.size());
        }
      }
      // Save IPTC
      if(exiv2.hasIptc()) {
        QByteArray exifArray = exiv2.getIptc(true);
        if (exifArray.size() < MAX_DATA_BYTES_IN_MARKER)
        {
          jpeg_write_marker(&cinfo, JPEG_APP0 + 13, (const JOCTET*)exifArray.data(), exifArray.size());
        }
      }
      // Save XMP
      if(exiv2.hasXmp()) {
        QByteArray exifArray = exiv2.getXmp();
        if (exifArray.size() < MAX_DATA_BYTES_IN_MARKER)
        {
          jpeg_write_marker(&cinfo, JPEG_APP0 + 14, (const JOCTET*)exifArray.data(), exifArray.size()); // TODO check which marker is suppose to host XMP data
        }
      }
    }
  }
  // Save data
  int pixelSize = sizeof( quint16 ) * 3;
  int lineWidth = pixelSize * width;
  
  QByteArray line;
  line.resize( lineWidth);
  
  PostProcessor processor(_processingOptions);
  JSAMPROW row_pointer = new JSAMPLE[width*cinfo.input_components];
  for (; cinfo.next_scanline < height;) {
    memcpy( line.data(), (_imageData.data() + cinfo.next_scanline * lineWidth ), lineWidth );
    processor.apply16( line );
    {
      quint16* srcIt = (quint16*)line.data();
      quint8* dstIt = (quint8*)row_pointer;
      for( int i = 0; i < 3 * width; ++i)
      {
        dstIt[ i ] = srcIt[ i ] / 0xFF;
      }
    }
    jpeg_write_scanlines(&cinfo, &row_pointer, 1);
  }
  // Writing is over
  jpeg_finish_compress(&cinfo);
  fclose(fp);

  delete [] row_pointer;
  // Free memory
  jpeg_destroy_compress(&cinfo);
  return true;
}
