/* Copyright 2008 Simon Richter <Simon.Richter@hogyros.de>
 *
 * Released under the GNU General Public Licence version 3.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "tree_builder.hpp"

#include "print_visitor.hpp"

#include "apply_configurations.hpp"

#include "unix_platform.hpp"
#include "windows_platform.hpp"

#include "gnu_toolchain.hpp"

#include "execute.hpp"

#include <dammit/damnfile.hpp>

#include <iostream>

extern "C" {
#include <limits.h>
}

using namespace dammit;

int main(int, char **)
try
{
	path cwd;
	{
		char buffer[PATH_MAX];
		char *res = ::getcwd(buffer, sizeof buffer);
		if(!res)
			return 1;
		cwd = res;
	}

	std::auto_ptr<project> project = load_projects_recursive(cwd);
	
	if(!project.get())
	{
		std::cerr << "E: no project here" << std::endl;
		return 1;
	}

	path homedir = getenv("HOME");

	path cache_dir = homedir / ".cache" / "dammit";

	cache c(cache_dir);

	tree_builder build_tree;

	intrusive_ptr<node> tree = build_tree(*project);

	{
		intrusive_ptr<configuration_node> native_config = new configuration_node;
		native_config->platform = "linux";
		native_config->temporary_dir = cache_dir / native_config->platform / "temporaries";
		native_config->output_dir = cache_dir / native_config->platform / "outputs";
		native_config->nodes.push_back(tree);
		tree = native_config;
	}

	{
		apply_configurations cfg_apply;
		tree = tree->apply(cfg_apply);
	}

	{
		unix_platform u;
		tree = tree->apply(u);

		windows_platform w;
		tree = tree->apply(w);
	}

	{
		gnu_toolchain tc;
		tree = tree->apply(tc);
	}

	/*
		print_visitor print(std::cerr);
		tree = tree->apply(print);
	*/

	{
		execute ex(c);
		ex.keep_going(true);
		tree->apply(ex);
	}

	return 0;
}
catch(std::exception &e)
{
	std::cerr << "E: " << e.what() << std::endl;
	return 1;
}
