# base.rb: basis of all plottable elements
# copyright (c) 2006,2007,2008 by Vincent Fourmond: 
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'Dobjects/Dvector'
require 'CTioga/debug'
require 'CTioga/log'

module CTioga

  Version::register_svn_info('$Revision: 818 $', '$Date: 2008-06-23 23:20:47 +0200 (Mon, 23 Jun 2008) $')

  # The base class for every single object that has to appear at one
  # point as a command in the tioga plot. It provides only one function:
  # do(f), which will be used by subclasses to do what they need.

  class TiogaElement
    include Debug
    include Log

    # The parent element.
    attr_accessor :parent

    # this function must be called with a FigureMaker object to
    # draw the contents of the TiogaElement onto it. It calls
    # the function real_do, which should be redefined by the
    # children. You can redefine do too if you need another
    # debugging output.
    def do(f)
      debug "plotting #{self.inspect}"
      real_do(f)
    end

    # this function returns true if the object needs a CurveStyle parameter.
    def need_style?
      return false
    end

    # We plot everything but parent. If a prefix is given, it is prepended
    # to all lines but the first (for indentation)
    def inspect(prefix="")
      ret = "#<#{self.class.name}:\n"
      for i in instance_variables
        next if i == "@parent"
        var = instance_variable_get(i)
        ret += "#{prefix}  - #{i} -> "
        if var.is_a? TiogaElement
          ret += "#{var.inspect("#{prefix}  ")}\n"
        else
          ret += "#{var.inspect}\n"
        end
      end
      ret += "#{prefix}>"
      return ret
    end
  end 
  

  # A class to represent a small function call to FigureMaker. Can represent
  # basically anything.
  class TiogaFuncall < TiogaElement

    # _symbol_ is the symbol to be called, and the remainder will be used
    # as arguments for the call.
    def initialize(symbol, *args)
      @symbol = symbol
      @args = args
    end
    
    def real_do(f)
      f.send(@symbol, *@args)
    end
  end

end
