/* $Id: cpl_fft-test.c,v 1.7 2010/12/22 07:56:33 llundin Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2008 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: llundin $
 * $Date: 2010/12/22 07:56:33 $
 * $Revision: 1.7 $
 * $Name: cpl-5_3_0-BRANCH $
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/
#include <float.h>
#include <complex.h>

#include "cpl_memory.h"
#include "cpl_stats.h"
#include "cpl_tools.h"
#include "cpl_test.h"
#include "cpl_math_const.h"

#include "cpl_image_io_impl.h"

#include "cpl_fft.h"

/*-----------------------------------------------------------------------------
                                Defines
 -----------------------------------------------------------------------------*/

#ifndef IMAGESZ
#define IMAGESZ         10
#endif

#ifndef CONSTANT
#define CONSTANT        200
#endif

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cpl_fit_test Testing of the CPL utilities
 */
/*----------------------------------------------------------------------------*/



/*-----------------------------------------------------------------------------
                            Private Function prototypes
 -----------------------------------------------------------------------------*/
static void cpl_fft_image_tests(void);

/*----------------------------------------------------------------------------*/
/**
   @brief   Unit tests of fit module
**/
/*----------------------------------------------------------------------------*/

int main(void)
{

    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    /* Insert tests below */

    cpl_fft_image_tests();

    /* End of tests */
    return cpl_test_end(0);
}


static void cpl_fft_image_tests(void)
{
    const cpl_type imtypes[] = {CPL_TYPE_DOUBLE, CPL_TYPE_FLOAT,
                                CPL_TYPE_INT, CPL_TYPE_DOUBLE_COMPLEX,
                                CPL_TYPE_FLOAT_COMPLEX};
    int            ityp;
    int            nok = 0; /* Number of successful calls */

    /* Insert tests below */

    /* Iterate through all pixel types */
    for (ityp = 0; ityp < (int)(sizeof(imtypes)/sizeof(imtypes[0])); ityp++) {
        const cpl_type imtype = imtypes[ityp];

        int ityp2;

        cpl_image * img1 = cpl_image_new(IMAGESZ, IMAGESZ, imtype);
        cpl_image * img3 = cpl_image_new(IMAGESZ, IMAGESZ, imtype);
        cpl_error_code error;

        /* Various error checks */
        error = cpl_fft_image(img3, NULL, CPL_FFT_FORWARD);
        cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

        error = cpl_fft_image(NULL, img3, CPL_FFT_FORWARD);
        cpl_test_eq_error(error, CPL_ERROR_NULL_INPUT);

        error = cpl_fft_image(img3, img3, CPL_FFT_FORWARD);
        cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);

        error = cpl_fft_image(img3, img1, CPL_FFT_FORWARD | CPL_FFT_BACKWARD);
        cpl_test_eq_error(error, CPL_ERROR_ILLEGAL_INPUT);

        if (!(imtype & CPL_TYPE_COMPLEX)) {
            error = cpl_image_fill_noise_uniform(img1, 0, CONSTANT);
            cpl_test_eq_error(error, CPL_ERROR_NONE);
        }

        for (ityp2 = 0; ityp2 < (int)(sizeof(imtypes)/sizeof(imtypes[0]));
             ityp2++) {
            const cpl_type imtype2 = imtypes[ityp2];
            cpl_image * img2 = cpl_image_new(IMAGESZ, IMAGESZ, imtype2);
            const cpl_image * imgin = img3;
            cpl_image * imgout = img2;
            int idir;
            unsigned mode = CPL_FFT_FORWARD;


            error = cpl_image_copy(img3, img1, 1, 1);
            cpl_test_eq_error(error, CPL_ERROR_NONE);

            /* Transform first forward, then backward */
            /* Those two iterations will succeed iff the input image
               in the first iteration is non-complex */

            for (idir = 0; idir < 2; idir++, mode = CPL_FFT_BACKWARD,
                     imgin = img2, imgout = img3) {

                error = cpl_fft_image(imgout, imgin, mode);

                if (cpl_image_get_type(img3) == CPL_TYPE_FLOAT &&
                           cpl_image_get_type(img2) ==
                           (CPL_TYPE_FLOAT | CPL_TYPE_COMPLEX)) {
#ifdef CPL_FFTWF_INSTALLED
                    cpl_test_eq_error(CPL_ERROR_NONE, error);
                    nok++;

                    if (mode == CPL_FFT_BACKWARD) {
                        /* Transformed forward and backwards, so the result
                           should equal the original input */
                        cpl_test_image_abs(img1, img3,
                                           3.0 * FLT_EPSILON * CONSTANT);
                    }
#else
                    cpl_test_eq_error(CPL_ERROR_UNSUPPORTED_MODE, error);
#endif
                } else if (cpl_image_get_type(img3) == CPL_TYPE_DOUBLE &&
                           cpl_image_get_type(img2) ==
                           (CPL_TYPE_DOUBLE | CPL_TYPE_COMPLEX)) {
#ifdef CPL_FFTW_INSTALLED
                    cpl_test_eq_error(CPL_ERROR_NONE, error);
                    nok++;

                    if (mode == CPL_FFT_BACKWARD) {
                        /* Transformed forward and backwards, so the result
                           should equal the original input */
                        cpl_test_image_abs(img1, img3,
                                           5.0 * DBL_EPSILON * CONSTANT);
                    }
#else
                    cpl_test_eq_error(CPL_ERROR_UNSUPPORTED_MODE, error);
#endif

                } else if (!((imtype  & CPL_TYPE_COMPLEX) ^
                             (imtype2 & CPL_TYPE_COMPLEX))) {
                    /* None or both are complex */
                    cpl_test_eq_error(CPL_ERROR_ILLEGAL_INPUT, error);
                } else if ((imtype  & (CPL_TYPE_FLOAT | CPL_TYPE_DOUBLE)) !=
                           (imtype2 & (CPL_TYPE_FLOAT | CPL_TYPE_DOUBLE))) {
                    cpl_test_eq_error(CPL_ERROR_TYPE_MISMATCH, error);
                } else {
                    cpl_test_eq_error(CPL_ERROR_ILLEGAL_INPUT, error);
                }
            }
            cpl_image_delete(img2);
        }
        cpl_image_delete(img1);
        cpl_image_delete(img3);
    }
#if defined CPL_FFTWF_INSTALLED && defined CPL_FFTW_INSTALLED
    cpl_test_eq(nok, 4); /* Forward and backward of float and double */
#elif defined CPL_FFTWF_INSTALLED
    cpl_msg_warning(cpl_func, "Double precision FFT not available for "
                    "unit testing");
    cpl_test_eq(nok, 2); /* Forward and backward of type float */
#elif defined CPL_FFTW_INSTALLED
    cpl_msg_warning(cpl_func, "Single precision FFT not available for "
                    "unit testing");
    cpl_test_eq(nok, 2); /* Forward and backward of type double */
#else
    cpl_msg_warning(cpl_func, "FFT not available for unit testing");
    cpl_test_zero(nok);
#endif

}
