/* $Id: image_smooth.cpp,v 1.7 2013-09-06 13:11:02 cgarcia Exp $
 *
 * This file is part of the MOSCA library
 * Copyright (C) 2013 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-09-06 13:11:02 $
 * $Revision: 1.7 $
 * $Name: not supported by cvs2svn $
 */


#ifndef IMAGE_SMOOTH_CPP
#define IMAGE_SMOOTH_CPP

#include <iostream>
#include <vector>
#include <algorithm>
#include <stdexcept>
#include "mosca_image.h"
#include "image_smooth.h"


/**
 * @brief
 *   Smooth an image with a 
 *
 * @param spectrum  A 1D emission line spectrum
 * @param level     Significance level
 * @param exp_width Expected lines FWHM (in pixels)
 *
 * @return List of peaks candidates positions
 *
 */
template<typename T>
void mosca::image_smooth_1d_median(mosca::image& image,
                                   int half_width, mosca::axis smooth_axis)
{
    cpl_size ny = image.size_y();
    cpl_size nx = image.size_x();
    cpl_size nlines;
    cpl_size nsmooth;
    cpl_size stride_line;
    cpl_size stride_pixel;
    mosca::axis image_axis;

    /* If the smooth direction is given in terms of dispersion or spatial
       axis, then decide whether it is X or Y. If not, it is already X or Y */
    image_axis = image.axis_to_image(smooth_axis);
    
    /* Lines are contiguos pixels along the smoothing direction.
       i.e., if image_axis == mosca::X_AXIS the lines are rows and there are
       ny lines.
       Each line contains nsmooth pixels. For image_axis == mosca::X_AXIS this
       means nsmooth = nx. */
    if(image_axis == mosca::X_AXIS)
    {
        nlines = ny;
        nsmooth = nx;
        stride_pixel = 1;
        stride_line = nx;
    }
    else
    {
        nlines = nx;
        nsmooth = ny;
        stride_pixel = nx;
        stride_line = 1;
    }
        
    if(2*half_width > nsmooth)
        throw std::out_of_range("2*half_width is larger than the image size");
    
    cpl_vector * line    = cpl_vector_new(nsmooth);
    double  * line_begin = cpl_vector_get_data(line);
    T       * im_data    = image.get_data<T>();
    for (cpl_size iline = 0; iline < nlines; ++iline)
    {
        cpl_vector * smooth_line;
        T          * im_data_line;
        double     * line_p;
        double     * smooth_line_p;
        double     * smooth_line_begin;
        
        im_data_line = im_data + iline * stride_line;
        for(line_p = line_begin; line_p != line_begin + nsmooth; ++line_p)
        {
            *line_p = *im_data_line;
            im_data_line += stride_pixel;
        }
        
        smooth_line = cpl_vector_filter_median_create(line, half_width);
        smooth_line_begin = cpl_vector_get_data(smooth_line);
        
        im_data_line = im_data + iline * stride_line;
        for(smooth_line_p = smooth_line_begin;
                smooth_line_p != smooth_line_begin + nsmooth; ++smooth_line_p)
        {
            *im_data_line = *smooth_line_p;
            im_data_line += stride_pixel;
        }
        cpl_vector_delete(smooth_line);
    }
    cpl_vector_delete(line);    
}
        
#endif
