/*
 * Author: Matthew Farrellee <matt@cs.wisc.edu>
 */

package birdbath;

import condor.*;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.rmi.RemoteException;

public class Transaction
{
    private CondorScheddPortType port;
    private condor.Transaction transaction;

    Transaction(CondorScheddPortType port)
    {
        this.port = port;
    }

    public void begin(int timeout)
    throws RemoteException
    {
        TransactionAndStatus result = port.beginTransaction(timeout);
        Utilities.checkAndThrowRemoteException(result.getStatus());

        this.transaction = result.getTransaction();
    }

    public void commit()
    throws RemoteException
    {
        Status status = port.commitTransaction(transaction);
        Utilities.checkAndThrowRemoteException(status);

        /*
         * A transaction should not be used after it is committed, but
         * the schedd should notify the user of this instead of storing
         * such informatin in these helper classes.
         port = null;
         transaction = null;
         */
    }

    public void abort()
    throws RemoteException
    {
        Status status = port.abortTransaction(transaction);
        Utilities.checkAndThrowRemoteException(status);
    }

    public int createCluster()
    throws RemoteException
    {
        IntAndStatus result = port.newCluster(transaction);
        Utilities.checkAndThrowRemoteException(result.getStatus());

        return result.getInteger().intValue();
    }

    public int createJob(int cluster)
    throws RemoteException
    {
        IntAndStatus result = port.newJob(transaction, cluster);
        Utilities.checkAndThrowRemoteException(result.getStatus());

        return result.getInteger().intValue();
    }

    public boolean removeJob(int cluster, int job, String reason)
    throws RemoteException
    {
        Status status = port.removeJob(transaction,
                                       cluster,
                                       job,
                                       reason,
                                       false);
        Utilities.checkAndThrowRemoteException(status);

        return true;
    }

    public void submit(int cluster,
                       int job,
                       String owner,
                       UniverseType universe,
                       String command,
                       String arguments,
                       String requirements,
                       ClassAdStructAttr[] attributes,
                       File[] files)
    throws RemoteException, FileNotFoundException, IOException
    {
        if (null == owner)
        {
            owner = System.getProperty("user.name");
        }

        if (null == universe)
        {
            universe = UniverseType.STANDARD;
        }

        if (null == command)
        {
            throw new IllegalArgumentException("command argument cannot be null.");
        }

        if (null == arguments)
        {
            arguments = "";
        }

        if (null == requirements)
        {
            requirements = "TRUE";
        }

        ClassAdStructAndStatus templateResult =
            port.createJobTemplate(cluster,
                                   job,
                                   owner,
                                   universe,
                                   command,
                                   arguments,
                                   requirements);
        Utilities.checkAndThrowRemoteException(templateResult.getStatus());

        ClassAdStructAttr[] template = templateResult.getClassAd();

        if (null != attributes)
        {
            template = Utilities.setAttributes(attributes, template);
        }

        if (null != files)
        {
            for (int index = 0; index < files.length; index++)
            {
                sendFile(cluster, job, files[index]);
            }
        }

        RequirementsAndStatus submitResult = port.submit(transaction,
                                                         cluster,
                                                         job,
                                                         template);
        Utilities.checkAndThrowRemoteException(submitResult.getStatus());
    }

    public void sendFile(int cluster, int job, File file)
    throws FileNotFoundException, RemoteException, IOException
    {
        /* NOTES:
         *  - Really long files are not supported.
         *  - Size of each chunk is fixed.
         */

        final int CHUNK_SIZE = 256 * 1024; // 256KB

        FileInputStream input = new FileInputStream(file);

        //System.out.println("file.length() == " + file.length());
        Status status = port.declareFile(transaction,
                                         cluster,
                                         job,
                                         file.getName(),
                                         (int) file.length(),
                                         HashType.NOHASH,
                                         null);
        Utilities.checkAndThrowRemoteException(status);

        byte[] readBuffer = new byte[CHUNK_SIZE];
        byte[] writeBuffer;
        int length;
        int offset = 0;
        while (-1 != (length = input.read(readBuffer)))
        {
            //System.out.println("offset: " + offset + " length: " + length);

            /*
             * Incase the file is not a multiple of the CHUNK_SIZE or
             * read() does not populate the full buffer, we must create
             * an output buffer that is exactly the size of the data. This
             * is only an issue because the base64 encoding encodes the
             * entire output buffer, not just a portion of it.
             */
            if (CHUNK_SIZE != length)
            {
                writeBuffer = new byte[length];
                System.arraycopy(readBuffer, 0, writeBuffer, 0, length);
            }
            else
            {
                writeBuffer = readBuffer;
            }

            status = port.sendFile(transaction,
                                   cluster,
                                   job,
                                   file.getName(),
                                   offset,
                                   writeBuffer);
            Utilities.checkAndThrowRemoteException(status);

            offset += length;
        }

        input.close();
    }

    public void getFile(int cluster,
                        int job,
                        String remoteName,
                        int size,
                        File localFile)
    throws RemoteException, FileNotFoundException, IOException
    {
        /* NOTES:
         *  - Really long files are not supported.
         *  - Size of each chunk is fixed.
         */

        final int CHUNK_SIZE = 256 * 1024; // 256KB

        FileOutputStream output = new FileOutputStream(localFile);

        Base64DataAndStatus result;
        int offset = 0;
        while (offset < size)
        {
            //System.out.println("offset: " + offset + "; length: " + Math.min(CHUNK_SIZE, size - offset - 1));
            result = port.getFile(transaction,
                                  cluster,
                                  job,
                                  remoteName,
                                  offset,
                                  Math.min(CHUNK_SIZE, size - offset));
            Utilities.checkAndThrowRemoteException(result.getStatus());

            output.write(result.getData());
            offset += result.getData().length;
        }

        output.close();
    }

    public FileInfo[] listSpool(int cluster,
                                int job)
    throws RemoteException
    {
        FileInfoArrayAndStatus result = port.listSpool(transaction, cluster, job);
        Utilities.checkAndThrowRemoteException(result.getStatus());

        return result.getInfo();
    }

    public void closeSpool(int cluster,
                           int job)
    throws RemoteException
    {
        Status status = port.closeSpool(transaction, cluster, job);
        Utilities.checkAndThrowRemoteException(status);
    }

    public ClassAdStructAttr[] getJobAd(int cluster,
                                        int job)
    throws RemoteException
    {
        ClassAdStructAndStatus result = port.getJobAd(transaction, cluster, job);
        Utilities.checkAndThrowRemoteException(result.getStatus());

        return result.getClassAd();
    }

    public ClassAdStructAttr[][] getJobAds(String constraint)
    throws RemoteException
    {
        ClassAdStructArrayAndStatus result = port.getJobAds(transaction, constraint);
        Utilities.checkAndThrowRemoteException(result.getStatus());

        return result.getClassAdArray();
    }
}

