/*
* Copyright (c) 2006 Rudi Cilibrasi, Rulers of the RHouse
* All rights reserved.     cilibrar@cilibrar.com
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions are met:
*
*     * Redistributions of source code must retain the above copyright
*       notice, this list of conditions and the following disclaimer.
*     * Redistributions in binary form must reproduce the above copyright
*       notice, this list of conditions and the following disclaimer in the
*       documentation and/or other materials provided with the distribution.
*     * Neither the name of the RHouse nor the
*       names of its contributors may be used to endorse or promote products
*       derived from this software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE RULERS AND CONTRIBUTORS "AS IS" AND ANY
* EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE RULERS AND CONTRIBUTORS BE LIABLE FOR ANY
* DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
* ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <assert.h>
#include <unistd.h>
#include <string.h>


extern "C" {
#include <complearn/complearn.h>
#include <qsearch/qsearch.h>
};

#include "springymatrix.h"

void SpringyMatrixTreeWatcher::wait_until_done(void) const {
  while (!this->is_done())
    sleep(1);
}

bool SpringyMatrixTreeWatcher::is_done(void) const { return _did_abort; }

void SpringyMatrixTreeWatcher::abort(void) { _should_abort = true; }

void SpringyMatrixTreeWatcher::calculate(void) {
  qsearch_treemaster_find_best_tree(_tm);
  _did_abort = true;
}

static void treeSearchStarted(gsl_matrix *dm, void *udata) {
}

static void treeTriedToImprove(gsl_matrix *dm, QSearchTree *old, QSearchTree *improved, void *udata) {
  SpringyMatrixTreeWatcher *smt = (SpringyMatrixTreeWatcher *) udata;
  if (improved)
    smt->moveToNextTree(dm, improved);
}

static void treeSearchDone(gsl_matrix *dm, QSearchTree *best, void *udata) {
  SpringyMatrixTreeWatcher *smt = (SpringyMatrixTreeWatcher *) udata;
  smt->moveToNextTree(dm, best);
}

static void *do_calculations(void *udata) {
  SpringyMatrixTreeWatcher *smt = (SpringyMatrixTreeWatcher *) udata;
  smt->calculate();
  return NULL;
}

void SpringyMatrixTreeWatcher::moveToNextTree(gsl_matrix *dm, QSearchTree *newguy) {
  pthread_mutex_lock(&_matlock);
  if (_should_abort) {
    qsearch_treemaster_stop_search(_tm);
  } else {
    if (_lastmat != 0)
      gsl_matrix_free(_lastmat);
    _lastmat = qsearch_tree_get_adjacency_matrix(newguy);
    _seqno += 1;
    if (dm)
      _tsco = qsearch_tree_score_tree(newguy, dm);
    else
      _tsco = -1;
  }
  pthread_mutex_unlock(&_matlock);
}

double SpringyMatrixTreeWatcher::get_score(void)
{
  return _tsco;
}
int SpringyMatrixTreeWatcher::get_leaf_node_count(void)
{
  if (_lastmat)
    return (_lastmat->size1+2)/2;
  return 0;
}

int SpringyMatrixTreeWatcher::get_seqno(void) {
  int result;
  pthread_mutex_lock(&_matlock);
  result = _seqno;
  pthread_mutex_unlock(&_matlock);
  return result;
}

gsl_matrix *SpringyMatrixTreeWatcher::get_springy_matrix(void) {
  pthread_mutex_lock(&_matlock);
  gsl_matrix *result = 0;
  if (_lastmat) {
    result = gsl_matrix_alloc(_lastmat->size1, _lastmat->size2);
    gsl_matrix_memcpy(result, _lastmat);
  }
  pthread_mutex_unlock(&_matlock);
  return result;
}

SpringyMatrixTreeWatcher::~SpringyMatrixTreeWatcher() {
  if (!_did_abort) {
    this->abort();
    this->wait_until_done();
  }
  pthread_mutex_lock(&_matlock);
  if (_lastmat) {
    gsl_matrix_free(_lastmat);
  }
  pthread_mutex_unlock(&_matlock);
  pthread_mutex_destroy(&_matlock);
  if (_tm)
    qsearch_treemaster_free(_tm);
}

SpringyMatrixTreeWatcher::SpringyMatrixTreeWatcher( QSearchTree *tree) {
  _tm = 0;
  _seqno = 0;
  _should_abort = false;
  _did_abort = false;
  _lastmat = 0;
  pthread_mutex_init(&_matlock, NULL);
  moveToNextTree(NULL, tree);
}

SpringyMatrixTreeWatcher::SpringyMatrixTreeWatcher(QSearchTreeMaster *tm) {
  _tm = tm;
  _seqno = 0;
  _should_abort = false;
  _did_abort = false;
  _lastmat = 0;
  pthread_mutex_init(&_matlock, NULL);
  qsearch_treemaster_add_observer(_tm, treeSearchStarted, treeTriedToImprove, treeSearchDone, this);
  pthread_t calcthread;
  pthread_create(&calcthread, NULL, do_calculations, (void *) this);
  //_matlock = PTHREAD_MUTEX_INITIALIZER;
}


