/* sph_math.c    generic spherical geometry routines */

/* gcc sph_math.c -o sph_math.o -c -g -lm */

#include "cp_types.h"
extern int s_pt_to_vec(),path_free();

/* Sphere is x^2 + y^2 + z^2 = 1. Store centers as complex numbers
with form (theta,phi); radii are spherical. To get the correct
orientation on the sphere when viewed from outside, we do NOT use
the normal stereographic projection with the plane; rather, the
origin is the NORTH pole and infinity is the SOUTH pole. */

int tangent(complex ctr1,complex ctr2,double *T)
     /* Given pts on sphere, return ptr to unit lenght 3-vector in 
tangent space of first, pointing toward second. If pts are
essentially equal or antipodal, return 0 and set T=<1,0,0>. */
{

  double d,vn,A[3],B[3],P[3];

  s_pt_to_vec(ctr1,A);
  s_pt_to_vec(ctr2,B);
  d=dot_prod(A,B);
  /* Find projection of B on plane normal to A */
  P[0]=B[0]-d*A[0];  P[1]=B[1]-d*A[1];  P[2]=B[2]-d*A[2];

  /* A and B essentially parallel? */
  if ((vn=vec_norm(P))<m_toler)
    {
      T[0]=1;
      T[1]=0;
      T[2]=0;
      return 0;
    }
      
  T[0]=P[0]/vn;T[1]=P[1]/vn;T[2]=P[2]/vn;
  return 1;
} /* tangent */

int s_compcenter(complex z1,complex z2,complex *z3,
		 double r1,double r2,double *r3,
		 double o1,double o2,double o3)
     /* Find center of third circle in ordered tangent triple. Note: 
orientation is counterclockwise looking at sphere from outside. Overlaps 
not yet used. (r3 is read as a ptr only for compatibility reasons.) */ 
{

  double a,b,c,h,j,k,angle;
  double TV[3], P[3], N[3];
  double mtan[3];

  a=sin(z1.im)*cos(z1.re);
  b=sin(z1.im)*sin(z1.re);
  c=cos(z1.im);
  h=sin(z2.im)*cos(z2.re);
  j=sin(z2.im)*sin(z2.re);
  k=cos(z2.im);
  /* angle is how far around from T we will rotate */
  angle=acos(( cos(r2+(*r3))-cos(r1+(*r3))*cos(r1+r2) )/
	     ( sin(r1+(*r3))*sin(r1+r2) ));
  /* TV is a tangent vector at z1 */
  tangent(z1,z2,TV);
  /* N is T x z1 */
  N[0]=b*TV[2]-c*TV[1];
  N[1]=c*TV[0]-a*TV[2];
  N[2]=a*TV[1]-b*TV[0];
  /* P will point toward the new center */
  P[0]=cos(angle)*TV[0]+sin(angle)*N[0];
  P[1]=cos(angle)*TV[1]+sin(angle)*N[1];
  P[2]=cos(angle)*TV[2]+sin(angle)*N[2];
  *mtan=cos(r1+(*r3))*a+sin(r1+(*r3))*P[0];
  *(mtan+1)=cos(r1+(*r3))*b+sin(r1+(*r3))*P[1];
  *(mtan+2)=cos(r1+(*r3))*c+sin(r1+(*r3))*P[2];
  if(mtan[2]>(1.0-m_toler))
    {z3->im=z3->re=0.0; return 1;}
  z3->im=acos(mtan[2]);
  z3->re=aTan2(mtan[1],mtan[0]);
  return 1;
} /* s_compcenter */

double s_comp_cos(double r1,double r2,double r3,int *flag)
     /* given ordered triple of spherical radii, compute the cosine 
of the angle at first circle in triangle formed by mutually tangent 
circles. Increment flag on error: r1+r2+r3>M_PI or denom zero. */
     /* fixup: Is this okay if r1+r2+r3 > M_PI? */
{
  double denom,sumr;

  sumr=r1+r2+r3;
  if (fabs(M_PI-sumr)<m_toler) /* circles centers on a great circle */
    return -1.0;
  if ((denom=sin(r1+r2)*sin(r1+r3))<m_toler)
    {
      (*flag)++; /* signal there's an error */
      return 0.0;
    }
  return ((cos(r2+r3)-(cos(r1+r2)*cos(r1+r3)))/denom);
} /* s_comp_cos */

struct Pathlist *s_geodesic(complex z1,complex z2,int num_plot)
     /* returns pointer to list of pts on geodesic which are on front 
of (apparent) sphere. num_plot is no. of divisions in plotting curve. */
{
  int gplot=(num_plot/2),n=0,length=0;
  double T[3], P[3], angle;
  double cf,sf,c1re,c1im,s1re,s1im,ag;
  struct Pathlist *trace,*pathlist;

  if ((angle=s_dist(z1,z2))<m_toler) return NULL; /* ess. equal */
  tangent(z1,z2,T);
  pathlist=trace=
    (struct Pathlist *)calloc((size_t)1,sizeof(struct Pathlist));
  gplot=(int)(gplot*angle/3.0);
  if (gplot<2) gplot=2;
  ag=angle/gplot;
  c1re=cos(z1.re);
  s1re=sin(z1.re);
  c1im=cos(z1.im);
  s1im=sin(z1.im);
  while (n<=gplot)
    {
      cf=cos(n*ag);
      sf=sin(n*ag);
      P[0]=cf*s1im*c1re+sf*T[0];
      P[1]=cf*s1im*s1re+sf*T[1];
      P[2]=cf*c1im+sf*T[2];
      /* note: unlike full_s_geodesic, here use direct y,z coords */    
      if (length && P[0]>-m_toler)
        {
	  trace->next=(struct Pathlist *)
	    calloc((size_t)1,sizeof(struct Pathlist)); 
	  trace=trace->next;
	  trace->x=P[1];trace->y=P[2];		
	  length++;
        }
      else if (!length && P[0]>-m_toler)
	{
	  pathlist->x=P[1];pathlist->y=P[2];
	  pathlist->next=NULL;
	  length=1;
	  trace=pathlist;
        }
      n++;
    }
  if (!length) {free(pathlist);return NULL;}
  return pathlist;
} /* s_geodesic */

struct Pathlist *full_s_geodesic(complex z1,complex z2,int num_plot)
     /* returns pointer to list of sph pts on whole geodesic; for use with 
closed faces. */
{
  int gplot=num_plot/2,n=0,length=0;
  double T[3], P[3], angle;
  complex pt;
  double cf,sf,c1re,c1im,s1re,s1im,ag;
  struct Pathlist *trace,*pathlist;
  
  if ((angle=s_dist(z1,z2))<m_toler) return NULL; /* ess. equal */
  tangent(z1,z2,T);
  pathlist=trace=
    (struct Pathlist *)calloc((size_t)1,sizeof(struct Pathlist));
  ag=angle/gplot;
  c1re=cos(z1.re);
  s1re=sin(z1.re);
  c1im=cos(z1.im);
  s1im=sin(z1.im);
  while (n<=gplot)
    {
      cf=cos(n*ag);
      sf=sin(n*ag);
      P[0]=cf*s1im*c1re+sf*T[0];
      P[1]=cf*s1im*s1re+sf*T[1];
      P[2]=cf*c1im+sf*T[2];
      pt=proj_vec_to_sph(P[0],P[1],P[2]); 
      /* note: unlike s_geodesic, need to proj to sph coords. */
      if (length)
        {
	  trace->next=(struct Pathlist *)
	    calloc((size_t)1,sizeof(struct Pathlist)); 
	  trace=trace->next;
	  trace->x=pt.re;trace->y=pt.im;		
	  length++;
        }
      else if (!length)
	{
	  pathlist->x=pt.re;pathlist->y=pt.im;
	  pathlist->next=NULL;
	  length=1;
	  trace=pathlist;
	}
      n++;
    }
  if (length==0) {free(pathlist);return NULL;}
  return pathlist;
} /* full_s_geodesic */

struct Pathlist *s_circle_list(complex ctr,double r,int num_plot)
     /* return oriented list of sph pts for circle. Circle is described 
first as circle in plane in 3-space using matrix propogation, then 
projected to sphere. */
{
  int i;
  double A[3],B[3],C[3],P[3];
  double step,ss,a11,a12,a21,a22,zr,zi,wr,wi,dist,rad,norm;
  complex pt;
  struct Pathlist *outlist,*trace;

  rad=sin(r);
  dist=cos(r);
  s_pt_to_vec(ctr,C);
  /* find unit vectors A, B orthog to C, 
     scale C to get center in plane of circle */ 
  if (fabs(C[0])<0.3)
    {A[0]=0.0;A[1]=(-1.0)*C[2];A[2]=C[1];}
  else
    {A[0]=(-1.0)*C[1];A[1]=C[0];A[2]=0.0;}
  cross_prod(A,C,B); /* B=AxC */
  C[0]=C[0]*dist;C[1]=C[1]*dist;C[2]=C[2]*dist;
  norm=vec_norm(A);
  A[0]=A[0]/norm;A[1]=A[1]/norm;A[2]=A[2]/norm;
  norm=vec_norm(B);
  B[0]=B[0]/norm;B[1]=B[1]/norm;B[2]=B[2]/norm;
  /* set up matrix to propogate circle */
  step=2.0*M_PI/num_plot;
  ss=step*step/4;
  a11=(1-ss)/(1+ss);a12= (-step)/(1+ss);a21= (-a12);a22=a11; 
  zr=rad;zi=0.0;
  outlist=(struct Pathlist *)
    calloc((size_t)1,sizeof(struct Pathlist));
  trace=outlist;
  for (i=0;i<=num_plot;i++)
    {
      P[0]=C[0]+zr*B[0]+zi*A[0];
      P[1]=C[1]+zr*B[1]+zi*A[1];
      P[2]=C[2]+zr*B[2]+zi*A[2];
      pt=proj_vec_to_sph(P[0],P[1],P[2]);
      trace->x=pt.re;trace->y=pt.im;
      trace=trace->next=(struct Pathlist *)
	calloc((size_t)1,sizeof(struct Pathlist));
      wr=zr;
      wi=zi;
      zr=a11*wr+a12*wi;
      zi=a21*wr+a22*wi;
    }
  trace->x=outlist->x;trace->y=outlist->y;
  return (outlist);
} /* s_circle_list */

struct Pathlist *sph_tri_list(complex z0,complex z1,complex z2,int num_plot)
     /* oriented list of geodesics in face */
{
  struct Pathlist *firstlist,*nlist,*trace;

  if ((firstlist=full_s_geodesic(z0,z1,num_plot))==NULL) return NULL;
  if ((nlist=full_s_geodesic(z1,z2,num_plot))==NULL) 
    {path_free(&firstlist);return NULL;}
  trace=firstlist;
  while (trace->next!=NULL) trace=trace->next;
  trace->next=nlist;/* link in nlist */
  if ((nlist=full_s_geodesic(z2,z0,num_plot))==NULL) 
    {path_free(&firstlist);return NULL;}
  while (trace->next!=NULL) trace=trace->next;
  trace->next=nlist;/* link in nlist */
  while (trace->next!=NULL) trace=trace->next;
  trace=trace->next=(struct Pathlist *)
    calloc((size_t)1,sizeof(struct Pathlist));;
  trace->x=firstlist->x;trace->y=firstlist->y;
  return firstlist;
} /* sph_tri_list */
	
struct Pathlist *fix_convex_list(struct Pathlist *inlist)
     /* given an oriented bdry list for convex region on (apparent) 
sphere, return oriented list (projected to screen plane) for display 
with 'fill' potential (i.e., if it goes over horizon, portion of 
horizon replaces that in back). */
{
  int i,n;
  double alpha,beta,delta;
  double step,ss,a11,a12,a21,a22,zr,zi,wr,wi;
  struct Pathlist *outlist,*trace,*last,*nexttrace;
  struct Pathlist *begin,*ending,*back;

  if (inlist==NULL || (inlist->next==NULL )) 
    return NULL; /* too short */
  /* whole list on front? */
  trace=inlist;
  while (trace!=NULL && cos(trace->x)>=(-1.0)*m_toler) /* on front */
    trace=trace->next;
  if (trace==NULL) /* whole list on front, project and return */
    {
      trace=inlist;
      while (trace!=NULL)
	{
	  trace->x=sin(trace->y)*sin(trace->x);/* y coord */
	  trace->y=cos(trace->y); /* z coord */
	  trace=trace->next;
	}
      return (inlist); 
    }
  back=trace; 
  /* whole list on back? */
  trace=inlist;
  while (trace!=NULL && cos(trace->x)<=m_toler) /* on back */
    trace=trace->next;
  if (trace==NULL) /* all on back */
    {
      path_free(&inlist);
      return NULL;
    }
  /* close up list temporarily */
  trace=begin=ending=inlist;
  while (trace!=NULL)
    {
      ending=trace;
      trace=trace->next;
    }
  ending->next=begin;
  /* find first on front */
  nexttrace=back;
  trace=back->next; /* 'back' is on back */
  while (trace!=back && cos(trace->x)<m_toler) 
    {
      nexttrace=trace;
      trace=trace->next;
    }
  if (trace==back) /* shouldn't happen */
    {
      ending->next=NULL;
      path_free(&begin);
      return NULL;
    }
  /* readjust begin, end */
  begin=trace;
  ending=nexttrace;
  ending->next=NULL;
  /* now find last on front */
  trace=last=begin;
  while (trace!=NULL && cos(trace->x)>=(-1.0)*m_toler) /* on front */
    {
      last=trace;
      trace=trace->next;
    }
  if (last==begin) /* only one pt on front - too short */
    {
      path_free(&begin);
      return NULL;
    }
  path_free(&(last->next)); /* discard after last, on back */
  last->next=NULL;
  /* find positive arc of horizon from end to beginning to close up;
     angles in (y,z)-plane mes'rd counterclockwise from n pole (+ z axis) */
  alpha=begin->y;
  if ((begin->x)>=0) alpha *=(-1.0);
  beta=last->y;
  if ((last->x)>=0) beta *=(-1.0);
  /* project points of circle */
  trace=outlist=begin;
  while (trace->next!=NULL)
    {
      trace->x=sin(trace->y)*sin(trace->x); /* y coord */
      trace->y=cos(trace->y); /* z coord */
      trace=trace->next;
    }
  trace->x=sin(trace->y)*sin(trace->x); /* y coord */
  trace->y=cos(trace->y); /* z coord */	
  /* want horizon counterclockwise, beta to alpha */
  delta=alpha-beta;
  if (delta<0.0) delta +=2.0*M_PI;
  n=delta/(2.0*M_PI)*10;
  n=n*10; /* n will be 0,10,20, .., 100. */
  if (n<10) n=10; /* tailor number of steps to proportion of circle */
  step=delta/n;
  ss=step*step/4;
  a11=(1-ss)/(1+ss);a12= (-step)/(1+ss);a21= (-a12);a22=a11; 
  zr=-sin(beta);zi=cos(beta);
  /* add these points */
  n++;
  for (i=0;i<n;i++)
    {
      trace=trace->next=(struct Pathlist *)
	calloc((size_t)1,sizeof(struct Pathlist));
      trace->x=zr;
      trace->y=zi;
      wr=zr;
      wi=zi;
      zr=a11*wr+a12*wi;
      zi=a21*wr+a22*wi;
    } 
  return (outlist);
} /* fix_convex_list */

double s_dist(complex z1,complex z2)
     /* spherical distance between pts on sphere. */
{
  double v1[3],v2[3],dotprod;

  if ( (fabs(z1.re-z2.re)<m_toler) && (fabs(z1.im-z2.im)<m_toler))
    return (0.0);
  s_pt_to_vec(z1,v1);
  s_pt_to_vec(z2,v2);
  dotprod= v1[0]*v2[0]+v1[1]*v2[1]+v1[2]*v2[2];
  if (fabs(dotprod)>(1.0-m_toler)) return M_PI;
  return acos(dotprod);
} /* s_dist */

int s_pt_to_vec(complex z,double *V)
     /* convert (theta,phi) pt on sphere to unit 3 vector */
{
  V[0]=sin(z.im)*cos(z.re);
  V[1]=sin(z.im)*sin(z.re);
  V[2]=cos(z.im);
  return 1;
} /* s_pt_to_vec */

complex s_pt_to_visual_plane(complex pt)
     /* return (y,z) of pt */
{
  complex z;

  z.im=cos(pt.im);
  z.re=sin(pt.im)*sin(pt.re);
  return z;
} /* s_pt_to_visual_plane */

complex screen_to_s_pt(complex pt,int *hit)
     /* given pt on screen, find pt on front of sphere. hit=1 if found */
{
  double V[3],xx;
  complex z;

  xx=1.0 - sqrt(pt.re*pt.re + pt.im*pt.im);
  if (xx<0)
    {
      *hit=0;
      z.re=z.im=0.0;
      return z;
    }
  V[0]=xx;
  V[1]=pt.re;
  V[2]=pt.im;
  *hit=1;
  return (proj_vec_to_sph(V[0],V[1],V[2]));
} /* screen_to_s_pt */

int s_to_e_data(complex z,double r,complex *e,double *er)
     /* projects circles from sphere to plane, when possible; circle
	properly enclosing infinity gets antipital point of z as fake 
	center and -(M_PI-rad) as fake radius, indicating that 
	the outside of the eucl circle is our actually our disc 
	-- generally need to change this negative sign on return to 
	the calling routine. Note: we don't permanently store the
	negative radius, so we loose track that eucl radii and center
	are fake; the user must be sure to use this info when it's
	available or make provisions to save it somehow. Also, a circle 
	essentially passing through infinity will have small expansion
	in radius to get euclidean data, but of course it's fake. 
	return -1 if setting negative radius. */
{
  int flipflag=1;
  double V[3],m,RR,rr,up,down;
  complex zz=z;
  extern complex proj_vec_to_sph();

  s_pt_to_vec(zz,V);
  if (fabs(zz.im+r-M_PI)<m_toler) /* essentially hit's infinity: increment
				  r slightly and proceed. */
    r += 2.0*m_toler; 
  if ((zz.im+r)>=(M_PI+m_toler)) /* encloses infinity */
    {
      r=M_PI-r;
      V[0] *= (-1.0);
      V[1] *= (-1.0);
      V[2] *= (-1.0);
      zz=proj_vec_to_sph(V[0],V[1],V[2]);
      flipflag=-1;
    }
  up=zz.im+r;
  down=zz.im-r;
  if (fabs(zz.im)<m_toler) 
    /* essentially centered at np */
    {
      *er=sin(up)/(1.0+cos(up));
      e->re=e->im=0.0;
      if (flipflag<0) *er *= -1.0;
      return flipflag;
    }
  if (fabs(zz.im-M_PI)< m_toler)
    /* essentially centered at sp, radius must be small since
       the circle wasn't flipped, so give it huge negative radius. */
    {
      *er=-100000;
      e->re=e->im=0.0;
      return -1;
    }
  if (fabs(up-M_PI)< .00001)
    /* circle essentially passes through infinity, so decrease up
       slightly */
    {
      up -= .00001;
    }
  RR=sin(up)/(1.0+cos(up));
  rr=sin(down)/(1.0+cos(down));
  *er=fabs(RR-rr)/2.0;
  if (flipflag<0) *er *= -1.0;
  m=(RR+rr)/2.0;
  e->re=V[0]*m/sin(zz.im);
  e->im=V[1]*m/sin(zz.im);
  return flipflag;
} /* s_to_e_data */
		
complex proj_vec_to_sph(double x,double y,double z)
     /* radially project (x,y,z) to sphere, origin to north pole */
{
  double dist;
  complex ans;

  if ((dist=sqrt(x*x+y*y+z*z))< m_toler) 
    {ans.re=ans.im=0.0;return ans;}
  /* default for things near origin */
  x /= dist;
  y /= dist;
  z /= dist;
  ans.re=aTan2(y,x);
  ans.im=acos(z);	
  return ans;
} /* proj_vec_to_sph */

int e_to_s_data(complex z,double r,complex *s,double *sr)
     /* converts circle data to the sphere. Caution: our
projection is NOT the standard "stereographic" projection. 
We have the origin map to the north pole, infinity to the 
south. It's more intuitive, since it preserves orientation 
(+ oriented triple in the plane is + oriented when looked at 
from OUTSIDE the sphere). */
{
  int midflag=0;
  complex v,w;
  double C[3], P1[3], P2[3],P3[3], T[3], E[3];
  double norm,rad,x,y,a,b,ang13,ang23,brk;
  double denom,tmpd,ns,d1,d2,mn,rr;

  ns=z.re*z.re+z.im*z.im;
  rr=fabs(r);
  if (rr<m_toler) /* r too small; project center and leave
			r unchanged. */
    {
      *sr=r;
      denom=ns+1.0;
      tmpd=1.0/denom;
      P3[0]=(2*z.re)*tmpd;  P3[1]=(2*z.im)*tmpd;  
      P3[2]=(2.0-denom)*tmpd;
      if(P3[2]>(1.0-m_toler))
	{s->re=s->im=0.0;return 0;}
      if (P3[2]<(m_toler-1.0))
	{s->re=0.0;s->im=M_PI;return 0;}
      s->im=acos(P3[2]);
      s->re=aTan2(P3[1],P3[0]);
      return 0;
    }
  norm=sqrt(ns);
  if (norm<m_toler) /* close to origin */
    {
      x=mn=-rr;
      y=0.0;
      a=rr;
      b=0.0;
    }
  else
    {
      denom=1/norm;
      mn=norm-rr;
      /* a point on the circle closest to origin */
      x=(mn*(z.re))*denom;
      y=(mn*(z.im))*denom;
      /* a point on the circle furthest from the origin */
      a=((norm+rr)*(z.re))*denom;
      b=((norm+rr)*(z.im))*denom;
    }

  /* now we project these two points onto the sphere */
  d1=(x*x + y*y +1.0);
  tmpd=1.0/d1;
  P1[0]=(2*x)*tmpd;  P1[1]=(2*y)*tmpd;  P1[2]=(2.0-d1)*tmpd;
  d2=a*a + b*b +1.0;
  tmpd=1.0/d2;
  P2[0]=(2*a)*tmpd;  P2[1]=(2*b)*tmpd;  P2[2]=(2.0-d2)*tmpd;

  /* We may need some point along the geo between these, for they
     themselves might be too far apart to get the correct angle
     between them or to get the right tangent direction from one 
     towards the other. 

     We may use the origin, the euclidean center, or a point
     on the unit circle, depending on which is well placed
     vis-a-vis the endpoints. */

  brk=100.0*m_toler;
  if (mn<=-brk) /* origin is well enclosed; use it. */
    {
      midflag=1;
      P3[0]=P3[1]=0;
      P3[2]=1.0;
    }
  else if (mn<=brk && norm>2) /* use pt on unit circle in direction
			       of center. */
    {
      midflag=1;
      P3[0]=z.re/norm;
      P3[1]=z.im/norm;
      P3[2]=0.0;
    }
  
  if (midflag) /* use pt along geo; radius in two parts */
    {
      if ((d1=P1[0]*P3[0]+P1[1]*P3[1]+P1[2]*P3[2])>=1.0) 
	d1=m_toler;
      if ((d2=P2[0]*P3[0]+P2[1]*P3[1]+P2[2]*P3[2])>=1.0) 
	d2=m_toler;
      ang13=acos(d1);
      ang23=acos(d2);
      rad=(ang13+ang23)/2.0;
      if (ang13<ang23) {E[0]=P1[0];E[1]=P1[1];E[2]=P1[2];}
      else {E[0]=P2[0];E[1]=P2[1];E[2]=P2[2];}

      /* Use E and P3 to find center; tangent direction from E
	 toward P3. */

      v.re=aTan2(E[1],E[0]);
      v.im=acos(E[2]);
      w.re=aTan2(P3[1],P3[0]);
      w.im=acos(P3[2]);
      tangent(v,w,T);
    }
  else
    {
      if ((d1=P1[0]*P2[0]+P1[1]*P2[1]+P1[2]*P2[2])>=1.0) 
	d1=m_toler;
      rad=acos(d1)/2.0;
      E[0]=P1[0];E[1]=P1[1];E[2]=P1[2];
      v.re=aTan2(E[1],E[0]);
      v.im=acos(E[2]);
      w.re=aTan2(P2[1],P2[0]);
      w.im=acos(P2[2]);
      tangent(v,w,T);
    }

  /* C will be the rectangular coordinates of the center */

  C[0]=E[0]*cos(rad)+T[0]*sin(rad);
  C[1]=E[1]*cos(rad)+T[1]*sin(rad);
  C[2]=E[2]*cos(rad)+T[2]*sin(rad);
  
  *sr=rad;

  if (r<0) /* wanted outside of circle */
    {
      *sr=M_PI-rad;
      C[0] *= -1.0;
      C[1] *= -1.0;
      C[2] *= -1.0;
    }
  if (C[2]>1-m_toler)
    {s->re=s->im=0.0;return 1;}
  if (C[2]<(m_toler-1.0))
    {s->re=0.0;s->im=M_PI;return 1;}
  s->im=acos(C[2]);
  s->re=aTan2(C[1],C[0]);
  return 1;
}/* e_to_s_data */

double aTan2(double y,double x)
     /* patched version of atan2 to error-check */
{
  if ((fabs(y)+fabs(x))<m_toler) return 0;
  return atan2(y,x);
} /* aTan2 */

complex sph_tri_center(complex z1,complex z2,complex z3)
     /* given points on sphere, finds barycenter of
triangle they form; inside determined by orientation. */
{
  double X[3],Y[3],Z[3],M[3],C[3],D[3],vn;

  s_pt_to_vec(z1,X);
  s_pt_to_vec(z2,Y);
  s_pt_to_vec(z3,Z);
  M[0]=(X[0]+Y[0]+Z[0])/3.0;
  M[1]=(X[1]+Y[1]+Z[1])/3.0;
  M[2]=(X[2]+Y[2]+Z[2])/3.0;
  vn=cross_prod(Y,X,C); /* C=YxX */
  if (vn<m_toler) /* almost parallel */
    {
      vn=cross_prod(Z,Y,D);
      if (vn<m_toler || dot_prod(D,X)<0) /* M should be good. */
	{
	  return (proj_vec_to_sph(M[0],M[1],M[2]));
	}
      return (proj_vec_to_sph((-1.0)*M[0],(-1.0)*M[1],(-1.0)*M[2]));
    }
  if (vec_norm(M)<m_toler) /* almost coplanar */
    return (proj_vec_to_sph(-C[0],-C[1],-C[2]));
  if (dot_prod(C,Z)<0) return (proj_vec_to_sph(M[0],M[1],M[2]));
  return (proj_vec_to_sph((-1.0)*M[0],(-1.0)*M[1],(-1.0)*M[2]));
} /* sph_tri_center */

int pt_in_sph_tri(complex pt,complex z1,complex z2,complex z3)
     /* true if pt in given tri, assuming corners in order for sph 
CONVEX tri; may fail for non-convex tri. */
{
  double X[3],Y[3],Z[3],P[3],C[3];

  s_pt_to_vec(pt,P);
  s_pt_to_vec(z1,X);
  s_pt_to_vec(z2,Y);
  s_pt_to_vec(z3,Z);
  cross_prod(Y,X,C);
  if (dot_prod(P,C)>0) return 0;
  cross_prod(Z,Y,C);
  if (dot_prod(P,C)>0) return 0;
  cross_prod(X,Z,C);
  if (dot_prod(P,C)>0) return 0;
  return 1;
} /* pt_in_sph_tri */


/* ======================= matrix/vector operations =============== */

int sph_rotation(double alpha,double beta,double gama,
		 double M[3][3],double invM[3][3])
     /* Apply new rotations to matrix and its inverse; order of 
application = z, y, then x */ 
{
  int i,j;
  double X[3][3],Y[3][3],Z[3][3],S[3][3],nM[3][3],ninvM[3][3];

  for (i=0;i<3;i++) 
    {
      for (j=0;j<3;j++)
	X[i][j]=Y[i][j]=Z[i][j]=nM[i][j]=ninvM[i][j]=0.0;
      X[i][i]=Y[i][i]=Z[i][i]=nM[i][i]=ninvM[i][i]=1.0;
    }
  /*  individual rotation matrices */

  X[1][1]=X[2][2]=cos(alpha);
  X[2][1]=sin(alpha);
  X[1][2]=(-1.0)*X[2][1]; 

  Y[0][0]=Y[2][2]=cos(beta);
  Y[2][0]=sin(beta);
  Y[0][2]= (-1.0)*Y[2][0];
	
  Z[0][0]=Z[1][1]=cos(gama);
  Z[1][0]=sin(gama);
  Z[0][1]=(-1.0)*Z[1][0];

  /* multiply together */
  mat_mult(X,Y,S);
  mat_mult(S,Z,nM); /* nM = new rotation matrix */
  /* for inverse, only adjustments needed are these */
  X[2][1] *= (-1.0);
  X[1][2] *= (-1.0);
  Y[2][0] *= (-1.0);
  Y[0][2] *= (-1.0);
  Z[0][1] *= (-1.0);
  Z[1][0] *= (-1.0);
  /* multiply in correct order for 'left' inverse */
  mat_mult(Z, Y, S);
  mat_mult(S, X, ninvM);
  /* apply to given matrices */
  mat_mult(nM,M,S);
  for (i=0;i<3;i++) for (j=0;j<3;j++) M[i][j]=S[i][j];
  mat_mult(invM,ninvM,S);
  for (i=0;i<3;i++) for (j=0;j<3;j++) invM[i][j]=S[i][j];
  return 1;
} /*rot_mat */

int mat_mult(double A[3][3],double B[3][3],double C[3][3])
     /* C=AB */
{
  int i, j;
  for(i=0;i<3;i++) for(j=0;j<3;j++)
    C[i][j]=A[i][0]*B[0][j]+A[i][1]*B[1][j]+A[i][2]*B[2][j];
  return 1;
} /* mat_mult */

double dot_prod(double V[3],double W[3])
     /* (V,W) */
{
  return (V[0]*W[0]+V[1]*W[1]+V[2]*W[2]);
} /* dot_prod */

double cross_prod(double X[3],double Y[3],double Z[3])
     /* Z=XxY, return |Z|. */
{
  Z[0]=X[1]*Y[2]-X[2]*Y[1];
  Z[1]=X[2]*Y[0]-X[0]*Y[2];
  Z[2]=X[0]*Y[1]-X[1]*Y[0];
  return (sqrt(Z[0]*Z[0]+Z[1]*Z[1]+Z[2]*Z[2]));
} /* cross_prod */

double vec_norm(double X[3])
     /* length of 3-vector */
{
  return (sqrt(X[0]*X[0]+X[1]*X[1]+X[2]*X[2]));
} /* vec_norm */

/* the following routines will each calculate a rotation matrix that fixes */
/* a particular axis (noted in routine's name)*/


/* ============= conversion routines ======================= */
/* NOTE: See book on Geometry of Complex Numbers, by Hans Schwertdfeger.
   Circle/line may be represented as 2x2 complex matrix [A B C D]
   where A and D are read and C= conj(B), so matrix is hermitian.

   When A is nonzero, then Det is -A^2(rho)^2, where rho is the
   eucl radius, B = -A*conj(z) where z is the center, and
   D=A*(z*conj(z)-(rho)^2). Can multiply by any positive real.
   Multiply by -1 and the matrix represents the circle with
   opposite orientation; thus A<0 means the circle bounds the
   disc exterior to it.
   
   When A=0 and |B| non zero, then a straight line. 

*/

int s_to_matrix_data(complex center,double rad,Mobius *C)
     /* given sph radius/center, find 2x2 matrix form of circle.
 
Have to look this up, but it appears that if matrix is [a,b,c,d]
as usual, then

  a=0: straight line (goes through S pole)
  d=0: goes through origin
  b: is -conj(z), where z is eucl center (if finite)
  c: always conj(b)

Always have to distinguish which side of circle is inside. 
     */
{
  double m,R,D;
  complex ez;

  C->a.re=C->a.im=0;
  C->b.re=C->b.im=0;
  C->d.re=C->d.im=0; /* note: C->c is set at end */
  C->flip=1;

  /* contains S pole, so projects to (straight) line */
  if (fabs(center.im+rad-M_PI)<m_toler) 
    {

    /* contains N pole, so line through origin */
      if (fabs(center.im-rad)<m_toler)
	{

	/* vertical line */
	  if (fabs(center.re)<m_toler)   /* center in right hp */
	    C->b.re=0.5;
	  else if (fabs(center.re-M_PI)<m_toler) /* center in left hp */
	    C->b.re=-0.5;

	/* horizontal */
	  else if (fabs(center.re-(M_PI_2))<m_toler) /* center in upper hp */
	    C->b.im=0.5;
	  else if (fabs(center.re-(3*M_PI_2))<m_toler) /* center in lower hp */
	    C->b.im=-0.5;

        /* skew straight line */
	  else if (sin(center.re)>0.0)  /* center in upper hp */
	    {
	      m=-1.0/tan(center.re);
	      C->b.re=m/2.0;C->b.im=0.5;
	    }
	  else  /* center in lower hp */
	    {
	      m=-1.0/tan(center.re);				
	      C->b.re=(-1.0)*m/2.0; C->b.im=-0.5;
	    }
	}

      /* straight line, but NOT through origin */
      else
	{
	  R=sin(center.im-rad)/(1-cos(center.im-rad));
	  /* R = signed eucl distance from origin to the line.
	     when R<0 (ie., (center.im-rad)<0), origin is interior 
	     to the circle */

	/* vertical line */
	  if (fabs(center.re)<m_toler) /* center on pos x-axis */
	    {
	      C->b.re=0.5;
	      C->d.re=R;
	    }
	  else if (fabs(center.re-M_PI)<m_toler) /* on neg x-axis */
	    {
	      C->b.re=-0.5;
	      C->d.re=-R;
	    }

	/* horizontal line*/
	  else if (fabs(center.re-(M_PI_2))<m_toler) /* on pos y-axis */
	    {
	      C->b.im=0.5;
	      C->d.re=R;
	    }
	  else if (fabs(center.re-(3*M_PI_2))<m_toler) /* on neg y-axis */
	    {
	      C->b.im=-0.5;
	      C->d.re=-R;
	    }

	/* skew line: note C->d positive if origin is not inside
	   circle, negative if it is. */
	  else if (rad<(M_PI_2)) /* origin not inside circle */
	    {
	      m=-1.0/tan(center.re);
	      D=R*(sin(center.re)-m*cos(center.re));
	      if (D>0) /* center in upper hp */
		{
		  C->b.re=m/2.0;C->b.im=0.5;
		  C->d.re=D;
		}
	      else	/* center in lower hp */
		{
		  C->b.re=-m/2.0; C->b.im=-0.5;
		  C->d.re=-D;
		}
	    }
	  
	  else if (rad>(M_PI_2))  /* origin is inside circle */
	    {
	      m=-1.0/tan(center.re);
	      D=R*(sin(center.re)-m*cos(center.re));
	      if (D<0)  /* center in upper hp */
		{
		  C->b.re=m/2.0;  C->b.im=0.5;
		  C->d.re=D;
		}
	      else     /* center in lower hp */
		{
		  C->b.re=-m/2.0; C->b.im=-0.5;
		  C->d.re=-D;
		}
	    }
	}
    }
  else                                    /* project to circle */
    {
      s_to_e_data(center,rad,&ez,&R);
      if (R>0.0) /* proper euclidean circle */
	{
	  C->a.re=1; C->a.im=0;
	  C->b.re=-ez.re;  C->b.im=ez.im;   /* B= - conj(z) */
	  C->d.re=cAbs(ez)*cAbs(ez) - R*R; C->d.im=0;
	}
      else  /* negative indicates we want outside of circle */
	{
	  C->a.re=-1; C->a.im=0;
	  C->b=cconj(ez);
	  C->d.re=-cAbs(ez)*cAbs(ez)+R*R; C->d.im=0;
	}
    }
  C->c=cconj(C->b);
  return 1;
} /* s_to_matrix_data */

int matrix_to_s_data(Mobius C,complex *center,double *rad)
     /* given 2x2 matrix form of circle, find sph radius/center. */
{
  double x,y,erad,m,DR;
  complex z;
	
  /* circle is a straight line */
  if (fabs(C.a.re)<m_toler)
    {

    /* through origin */
      if (fabs(C.d.re)<m_toler)
	{
	  center->im=*rad=M_PI_2;

	/* vertical */
	  if (fabs(C.b.im)<m_toler)
	    {
	      if (C.b.re>0) center->re=0;  /* center in right hp */
	      else center->re=M_PI; /* center in left hp */
	    }

	/* horizonal line */
	  else if (fabs(C.b.re)<m_toler)		
	    {
	      if (C.b.im>0) center->re=M_PI_2; /* center in upper hp */
	      else center->re=3*M_PI_2; /* center in lower hp */
	    }
	  else		/* skew line thru origin */		
	    {
	      m=C.b.re/C.b.im;
	      if (C.b.im>0) center->re=aTan2(-1.0,m)+M_PI;
	         /* center in upper hp */
	      else center->re=aTan2(-1.0,m); /* cent in lower hp */
	    }
	  return 1; 
	}

      /* straight line, but NOT through origin */
      else
	{
	  /* the C.d entry determines whether the origin is or
	     is not inside the circle. Compute DR (which is
	     either the value R or the value D in s_to_matrix,
	     depending on the situation): if DR>0, then the
	     origin is NOT inside the circle, else origin IS
	     inside the circle. We basically find the point
	     closest to the origin on the line, project to 
	     point z on sphere, use that for radius and
	     center.im. */


	  if (fabs(C.b.im)<m_toler) /* special case, vertical */
	    DR=(C.b.re<0) ? 0.5*(-C.d.re)/C.b.re :
	      0.5*C.d.re/C.b.re;

	  else 	/* generic case, not vertical */
	    DR=(C.b.im<0) ? 0.5*(-C.d.re)/C.b.im :
	      0.5*C.d.re/C.b.im;

	  if (fabs(C.b.im)<m_toler || fabs(C.b.re)<m_toler) 
	    /* horizontal or vertical */
	    {
	      z=proj_pt_to_sph(DR,0);



	    /* vertical */
	      if (fabs(C.b.im)<m_toler)
		{
		  if (C.b.re > 0.0) /* center on pos x-axis */
		    center->re=0.0;
		  else /* center on neg x-axis */
		    center->re=M_PI;
		}

	    /* horizontal */
	      else if (fabs(C.b.re)<m_toler)
		{
		  if (C.b.im>0.0) /* center on pos y-axis */
		    center->re=M_PI_2;
		  else /* center on neg y-axis */
		    center->re=-M_PI_2;
		}
	    }

	/* skew line */
	  else
	    {
	      m=C.b.re/C.b.im;
	      x=(-DR*m)/(m*m + 1); 
	      y=DR/(m*m+1);  /* x,y coords of point on line 
				nearest the origin (or its
				reflection thru the origin) */
	      z=proj_pt_to_sph(x,y); /* project to sph */

	      if ((DR>0 && C.b.im>0) 
		  || (DR<0 && C.b.im<0)) 
		center->re=z.re; 
	      else center->re=(z.re<=0.0) ? z.re+M_PI :
		    z.re-M_PI;

	    }

	  /* can set center->im and rad from z */
	  if (DR>0) /* origin not in circle */
	    {
	      center->im=z.im/2+M_PI_2;
	      *rad=M_PI_2-z.im/2.0;
	    }
	  else /* origin not in circle */
	    {
	      center->im=M_PI_2-z.im/2.0;
	      *rad=z.im/2.0+M_PI_2;
	    }
	}
      return 1;
    }
  else	/* circle */
    {
      z.re=-C.c.re/C.a.re;
      z.im=-C.c.im/C.a.re;	
      erad=sqrt(z.re*z.re+z.im*z.im-(C.d.re/C.a.re)); 
      if (C.a.re<0) erad *=-1.0;
      /* get eucl center, radius */
      e_to_s_data(z,erad,center,rad);
      return 1;
    }
} /* matrix_to_s_data */

complex proj_pt_to_sph(double x,double y)
{
  double x1,x2,x3,denom;

  denom=x*x+y*y+1.0;
  x1=(2.0*x)/denom;
  x2=(2.0*y)/denom;
  x3=(denom-2.0)/denom;
  return (proj_vec_to_sph(x1,x2,x3));
} /* proj_pt_to_sph */

		 
       
 

  
 




