#include "cp_types.h"
#include "cp_proto.h"

/* 7/2000

Open a file (*.vl) containing a parent packing (optional, occurs first) 
and a sequence of lists of vertices to be the 'variable' vertices
of associated child light packings. 'parentpacking' must
be created by calling program; it will be replaced or used.

If PARENT is given, then allocate the packing space and read it.
Else, check if 'parentpacking' is given and CHECK_COUNT matches its
nodecount, and use that as the parent. Allocate the light packing 
space, read in the lists, create the light packings.

Idea is to create these data files using various decompositions
to specify decompositions into light packings, generally as input 
to various repacking routines. 

Return 0 on error or n, number of light packings.

Input file format (at least for developmental purposes):

CHECK_COUNT: <nodecount of parent>
VERT_LIST_COUNT: n

PARENT: <parent packing>

VERT_LIST: <list-1> (done)
..
..
VERT_LIST: <list-n> (done)

 */

int read_parent_child(char *filename,struct p_data *parentpacking,
		      struct p_light ***plight)
{
  int i,j,v,N,checkcount;
  long ftel;
  char buf[1024],command[64];
  struct p_data *p=parentpacking;
  struct K_data *pK_ptr;
  struct p_light **pl_ptr=NULL;
  FILE *fp=NULL;

  if (!parentpacking || !(fp=fopen(filename,"r"))) 
    {
      sprintf(msgbuf,"parentpacking space not allocated or failure "
	      "to open file %s.",filename);
      emsg();
      goto BOMB;
    }

  /* must find CHECK_COUNT: and VERT_LIST_COUNT: first */
  if (!fscanf(fp,"%127s",buf) || (strcmp(buf,"CHECK_COUNT:"))
      || !fscanf(fp,"%d",&checkcount) || checkcount<4 
      || !fscanf(fp,"%127s",buf) || (strcmp(buf,"VERT_LIST_COUNT:"))
      || !fscanf(fp,"%d",&N) || N<1)
    {
      sprintf(msgbuf,"A parent/child *.vl file must begin with "
	      "'CHECK_COUNT:' and 'VERT_LIST_COUNT:'.");
      emsg();
      goto BOMB;
    }

  /* if PARENT is included and nodecount matches CHECK_COUNT, then
     read that as the parent packing. Otherwise, use given parentpacking 
     if its nodecount matches CHECK_COUNT. Otherwise, error. */
  
  ftel=ftell(fp);
  if (fscanf(fp,"%127s",buf) && !(strcmp(buf,"PARENT:")))
    {
      if (!alloc_pack_space(p,10000,0) || !readpack(fp,p))
	/* if PARENT is indicated but reading it fails, error */
	{
	  sprintf(msgbuf,"Failed in reading given parent or "
		  "allocating space.");
	  emsg();
	  goto BOMB;
	}
    }
  else /* didn't have packing in data, use parentpacking */
    {
      fseek(fp,ftel,SEEK_SET); /* reset file position */
      if (p->nodecount!=checkcount)
	{
	  sprintf(msgbuf,"No packing given in data or as arguement.");
	  emsg();
	  goto BOMB;
	}
    }
  pK_ptr=p->packK_ptr;
  
  /* now create light packing space and read vert lists */
  if (!(pl_ptr=(struct p_light **)calloc(N+1,sizeof(struct p_light *))))
    {
      sprintf(msgbuf,"Failed to allocate space for %d light packings.",
	      N);
      emsg();
      goto BOMB;
    }
    
  for(i=1;i<=N;i++) 
    {
      for (j=1;j<=p->nodecount;j++) pK_ptr[j].mark=0;
      while ((fscanf(fp,"%63s",command)!=EOF)
	     && (strcmp(command,"VERT_LIST:")) );
      if (!strcmp(command,"")) 
	{
	  sprintf(msgbuf,"Failed to locate list #%d.",i);
	  emsg();
	  goto BOMB;
	}
      while ((fscanf(fp,"%63s",command)!=EOF)
	     && sscanf(command,"%d",&v) && v>0 && v<=p->nodecount)
	pK_ptr[v].mark=1;
      if ((!strcmp(command,"") && i<N))
	{
	  sprintf(msgbuf,"Failed to locate list #%d.",i+1);
	  emsg();
	  goto BOMB;
	}
      if (!(pl_ptr[i]=convert_to_p_light(p)))
      {
	sprintf(msgbuf,"Failed to convert list #%d to light packing.",i);
	emsg();
	goto BOMB;
      }
    }
  fclose(fp);
  *plight=pl_ptr;
  return N;
    
 BOMB:
  if (fp) fclose(fp);
  for (i=1;i<=N;i++)
    if (pl_ptr[i]) free_p_light(pl_ptr+i);
  if (pl_ptr) free(pl_ptr);
  return 0;
} /* read_parent_child */
  
  






