#include "cp_types.h"
#include "cp_proto.h"

/* Granulator.c, 8/2001. Taken from ActivePack.c code; decompose 
complex into simply connected pieces. */

/* ==================================================================== 
   global declarations
 ==================================================================== */

/* basic packings */
struct p_data *packdata;
int grain_count;
int **face_org;

/* utility buffers. */
char packname[256];
char emsgbuf[BUFSIZE],buf[BUFSIZE],data[2*BUFSIZE],time_buf[64],*msgbuf;
char param_name[BUFSIZE],param_option[BUFSIZE]; 

/* persistent storage of info for routines */
int *vert_indices,aimnum; /* storage/count of verts subject to adjustment */
double *R0;              /* storage for superstep routines */
int residual_count;
struct Vertlist *residual_list,*residual_ptr;

/* timing stuff */
int elapse_hr,elapse_min,elapse_sec;
long time_in=0.0,time_out=0.0,elapse_time=0.0,global_elapsed=0.0;

/* file pointers */
FILE *data_fp, *packfp=NULL,*cmdfp;

/* procedure prototypes */
struct RedList *build_grain_redlist(struct p_data *p,int seed,
	       int *vec_seed,struct Vertlist *tether,int gen);
int simplify_grain_redchain(struct p_data *p,struct RedList **redlist,
			    struct Vertlist *tether);
struct p_light *extract_residual(struct p_data *p,int **green_list,int *okay);
struct p_light *extract_negotiate(struct p_data *p,int *okay);
int find_far_seed(struct p_data *p,int *base_verts,
		  struct Vertlist *tether,int max);
struct Vertlist *create_tether(struct p_data *p,int *green_list);
extern int e_pack_light_uniform(), h_pack_light_uniform();


/* Idea is to take given complex and break it into simply connected
sub-complexes, approx size determined by granularity n. Should be 
easier to pack and communicate data on these sub-complexes.
We build vertex lists identifying the 'variable' vertices in the
various pieces It's all rather complicated. 


Main Routines:
   Granulator
     -- master control routine, creates/cleans up temp data
   find_far_seed
     -- find a seed for a new grain which is far removed from 
        grains already produced.
   build_grain_redlist
     -- use build_redchain and simplify_grain_redchain, as modified
        for special grain properties.
   label_seed_generations
     -- label generations measured from vec_seed; returns vector
        with info
   build_redchain
     -- Build redchain around seed avoiding poison verts (labeled in 
        util_flag)
   simplify_grain_redchain
     -- cut out inclusions formed by red chain
   extract_light_pack
     -- use redlist to build grain data structures.
   extract_residual
     -- gather left over vert.

We need various temporary information. Status of vertices is kept 
in 'green_list': vert v in a grain has green_list[v]=-nmbr, where 
'nmbr' is the number of that grain. Note, in calling 
'find_far_seed', green_list values changed to positive for 
earliest grain number which is not yet surrounded to identify 
'base_verts' used in search for next seed.

(Careful, util_flag needs to be set before certain
subroutines; in most cases it's used for local info.)
*/

struct Vertlist **Granulator(struct p_data *p,int *grain_count,int n)
/* granulate p with grain size n. There remains a problem: 
A grain might entirely engulf one or more previous grains. I 
think previous grains will form connected union, hence should be 
able to go around preliminary bdry of new grain to see that it's 
engulfing former ones. However, how do we fix it? prevent it, 
as it cleans up repeats on its own bdry, from completing the engulfing? 
Return count of grains, set grains pointer. */
{
  int i,j,g,*green_list=NULL,npl=0,nb=0,na=0,max,size,okay,grain_num=1;
  int new_seed,verts_left,cur_target=1;
  struct RedList *redlist=NULL;
  struct p_light *pltrace;
  struct Vertlist **Verts,*vertlist,*trace,*tether=NULL,*strace;
  struct p_light *tmptrace;

  int debg=0;

  if (!p->status || n<3) return 0;
  max=2*((int)(p->nodecount/((double)(n*n)))+1); /* max number of
						    grains */
  /* set up 'grain' to keep track of grains generated, 
     green_list to keep track of vertices handled, and
     residual_list for those destined to residual grain. */

  Verts=(struct Vertlist **)
    calloc((size_t)(max+1),sizeof(struct Vertlist *));
  *grain_count=0;
  green_list=(int *)calloc((size_t)(p->nodecount+1),sizeof(int));
  if (residual_list) vert_free(&residual_list);
  residual_count=0;
  catalog_faces(p);

  /* do central granule starting at alpha */

  if (!(redlist=build_grain_redlist(p,p->alpha,green_list,tether,n))
      || !(vertlist=mark_light_pack(p,p->alpha,
         &redlist,&green_list,&okay,n,grain_count+1)))
    {
      printf("Failed to form central grain.\n");
      goto KILLIT;
    }
  (*grain_count)++;
  Verts[*grain_count]=vertlist;

  /* create 'tether', list of verts from bdry to first grain */
  tether=create_tether(p,green_list);

  npl++;
  na=pltrace->counts[0];
  nb=pltrace->counts[0]-pltrace->counts[2];
  cur_target=*grain_count; /* cur_target points to earliest grain not yet
			 surrounded -- hence we try to find seed for
			 new grain abutting it. */

  /* loop, adding granules until every vert up to small residual
     set is accounted for. 
     Note: when looking for vert to seed next grain, we try to
     find verts far from the earliest grain in linked list still 
     not totally surrounded. Pass info in green_list: entry > 0 
     means it's poison, but is in the earliest grain, < 0 means it
     is poison because it's in an already surrounded grain. */

  size=(int)(3.0*n/4.0);
  if (size<3) size=3;
  verts_left=p->nodecount-na;

  while (verts_left>n*n)
    {
      new_seed=0;

/* find seed for next new grain */
      grain_num=cur_target;
      while(grain_num<max && grains[grain_num].pl && !new_seed)
	{
	  tmptrace=grains[grain_num].pl;
	  for (j=1;j<=tmptrace->counts[0];j++)
	    /* need temporary change to positive sign in green_list
	       for verts from cur_target grain; these then serve
	       as starting points for search for seed. */
	    if (green_list[(i=tmptrace->orig_indices[j])]==(-grain_num))
	      green_list[i]=grain_num;
	  new_seed=find_far_seed(p,green_list,tether,size);
	  for (j=1;j<=tmptrace->counts[0];j++) /* revert signs */
	    if (green_list[(i=tmptrace->orig_indices[j])]==grain_num)
	      green_list[i]=-grain_num;
	  if (!new_seed) 
	      tmptrace=grains[++grain_num].pl;
	}
      if (!grains[grain_num].pl) /* must have captured all vertices, success */
	goto FINISHUP;

/* create next grain */
      cur_target=grain_num;
      if (redlist) free_redfaces(&redlist);
      if (!(redlist=build_grain_redlist(p,new_seed,green_list,tether,n)))
	goto KILLIT;
      if (!(vertlist=mark_light_pack(p,new_seed,
	 &redlist,&green_list,&okay,n,(*grain_count)+1)))
	{
	  if (okay) goto FINISHUP;
	  else goto KILLIT;    /* ran into some problem */
	}
      (*grain_count)++;
      Verts[*grain_count]=vertlist;

/* maif (debg) goto KILLIT;

y be able to shorten tether */
      if (tether && green_list[tether->v])
	{
	  vert_free(&tether);
	  tether=NULL;
	}
      else if (tether)
	{
	  trace=tether->next;
	  while (trace && trace->next) 
	    {
	      if (!green_list[trace->v])
		trace=trace->next;
	      else /* rest of tether not needed */
		{
		  vert_free(&(trace->next)); 
		  trace->next=NULL;
		}
	    }
	}

/* update counts */
      na += pltrace->counts[0];
      nb += pltrace->counts[0]-pltrace->counts[2];

/* count verts not yet in some grain */
      verts_left=0;
      for (i=1;i<=p->nodecount;i++) if (!green_list[i]) verts_left++;

    } /* end of while */

 FINISHUP:
  destroy_catalog(p->nodecount);
  if (tether) vert_free(&tether);
  if (redlist) free_redfaces(&redlist);

  /* announce results */
  printf("Granulate: %d sub-complexes, avg area %d, avg circum %d.\n"
	 "   Seeds are: ",*grain_count,(int)(na/*grain_count),
	 (int)(nb/*grain_count));
  for (g=1;g<=*grain_count;g++)
    printf("Grain %d: area %d, interiors %d\n",
	   g,(grains[g].pl)->counts[0],(grains[g].pl)->counts[2]);


  /* take care of remaining 'residual' and 'negotiation' vertices. */

if (debg) goto KILLIT;


  if ((vertlist=mark_residual(p,&green_list,&okay)) 
      && okay>0)
    {
      (*grain_count)++;
      Verts[*grain_count]=vertlist;
    }
  else if (!okay) printf("Some error extracting the 'residual' grain.");
  else printf("No residual grain.");

if (debg) goto KILLIT;


  if ((pltrace=extract_negotiate(p,&okay)) 
      && okay>0)
    {
      (*grain_count)++;
      Verts[*grain_count]=vertlist;
    }
  else printf("Some error extracting the 'negotiate' grain.");

if (debg) goto KILLIT;

  if (green_list) {free(green_list);green_list=NULL;}
  /*  vert_free(&residual_list); */
  strace=residual_list;
  while (strace)
    {
      residual_list=residual_list->next;
      free(strace);
      strace=residual_list;
    }

 KILLIT:

if (debg) return 0;

  destroy_catalog(p->nodecount);
  if (tether) vert_free(&tether);
  if (redlist) free_redfaces(&redlist);
  vert_free(&residual_list);
  if (green_list) {free(green_list);green_list=NULL;}
  printf("granulate failed for some reason.\n");
  return 0;
} /* Granulator */


/* Given vertex seed v0 and verts designated as 'poison' want the
smallest simply connected complex containing v0 and avoiding poison
verts. E.g., for n>1 generations, make larger gen verts poison.
Main difficulty: need to identify and fill "holes" formed by 
poison verts -- inclusions. 

  Main approach: verts with util_flag < 0 will be off-limits.
First, make up red chain in some way, eg. by generation. Also
have to find "persistent" entry in red chain, one which is kept
in red chain throughout: we use it as we modify the red chain to
determine which part to keep. Next, walk around red chain; when
outside vert is discovered which is repeated elsewhere on the
outside of the red chain, cut out the intervening chain which
forms an inclusion.*/

struct RedList *build_grain_redlist(struct p_data *p,int seed,
	       int *vec_seed,struct Vertlist *tether,int gen)
/* Given seed and number gen>1, find red chain enclosing smallest 
simply connected sub-complex containing vertices up to and including 
generation gen from seed, but stopping at any 'green' v (ie, 
vec_seed[v]<0). Note vec_seed and tether unchanged. */
{
  int i,j,k,num,o_gen=0,*gen_list,dum;
  int debg=0,out_vert=1,in_vert=0,tick=0,wflag=0;
  struct Vertlist *dum_fdo=NULL;
  struct RedList *redlist=NULL,*rtrace,*keep_ptr=NULL;
  struct K_data *pK_ptr;

  if (!(gen_list=
	label_seed_generations(p,seed,vec_seed,gen-1,&out_vert,&dum))) 
    return NULL; 
  pK_ptr=p->packK_ptr;

/* set generation gen and above verts and green verts to poison */
  for (i=1;i<=p->nodecount;i++) 
    {
      if (gen_list[i]<=0 || gen_list[i]>=gen) 
	pK_ptr[i].util_flag=-1;
      else pK_ptr[i].util_flag=0;
    }
/* remove any isolated poison verts */
  for (i=1;i<=p->nodecount;i++)
    if (pK_ptr[i].util_flag==-1)
      {
	k=0;
	for (j=0;j<=pK_ptr[i].num;j++) 
	  k+= pK_ptr[pK_ptr[i].flower[j]].util_flag;
	if (!k)
	  {
	    pK_ptr[i].util_flag=0;
	    tick++;
	  }
      }

/* build the redchain and set its data */
  dum_fdo=(struct Vertlist *)
    calloc((size_t)1,sizeof(struct Vertlist));
  if (p->rwb_flags) free(p->rwb_flags);
  p->rwb_flags=(int *)calloc((size_t)(p->facecount+1),sizeof(int));
  if (!(redlist=build_redchain(p,seed,dum_fdo)))
    {
      fprintf(stderr,"Failed to build redchain.");
      free(p->rwb_flags);
      p->rwb_flags=NULL;
      goto BOMB;
    }

if (debg) record_redlist(redlist,p->facecount);

/* define a persistent redlist entry (so we don't discard the
   wrong portion of redlist) */

/* if out_vert isn't green, it's a bdry vert and we need to work 
   our way back in generation towards gen. */
  if (gen_list[out_vert]>0)
    while ((o_gen=gen_list[out_vert])>gen)
      {
	num=pK_ptr[out_vert].num;
	for (j=0;j<=num;j++)
	  if (gen_list[pK_ptr[out_vert].flower[j]]>0 
	      && gen_list[pK_ptr[out_vert].flower[j]]<o_gen)
	    {
	      out_vert=pK_ptr[out_vert].flower[j];
	      j=num+1;
	    }
      }
  /* out_vert should now be in edge of desired red chain; get
     in_vert, one earlier generation nghb. */
      
  num=pK_ptr[out_vert].num;
  tick=0;
  for (j=0;j<=num;j++)
    if ((gen_list[out_vert]>0 && gen_list[(i=pK_ptr[out_vert].flower[j])]>0
	 && gen_list[i]<o_gen)
	|| (gen_list[out_vert]<=0 && gen_list[pK_ptr[out_vert].flower[j]]>0))
      {
	in_vert=pK_ptr[out_vert].flower[j];
	j=num+1;
	tick++;
      }
  if (!tick) 
    {
      fprintf(stderr,"Didn't find in_vert.");
      goto BOMB;
    }

  /* identify a redlist entry whose face contains in/out_vert's */

  rtrace=redlist;
  while (rtrace!=redlist || !(wflag++))
    {
      if (check_face(p,rtrace->face,in_vert,out_vert)>=0
	  || check_face(p,rtrace->face,out_vert,in_vert)>=0)
	{
	  keep_ptr=rtrace;
	  rtrace=redlist;
	}
      else rtrace=rtrace->next;
    }
  if (!keep_ptr) 
    {
      fprintf(stderr,"Didn't find persistent face.");
      goto BOMB;
    } 

  /* set redlist to start at persistent entry, then process red 
     chain to remove inclusions */

  redlist=keep_ptr;
  if (!simplify_grain_redchain(p,&redlist,tether)) 
    {
      fprintf(stderr,"Failed to simplify properly.");
      goto BOMB;
    }
  return redlist;

 BOMB:
  if (gen_list) free(gen_list);
  if (redlist) free_redfaces(&redlist);
  if (dum_fdo) vert_free(&dum_fdo);
  return NULL;
} /* build_grain_redlist */

int simplify_grain_redchain(struct p_data *p,struct RedList **redlist,
			    struct Vertlist *tether)
/* Given a red chain, cut 'inclusions' to form new red chain. 
E.g., if p is simply connected, can cookie this to get smallest 
simply connected sub-complex containing the original red chain. 
Note that the first entry of redlist is considered 'persistent'; 
that is, it must remain in the red chain, hence it helps define 
which parts of chain are kept and which removed as inclusions 
during processing. Tether is chain of verts which shouldn't be 
crossed: it prevents previous grains from being treated as an 
inclusion to be filled in. Return 0 on error. */
{
  int i,n,nn,*face_ptr,findex,bindex,eindex,indx,nindex,num;
  int trip_flag,stop_flag=0,cface,vert,switch_flag,hits,w,u,v,f;
  int debg=0,last_vert,wflag=0,prev_vert=0,next_vert,dont_cut;
  struct RedList *rtrace,*upstream,*stop_ptr,*backtrace,*foretrace;
  struct RedList *check_ptr,*keep_ptr,*newface;
  struct RedList *cut_begin,*cut_end,*new_red,*hold;
  struct Vertlist *ttrace;
  struct K_data *pK_ptr;

  if (!p->status || (*redlist)==NULL) return 0;
  keep_ptr=*redlist; /* first entry of incoming redlist must be
			the chosen 'persistent' entry; this is the
			responsibility of the calling routine. */
  pK_ptr=p->packK_ptr;

if (debg) record_redlist(*redlist,p->facecount);

/* Modify red chain to fill 'inclusions'. Method: 
   walk around outer edge verts, and when one is repeated, 
   discard the intervening red chain (insuring that keep_ptr
   is kept). Repeat as needed. Resulting red chain should define 
   simply connected sub-complex (assuming p is simply connected).
*/
   
  while (!stop_flag) /* main loop: go until pruning done */
    {
      stop_flag=1;
      trip_flag=0;
      stop_ptr=*redlist;
      last_vert=0;
      while (!trip_flag) /* go through whole red chain */
	{
	  upstream=*redlist;
	  cface=(*redlist)->face;
	  indx=nghb_tri(p,(*redlist)->next->face,cface);
	  vert=p->faces[cface].vert[indx];
	  /* if blue, take care of isolated vert first */
	  if ((*redlist)->prev->face==(*redlist)->next->face
	      && p->faces[cface].vert[(indx+2) % 3]!=last_vert)
	    last_vert=vert=p->faces[cface].vert[(indx+2) % 3];
	  else last_vert=0;
	  face_ptr=face_org[vert]+1;
	  num=pK_ptr[vert].num;
/* find index of cface */
	  findex=0;
	  while (*(face_ptr+findex)!=cface && findex<(num-1)) findex++;
/* find index of downstream face sharing vert; note, redlist gets 
   shifted at least one position downstream. */
	  i=findex;
	  while ( (*redlist)->next->face==*(face_ptr+((i+num-1) % num))
		  && (*redlist)->next->face!=cface ) 
	    {
	      *redlist=(*redlist)->next;
	      if ((*redlist)==stop_ptr) 
		trip_flag++;
	      i--;
	    }
	  if ((*redlist)->next->face==cface 
	      && (*redlist)->prev->face!=cface) /* red wraps single vert */
	    return 0;
	  eindex= (num+i) % num;

/* find index of upstream face */
	  i=findex;
	  while ( upstream->prev->face==*(face_ptr+((i+1) % num))
		  && upstream->prev->face!=cface )
	    {
	      upstream=upstream->prev;
	      i++;
	    }
	  if (upstream->prev->face==cface 
	      && upstream->next->face!=cface) 
	    return 0; 
	  bindex=(i % num);
/* go around outer verts to see if we come across vert again */
	  rtrace=*redlist;
	  while (rtrace!=upstream
		 && p->faces[rtrace->face].vert[nghb_tri(p,
		    rtrace->next->face,rtrace->face)]!=vert)
	    rtrace=rtrace->next;
	  if (rtrace!=upstream) /* got a hit */
	    {
	      /* save beginning/end of red chain segment to kill */
	      cut_begin=(*redlist)->next;
	      cut_end=rtrace->prev;
	      /* get index of this face */
	      nindex=0;
	      while (*(face_ptr+nindex)!=rtrace->face && nindex<(num-1))
		nindex++;
	      /* decide which portion of chain to toss */
	      if (pK_ptr[vert].bdry_flag)
		{
		  if (eindex>nindex) switch_flag=1;
		  else switch_flag=0;
		}
	      else
		{
		  switch_flag=0;
		  check_ptr=cut_begin;
		  while (check_ptr!=rtrace)
		    {
		      if (check_ptr==keep_ptr)
			{
			  switch_flag=1;
			  check_ptr=rtrace;
			}
		      else check_ptr=check_ptr->next;
		    }
		}

	      if (switch_flag) /* switch portion to be tossed */
		{
		  eindex=nindex;
		  while (rtrace->next->face
			 ==*(face_ptr+((eindex+num-1) % num)))
		    {
		      rtrace=rtrace->next;
		      eindex--;
		    }
		  cut_begin=rtrace->next;
		  cut_end=upstream->prev;
		  *redlist=rtrace;
		  nindex=bindex;
		}
	      n=(eindex+num-nindex-1) % num;
	      indx=(eindex+num-1) % num;

	      /* don't cut if new segment would cross tether */
	      ttrace=tether;
	      if (ttrace) prev_vert=ttrace->v;
	      hits=next_vert=0;
	      while (ttrace && !hits)
		{
		  if (ttrace->v==vert) 
		    {
		      hits=1;
		      if (ttrace->next) next_vert=ttrace->next->v;
		    }
		  else
		    {
		      prev_vert=ttrace->v;
		      ttrace=ttrace->next;
		    }
		}
	      dont_cut=0;
	      if (hits) /* would we cross tether? */
		for (i=nindex;i<=n;i++)
		  if ((nn=pK_ptr[vert].flower[i % num])==prev_vert
		      || nn==next_vert)
		    {
		      dont_cut=1;
		      i=n;
		    }
	      if (!dont_cut)
		{     
		  /* add new section of red chain */
		  while (n>0)
		    {
		      (*redlist)->next=new_red=(struct RedList *)
			calloc((size_t)1,sizeof(struct RedList));
		      new_red->prev=*redlist;
		      new_red->face=*(face_ptr+indx);
		      *redlist=new_red;
		      indx=(indx+num-1) % num;
		      n--;
		    }
		  rtrace->prev=*redlist;
		  stop_ptr=*redlist=(*redlist)->next=rtrace;
		  stop_flag=0; 
		  /* destroy cutoff segment */
		  cut_end->next=NULL;
		  rtrace=cut_begin;
		  while (rtrace!=NULL)
		    {
		      hold=rtrace->next;
		      free(rtrace);
		      rtrace=hold;
		    }
		  last_vert=0;
		}
	    } /* took care of hit */
	  if (!last_vert) *redlist=(*redlist)->next; /* last_vert true would 
						    mean face was blue,
						    nothing happened, but
						    now need to handle 
						    its second vert */
	  if ((*redlist)==stop_ptr) trip_flag++;
	} /* end of while (go to new 'vert') */
    } /* end of main loop */

  /* One final processing step for red chain: catch instances
     at vert v (int in parent) with red chain wrapping clockwise
     around v and getting all but one face in star of v. We will
     make v interior to sub-complex by shorting red chain across
     omitted face. One adjustment can lead to others,
     so keep cycling through list until no more adjustments.
     (Note: this adjustment can't cause complex to cross tether.) */

  hits=1;
  while (hits)
    {
      hits=0;
      rtrace=*redlist;
      wflag=0;
      while (rtrace!=(*redlist) || !(wflag++))
	{
	  indx=nghb_tri(p,rtrace->next->face,rtrace->face);
	  v=p->faces[rtrace->face].vert[indx];
	  if (!pK_ptr[v].bdry_flag)
	    {
	      backtrace=rtrace;
	      while (p->faces[backtrace->prev->face].vert[
		nghb_tri(p,backtrace->face,backtrace->prev->face)]==v)
		backtrace=backtrace->prev;
	      foretrace=rtrace->next;
	      while (p->faces[foretrace->face].vert[
		nghb_tri(p,foretrace->next->face,foretrace->face)]==v)
		foretrace=foretrace->next;
	      w=p->faces[backtrace->face].vert[
		(nghb_tri(p,backtrace->next->face,backtrace->face)+2) % 3];
	      u=p->faces[foretrace->face].vert[
		(nghb_tri(p,foretrace->prev->face,foretrace->face)+2) % 3];
	      if ((f=what_face(p,u,v,w))) /* <u,v,w> form face? do shortcut */
		{
		  newface=(struct RedList *)
		    calloc((size_t)1,sizeof(struct RedList));
		  newface->face=f;
		  newface->prev=backtrace;
		  newface->next=backtrace=backtrace->next;
		  /* cut out part of redchain going clockwise about v */
		  while (newface->next!=foretrace)
		    {
		      newface->next=backtrace->next;
		      if (backtrace==(*redlist)) *redlist=newface;
		      free(backtrace);
		      backtrace=newface->next;
		    }
		  newface->prev->next=newface;
		  newface->next->prev=newface;
		  rtrace=newface->prev;
		  wflag=0;
		  hits=1;
		}
	    }
	  rtrace=rtrace->next;
	} /* end of inner while */
    } /* end of outer while */

  return (1); /* success */
} /* simplify_grain_redchain */


struct p_light *extract_residual(struct p_data *p,int **green_list,int *okay)
/* Form p_light structure with remaining vertices, ie. green_list[v]==0 
or in rsidual_list -- even if they don't form a legit complex,  
okay=1 means no error, even if we return a NULL. */
{
  int i,j,k,m,v,tick=1,w,seed=0,vcount,icount=0;
  struct Vertlist *strace=NULL;
  struct p_light *pl;
  struct K_data *pK_ptr;

  int debg=0;
  
  pK_ptr=p->packK_ptr;
  if (!(*green_list) && !residual_list)
    {*okay=-1;return NULL;}

  /* indicate residual verts which are interior to parent packing */
  for (i=1;i<=p->nodecount;i++) pK_ptr[i].util_flag=0;
  strace=residual_list;
  while (strace) /* from accumulated residual list */
    {
      v=strace->v;
      if (!pK_ptr[v].bdry_flag) pK_ptr[v].util_flag=1;
      icount++;
      strace=strace->next;
    }
  for (i=p->nodecount;i>0;i--) /* other verts not picked up */
    if (!((*green_list)[i]) && !pK_ptr[i].bdry_flag) 
      {
	pK_ptr[i].util_flag=1;
	icount++;
	seed=i;
      }
   if (!seed) /* no interior verts */
    {
      *okay=-1;
      return NULL;
    }

   /* bdry are just nghb's of the relative interiors; ie,
      neglect isolated bdry. */
   vcount=icount;

   for (i=0;i<=p->nodecount;i++) 
     if (pK_ptr[i].util_flag>0)
       for (j=0;j<=pK_ptr[i].num;j++)
	 if (!pK_ptr[(k=pK_ptr[i].flower[j])].util_flag)
	   {
	     vcount++;
	     pK_ptr[k].util_flag=-vcount;
	   }

  *okay=0;
  pl=(struct p_light *)calloc((size_t)1,sizeof(struct p_light));
  pl->counts=(int *)calloc((size_t)5,sizeof(int));
  pl->counts[0]=vcount;
  pl->counts[1]=(int)(p->hes);
  pl->counts[2]=icount;
  pl->orig_indices=(int *)calloc((size_t)(pl->counts[0]+1),sizeof(int));
  pl->radii=(double *)
    calloc((size_t)(pl->counts[0]+1),sizeof(double));

if (debg) return NULL;

  /* set util_flag to new index and find size of var_indices block */

  for (i=1;i<=p->nodecount;i++)
    if (pK_ptr[i].util_flag>0) /* interior */
      {
	icount++;
	pK_ptr[i].util_flag=icount;
	pl->counts[3] += pK_ptr[i].num+1;
      }

if (debg) return NULL;

  /* store interior block info, orig_indices, and radii */
  pl->var_indices=(int *)calloc((size_t)(pl->counts[3]+1),sizeof(int));
  for (i=1;i<=p->nodecount;i++)
    {
      k=pK_ptr[i].util_flag;
      if (k>0) /* interior */
	{
	  pl->var_indices[tick++]=pK_ptr[i].num;
	  /* create flower list. Note: every flower should be
	     interior, so don't record the redundant last petal index. */
	  for (j=0;j<pK_ptr[i].num;j++)
	    {
	      w=pK_ptr[i].flower[j];
	      if (!(m=pK_ptr[w].util_flag)) 
		goto SCREWUP2; /* every petal should be an interior or 
				 bdry, hence >0 or <0 util_flag, resp. */
	      m = (m<0) ? -m : m;
	      pl->var_indices[tick++] = m;
	    }
	  pl->radii[k]=p->packR_ptr[i].rad;
	  pl->orig_indices[k]=i;
	}
      else if (k<0) /* bdry */
	{
	  pl->radii[-k]=p->packR_ptr[i].rad;
	  pl->orig_indices[-k]=i;
	}
    }

if (debg) return NULL;

  /* success */
   for (i=p->nodecount;i>0;i--) 
     if (pK_ptr[i].util_flag) (*green_list)[i]=-(grain_count+1);
  pl->counts[4]=p->nodecount;
  *okay=seed;
  return pl; 

 SCREWUP2:
  free_p_light(&pl);
  *okay=0;
  return NULL;
} /* extract_residual */

struct p_light *extract_negotiate(struct p_data *p,int *okay)
     /* Grain bdry's intersect and we need a p_light structure 
to pack these. Thus we identify all interior verts which lie in
the bdry of some grain and create p_light. (Experience may show
we should include this in the residual pack.) okay=1 means
no packing, but no error. */
{
  int i,j,k=0,m,tick=1,w,seed=0,vcount,icount=0;
  struct p_light *pl;
  struct K_data *pK_ptr;
  
  pK_ptr=p->packK_ptr;
  for (i=1;i<=p->nodecount;i++) pK_ptr[i].util_flag=0;

  for (i=1;i<=grain_count;i++)
    {
      pl=grains[i].pl;
      for (j=pl->counts[2]+1;j<=pl->counts[0];j++)
	if (!pK_ptr[(k=pl->orig_indices[j])].bdry_flag
	    && !pK_ptr[k].util_flag)
	  pK_ptr[k].util_flag=++icount; /* interior of negotiate packing */
    }
  seed=k; /* treat last indexed interior as 'seed' */
  if (!icount) /* must be something wrong unless there is only
		  one grain = whole original packing. */
    {
      if (grain_count==1) *okay=1; 
      else *okay=0;
      return NULL;
    }

   /* bdry are just nghb's of the relative interiors; ie,
      neglect isolated bdry. */
   vcount=icount;

   for (i=p->nodecount;i>0;i--) 
     if (pK_ptr[i].util_flag>0)
       for (j=0;j<=pK_ptr[i].num;j++)
	 if (!pK_ptr[(k=pK_ptr[i].flower[j])].util_flag)
	   {
	     vcount++;
	     pK_ptr[k].util_flag=-vcount;
	   }

  *okay=0;
  pl=(struct p_light *)calloc((size_t)1,sizeof(struct p_light));
  pl->counts=(int *)calloc((size_t)5,sizeof(int));
  pl->counts[0]=vcount;
  pl->counts[1]=(int)p->hes;
  pl->counts[2]=icount;
  pl->orig_indices=(int *)calloc((size_t)(pl->counts[0]+1),sizeof(int));
  pl->radii=(double *)
    calloc((size_t)(pl->counts[0]+1),sizeof(double));

  /* set util_flag to new index and find size of var_indices block */

  icount=0;
  for (i=1;i<=p->nodecount;i++)
    if (pK_ptr[i].util_flag>0) /* interior */
      {
	icount++;
	pl->counts[3] += pK_ptr[i].num+1;
	pK_ptr[i].util_flag=icount;
      }

  /* store interior block info, orig_indices, and radii */
  pl->var_indices=(int *)calloc((size_t)(pl->counts[3]+1),sizeof(int));
  for (i=1;i<=p->nodecount;i++)
    {
      k=pK_ptr[i].util_flag;
      if (k>0) /* interior */
	{
	  pl->var_indices[tick++]=pK_ptr[i].num;
	  /* create flower list. Note: every flower should be
	     interior, so don't record the redundant last petal index. */
	  for (j=0;j<pK_ptr[i].num;j++)
	    {
	      w=pK_ptr[i].flower[j];
	      if (!(m=pK_ptr[w].util_flag)) 
		goto SCREWUP3; /* every petal should be an interior or 
				 bdry, hence >0 or <0 util_flag, resp. */
	      m = (m<0) ? -m : m;
	      pl->var_indices[tick++] = m;
	    }
	  pl->radii[k]=p->packR_ptr[i].rad;
	  pl->orig_indices[k]=i;
	}
      else if (k<0) /* bdry */
	{
	  pl->radii[-k]=p->packR_ptr[i].rad;
	  pl->orig_indices[-k]=i;
	}
    }
  /* success */
  pl->counts[4]=p->nodecount;
  *okay=seed;
  return pl; 

 SCREWUP3:
  free_p_light(&pl);
  *okay=0;
  return NULL;
} /* extract_negotiate */

int find_far_seed(struct p_data *p,int *base_verts,
		  struct Vertlist *tether,int max) 
/* Return index of vert v as far (up to max) generations from 
'base_verts' v (ie, base_verts[v]>0) as possible, staying away 
from 'poison' w (base_verts[w]<0) as far as possible, and trying
(a little) to stay away from bdry verts. 

Status is kept in util_flag: 0=still free, >0 marks generation
from base_verts, -1=poison or 'tether' vert, -2 fender layers to
keep away from poison; current best choice is 'return_vert'. 

Idea is to use dynamic linked lists: targens is list of next 
generation, poigens is list of latest generation fending off 
from the poison verts. Each pass through, (a) label free verts 
next to current targens verts and put them in new targens list.
(b) pass through poigens and label as -2 any free vert ngbh's.
In step (a) also relabel any -2 verts with their generation,
but only add to targens if 'strategy' flag is set, meaning we
ran out of free verts and are forced to eat into the layers 
built up around the poison verts to try to reach max.

Return 0 if no nonpoison verts are found in first generation 
next to base_verts or if non is found in generation <= max. */
{
  int i,j,v,gen_count=2,hits,return_vert=0;
  int strategy=0;
  struct K_data *pK_ptr;
  struct Vertlist *targens=NULL,*poigens=NULL,*ttrace,*ptrace;
  struct Vertlist *hold_targens=NULL,*vertlist=NULL,*vtrace;

  if (!p->status || !base_verts) return 0;
  pK_ptr=p->packK_ptr;
  for (i=1;i<=p->nodecount;i++) pK_ptr[i].util_flag=0;
  if (max<=0) max=p->nodecount; /* no bound on generations */

  /* set up target/poison lists */
  targens=ttrace=(struct Vertlist *)
    calloc((size_t)1,sizeof(struct Vertlist));
  poigens=ptrace=(struct Vertlist *)
    calloc((size_t)1,sizeof(struct Vertlist));
  hits=0;
  for (i=1;i<=p->nodecount;i++)
    {
      if (base_verts[i]>0)
	{
	  ttrace=ttrace->next=(struct Vertlist *)
	  calloc((size_t)1,sizeof(struct Vertlist));
	  ttrace->v=i;
	  pK_ptr[i].util_flag=1;
	  hits++;
	}
      else if (base_verts[i]<0 || pK_ptr[i].bdry_flag)
	{
	  ptrace=ptrace->next=(struct Vertlist *)
	    calloc((size_t)1,sizeof(struct Vertlist));
	  ptrace->v=i;
	  if (base_verts[i]<0) pK_ptr[i].util_flag=-1;
	  else pK_ptr[i].util_flag=-2;
	}
    }
  /* 'tether' is a linked list of verts from first grain out to
     some bdry vert; make these poison to prevent subsequent grains
     from wrapping around existing ones. */
  vtrace=tether;
  while (vtrace)
    {
      if (pK_ptr[i=vtrace->v].util_flag==1)
	{
	  pK_ptr[i].util_flag=-1;
	  hits--;
	}
      else pK_ptr[i].util_flag=-1; 
      vtrace=vtrace->next;
    }
  if (!hits) /* base_verts verts have no free neighbors */ 
    {
      return_vert=0;
      goto OUTAHERE;
    }
  ttrace=targens;
  targens=targens->next; /* first location wasn't used */
  free(ttrace);
  ptrace=poigens;
  poigens=poigens->next; /* first location wasn't used */
  free(ptrace);

  /* main loop */
  hits=0;
  while (!(hits++) && gen_count<=max 
	 && (targens || (strategy && hold_targens)))
    {
      /* use or throw out previous target list */
      if (!targens) targens=hold_targens; 
      else vert_free(&hold_targens);
      vertlist=hold_targens=targens; /* process old target list */
      targens=ttrace=(struct Vertlist *)  /* create next target list */
	calloc((size_t)1,sizeof(struct Vertlist));
      /* label any unlabeled nghb's of base_verts, or if strategy==1,
	 any nghb's labeled <= -2 (note we don't relabel poison) */
      do
	{
	  v=vertlist->v;
	  for (i=0;i<=pK_ptr[v].num;i++)
	    {
	      if (!pK_ptr[(j=pK_ptr[v].flower[i])].util_flag)
		{
		  strategy=0;
		  return_vert=j;
		  pK_ptr[j].util_flag=gen_count;
		  ttrace=ttrace->next=(struct Vertlist *)
		    calloc((size_t)1,sizeof(struct Vertlist));
		  ttrace->v=j;
		  hits=0;
		}
	      else if (pK_ptr[j].util_flag <= -2)
		{
		  if (strategy) /* no free verts, so add this to list */
		    {
		      return_vert=j;
		      pK_ptr[j].util_flag=gen_count;
		      ttrace=ttrace->next=(struct Vertlist *)
			calloc((size_t)1,sizeof(struct Vertlist));
		      ttrace->v=j;
		      hits=0;
		    }	      
		  else /* don't add to list, but label generation */
		    pK_ptr[j].util_flag=gen_count;
		}
	    }
	  vertlist=vertlist->next;
	} while (vertlist);
      ttrace=targens;
      targens=targens->next; /* first position was empty */
      free(ttrace);

      /* what's status? */
      if (targens) /* found verts in new generation, so throw
		      old list out, continue as normal */
	vert_free(&hold_targens);
      else if (gen_count==2) /* first pass and didn't find anything */
	{
	  return_vert=0;
	  goto OUTAHERE;
	}
      else  /* ran out of candidates; shift strategy to allow moving 
	       closer to poison */
	{
	  strategy=1;
	  gen_count--;
	}

      /* now go through poison, mark nghb's with -2 */

      if (poigens)
	{
	  vertlist=poigens; /* process old poison list */
	  poigens=ptrace=   /* create next poison list */
	    (struct Vertlist *)calloc((size_t)1,sizeof(struct Vertlist));
	  do
	    {
	      v=vertlist->v;
	      for (i=0;i<=pK_ptr[v].num;i++)
		if (!pK_ptr[(j=pK_ptr[v].flower[i])].util_flag)
		  {
		    pK_ptr[j].util_flag = -2;
		    ptrace=ptrace->next=(struct Vertlist *)
		      calloc((size_t)1,sizeof(struct Vertlist));
		    ptrace->v=j;
		  }
	      vertlist=vertlist->next;
	    } while (vertlist);
	  ptrace=poigens;
	  poigens=poigens->next; /* first position was empty */
	  free(ptrace);
	}

      gen_count++;
    } /* end of outer while */

    OUTAHERE:
      vert_free(&targens);
      vert_free(&poigens);
      return return_vert;
} /* find_far_seed */

struct Vertlist *create_tether(struct p_data *p,int *green_list)
/* list of verts running from bdry to verts set in green_list. */
{
  int hits=0,gen_count=1,v,vv=0,i,j;
  struct Vertlist *genlist=NULL,*final_list=NULL,*vertlist=NULL;
  struct Vertlist *trace,*gtrace;
  struct K_data *pK_ptr;

  if (!p->status || !p->num_bdry_comp || !green_list) 
    return final_list;
  pK_ptr=p->packK_ptr;
  genlist=trace=(struct Vertlist *)
    calloc((size_t)1,sizeof(struct Vertlist));
  for (i=1;i<=p->nodecount;i++) 
    {
      pK_ptr[i].util_flag=0;
      if (green_list[i])
	{
	  if (pK_ptr[i].bdry_flag)  /* green_list has a bdry vert */
	    {
	      vert_free(&genlist);
	      return final_list; 
	    }
	  pK_ptr[i].util_flag=gen_count;
	  trace=trace->next=(struct Vertlist *)
	    calloc((size_t)1,sizeof(struct Vertlist));
	  trace->v=i;
	}
    }
  trace=genlist;
  genlist=genlist->next; /* first spot empty */
  free(trace);

  /* mark generations out to first bdry vert */
  while (!(hits++))
    {
      gen_count++;
      vertlist=genlist; /* use up old list */
      genlist=gtrace=(struct Vertlist *) /* create new list */
	calloc((size_t)1,sizeof(struct Vertlist));
      do
	{
	  v=vertlist->v;
	  for (i=0;i<=pK_ptr[v].num;i++)
	    if (!pK_ptr[(j=pK_ptr[v].flower[i])].util_flag)
	      {
		pK_ptr[j].util_flag=gen_count;
		if (pK_ptr[j].bdry_flag) goto GOTONE;
		gtrace=gtrace->next=(struct Vertlist *)
		  calloc((size_t)1,sizeof(struct Vertlist));
		gtrace->v=j;
		hits=0;
	      }
	  trace=vertlist;
	  vertlist=vertlist->next;
	  free(trace);
	} while (vertlist);
      gtrace=genlist;
      genlist=genlist->next;
      free(gtrace);
    }
  /* reaching here means some error; didn't find a bdry vert */  
  goto BUMBLE;

  /* create the list */
 GOTONE:
  final_list=trace=(struct Vertlist *)
    calloc((size_t)1,sizeof(struct Vertlist));
  trace->v=j;
  while (gen_count>1)
    {
      v=trace->v;
      for (i=0;i<=pK_ptr[v].num;i++)
	if (pK_ptr[(vv=pK_ptr[v].flower[i])].util_flag < gen_count
	    && pK_ptr[vv].util_flag>0)
	  {
	    i=-1;
	    break;
	  }
      if (i!=-1) goto BUMBLE; /* something wrong */
      trace=trace->next=(struct Vertlist *)
	calloc((size_t)1,sizeof(struct Vertlist));
      trace->v=vv;
      gen_count--;
    }
  if (gen_count<1) goto BUMBLE; /* something wrong */
  trace=final_list;
  final_list=final_list->next; /* first spot empty */
  free(trace);
  vert_free(&genlist);
  return final_list;
  
 BUMBLE:
  vert_free(&genlist);
  vert_free(&vertlist);
  vert_free(&final_list);
  return NULL;
} /* create_tether */







