# +==========================================================================+
# || CipUX::Storage::Client                                                 ||
# ||                                                                        ||
# || CipUX storage abstraction layer                                        ||
# ||                                                                        ||
# || Copyright (C) 2008 - 2009 by Christian Kuelker. All rights reserved!   ||
# ||                                                                        ||
# || License: GNU GPL version 2 or any later version.                       ||
# ||                                                                        ||
# +==========================================================================+
# ID:       $Id$
# Revision: $Revision$
# Head URL: $HeadURL$
# Date:     $Date$
# Source:   $Source$

package CipUX::Storage::Client;

use 5.008001;
use strict;
use warnings;
use Carp;
use Class::Std;
use Data::Dumper;
use English qw( -no_match_vars);
use Getopt::Long;
use Log::Log4perl qw(:easy);
use Pod::Usage;
use Readonly;
use base qw(CipUX CipUX::Storage);

{    # BEGIN CLASS

    use version; our $VERSION = qv('3.4.0.2');
    use re 'taint';    # Keep tainted any data captured by parens
    delete @ENV{qw(PATH IFS CDPATH ENV BASH_ENV)};    # Make %ENV safe

    # +======================================================================+
    # || CONST                                                              ||
    # +======================================================================+
    Readonly::Scalar my $EMPTY_STRING => q{};
    Readonly::Scalar my $LINEWIDTH    => 78;
    Readonly::Scalar my $L4PCONF      => '/etc/cipux/log4perl.conf';
    Readonly::Array my @ACTION        => qw(
        get_all_values
        set_all_values
        add_node
        get_value
        set_value
        list
        delete_node
        rename_node
    );

    # +======================================================================+
    # || INIT ARGS                                                          ||
    # +======================================================================+

    # register client with name:
    #     name : cipux_storage_client
    my %name_of : ATTR( init_arg => 'name');

    # +======================================================================+
    # || GLOBAL                                                             ||
    # +======================================================================+
    my $L = q{=} x $LINEWIDTH;
    $L .= "\n";

    my $maxcol_ar     = [];
    my $value_ar      = [];
    my $attrvalue_hr  = {};
    my $mattrvalue_hr = {};
    my %opt           = ();
    my %option        = (
        'cipux_storage_client' => {
            'must' => [qw(t=storage_task l=list)],
            'may' => [qw(c cfg h ? help D debug p pretty V verbose version )],
            'not' => [qw(a add c d e r t v x y)],
        },
        'add_node' => {
            'must' => [qw(t=storage_task x=mattrvalue s=type o=object)],
            'may'  => [qw(c cfg h ? help D debug V verbose version )],
            'not'  => [
                qw(a add d delete e erase p prettey r replace v value y attrvalue)
            ],
        },
        'delete_node' => {
            'must' => [qw(t=storage_task s=type o=object)],
            'may'  => [qw(c cfg h ? help D debug V verbose version )],
            'not'  => [
                qw(a add d delete e erase p prettey r replace v value x mattrvalue y attrvalue)
            ],
        },
        'rename_node' => {
            'must' => [qw(t=storage_task s=type o=object v=value)],
            'may'  => [qw(c cfg h ? help D debug V verbose version )],
            'not'  => [
                qw(a add d delete e erase p prettey r replace y attrvalue x mattrvalue)
            ],
        },
        'get_value' => {
            'must' => [qw(t=storage_task y=attrvalue s=type o=object)],
            'may'  => [qw(c cfg h ? help D debug pretty V verbose version )],
            'not' =>
                [qw(a add d delete e erase r replace v value x mattrvalue)],
        },
        'get_all_values' => {
            'must' => [qw(t=storage_task y=attrvalue s=type)],
            'may' => [qw(c cfg h ? help D debug p pretty V verbose version )],
            'not' => [
                qw(a add d delete e erase o object r replace v value x mattrvalue)
            ],
        },
        'set_value' => {
            'must' => [qw(t=storage_task y=attrvalue s=type o=object)],
            'may'  => [
                qw(a add d delete c cfg e erase h ? help D debug r replace v value V verbose version )
            ],
            'not' => [qw(p prettey x mattrvalue)],
        },
        'set_all_values' => {
            'must' => [qw(t=storage_task y=attrvalue s=type)],
            'may'  => [
                qw(a add d delete c cfg erase h ? help D debug r replace v value V verbose version )
            ],
            'not' => [qw(o object p prettey x mattrvalue)],
        },
    );

    # +======================================================================+
    # || run                                                                ||
    # +======================================================================+
    sub run {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        # constructor parameter from CipUX::Storage::Client
        my $script = $name_of{ ident $self};

        # test right away if we have a valid name
        if ( $script ne 'cipux_storage_client' ) {
            $self->exc( { msg => 'unknown script name', value => $script } );
        }

        # +------------------------------------------------------------------+
        # | Environment
        Getopt::Long::Configure('bundling');

        my $store_attrvalue
            = sub { $self->store_attrvalue( $attrvalue_hr, @_ ) };
        my $store_mattrvalue
            = sub { $self->store_mattrvalue( $mattrvalue_hr, @_ ) };

        GetOptions(
            \%opt,
            'add|a',
            'cfg|c=s',
            'debug|D',
            'delete|d',
            'erase|e',
            'help|h',
            'list|l',
            'gum|g=i',
            'object|o=s',
            'pretty|p',
            'replace|r',
            'type|s=s',
            'storage_task|t=s',
            'value|v=s',
            'version|V',
            'verbose',
            'mattrvalue|x=s%' =>
                sub { $self->store_mattrvalue( $mattrvalue_hr, \%opt, @_ ); },
            'attrvalue|y:s%' =>
                sub { $self->store_attrvalue( $attrvalue_hr, \%opt, @_ ); },
            )
            or pod2usage(
            -exitstatus => 2,
            -msg        => "$L problems parsing command line!\n$L"
            );

        if ( defined $opt{debug} ) {
            Log::Log4perl->init_once($L4PCONF);
        }

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        $logger->debug( 'opt: ', { filter => \&Dumper, value => \%opt } );
        $logger->debug( 'attrvalue_hr: ',
            { filter => \&Dumper, value => $attrvalue_hr } );
        $logger->debug( 'mattrvalue_hr: ',
            { filter => \&Dumper, value => $mattrvalue_hr } );

        # display help page
        if ( exists $opt{help} ) {
            pod2usage( -exitstatus => 0, -verbose => 1 );
        }

        my $storage_task
            = exists $opt{storage_task}
            ? $opt{storage_task}
            : undef;

        if ( not defined $storage_task ) {
            my $msg = "$L Please give parameter --storage_task or -t! \n";
            $msg .= "valid storage tasks are: \n";
            foreach my $a (@ACTION) {
                $msg .= "$a\n";
            }
            $msg .= $L;
            pod2usage( -exitstatus => 1, -verbose => 0, -msg => $msg );
        }
        $logger->debug( 'storage_task: ', $storage_task );

        if ( exists $opt{version} ) {
            $self->out("$script $VERSION\n");
            exit 0;
        }

        my $ret = $self->test_cli_option(
            {
                script   => $storage_task,
                logic_hr => \%option,
                opt_hr   => \%opt,
            }
        );

        # +------------------------------------------------------------------+
        # | main

        my $cfg
            = exists $opt{cfg}
            ? $opt{cfg}
            : undef;

        if ( defined $cfg ) {
            $logger->debug( 'cfg: ', $cfg );
        }

        my $escope
            = exists $opt{erase}
            ? 'all'
            : 'one';

        my $changes
            = exists $opt{erase}   ? 'delete'
            : exists $opt{delete}  ? 'delete'
            : exists $opt{add}     ? 'add'
            : exists $opt{replace} ? 'replace'
            :                        'replace';
        my $scope
            = $storage_task eq 'get_value'      ? 'one'
            : $storage_task eq 'set_value'      ? 'one'
            : $storage_task eq 'get_all_values' ? 'all'
            : $storage_task eq 'set_all_values' ? 'all'
            :                                     'one';

        my $pretty
            = exists $opt{pretty}
            ? 1
            : 0;

        my $object
            = exists $opt{object}
            ? $opt{object}
            : $EMPTY_STRING;

        # s = type
        my $type
            = exists $opt{type}
            ? $opt{type}
            : $EMPTY_STRING;

        my $value
            = exists $opt{value}
            ? $opt{value}
            : $EMPTY_STRING;

        my $configfile
            = exists $opt{cfg}
            ? $opt{cfg}
            : $EMPTY_STRING;

        if ( scalar( grep { $_ eq $storage_task } @ACTION ) < 1 ) {
            my $msg = "unknown storage task [$storage_task]\n";

            $msg .= "valid storage tasks are: \n";
            foreach my $a (@ACTION) {
                $msg .= "$a\n";
            }
            $msg .= $L;
            $self->exc( { msg => $msg } );
        }

        my $sub = 'cipux_storage_' . $storage_task;
        $self->$sub(
            {
                scope  => $scope,
                escope => $escope,
                pretty => $pretty,
                type   => $type,
                object => $object,
                value  => $value,
                cfg    => $cfg,
            }
        );

        return;
    }

    # +----------------------------------------------------------------------+
    # | cipux_storage_list                                                   |
    # +----------------------------------------------------------------------+
    sub cipux_storage_list {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $cfg
            = exists $arg_r->{cfg}
            ? $self->l( $arg_r->{cfg} )
            : $self->perr('cfg');

        my $pretty
            = exists $arg_r->{pretty}
            ? $self->l( $arg_r->{pretty} )
            : $self->perr('pretty');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug( 'pretty: ', $pretty );
        $logger->debug( 'cfg: ', { filter => \&Dumper, value => $cfg } );

        $self->_print_list_type( { pretty => $pretty, cfg => $cfg } );

        # +------------------------------------------------------------------+
        # | API
        return;
    }

    # +----------------------------------------------------------------------+
    # | cipux_storage_get_value                                              |
    # +----------------------------------------------------------------------+
    sub cipux_storage_get_value {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $cfg
            = exists $arg_r->{cfg}
            ? $self->l( $arg_r->{cfg} )
            : $self->perr('cfg');

        my $scope
            = exists $arg_r->{scope}
            ? $self->l( $arg_r->{scope} )
            : $self->perr('scope');

        my $pretty
            = exists $arg_r->{pretty}
            ? $self->l( $arg_r->{pretty} )
            : $self->perr('pretty');

        my $type
            = exists $arg_r->{type}
            ? $self->l( $arg_r->{type} )
            : $self->perr('type');

        my $object
            = exists $arg_r->{object}
            ? $self->l( $arg_r->{object} )
            : $self->perr('object');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');

        $logger->debug( 'attrvalue_hr: ',
            { filter => \&Dumper, value => $attrvalue_hr } );

        #$attrvalue_hr->{cipuxAddress} = 'cipuxAddress';

        my $storage = CipUX::Storage->new();

        my $value_hr = $storage->get_value(
            {
                scope   => $scope,
                type    => $type,
                obj     => $object,
                attr_ar => [ %{$attrvalue_hr} ],
            }
        );

        $self->_print_values(
            { scope => $scope, pretty => $pretty, value_hr => $value_hr } );

        # +------------------------------------------------------------------+
        # | API
        return;
    }

    # +----------------------------------------------------------------------+
    # | cipux_storage_get_all_values                                         |
    # +----------------------------------------------------------------------+
    sub cipux_storage_get_all_values {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $cfg
            = exists $arg_r->{cfg}
            ? $self->l( $arg_r->{cfg} )
            : $self->perr('cfg');

        my $scope
            = exists $arg_r->{scope}
            ? $self->l( $arg_r->{scope} )
            : $self->perr('scope');

        my $pretty
            = exists $arg_r->{pretty}
            ? $self->l( $arg_r->{pretty} )
            : $self->perr('pretty');

        my $type
            = exists $arg_r->{type}
            ? $self->l( $arg_r->{type} )
            : $self->perr('type');

        my $object
            = exists $arg_r->{object}
            ? $self->l( $arg_r->{object} )
            : $self->perr('object');

        # +------------------------------------------------------------------+
        # | main
        my $storage = CipUX::Storage->new();

        my $value_hr = $storage->get_value(
            {
                scope   => $scope,
                type    => $type,
                obj     => $object,
                attr_ar => [ %{$attrvalue_hr} ]

            }
        );
        $self->_print_values(
            { scope => $scope, pretty => $pretty, value_hr => $value_hr } );

        # +------------------------------------------------------------------+
        # | API
        return;
    }

    # +----------------------------------------------------------------------+
    # | cipux_storage_set_value                                              |
    # +----------------------------------------------------------------------+
    sub cipux_storage_set_value {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $cfg
            = exists $arg_r->{cfg}
            ? $self->l( $arg_r->{cfg} )
            : $self->perr('cfg');

        my $scope
            = exists $arg_r->{scope}
            ? $self->l( $arg_r->{scope} )
            : $self->perr('scope');

        my $escope
            = exists $arg_r->{escope}
            ? $self->l( $arg_r->{escope} )
            : $self->perr('escope');

        my $changes
            = exists $arg_r->{changes}
            ? $self->l( $arg_r->{changes} )
            : $self->perr('changes');

        my $type
            = exists $arg_r->{type}
            ? $self->l( $arg_r->{type} )
            : $self->perr('type');

        my $object
            = exists $arg_r->{object}
            ? $self->l( $arg_r->{object} )
            : $self->perr('object');

        my $value
            = exists $arg_r->{value}
            ? $self->l( $arg_r->{value} )
            : $self->perr('value');

        # +------------------------------------------------------------------+
        # | main
        my $storage = CipUX::Storage->new();

        my $value_hr = $storage->set_value(
            {
                scope   => $scope,    # scope referencing objects
                escope  => $escope,   # scope referencing attributes (e=erase)
                changes => $changes,
                type    => $type,
                obj     => $object,
                value   => $value,
                attr_ar => [ %{$attrvalue_hr} ]
            }
        );

        # +------------------------------------------------------------------+
        # | API
        return;

    }

    # +----------------------------------------------------------------------+
    # | cipux_storage_set_all_values                                         |
    # +----------------------------------------------------------------------+
    sub cipux_storage_set_all_values {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $cfg
            = exists $arg_r->{cfg}
            ? $self->l( $arg_r->{cfg} )
            : $self->perr('cfg');

        my $scope
            = exists $arg_r->{scope}
            ? $self->l( $arg_r->{scope} )
            : $self->perr('scope');

        my $escope
            = exists $arg_r->{escope}
            ? $self->l( $arg_r->{escope} )
            : $self->perr('escope');

        my $changes
            = exists $arg_r->{changes}
            ? $self->l( $arg_r->{changes} )
            : $self->perr('changes');

        my $type
            = exists $arg_r->{type}
            ? $self->l( $arg_r->{type} )
            : $self->perr('type');

        my $object
            = exists $arg_r->{object}
            ? $self->l( $arg_r->{object} )
            : $self->perr('object');

        my $value
            = exists $arg_r->{value}
            ? $self->l( $arg_r->{value} )
            : $self->perr('value');

        # +------------------------------------------------------------------+
        # | main
        my $storage = CipUX::Storage->new();

        my $value_hr = $storage->set_value(
            {
                scope   => $scope,    # scope referencing objects
                escope  => $escope,   # scope referencing attributes (e=erase)
                changes => $changes,
                type    => $type,
                obj     => $object,
                value   => $value,
                attr_ar => [ %{$attrvalue_hr} ]
            }
        );

        # +------------------------------------------------------------------+
        # | API
        return;

    }

    # +----------------------------------------------------------------------+
    # | cipux_storage_add_node                                               |
    # +----------------------------------------------------------------------+
    sub cipux_storage_add_node {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $cfg
            = exists $arg_r->{cfg}
            ? $self->l( $arg_r->{cfg} )
            : $self->perr('cfg');

        my $object
            = exists $arg_r->{object}
            ? $self->l( $arg_r->{object} )
            : $self->perr('object');

        my $type
            = exists $arg_r->{type}
            ? $self->l( $arg_r->{type} )
            : $self->perr('type');

        # +------------------------------------------------------------------+
        # | main
        my $storage = CipUX::Storage->new();

        my %merged = ( %{$attrvalue_hr}, %{$mattrvalue_hr} );

        # If to print that out, better not relay on Data::Dumper
        # if ($debug) {
        #     my $c = Data::Dumper->new( [ \%merged ], ['*merged'] );
        #     $c->Varname('merged');
        #     $c->Purity(1);
        #     $c->Deepcopy(1);
        #     print q{=} x 56, "\n";
        #     print "\n" . $c->Dump;
        #     print q{=} x 56, "\n";
        # } ## end if($debug)

        my $value_hr = $storage->add_node(
            {
                type    => $type,
                obj     => $object,
                attr_hr => \%merged
            }
        );

        # +------------------------------------------------------------------+
        # | API
        return;
    }

    # +----------------------------------------------------------------------+
    # | cipux_storage_rename_node                                            |
    # +----------------------------------------------------------------------+
    sub cipux_storage_rename_node {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $cfg
            = exists $arg_r->{cfg}
            ? $self->l( $arg_r->{cfg} )
            : $self->perr('cfg');

        my $type
            = exists $arg_r->{type}
            ? $self->l( $arg_r->{type} )
            : $self->perr('type');

        my $object
            = exists $arg_r->{object}
            ? $self->l( $arg_r->{object} )
            : $self->perr('object');

        my $value
            = exists $arg_r->{value}
            ? $self->l( $arg_r->{value} )
            : $self->perr('value');

        # +------------------------------------------------------------------+
        # | main
        my $storage = CipUX::Storage->new();

        my $value_hr = $storage->rename_node(
            {
                type  => $type,
                obj   => $object,
                value => $value
            }
        );

        # +------------------------------------------------------------------+
        # | API
        return;
    }

    # +----------------------------------------------------------------------+
    # | cipux_storage_delete_node                                            |
    # +----------------------------------------------------------------------+
    sub cipux_storage_delete_node {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $cfg
            = exists $arg_r->{cfg}
            ? $self->l( $arg_r->{cfg} )
            : $self->perr('cfg');

        my $type
            = exists $arg_r->{type}
            ? $self->l( $arg_r->{type} )
            : $self->perr('type');

        my $object
            = exists $arg_r->{object}
            ? $self->l( $arg_r->{object} )
            : $self->perr('object');

        # +------------------------------------------------------------------+
        # | main
        my $storage = CipUX::Storage->new();

        my $value_hr = $storage->delete_node(
            {
                type => $type,
                obj  => $object,
            }
        );

        # +------------------------------------------------------------------+
        # | API
        return;

    }

    # +----------------------------------------------------------------------+
    # | _print_values                                                        |
    # +----------------------------------------------------------------------+
    sub _print_values : PRIVATE {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $scope
            = exists $arg_r->{scope}
            ? $self->l( $arg_r->{scope} )
            : $self->perr('scope');

        my $pretty
            = exists $arg_r->{pretty}
            ? $self->l( $arg_r->{pretty} )
            : $self->perr('pretty');

        my $value_hr
            = exists $arg_r->{value_hr}
            ? $self->h( $arg_r->{value_hr} )
            : $self->perr('value_hr');

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);
        $logger->debug('BEGIN');
        $logger->debug( 'scope: ',  $scope );
        $logger->debug( 'pretty: ', $pretty );
        $logger->debug( 'value_hr: ',
            { filter => \&Dumper, value => $value_hr } );

        if ($pretty) {

            my $mc1   = 0;    # max col ID
            my $mc2   = 0;    # max col ATTR
            my $mc3   = 0;    # max col VALUE
            my $width = 0;

            # calculate max count of attribute
            my $z = 0;        # counts attr
            foreach my $attr ( keys %{$attrvalue_hr} ) {
                $z++;         # this will count only given attr
            }

            # one OID, scope = one:
            #  l=1 => VALUE           :1 val, 1 attr, only value printed
            #  l=4 => ATTR|VALUE      :1 val, x attr, attr + value printed
            #
            # many OID, scope = all:
            #  l=2 => OID|VALUE|      :1 val, 1 attr, no need to print attr
            #  l=3 => OID|ATTR |VALUE :1 val, x attr, x oid
            #
            #  l=layout
            #
            my $layout
                = ( $z < 2 and $scope eq 'one' ) ? 1
                : ( $z < 2  and $scope eq 'all' ) ? 2
                : ( $z >= 2 and $scope eq 'all' ) ? 3
                :                                   4;    # type 4

            # count of columns
            my $col
                = ( $layout == 1 ) ? 1
                : ( $layout == 2 ) ? 2
                : ( $layout == 3 ) ? 3
                : ( $layout == 4 ) ? 2
                :                    2;

            # calculate max column width
            foreach my $oid ( sort keys %{$value_hr} ) {
                my $v = 'NULL';
                foreach my $attr ( keys %{ $value_hr->{$oid} } ) {
                    if ( defined $value_hr->{$oid}->{$attr} ) {
                        $v = join '; ', @{ $value_hr->{$oid}->{$attr} };
                    }
                    if ( $mc1 < length $oid ) {
                        $mc1 = length $oid;
                    }
                    if ( $mc2 < length $attr ) {
                        $mc2 = length $attr;
                    }
                    if ( $mc3 < length $v ) {
                        $mc3 = length $v;
                    }
                }
            }

            # col = 1 => c+2
            # col = 2 => a+c+5, b+c+5
            # col = 3 => a+b+c+8
            $width
                = ( $layout == 1 ) ? $mc3 + ( $col * 2 + $col - 1 )
                : ( $layout == 2 ) ? $mc1 + $mc3 + ( $col * 2 + $col - 1 )
                : ( $layout == 3 )
                ? $mc1 + $mc2 + $mc3 + ( $col * 2 + $col - 1 )
                : ( $layout == 4 ) ? $mc2 + $mc3 + ( $col * 2 + $col - 1 )
                :                    0;

            $logger->debug( 'ATTR  : ', $z );
            $logger->debug( 'COL   : ', $col );
            $logger->debug( 'LAYOUT: ', $layout );
            $logger->debug( 'mc1   : ', $mc1 );
            $logger->debug( 'mc2   : ', $mc2 );
            $logger->debug( 'mc3   : ', $mc3 );
            $logger->debug( 'width : ', $width );

            # print header
            print q{+} . q{-} x $width . "+\n";
            print $self->_row(
                {
                    layout => $layout,
                    pretty => $pretty,
                    value  => [ 'ID', 'ATTR', 'VALUE' ],
                    max    => [ $mc1, $mc2, $mc3 ]
                }
            );
            print q{+} . q{=} x $width . "+\n";

            foreach my $oid ( sort keys %{$value_hr} ) {
                my $v = 'NULL';
                foreach my $attr ( keys %{ $value_hr->{$oid} } ) {
                    if ( defined $value_hr->{$oid}->{$attr} ) {
                        $v = join ', ', @{ $value_hr->{$oid}->{$attr} };
                    }

                    if ( defined $attrvalue_hr->{$attr} ) {
                        print $self->_row(
                            {
                                layout => $layout,
                                max    => [ $mc1, $mc2, $mc3 ],
                                value  => [ $oid, $attr, $v ],
                                pretty => $pretty
                            }
                        );
                    }

                }
            }

            print q{+} . q{-} x $width . "+\n";
        }
        else {
            $logger->debug('Non pretty output: ');
            my $z = 0;
            foreach my $attr ( keys %{$attrvalue_hr} ) {
                $z++;    # this will count only given attr
            }
            $logger->debug( 'attr count: ', $z );
            $logger->debug( 'scope:      ', $scope );

            foreach my $oid ( sort keys %{$value_hr} ) {
                $logger->debug( 'oid: ', $oid );
                my $v = 'NULL';
                foreach my $attr ( keys %{ $value_hr->{$oid} } ) {
                    $logger->debug( 'attr:      ', $attr );

                    if ( defined $value_hr->{$oid}->{$attr} ) {
                        $v = join q{ }, @{ $value_hr->{$oid}->{$attr} };
                    }

                 #$logger->debug( 'attr2:     ',$attrvalue_hr->{$attr} ,"\n");
                 # next if not defined $attrvalue_hr->{$attr};
                    if ( $z < 2 and $scope eq 'one' )
                    {    # one value with one attribute
                        printf "%s\n", $v;
                    }
                    else {   # more the one value: (1) attr > 1 or (2) obj > 1
                        if ( $scope eq 'all' ) {

                            # do not print cn or oid line
                            # OID\tVALUE
                            if ( $z < 2 )
                            {    # with one attr, we do not have to print it
                                printf "%s\t%s\n", $oid, $v;
                            }
                            else {
                                printf "%s\t%s\t%s\n", $oid, $attr, $v;
                            }
                        }
                        else {

                            # do not print cn or oid line
                            # ATTR\tVALUE
                            printf "%s\t%s\n", $attr, $v;
                        }
                    }
                }
            }
        }

        # +------------------------------------------------------------------+
        # | API
        return;

    }

    # +======================================================================+
    # || _list_type                                                         ||
    # +======================================================================+
    sub _print_list_type : PRIVATE {

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        my $cfg
            = exists $arg_r->{cfg}
            ? $self->l( $arg_r->{cfg} )
            : $self->perr('cfg');

        my $pretty
            = exists $arg_r->{pretty}
            ? $self->l( $arg_r->{pretty} )
            : $self->perr('pretty');

        # +------------------------------------------------------------------+
        # | main
        my $storage = CipUX::Storage->new();

        my $list_ar = $storage->list_type();

        # output
        #
        #    foreach my $line (@$list_ar){
        #        chomp($line);
        #        print "$line\n";
        #    }

        # mostly this stuff is for pretty print
        my $max_col = 0;
        my $width   = 0;

        if ($pretty) {
            foreach my $line ( @{$list_ar} ) {
                if ( $max_col < length $line ) {
                    $max_col = length $line;
                }
            }
            $width = 2 + $max_col;
            print q{+}, q{-} x $width, "+\n";
            printf '| %-' . $max_col . "s |\n", 'storage type';
            print q{+}, q{=} x $width, "+\n";
        } ## end if($pretty)

        foreach my $line ( @{$list_ar} ) {
            chomp $line;
            if ($pretty) {
                printf '| %-' . $max_col . "s |\n", $line;
            }
            else {
                print "$line\n";
            }
        } ## end foreach my $line (@$list_ar)
        if ($pretty) {
            print q{+}, q{-} x $width, "+\n";
        }

        # +------------------------------------------------------------------+
        # | API
        return;

    }

    # +======================================================================+
    # || _row                                                               ||
    # +======================================================================+
    sub _row : PRIVATE {

        # API
        my ( $self, $arg_r ) = @_;
        my $pretty      = $arg_r->{pretty} || 0;     # 0|1 pretty print
        my $layout      = $arg_r->{layout} || 2;     # layout 1..4
        my $mymaxcol_ar = $arg_r->{max}    || ();    # max colum width
        my $myvalue_ar  = $arg_r->{value}  || ();    # max colum width

        # +------------------------------------------------------------------+
        # | main
        my $logger = get_logger(__PACKAGE__);

        $logger->debug( 'pretty: ', $pretty );
        $logger->debug( 'layout: ', $layout );

        # one OID, scope = one
        #  l=1 => VALUE           :1 val, 1 attr, only value printed
        #  l=4 => ATTR| VALUE     :1 val, x attr, attr + value printed
        # many OID, scope = all
        #  l=2 => OID|VALUE|      :1 val, 1 attr, no need to print attr
        #  l=3 => OID|ATTR |VALUE :1 val, x attr, x oid
        #
        #  l=layout
        #

        my $c1 = ${ $mymaxcol_ar->[0] } || 0;
        my $c2 = ${ $mymaxcol_ar->[1] } || 0;
        my $c3 = ${ $mymaxcol_ar->[2] } || 0;
        my $v1 = ${ $myvalue_ar->[0] }  || 0;
        my $v2 = ${ $myvalue_ar->[1] }  || 0;
        my $v3 = ${ $myvalue_ar->[2] }  || 0;

        my $l = ($pretty) ? '| '  : $EMPTY_STRING;    # left border
        my $r = ($pretty) ? ' |'  : $EMPTY_STRING;    # right border
        my $m = ($pretty) ? ' | ' : $EMPTY_STRING;    # middle line

        my $v = "%-${c3}s";                           # value
        my $t = "%-${c2}s";                           # attr
        my $o = "%-${c1}s";                           # oid

        $logger->debug( 'c1: ', $c1 );
        $logger->debug( 'c2: ', $c2 );
        $logger->debug( 'c3: ', $c3 );
        $logger->debug( 'l: ',  $l );
        $logger->debug( 'r: ',  $r );
        $logger->debug( 'm: ',  $m );

        my $row
            = ( $layout == 1 ) ? sprintf( "$l$v$r", $v3 )
            : ( $layout == 2 ) ? sprintf( "$l$o$m$v$r",     $v1, $v3 )
            : ( $layout == 3 ) ? sprintf( "$l$o$m$t$m$v$r", $v1, $v2, $v3 )
            : ( $layout == 4 ) ? sprintf( "$l$o$m$v$r",     $v2, $v3 )
            :                    "$l$r";

        # +------------------------------------------------------------------+
        # | API
        return $row . "\n";

    }

}    # END INSIDE-OUT CLASS

1;

__END__

=pod

=head1 NAME

CipUX::Storage::Client - Command line client library for CipUX-Storage

=head1 VERSION

version 3.4.0.2

=head1 SYNOPSIS

 use CipUX::Storage::Client;

 my $c = CipUX::Storage::Client->new( { name => 'cipux_storage_client' } );

 $c->run();


=head1 DESCRIPTION

Provides the functions get_value, set_value to modify LDAP attribute values.
The function add_node, delete_node and rename_node for adding, deleting and
renaming LDAP objects.

=head1 SUBROUTINES/METHODS

The following functions will be exported by CipUX::Storage::Client.

=head2 new

Constructor

B<Syntax:>


  my $storage = CipUX::Storage::Client->new({});

=head2 cipux_storage_add_node

Wrapper subroutine for CipUX::Storage->add_node.

=head2 cipux_storage_delete_node

Wrapper subroutine for CipUX::Storage->delete_node.

=head2 cipux_storage_get_all_values

Wrapper subroutine for CipUX::Storage->get_all_values.

=head2 cipux_storage_get_value

Wrapper subroutine for CipUX::Storage->get_value.

=head2 cipux_storage_list

Wrapper subroutine for CipUX::Storage->print_list.

=head2 cipux_storage_rename_node

Wrapper subroutine for CipUX::Storage->rename_node.

=head2 cipux_storage_set_all_values

Wrapper subroutine for CipUX::Storage->set_all_values.

=head2 cipux_storage_set_value

Wrapper subroutine for CipUX::Storage->set_value.

=head2 run

Initialize the library. Should be caleld by a script.


=head1 CONFIGURATION AND ENVIRONMENT

See cipux-storage-access.conf and cipux-storage-structure.conf man page for
details on configuration. CipUX::Storage do not use the environment for
configuration.

=head1 DEPENDENCIES

 Carp
 Class::Std;
 CipUX
 CipUX::Storage
 Data::Dumper
 English
 Getopt::Long
 Log::Log4perl
 Pod::Usage
 Readonly

=head1 INCOMPATIBILITIES

Not known.

=head1 BUGS AND LIMITATIONS

Not known.

=head1 SEE ALSO

See the CipUX web page and the manual at L<http://www.cipux.org>

See the mailing list L<http://sympa.cipworx.org/wws/info/cipux-devel>

=head1 AUTHOR

Christian Kuelker  E<lt>christian.kuelker@cipworx.orgE<gt>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2007 - 2009 by Christian Kuelker

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA

=cut

