/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <string.h>
#include <stdlib.h>

#include "cdw_task.h"
#include "cdw_disc.h"
#include "cdw_drive.h"
#include "cdw_config.h"
#include "cdw_ext_tools.h"
#include "cdw_dvd_rw_mediainfo.h"
#include "cdw_cdrecord.h"
#include "cdw_xorriso.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_main_window.h"
#include "cdw_widgets.h"
#include "cdw_processwin.h"
#include "cdw_fs.h"
#include "cdw_utils.h"
#include "cdw_sys.h"


/**
   \file cdw_disc.c

   \brief Few functions that operate on variable of type cdw_disc_t.
   The variable - if correctly updated - represents most information
   about a disc in drive that cdw can get.


\verbatim
CD-R and CD-RW Disc Capacities
(capacities indicated in bytes)

(source: http://www.osta.org/technology/cdqa7.htm)

Disc    Playing   Audio       CD-ROM        CD-ROM        CD-i/XA       CD-i/XA
Size    Time                  Mode 1        Mode 2        Form 1        Form 2
                  2352        2048          2336          2048          2324     <-- bytes per sector

 8 cm  18 min   190 512 000   165 888 000   189 216 000   165 888 000   188 244 000
 8 cm  21 min   222 264 000   193 536 000   220 752 000   193 536 000   219 618 000
12 cm  63 min   666 792 000   580 608 000   662 256 000   580 608 000   658 854 000
12 cm  74 min   783 216 000   681 984 000   777 888 000   681 984 000   773 892 000
12 cm  80 min   846 720 000   737 280 000   840 960 000   737 280 000   836 640 000
\endverbatim


   Sources of information printed below are (in no particular order):
   \li http://www.mscience.com/faq502.html
   \li http://www.mscience.com/faq62.html
   \li http://www.mscience.com/faq60.html
   \li http://www.cdrlabs.com/forums/mode-versus-mode-discs-t8075.html
   \li "Upgrading and Repairing PCs" by Scott Mueller, Published by Que Publishing, 2003; ISBN 0789729741, 9780789729743
   \li http://www.cdrfaq.org/
   \li http://sony.storagesupport.com/node/6405
   \li http://www.ecma-international.org/publications/files/ECMA-ST/Ecma-130.pdf
   \li http://www.pcguide.com/ref/cd/formatXA-c.html

   Information goes as follows:
   - Red Book: defines CD-DA, audio only (no sector structure)
   - Yellow Book: defines:
       - Mode 0 (12B sync + 4B header + 2336B data (all zeros))
       - Mode 1 (12B sync + 4B header + 2048B data + 4B EDC + 8B blank + 276B data protection);
         Referred to as: "CD-ROM Mode 1", "Mode 1"
       - Mode 2 (12B sync + 4B header + 2336B data);
         Referred to as: "CD-ROM Mode 2", "Mode 2", "Mode2 Formless"

   - Yellow Book extension defines CD-ROM XA (CD-ROM/XA, eXtended Architecture):
       - Mode 1: (12B sync + 4B header + 2048B data + 4B EDC + 8B blank + 276B ECC)
       - Mode 2: for mixing sectors with and without data correction on the
         same track; introduces 8B subheader; Referred to as "CD-ROM/XA MODE 2"
       - Mode 2 Form 1: (12B sync + 4B header + 8B subheader + 2048B data + 4B EDC + 276B ECC)
         for error-sensitive data;
         Referred to as "CD-ROM/XA MODE 2 Form 1", "CD-ROM/XA Form 1", "Form 1"
       - Mode 2 Form 2: (12B sync + 4B header + 8B subheader + 2324B data + 4B EDC)
         for error-tolerant data (e.g. multimedia)
         Referred to as "CD-ROM/XA MODE 2 Form 2", "CD-ROM/XA Form 2", "Form 2"
   - Green Book: defines CD-i (CD-interactive, obsolete?, www.icdia.com);
     it also modified Yellow Book Mode 2, by adding:
       - Mode 2 Form 1 (8 bytes of subheader) + (2048 bytes of data)
       - Mode 2 Form 2 (2324 bytes of data)


   Some excerpts:
   http://www.mscience.com/faq62.html:
   "In practice, CD-ROM/XA should only be used for multimedia applications,
   because not all drives recognize the CD-ROM/XA format, while all data
   drives recognize CD-ROM Mode 1."

   http://www.cdrfaq.org/faq02.html#S2-2:
   "ECMA-119 describes ISO-9660, and ECMA-130 sounds a lot like "yellow book"
   if you say it slowly." ("yellow book" without XA extension)

   http://www.mscience.com/faq502.html
   "CD-ROM/XA added two new formats for multimedia applications, neither of
   which is the same as ISO 10149 Mode 2."
*/




/** \brief Variable storing information about disc currently in drive */
cdw_disc_t current_disc;

extern cdw_config_t global_config; /* main cdw configuration variable */

static cdw_rv_t cdw_disc_get_meta_info_with_cdio(cdw_disc_t *disc);
static cdw_rv_t cdw_disc_get_meta_info_with_external_tools(cdw_disc_t *disc);

static cdw_rv_t cdw_disc_validate(cdw_disc_t *disc);
static int cdw_disc_get_default_drive_speed_id(cdw_disc_t *disc);
static void cdw_disc_resolve_type_label(cdw_disc_t *disc);
static void cdw_disc_resolve_write_speeds(cdw_disc_t *disc);
static void cdw_disc_resolve_capacities(cdw_disc_t *disc);

static void cdw_disc_debug_print_disc(cdw_disc_t *disc);


/* these labels are in the same order as in cdw_disc_type_t type
   declaration in cdw_disc.h */
static const char *cdw_disc_type_labels[] = {
	(char *) NULL, /* place for "unknown" */
	(char *) NULL, /* place for "none" */
	(char *) NULL, /* place for "CD-Audio" */
	"CD-R",
	"CD-RW",
	"CD-ROM",
	"DVD-R",
	"DVD-R Seq",
	"DVD-R Res",
	"DVD+R",
	"DVD-RW",
	"DVD-RW Seq",
	"DVD-RW Res",
	"DVD+RW",
	"DVD-ROM",
	"DVD+R DL" };


enum {
	CDW_DISC_NOT_READABLE_EMPTY,
	CDW_DISC_NOT_READABLE_NO_ISO9660,     /* no support for iso9660 in operating system kernel */
	CDW_DISC_NOT_READABLE_NO_UDF,         /* no support for udf in operating system kernel */
	CDW_DISC_NOT_READABLE_UNSUPPORTED_FS, /* file system not supported by cdw */
	CDW_DISC_NOT_READABLE_OTHER
};

cdw_id_clabel_t disc_not_readable_reasons[] = {
	{ CDW_DISC_NOT_READABLE_EMPTY,          (char *) NULL },
	{ CDW_DISC_NOT_READABLE_NO_ISO9660,     (char *) NULL },
	{ CDW_DISC_NOT_READABLE_NO_UDF,         (char *) NULL },
	{ CDW_DISC_NOT_READABLE_UNSUPPORTED_FS, (char *) NULL },
	{ CDW_DISC_NOT_READABLE_OTHER,          (char *) NULL },
	{ -1,                                   (char *) NULL }};



/**
   \brief Initialize cdw_disc module
*/
void cdw_disc_init(void)
{
	/* values returned by gettext() / _() are static strings
	   and they must not be free()d */

	/* 2TRANS: "unknown" as in "Disc type: unknown"; keep short */
	cdw_disc_type_labels[CDW_DISC_TYPE_UNKNOWN] = _("unknown");
	/* 2TRANS: this string means "there is no disc in drive"; keep short */
	cdw_disc_type_labels[CDW_DISC_NONE] = _("No disc");
	/* 2TRANS: this string means "regular CD disc with music"; keep short */
	cdw_disc_type_labels[CDW_CD_AUDIO] = _("Audio CD");

	/* 2TRANS: this is message printed in dialog window */
	disc_not_readable_reasons[CDW_DISC_NOT_READABLE_EMPTY].label = _("cdw can't read the disc because the disc is empty.");
	/* 2TRANS: this is message printed in dialog window */
	disc_not_readable_reasons[CDW_DISC_NOT_READABLE_NO_ISO9660].label = _("cdw can't read the disc because your operating system does not support ISO9660 file system.");
	/* 2TRANS: this is message printed in dialog window */
	disc_not_readable_reasons[CDW_DISC_NOT_READABLE_NO_UDF].label = _("cdw can't read the disc because your operating system does not support UDF file system.");
	/* 2TRANS: this is message printed in dialog window;
	   "not supported" means "cdw can't handle it" */
	disc_not_readable_reasons[CDW_DISC_NOT_READABLE_UNSUPPORTED_FS].label = _("cdw can't read the disc because file system on the disc is not supported by cdw.");
	/* 2TRANS: this is message printed in dialog window;
	   there is some generic/unknown issue/problem with
	   checking the disc */
	disc_not_readable_reasons[CDW_DISC_NOT_READABLE_OTHER].label = _("cdw can't read the disc.");

	cdw_disc_reset(&current_disc);

	return;
}





/**
   \brief Simple access function

   Function returns pointer to global variable representing current disc.
   The pointer is always valid, but information in the variable may be
   invalid or outdated. Use other functions to bring state of the variable
   up to date.
*/
cdw_disc_t *cdw_disc_get(void)
{
	return &current_disc;
}





/**
   \brief Reset values in a disc

   Set some values of \p disc to their initial state, which corresponds
   to 'no disc in drive'.

   \param disc - disc to reset
*/
void cdw_disc_reset(cdw_disc_t *disc)
{
	cdw_assert (disc != (cdw_disc_t *) NULL, "ERROR: disc pointer is null\n");

	disc->simple_type = CDW_DISC_SIMPLE_TYPE_UNKNOWN;
	disc->simple_type_label[0] = '\0';
	disc->type = CDW_DISC_TYPE_UNKNOWN;
	disc->type_label[0] = '\0';

	disc->cdio_disc_mode = CDIO_DISC_MODE_NO_INFO;

	disc->n_tracks = -1;

	disc->state_empty = CDW_UNKNOWN;
	disc->state_writable = CDW_UNKNOWN;
	disc->type_erasable = CDW_UNKNOWN;
	disc->type_writable = CDW_UNKNOWN;

	disc->capacity.sectors_total = 0;
	disc->capacity.sectors_used = 0;

	disc->fs.type = -1;
	disc->fs.type_label = (char *) NULL;
	disc->fs.volume_id[0] = '\0';
        disc->fs.n_sectors = 0;
	disc->write_speeds.n_speeds = -1;
	disc->write_speeds.drive_max_speed = -1;
	disc->write_speeds.drive_default_speed = -1;
	for (int i = 0; i < CDW_DISC_N_SPEEDS_MAX; i++) {
		disc->write_speeds.speeds[i] = 0;
	}

	disc->accepted_disc_modes = CDW_DISC_MODE_INIT;

	disc->media_info_source = CDW_TOOL_NONE;

	disc->cdrecord_info.last_sess_start = 0;
	disc->cdrecord_info.next_sess_start = 0;
	disc->cdrecord_info.has_toc = false;
	disc->cdrecord_info.phys_size = -1;
	disc->cdrecord_info.start_of_lead_out = -1;
	disc->cdrecord_info.rzone_size = 0;

	disc->dvd_rw_mediainfo_info.read_capacity = -1;
	/* 'e' = "error"; dvd+rw-mediainfo aborts reading metainfo of DVD-ROM
	   disc before getting to "status", so for DVD-ROM status stays as 'e';
	   not a big problem because dvd+rw-mediainfo pipe regex code will
	   properly recognize type of disc as DVD-ROM, and everything will be
	   handled correctly */
	disc->dvd_rw_mediainfo_info.disc_status = 'e';

	disc->dvd_rw_mediainfo_info.end_lba = -1;
	disc->dvd_rw_mediainfo_info.all_tracks_size = -1;
	disc->dvd_rw_mediainfo_info.data_tracks_size = -1;
	disc->dvd_rw_mediainfo_info.last_track_size = -1;

	disc->xorriso_info.is_written = CDW_UNKNOWN;
	disc->xorriso_info.is_blank = CDW_UNKNOWN;
	disc->xorriso_info.is_appendable = CDW_UNKNOWN;
	disc->xorriso_info.is_closed = CDW_UNKNOWN;
	disc->xorriso_info.sectors_used = -1;
	disc->xorriso_info.sectors_total = -1;

	disc->libburn_info.unformatted_size = 0;
	disc->libburn_info.formatting_status = BURN_FORMAT_IS_UNKNOWN;
	disc->libburn_info.read_capacity = -1;


	/* take care of other things that should be recalculated based
	   on the assignments made in this function */
	cdw_disc_resolve(disc);

	return;
}





/**
   \brief Inspect given disc and current cdw configuration to get index of initial speed value

   Inspect speed range (selected by user in configuration) to calculate
   index that can be used to initialize dropdown with writing speeds or
   to get initial value of writing speed.

   You need to call cdw_disc_validate() before calling this
   function to ensure that disc->write_speeds.n_speeds > 0.

   \param disc - disc for which to get initial write speed index

   \return correct index of speed in writing_speeds on success
*/
int cdw_disc_get_initial_write_speed_id(cdw_disc_t *disc)
{
	cdw_assert (disc != (cdw_disc_t *) NULL, "ERROR: disc pointer is null\n");
	cdw_assert (disc->write_speeds.n_speeds > 0, "ERROR: you forgot to call cdw_disc_validate()\nand now n_speeds is non-positive\n");

	if (disc->write_speeds.n_speeds == 1) {
		return disc->write_speeds.speeds[0];
	} else { /* n_speeds > 1 */
		return cdw_disc_get_default_drive_speed_id(disc);
	}
}





/**
   \brief Get from disc all available meta information about the disc

   Get as much meta information about disc as possible, using available
   tools: libcdio calls, and calls to cdrecord or dvd+rw-mediainfo.

   The function is smart enough to determine when to call cdrecord
   and when to call dvd+rw-mediainfo.

   \param disc - disc variable into which to store the information

   \return CDW_OK if media info retrieved successfully
   \return CDW_ERROR if function failed to retrieve full info
*/
cdw_rv_t cdw_disc_get_meta_info(cdw_disc_t *disc)
{
	cdw_assert (disc != (cdw_disc_t *) NULL, "ERROR: disc pointer is null\n");

	cdw_rv_t crv = cdw_disc_get_meta_info_with_cdio(disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to open disc with cdio calls\n");
		return CDW_ERROR;
	}

	if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD
	    && disc->fs.type == CDIO_FS_AUDIO) {

		/* we won't call external tools to get more meta information
		   about a disc, so for audio CDs we have to set type here */
		disc->type = CDW_CD_AUDIO;

		cdw_disc_resolve(disc);
		cdw_disc_debug_print_disc(disc);

		/* not much else to do, everything that we needed
		   to know about Audio CD has been checked with cdio */
		return CDW_OK;
	}

	crv = cdw_disc_get_meta_info_with_external_tools(disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to get metainfo from a disc\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message dialog window,
				      program cannot get metadata from disc */
				   _("Cannot get information about disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		cdw_main_ui_main_window_wrefresh();

		return CDW_ERROR;
	}


	if (disc->type == CDW_DVD_RW
	    || disc->type == CDW_DVD_RW_SEQ
	    || disc->type == CDW_DVD_RW_RES) {

		/* DVD-RW has some quirks and they are not fully
		   supported (this was more true in cdw v. 0.3.91,
		   situation has improved in 0.3.92). */

		if (global_config.show_dvd_rw_support_warning) {

			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Warning"),
					   /* 2TRANS: this is message dialog window */
					   _("This is DVD-RW disc, it is not fully supported by cdw. You may have some problems."),
					   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
			cdw_main_ui_main_window_wrefresh();

			/* don't annoy user with this message anymore
			   in current run of cdw; this option is
			   not saved to configuration file */
			global_config.show_dvd_rw_support_warning = false;
		}
	}

	/* at this point we know that we have _some_ meta data
	   in "disc" data structure */

	/* calculate some more high-level information */
	cdw_disc_resolve(disc);

	/* validate basic meta data - check for obvious errors */
	crv = cdw_disc_validate(disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to validate disc information\n");
		return CDW_ERROR;
	} else {
		cdw_disc_debug_print_disc(disc);
		return CDW_OK;
	}
}





/* *** local functions *** */




/**
   \brief Get basic disc meta information using libcdio calls

   Get information such as simple disc type, file system on a disc,
   file system readable label, volume id.

   \param disc - disc variable into which to store the information

   \return CDW_OK on success
   \return CDW_ERROR on error
*/
cdw_rv_t cdw_disc_get_meta_info_with_cdio(cdw_disc_t *disc)
{
	cdw_assert (disc != (cdw_disc_t *) NULL, "ERROR: disc pointer is null\n");

	cdw_rv_t crv = cdw_cdio_open_disc();
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to open disc with cdio calls\n");
		return CDW_ERROR;
	}

	disc->simple_type = cdw_cdio_get_simple_type();
	disc->cdio_disc_mode = cdw_cdio_get_disc_mode();
	disc->state_empty = cdw_cdio_is_blank() ? CDW_TRUE : CDW_FALSE;
	disc->fs.type = cdw_cdio_get_fs_type();
	disc->fs.type_label = cdw_cdio_get_fs_type_label();
	disc->fs.n_sectors = cdw_cdio_get_fs_n_sectors();
	disc->n_tracks = cdw_cdio_get_number_of_tracks();
	/* cdw_cdio_get_volume_id() always returns pointer to valid C string
	   no longer than ISO_MAX_VOLUME_ID + 1 (including ending '\0') */
	strncpy(disc->fs.volume_id, cdw_cdio_get_volume_id(), ISO_MAX_VOLUME_ID + 1);

	cdw_cdio_close_disc();

	if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
		strcpy(disc->simple_type_label, "CD");
	} else if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_DVD) {
		strcpy(disc->simple_type_label, "DVD");
	} else {
		cdw_vdm ("ERROR: failed to get meta info for disc %d, which is not CD nor DVD\n", disc->simple_type);
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message dialog window,
				      program cannot get metadata from disc */
				   _("Cannot get basic information about disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		cdw_main_ui_main_window_wrefresh();
		return CDW_ERROR;
	}
	cdw_vdm ("INFO: disc simple type is \"%s\"\n", disc->simple_type_label);

	return CDW_OK;
}





/**
   \brief Call external tools to collect meta information about a disc

   Function checks disc type, then checks which tool should check
   disc meta information, and then calls function calling the tool
   to do the job.

   \param disc - disc variable in which to store disc meta information

   \return CDW_OK on success
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_disc_get_meta_info_with_external_tools(cdw_disc_t *disc)
{
	cdw_assert (disc != (cdw_disc_t *) NULL, "ERROR: disc pointer is null\n");

	cdw_task_t *task = cdw_task_new(CDW_TASK_MEDIA_INFO, disc);
	if (task == (cdw_task_t *) NULL) {
		cdw_vdm ("ERROR: failed to create a task\n");
		return CDW_ERROR;
	}
	cdw_id_t tool_id = task->media_info.tool.id;
	cdw_assert (tool_id == CDW_TOOL_CDRECORD
		    || tool_id == CDW_TOOL_XORRISO
		    || tool_id == CDW_TOOL_DVD_RW_MEDIAINFO,
		    "ERROR: tool selector returns unsupported tool %lld\n", tool_id);
	cdw_assert (task->media_info.tool.label != (char *) NULL,
		    "ERROR: tool selector returns NULL path to tool\n");

	disc->media_info_source = tool_id;
	cdw_rv_t crv = CDW_OK;
	if (tool_id == CDW_TOOL_CDRECORD) {
		/* 2TRANS: this is message displayed in process window */
		cdw_processwin_display_main_info(_("Getting disc info with cdrecord"));
		crv = cdw_cdrecord_run_task(task, disc);

	} else if (tool_id == CDW_TOOL_DVD_RW_MEDIAINFO) {
		/* 2TRANS: this is message displayed in process window */
		cdw_processwin_display_main_info(_("Getting disc info with dvd+rw-mediainfo"));
		crv = cdw_dvd_rw_mediainfo_run_task(task, disc);

	} else if (tool_id == CDW_TOOL_XORRISO) {
		/* 2TRANS: this is message displayed in process window */
		cdw_processwin_display_main_info(_("Getting disc info with xorriso"));
		crv = cdw_xorriso_run_task(task, disc);
	} else {
		; /* covered by assert above */
	}
	cdw_rv_t tool_status = cdw_task_check_tool_status(task);
	cdw_task_delete(&task);

	if (tool_status != CDW_OK ||  crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to get media info with %s\n",
			 tool_id == CDW_TOOL_CDRECORD ? "cdrecord" :
			 tool_id == CDW_TOOL_XORRISO ? "xorriso" :
			 tool_id == CDW_TOOL_DVD_RW_MEDIAINFO ? "dvd+rw-mediainfo" : "unknown tool");
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





int cdw_disc_get_default_drive_speed_id(cdw_disc_t *disc)
{
	cdw_assert (disc != (cdw_disc_t *) NULL, "ERROR: disc pointer is null\n");

	if (disc->write_speeds.drive_default_speed == -1) {
		/* default drive speed is unknown */
		return disc->write_speeds.speeds[0];
	} else {
		return disc->write_speeds.drive_default_speed;
	}
}





/**
   \brief Remove any repeating values from table of writing speeds, sort the speeds

   When parsing output of "cdrecord -prcap", cdw puts speeds available
   for a disc in table. The problem is that the speed values can repeat
   in the table (cdrecord prints them this way). This function removes
   duplicates from the table.

   There is also another problem with writing speeds table, as printed by
   "cdrecord -prcap" or dvd+rw-mediainfo: values in the table may be in
   order from highest to lowest. This function reverses the table to have
   values from lowest to highest.

   Similar problem occurs when getting write speeds table with libburn.

   The function works on disc->write_speeds.speeds[].

   Caller has to make sure that number of (non-sorted) speeds is greater
   than zero.

   \param disc - disc in which to fix write speeds table
*/
void cdw_disc_resolve_write_speeds(cdw_disc_t *disc)
{
#ifndef NDEBUG
	cdw_assert (disc != (cdw_disc_t *) NULL, "ERROR: disc pointer is null\n");
	cdw_vdm ("INFO: table of write speeds before resolving:\n");
	for (int i = 0; i < CDW_DISC_N_SPEEDS_MAX; i++) {
		if (disc->write_speeds.speeds[i] != 0) {
			cdw_vdm ("INFO:   write_speeds[%d] = '%d'\n", i, disc->write_speeds.speeds[i]);
		}
	}
#endif

	/* number of speeds in table has been limited at the level of parsing
	   cdrecord / dvd+rw-mediainfo output, so at this point we are
	   sure that there is no more than CDW_DISC_N_SPEEDS_MAX
	   speed values in write_speeds[] table

	   output from libburn or dvd+rw-mediainfo may contain duplicates:
	   example list of speeds from dvd+rw-mediainfo: 16, 16, 12, 10, 8, 4
	   example list of speeds from libburn: 16, 12, 10, 8, 4, 16, 16, 12, 10, 8, 4, 16 */

	/* algorithm of resolving speeds table:
	   1. sort from highest to lowest (with possible zeros at the end of result table);
	   2. remove duplicates (compress);
	   3. count non-zero speeds to get number of valid speeds: n_speeds;
	   4. sort from lowest to highest, but only first n_speeds items; */

	qsort(disc->write_speeds.speeds, CDW_DISC_N_SPEEDS_MAX, sizeof(disc->write_speeds.speeds[0]), cdw_utils_compare_ints_reverse);

	cdw_utils_compress_table_of_ints(disc->write_speeds.speeds, CDW_DISC_N_SPEEDS_MAX);

	for (int i = 0; i < CDW_DISC_N_SPEEDS_MAX; i++) {
		if (disc->write_speeds.speeds[i] == 0) {
			disc->write_speeds.n_speeds = i;
			break;
		}
	}

	qsort(disc->write_speeds.speeds, (size_t) disc->write_speeds.n_speeds, sizeof(disc->write_speeds.speeds[0]), cdw_utils_compare_ints);

#ifndef NDEBUG
	cdw_vdm ("INFO: table of write speeds after resolving (%d speeds total):\n", disc->write_speeds.n_speeds);
	for (int i = 0; i < disc->write_speeds.n_speeds; i++) {
		cdw_vdm ("INFO:   write_speeds[%d] = '%d'\n", i, disc->write_speeds.speeds[i]);
	}
#endif

	return;
}





/**
   \brief Check disc data structure for any obvious errors

   The function checks if fields of \p disc don't contain any
   obvious errors or inconsistencies.

   NOTE: the function should avoid calculating any values of \p disc
   fields, it should only check them!

   \param disc - disc to be checked

   \return CDW_ERROR on errors in disc
   \return CDW_OK if disc seems to be ok
*/
cdw_rv_t cdw_disc_validate(cdw_disc_t *disc)
{
	cdw_assert (disc != (cdw_disc_t *) NULL, "ERROR: disc pointer is null\n");

	if (disc->simple_type != CDW_DISC_SIMPLE_TYPE_CD
	    && disc->simple_type != CDW_DISC_SIMPLE_TYPE_DVD) {

		/* simple_type _has to_ be known to do anything useful */
		cdw_vdm ("ERROR: disc is neither CD nor DVD\n");
		return CDW_ERROR;
	}

	if (disc->type == CDW_CD_AUDIO
	    || disc->type == CDW_CD_ROM
	    || disc->type == CDW_DVD_ROM) {

		cdw_assert (disc->type_writable == CDW_FALSE,
			    "ERROR: read only disc not marked as non-writable\n");
	}

	if (disc->type_writable == CDW_TRUE) {
		if (disc->write_speeds.n_speeds == 0) {
			cdw_vdm ("ERROR: no writing speeds for writable disc!\n");
			return CDW_ERROR;
		} else if (disc->write_speeds.n_speeds == 1) {
			if (disc->write_speeds.speeds[0] == 0) {
				cdw_vdm ("ERROR: first and only writing speed is zero!\n");
				return CDW_ERROR;
			}
		} else {
			; /* speeds table is non-empty, so it seems to be ok */
		}
	} else {
		; /* read-only disc, not much to validate */
	}

	return CDW_OK;
}





/**
   \brief Calculate values of some fields in disc that represent
   higher-level information about disc

   The function checks some fields in \p disc that were set by
   cdrecord- or dvd+rw-mediainfo-related code (let's call them lower-level
   data), and based on values of these fields it sets values of some fields
   representing higher-level information.

   Simple example: "if disc type is DVD-RW then disc is erasable", or
   "if disc type is CD-RW and certain values related to session information
   have certain values then disc is or is not writable".

   Rest of code in cdw can then check these higher-level fields instead
   figuring it out itself over and over again.

   Function also sets appropriate value of disc->type_label
   so nice label of disc type can also be used.

   Call this function in function getting meta information from
   a disc (cdw_disc_get_meta_info()) or in function resetting
   a disc (cdw_disc_reset()).

   \param disc - disc to be resolved

   \return CDW_OK when there were no problems
   \return CDW_NO if disc type is NONE or UNKNOWN
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_disc_resolve(cdw_disc_t *disc)
{
	cdw_assert (disc != (cdw_disc_t *) NULL, "ERROR: disc pointer is null\n");

	/* first set disc label, it may be needed later */
	cdw_disc_resolve_type_label(disc);

	/* safe defaults; these values should be assigned
	   in disc reset function, but anyway... */
	disc->type_erasable = CDW_UNKNOWN;
	disc->type_writable = CDW_UNKNOWN;
	disc->state_writable = CDW_UNKNOWN;
	/* don't modify disc->state_empty, it was already set using
	   information about file system on the disc */

	const char *drive = cdw_drive_get_drive_fullpath();
	int status = cdw_drive_status(drive);
	if (status == CDS_NO_DISC || status == CDS_TRAY_OPEN) {
		/* this check is to avoid further unnecessary checks
		   and possible printing of warning messages that
		   disc type is unknown */
		return CDW_NO;
	}


	/* types: erasable and writable, this is quite easy part */
	cdw_disc_type_t t = disc->type;
	if (t == CDW_DISC_TYPE_UNKNOWN) {
		cdw_vdm ("WARNING: disc in drive is of type \"unknown\"\n");
		return CDW_NO;

	} else if (t == CDW_DISC_NONE) {
		cdw_vdm ("INFO: no disc in drive\n");
		/* TODO: should this happen at all? should we call this
		   function when there is no disc in drive? */
		return CDW_NO;

	} else if (t == CDW_CD_AUDIO
		   || t == CDW_CD_ROM
		   || t == CDW_DVD_ROM) {

		disc->type_erasable = CDW_FALSE;
		disc->type_writable = CDW_FALSE;

	} else if (t == CDW_CD_R
		   || t == CDW_DVD_R
		   || t == CDW_DVD_R_SEQ
		   || t == CDW_DVD_R_RES
		   || t == CDW_DVD_RP
		   || (t == CDW_DVD_RP_DL && global_config.support_dvd_rp_dl)) {

		disc->type_erasable = CDW_FALSE;
		disc->type_writable = CDW_TRUE;

	} else if (t == CDW_CD_RW
		   || t == CDW_DVD_RW
		   || t == CDW_DVD_RW_SEQ
		   || t == CDW_DVD_RW_RES
		   || t == CDW_DVD_RWP) {

		disc->type_erasable = CDW_TRUE;
		disc->type_writable = CDW_TRUE;
	} else {
		cdw_vdm ("ERROR: unknown disc type %d\n", t);
		return CDW_ERROR;
	}


	/* now more difficult part: it is tool-dependent
	   and (since tool support varies between tools
	   and tool versions) may be tricky */

	int empty = disc->state_empty;
	if (disc->media_info_source == CDW_TOOL_CDRECORD) {
		cdw_cdrecord_set_disc_states(disc);
	} else if (disc->media_info_source == CDW_TOOL_DVD_RW_MEDIAINFO) {
		cdw_dvd_rw_mediainfo_set_disc_states(disc);
	} else if (disc->media_info_source == CDW_TOOL_XORRISO) {
		cdw_xorriso_set_disc_states(disc);
	} else {
		cdw_vdm ("ERROR: unknown media info tool in disc struct: %lld\n",
			 disc->media_info_source);

		return CDW_ERROR;
	}
	if (empty != disc->state_empty) {
		cdw_vdm ("WARNING: external tool changes \"state empty\"\n");
		disc->state_empty = empty;
	}


	/* now easy part */

	/* disc->state_empty was set earlier */

	/* some common tool independent settings;
	   these "state values" may have been set by tool-dependent
	   code, but here cover some special cases, in tool independent fashion */
	if (disc->type == CDW_DVD_RWP
	    || disc->type == CDW_DVD_RW_RES) {

		/* The two types of disc receive special treatment.
		   I'm not sure if this is 100% true, but I think that you
		   can always expand (grow) iso file system that exists on
		   'DVD+RW' and 'DVD-RW Restricted Overwrite'; search for
		   "it seems that with DVD+RW and DVD-RW Restricted"
		   comment in write_wizard.c */

		if (disc->state_writable != CDW_TRUE) {
			cdw_vdm ("WARNING: tool \"%s\" didn't set \"writable=true\" for \"%s\" disc\n",
				 cdw_ext_tools_get_tool_name(disc->media_info_source),
				 disc->type_label);
		}

		disc->state_writable = CDW_TRUE;

	} else if (disc->type == CDW_CD_ROM
		   || disc->type == CDW_CD_AUDIO
		   || disc->type == CDW_DVD_ROM) {

		if (disc->state_writable != CDW_FALSE) {
			cdw_vdm ("WARNING: tool \"%s\" didn't set \"writable=false\" for \"%s\" disc\n",
				 cdw_ext_tools_get_tool_name(disc->media_info_source),
				 disc->type_label);
		}

		disc->state_writable = CDW_FALSE;
	} else {
		; /* pass */
	}

	if (disc->type_writable == CDW_TRUE) {
		/* writable disc has some write speeds that
		   need to be taken care of */
		cdw_disc_resolve_write_speeds(disc);

		/* calculating capacity makes sense only for
		   writable discs */
		cdw_disc_resolve_capacities(disc);
	}

	return CDW_OK;
}





/**
   \brief Function sets proper disc type label

   Function checks type of given \p disc (an integer value) and
   copies proper human-readable label to \p disc->type_label field.
   Disc type must be set before calling this function.

   If a disc type is DVD+R DL, function checks if support for this
   disc type is enabled, and acts accordingly.

   \param disc - disc to be processed
*/
void cdw_disc_resolve_type_label(cdw_disc_t *disc)
{
	cdw_assert (disc != (cdw_disc_t *) NULL, "ERROR: disc pointer is null\n");

	char *l = disc->type_label;

	if (disc->type == CDW_DVD_RP_DL &&
	    ! global_config.support_dvd_rp_dl) {

		strncpy(l, cdw_disc_type_labels[CDW_DISC_TYPE_UNKNOWN], CDW_DISC_TYPE_LABEL_LEN);
	} else {
		strncpy(l, cdw_disc_type_labels[disc->type], CDW_DISC_TYPE_LABEL_LEN);
	}
	l[CDW_DISC_TYPE_LABEL_LEN] = '\0';

	cdw_vdm ("INFO: disc type label %d resolved as \"%s\"\n", disc->type, l);
	return;
}





/**
   \brief Function checking if given disc is readable

   Disc is readable if it is not empty and is of proper (supported) type
   and has supported file system.

   \param disc - disc to be checked

   \return (char *) NULL pointer if cdw will be able to read data or audio from the disc
   \return pointer to const char string describing reason why disc can't be read, if it can't be read by cdw
*/
const char *cdw_disc_check_if_is_readable(cdw_disc_t *disc)
{
	cdw_assert (disc != (cdw_disc_t *) NULL, "ERROR: disc pointer is null\n");

	if (disc->state_empty == CDW_TRUE) {
		cdw_vdm ("ERROR: disc is empty, shouldn't you have checked this before calling this function?\n");
		return disc_not_readable_reasons[CDW_DISC_NOT_READABLE_EMPTY].label;;
	}

	if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
		if (disc->type == CDW_CD_AUDIO) {
			/* reading Audio CDs with cdio calls */
			return (char *) NULL;
		}
		if (disc->cdio_disc_mode == CDIO_DISC_MODE_CD_DATA /* Mode1/Mode2 Formless? */
		    || disc->cdio_disc_mode == CDIO_DISC_MODE_CD_XA /* Mode2 Form1 / Mode2 Form2 ? */

		    /* well, this is redundant, covered by "if" above */
		    || disc->cdio_disc_mode == CDIO_DISC_MODE_CD_DA

		    /* not 100% sure what this is, but cdw doesn't support it */
		    /* ||  disc->cdio_disc_mode == CDIO_DISC_MODE_CD_MIXED */ ) {

			/* reading CDs with cdio calls, so I'm not
			   caring about file system support in operating system */
			return (char *) NULL;
		}

		return disc_not_readable_reasons[CDW_DISC_NOT_READABLE_OTHER].label;
	} else if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_DVD) {
		if (cdw_cdio_is_fs_iso(disc->fs.type)) {

			return (char *) NULL;

			/* For some time I thought that in order to
			   read data from optical disc with ISO9660 file
			   system you have to have support for such file
			   system enabled in operating system kernel.
			   It turns out that this is not the case - I was
			   able to successfully read such disc with support
			   for ISO9660 completely disabled in kernel.
			   The disc can't be mounted, but it can be read
			   with read().
			   So - this piece of code below is not necessary,
			   but I will keep it for posterity. */
#if 0
			/* reading DVDs with UNIX read(), so I still need to
			   check for support of ISO9660 in kernel */

			cdw_rv_t crv = cdw_sys_check_file_system_support(CDIO_FS_ISO_9660);
			if (crv == CDW_OK) {
				return (char *) NULL;
			} else if (crv == CDW_NO) {
				return disc_not_readable_reasons[CDW_DISC_NOT_READABLE_NO_ISO9660].label;
			} else { /* CDW_CANCEL || CDW_ERROR */
				return disc_not_readable_reasons[CDW_DISC_NOT_READABLE_OTHER].label;
			}
#endif
		} else {
			cdw_vdm ("INFO: disc is not readable because file system type is not readable\n");
			return disc_not_readable_reasons[CDW_DISC_NOT_READABLE_UNSUPPORTED_FS].label;;
		}
	} else {
		cdw_assert (0, "ERROR: incorrect simple disc type %d\n", disc->simple_type);
		return disc_not_readable_reasons[CDW_DISC_NOT_READABLE_OTHER].label;
	}
}





/**
   \brief Debug function displaying on stderr basic information about a disc

   \param disc - disc to display
*/
void cdw_disc_debug_print_disc(cdw_disc_t *disc)
{
	cdw_assert (disc != (cdw_disc_t *) NULL, "ERROR: disc pointer is null\n");

	cdw_vdm ("CURRENT DISC:    simple type = \"%s\"\n", disc->simple_type_label);
	cdw_vdm ("CURRENT DISC:           type = \"%s\" (%d)\n", disc->type_label, disc->type);
	cdw_vdm ("CURRENT DISC:    state empty = \"%s\"\n", cdw_utils_get_cdw_bool_type_label(disc->state_empty));
	cdw_vdm ("CURRENT DISC: state writable = \"%s\"\n", cdw_utils_get_cdw_bool_type_label(disc->state_writable));
	cdw_vdm ("CURRENT DISC:  type writable = \"%s\"\n", cdw_utils_get_cdw_bool_type_label(disc->type_writable));
	cdw_vdm ("CURRENT DISC:  type erasable = \"%s\"\n", cdw_utils_get_cdw_bool_type_label(disc->type_erasable));

	float coeff = 512.0; /* (n_sectors * 2048) / (1024.0 * 1024.0) = n_sectors / 512.0 */
	cdw_vdm ("CURRENT DISC CAPACITY:  used = %lld (%.3f MB)\n", disc->capacity.sectors_used, (double) disc->capacity.sectors_used / coeff);
	cdw_vdm ("CURRENT DISC CAPACITY: total = %lld (%.3f MB)\n", disc->capacity.sectors_total, (double) disc->capacity.sectors_total / coeff);

	if (disc->state_empty == CDW_TRUE) {
		return;
	}
	cdw_vdm ("CURRENT DISC:   file system: = \"%s\"\n", disc->fs.type_label);
	if (cdw_cdio_is_fs_iso(disc->fs.type)) {
		cdw_vdm ("CURRENT DISC:     volume id: = \"%s\"\n", disc->fs.volume_id);
	}

	if (disc->type_writable) {
		cdw_vdm ("CURRENT DISC: %d write speed(s):\n", disc->write_speeds.n_speeds);
		for (int i = 0; i < disc->write_speeds.n_speeds; i++) {
			cdw_vdm ("CURRENT DISC:    %d\n", disc->write_speeds.speeds[i]);
		}
	}
	return;
}





void cdw_disc_resolve_capacities(cdw_disc_t *disc)
{
	if (disc->cdrecord_info.rzone_size > 0
	    && disc->cdrecord_info.rzone_size > disc->cdrecord_info.phys_size) {

		/* sometimes rzone size is a great indicator of total
		   disc size, but, alas, only sometimes */

		cdw_vdm ("INFO: using cdrecord info (DVD rzone size)\n");
		disc->capacity.sectors_total = disc->cdrecord_info.rzone_size;
		disc->capacity.sectors_used = disc->fs.n_sectors;
	} else if (disc->cdrecord_info.phys_size != -1) {
		cdw_vdm ("INFO: using cdrecord info (DVD phys size)\n");
		disc->capacity.sectors_total = disc->cdrecord_info.phys_size;
		disc->capacity.sectors_used = disc->fs.n_sectors;
	} else if (disc->cdrecord_info.start_of_lead_out != -1) {
		cdw_vdm ("INFO: using cdrecord info (CD)\n");
		disc->capacity.sectors_total = disc->cdrecord_info.start_of_lead_out;
		disc->capacity.sectors_used = disc->fs.n_sectors;
	} else if (disc->dvd_rw_mediainfo_info.end_lba != -1) {
		cdw_vdm ("INFO: using dvd+rw-mediainfo info\n");
		disc->capacity.sectors_total = disc->dvd_rw_mediainfo_info.end_lba;
		disc->capacity.sectors_used = disc->fs.n_sectors;
	} else if (disc->xorriso_info.sectors_total != -1) {
		cdw_vdm ("INFO: using xorriso info\n");
		disc->capacity.sectors_total = disc->xorriso_info.sectors_total;
		disc->capacity.sectors_used = disc->xorriso_info.sectors_used >= 0 ? disc->xorriso_info.sectors_used : disc->fs.n_sectors;
	} else if (disc->libburn_info.unformatted_size != 0) {
		cdw_vdm ("INFO: using libburn info\n");
		disc->capacity.sectors_total = disc->libburn_info.unformatted_size / 2048;
		disc->capacity.sectors_used = 0;
		if (disc->fs.n_sectors > 0) {
			cdw_vdm ("WARNING: assuming that disc is empty while fs size != 0 (%ld)\n", disc->fs.n_sectors);
		}
	} else {
		cdw_vdm ("WARNING: no info available\n");
	}
	return;
}





bool cdw_disc_is_disc_type_writable(cdw_disc_t *disc)
{
	if (disc->type == CDW_DISC_TYPE_UNKNOWN
	    || disc->type == CDW_CD_AUDIO
	    || disc->type == CDW_CD_ROM
	    || disc->type == CDW_DVD_ROM) {

		return false;
	} else {
		return true;
	}
}
