/**
 * Copyright (C) 2009-2013 Paul Fretwell - aka 'Footleg' (drfootleg@gmail.com)
 * 
 * This file is part of Cave Converter.
 * 
 * Cave Converter is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Cave Converter is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Cave Converter.  If not, see <http://www.gnu.org/licenses/>.
 */
package cavesurvey.data.model;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import org.junit.Test;

import footleg.cavesurvey.data.model.SurveyLeg;
import footleg.cavesurvey.data.model.SurveyStation;

/**
 * Unit test class for {@link footleg.cavesurvey.data.model.SurveyLeg}
 * 
 * @author      Footleg <drfootleg@gmail.com>
 * @version     2012.12.26                                (ISO 8601 YYYY.MM.DD)
 * @since       1.6                                       (The Java version used)
 */
public class SurveyLegTest {

//	double easting = 4800456.0;
//	double northing = 4800789.75;
//	double altitude = 8900.45;
	
	/**
	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#setFromStn(int)}.
	 */
	@Test
	public void testSetFromStn() {
		SurveyLeg leg = new SurveyLeg();
		
		int stnID = 543;
		SurveyStation stn = new SurveyStation( stnID );
		leg.setFromStn(stn);
		
		assertEquals( stnID, leg.getFromStn().getId() );
	}

	/**
	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#setToStn(int)}.
	 */
	@Test
	public void testSetToStn() {
		SurveyLeg leg = new SurveyLeg();
		int stnID = 12345;
		SurveyStation stn = new SurveyStation( stnID );
		leg.setToStn(stn);
		
		assertEquals( stnID, leg.getToStn().getId() );
	}

	/**
	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#setLength(double)}.
	 */
	@Test
	public void testSetLength() {
		SurveyLeg leg = new SurveyLeg();
		double length = 123.45;
		
		leg.setLength(length);
		
		assertEquals( length, leg.getLength(), 0.0 );
	}

	/**
	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#setCompass(double)}.
	 */
	@Test
	public void testSetCompass() {
		SurveyLeg leg = new SurveyLeg();
		double compass = 123.45;
		
		leg.setCompass(compass);
		
		assertEquals( compass, leg.getCompass(), 0.0 );
	}

	/**
	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#setClino(double)}.
	 */
	@Test
	public void testSetClino() {
		SurveyLeg leg = new SurveyLeg();
		double clino = 123.45;
		
		leg.setClino(clino);
		
		assertEquals( clino, leg.getClino(), 0.0 );
	}

	/**
	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#setLeft(double)}.
	 */
	@Test
	public void testSetLeft() {
		SurveyLeg leg = new SurveyLeg();
		double left = 0.45;
		leg.setLeft(left);
		assertEquals( left, leg.getLeft(), 0.0 );

		left = 0.0;
		leg.setLeft(left);
		assertEquals( left, leg.getLeft(), 0.0 );

		//TODO: Should error 
		left = -0.1;
		leg.setLeft(left);
		assertEquals( left, leg.getLeft(), 0.0 );
}

	/**
	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#setRight(double)}.
	 */
	@Test
	public void testSetRight() {
		SurveyLeg leg = new SurveyLeg();
		double right = 0.45;
		leg.setRight(right);
		assertEquals( right, leg.getRight(), 0.0 );

		right = 0.0;
		leg.setRight(right);
		assertEquals( right, leg.getRight(), 0.0 );
	}

	/**
	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#setUp(double)}.
	 */
	@Test
	public void testSetUp() {
		SurveyLeg leg = new SurveyLeg();
		double up = 0.45;
		leg.setUp(up);
		assertEquals( up, leg.getUp(), 0.0 );

		up = 0.0;
		leg.setUp(up);
		assertEquals( up, leg.getUp(), 0.0 );
	}

	/**
	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#setDown(double)}.
	 */
	@Test
	public void testSetDown() {
		SurveyLeg leg = new SurveyLeg();
		double down = 0.45;
		leg.setDown(down);
		assertEquals( down, leg.getDown(), 0.0 );

		down = 0.0;
		leg.setDown(down);
		assertEquals( down, leg.getDown(), 0.0 );
	}

//	/**
//	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#setFixedStn(boolean, double, double, double)},
//	 * {@link footleg.cavesurvey.data.model.SurveyLeg#clearFixedStn()} and associated getters.
//	 */
//	@Test
//	public void testSetFixedStnFrom() {
//		SurveyLeg leg = new SurveyLeg();
//		
//		leg.setFixedStn(true, easting, northing, altitude);
//		
//		assertEquals( easting, leg.getEasting(), 0.0 );
//		assertEquals( northing, leg.getNorthing(), 0.0 );
//		assertEquals( altitude, leg.getAltitude(), 0.0 );
//		assertTrue( leg.hasFixedStn() );
//		assertTrue( leg.isFromStnFixed() );
//		assertTrue( leg.isToStnFixed() == false );
//		
//		leg.clearFixedStn();
//		assertTrue( leg.hasFixedStn() == false );
//		assertTrue( leg.isFromStnFixed() == false );
//		assertTrue( leg.isToStnFixed() == false );
//	}
//
//	/**
//	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#setFixedStn(boolean, double, double, double)},
//	 * {@link footleg.cavesurvey.data.model.SurveyLeg#clearFixedStn()} and associated getters.
//	 */
//	@Test
//	public void testSetFixedStnTo() {
//		SurveyLeg leg = new SurveyLeg();
//		
//		leg.setFixedStn(false, easting, northing, altitude);
//		
//		assertEquals( easting, leg.getEasting(), 0.0 );
//		assertEquals( northing, leg.getNorthing(), 0.0 );
//		assertEquals( altitude, leg.getAltitude(), 0.0 );
//		assertTrue( leg.hasFixedStn() );
//		assertTrue( leg.isFromStnFixed() == false );
//		assertTrue( leg.isToStnFixed() );
//		
//		leg.clearFixedStn();
//		assertTrue( leg.hasFixedStn() == false );
//		assertTrue( leg.isFromStnFixed() == false );
//		assertTrue( leg.isToStnFixed() == false );
//	}

	/**
	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#setSplay(boolean)}.
	 */
	@Test
	public void testSetSplay() {
		SurveyLeg leg = new SurveyLeg();
		
		leg.setSplay( true );
		assertTrue( leg.isSplay() );
		
		leg.setSplay( false );
		assertTrue( leg.isSplay() == false );
	}

	/**
	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#compareTo(footleg.cavesurvey.data.model.SurveyLeg)}.
	 */
	@Test
	public void testCompareTo() {
		SurveyLeg leg1 = new SurveyLeg();
		SurveyLeg leg2 = new SurveyLeg();
		SurveyLeg leg3 = new SurveyLeg();
		SurveyLeg leg4 = new SurveyLeg();
		
		int stnID1 = 1;
		int stnID2 = 2;
		int stnID3 = 3;
		int stnID4 = 4;
		
		SurveyStation stn1 = new SurveyStation(stnID1);
		SurveyStation stn2 = new SurveyStation(stnID2);
		SurveyStation stn3 = new SurveyStation(stnID3);
		SurveyStation stn4 = new SurveyStation(stnID4);
		
		leg1.setFromStn(stn1);
		leg1.setToStn(stn2);
		leg2.setFromStn(stn2);
		leg2.setToStn(stn3);
		leg3.setFromStn(stn2);
		leg3.setToStn(stn4);
		leg4.setFromStn(stn2);
		leg4.setToStn(stn1);
		
		assertTrue( leg1.compareTo(leg2) < 0 );
		assertTrue( leg2.compareTo(leg1) > 0 );
		assertTrue( leg2.compareTo(leg3) < 0 );
		assertTrue( leg3.compareTo(leg2) > 0 );
		assertTrue( leg2.compareTo(leg4) > 0 );
		assertTrue( leg4.compareTo(leg2) < 0 );
		
	}

	/**
	 * Test method for {@link footleg.cavesurvey.data.model.SurveyLeg#clone()}.
	 */
	@Test
	public void testClone() {
		SurveyLeg leg = new SurveyLeg();
		int fromStnId = 123;
		int toStnId = 456;
		SurveyStation fromStn = new SurveyStation( fromStnId );
		SurveyStation toStn = new SurveyStation( toStnId );
		double length = 123.45;
		double compass = 123.45;
		double clino = 123.45;
		double left = 1.2;
		double right = 0.1;
		double up = 0.23;
		double down = 0.92;
		boolean splay = true;
//		boolean fixFromStn = true;
		
		//Create leg fully populated with test data
		leg.setFromStn(fromStn);
		leg.setToStn(toStn);
		leg.setLength(length);
		leg.setCompass(compass);
		leg.setClino(clino);
		leg.setLeft(left);
		leg.setRight(right);
		leg.setUp(up);
		leg.setDown(down);
		leg.setSplay(splay);
//TODO remove		leg.setFixedStn(fixFromStn, easting, northing, altitude);
		
		//Clone leg
		SurveyLeg legClone = leg.clone();
		
		//Set clone has the correct data values
		assertEquals( fromStnId, legClone.getFromStn().getId() );
		assertEquals( toStnId, legClone.getToStn().getId() );
		assertEquals( length, legClone.getLength(), 0.0 );
		assertEquals( compass, legClone.getCompass(), 0.0 );
		assertEquals( clino, legClone.getClino(), 0.0 );
		assertEquals( left, legClone.getLeft(), 0.0 );
		assertEquals( right, legClone.getRight(), 0.0 );
		assertEquals( up, legClone.getUp(), 0.0 );
		assertEquals( down, legClone.getDown(), 0.0 );
		assertTrue( legClone.isSplay() == splay );

//TODO remove		assertEquals( easting, legClone.getEasting(), 0.0 );
//TODO remove		assertEquals( northing, legClone.getNorthing(), 0.0 );
//TODO remove		assertEquals( altitude, legClone.getAltitude(), 0.0 );
//TODO remove		assertTrue( legClone.hasFixedStn() );
//TODO remove		assertTrue( legClone.isFromStnFixed() == fixFromStn );
//TODO remove		assertTrue( legClone.isToStnFixed() != fixFromStn );

	}
}
