/**
   \class CWebCamViewer

   A class for the usual webcam activity: take pictures, save them do disk
   and/or upload to a server.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef INCLUDE_UNISTD_H
#include <unistd.h>
#endif
#ifdef INCLUDE_STDLIB_H
#include <stdlib.h>
#endif

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <qaccel.h>
#include <qbitmap.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qmultilineedit.h>
#include <qpixmap.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qstatusbar.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qtooltip.h>
#include <qurl.h>

#include "tracer.h"

#include "AudioCollector.h"

#include "CamStreamApp.h"
#include "WebCamViewer.h"

#include "icons/configure.xpm"
#include "icons/ctl_input.xpm"
#include "icons/ctl_size.xpm"
#include "icons/ctl_video.xpm"
#include "icons/ctl_tuner.xpm"
#include "icons/filesave.xpm"
#include "icons/mute.xpm"
#include "icons/savetimer.xpm"
#include "icons/showsnap.xpm"
#include "icons/snd_config.xpm"

TR_MODULE("WebCamViewer");

CWebCamViewer::CWebCamViewer(CVideoDevice *video, QWidget *parent, const char *name)
              : CCamWindow(parent, name)
{
   int r;
///   uint t;
   QWidget *pWidget;
   QToolBar *bar;
   QStatusBar *status;
   QString nullstring;
   QDomNode vconfig, node;

   TR_ENTER();

   /* Our central widget contains space for the view image, and optionally
      the last snapshot. The widget itself is empty (except for a background)
    */
   pVideo = video;
   pViewer = 0;
   m_SnapInterval = -1;
   m_SnapCounter = 0;
   pLastSnapshot = NULL;
   Upload.pClient = NULL;
   Upload.ErrorCondition = false;
   Upload.Commands.setAutoDelete(true);

#if QT_VERSION >= 0x30000
   connect(&m_ExternalCommand.Process, SIGNAL(processExited()), this, SLOT(SubProcessDone()));
#endif

   /* Get video configuration, set nodes in dialogs & devices */
   vconfig = g_pCamApp->FindVideoDeviceConfig(pVideo->GetIntfName(), pVideo->GetNodeName(), true);
   pVOptions = new CVideoOptions();
   pVOptions->SetXML(vconfig);
   pVideo->SetConfiguration(vconfig);

   m_pSnapshotSettingsDlg = new CSnapshotSettingsDlg(pVOptions, 0, "snapshot settings " + pVideo->GetIntfName());
#if QT_VERSION < 0x30000
   m_pSnapshotSettingsDlg->RunCommandCheckBox->setDisabled(true);
   m_pSnapshotSettingsDlg->CommandLineEdit->setDisabled(true);
#endif

   pSnapTimer = new QTimer(this);
   connect(pSnapTimer, SIGNAL(timeout()), this, SLOT(TimeSnapTick()));

   r = pVideo->Open();
   if (r < 0) {
     qWarning("Error opening video device: %d.", r);
     TR_RETURN();
   }
   setCaption(pVideo->GetIntfName());

   /* Connect the video device to us. You may wonder why the Notify and
      Resized signal aren't connected to us. Well, Notify is handled
      in the Viewer widget (which updates itself), and Resized() indirectly
      through the same Viewer. Only the error condition matters to us.
    */
   connect(pVideo, SIGNAL(Error(int)), this, SLOT(DeviceError(int)));

   /* Add buttons to toolbar */
   bar = new QToolBar("main", this);

   pButton[pbt_ctl_size  ] = new QToolButton(QPixmap((const char **)ctl_size_xpm),  "Size and framerate",                 "", pVideo, SLOT(ShowFormatDialog()),      bar);
   pButton[pbt_ctl_video ] = new QToolButton(QPixmap((const char **)ctl_video_xpm), "Video adjustments",                  "", pVideo, SLOT(ShowDisplayDialog()),     bar);
   pButton[pbt_ctl_input ] = new QToolButton(QPixmap((const char **)ctl_input_xpm), "Input",                              "", pVideo, SLOT(ShowSourceDialog()),      bar);
   pButton[pbt_ctl_tuner ] = new QToolButton(QPixmap((const char **)ctl_tuner_xpm), "Tuner",                              "", pVideo, SLOT(ShowTunerDialog()),       bar);
   pButton[pbt_mute      ] = new QToolButton(QPixmap((const char **)mute_xpm),      "Sound on/off",                       nullstring, this, SLOT(ClickedSoundOnOff()),       bar);
   pButton[pbt_config    ] = new QToolButton(QPixmap((const char **)configure_xpm), "Configuration",                      nullstring, this, SLOT(ClickedVideoConfig()),      bar);
   pButton[pbt_showsnap  ] = new QToolButton(QPixmap((const char **)showsnap_xpm),  "Show last snapshot",                 nullstring, this, SLOT(ClickedShowLastSnapshot()), bar);
   pButton[pbt_snapshot  ] = new QToolButton(QPixmap((const char **)filesave_xpm),  "Take snapshot",                      nullstring, this, SLOT(TakeSnapshot()),            bar);
   pButton[pbt_timesnap  ] = new QToolButton(QPixmap((const char **)savetimer_xpm), "Take snapshot at regular intervals", nullstring, this, SLOT(ClickedTimeSnapDlg()),      bar);

   pButton[pbt_ctl_size ]->setEnabled(pVideo->HasFormatDialog());
   pButton[pbt_ctl_video]->setEnabled(pVideo->HasDisplayDialog());
   pButton[pbt_ctl_input]->setEnabled(pVideo->HasSourceDialog());
   pButton[pbt_ctl_tuner]->setEnabled(pVideo->HasTunerDialog());
   pButton[pbt_mute     ]->setToggleButton(true);
   pButton[pbt_mute     ]->setEnabled(pVideo->IsMutable());
   pButton[pbt_showsnap ]->setToggleButton(true);
   pButton[pbt_timesnap ]->setToggleButton(true);
   addToolBar(bar);

   /* Create statusbar, add space for countdown timer */
   status = statusBar();
   pSnapLabel = new QLabel(status);
   pSnapLabel->setText("--:--");
   status->addWidget(pSnapLabel, 0, TRUE);

   /* Create auxilary timer & workaround for adjustSize() */
   connect(&m_SizeTimer, SIGNAL(timeout()), this, SLOT(CallAdjustSize()));
#if QT_VERSION >= 0x30000
   connect(this, SIGNAL(dockWindowPositionChanged(QDockWindow *)), this, SLOT(CallAdjustSize()));
#else
   connect(this, SIGNAL(endMovingToolBar(QToolBar *)), this, SLOT(CallAdjustSize()));
#endif

   /* See if we can mute/unmute the device. If yes, enable sound now */
   if (pVideo->IsMutable())
     pVideo->Mute(false);

   /* If the device has a tuner, add PageUp/PageDown slots to cycle through the channels */
   if (pVideo->GetNumberOfTuners() > 0)
   {
     QAccel *PageDown = new QAccel(this);
     QAccel *PageUp   = new QAccel(this);
     TVChannel tvc;

     TR(1, "Enabling TV channel selection with PgUp/PgDn");
     PageDown->connectItem(PageDown->insertItem(Key_Next),
                           this,
                           SLOT(CycleNextChannel()));

     PageUp->connectItem(PageUp->insertItem(Key_Prior),
                         this,
                         SLOT(CyclePrevChannel()));
     connect(pVideo, SIGNAL(TVChannelChanged()), this, SLOT(NewChannelSelected()));
     tvc = pVideo->GetCurrentTVChannel();
     setCaption(pVideo->GetIntfName() + " : " + tvc.Name);
   }

   /* Create an empty widget upon we place our viewer and snapshot panels */
   pWidget = new QWidget(this, "empty widget");
   assert(pWidget != NULL);
   pWidget->setBackgroundMode(QWidget::PaletteBase);
   pWidget->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
   setCentralWidget(pWidget);

   pViewer = new CImagePanelRGB(pVideo, pWidget);
//   pViewer = new CImagePanelYUV(pVideo, pWidget);
   pLastSnapshot = new CBasicPanel("snapshot", "Last snapshot", CCamPanel::RGB, pWidget);

   if (pViewer) {
     QPixmap blackimg;
     QSize viewsize;
     QPainter painter;

     connect(pViewer, SIGNAL(ChangedVisibleSize(const QSize &)), this, SLOT(DeviceChangedSize(const QSize &)));

     viewsize = pVideo->GetSize();
     blackimg.resize(viewsize);
     painter.begin(&blackimg);
     painter.fillRect(0, 0, viewsize.width(), viewsize.height(), black);
     painter.setPen(yellow);
     painter.drawText(0, 0, viewsize.width(), viewsize.height(), AlignCenter, tr("Your last saved snapshot appears here"));
     painter.end();
     if (pLastSnapshot != 0)
     {
       pLastSnapshot->SetSize(viewsize);
       pLastSnapshot->SetImage(0, blackimg.convertToImage());
       pLastSnapshot->hide();
     }

     pViewer->show();
   }

   RecalcTotalViewSize();
   TR_LEAVE();
}


CWebCamViewer::~CWebCamViewer()
{
   TR_ENTER();
   StopTimeSnap();

   if (pVideo) {
     QDomNode vconfig;

     if (pVideo->IsMutable())
       pVideo->Mute(true);
     pVideo->Close();

     vconfig = g_pCamApp->FindVideoDeviceConfig(pVideo->GetIntfName(), pVideo->GetNodeName(), true);
     pVideo->GetConfiguration(vconfig);
     pVOptions->GetXML(vconfig);
   }
   m_pSnapshotSettingsDlg->hide();
   delete pViewer;
   delete pLastSnapshot;
   delete Upload.pClient;
   TR_LEAVE();
}

// private

void CWebCamViewer::StartFTPUpload(const QString &local_name, const QString &remote_name, bool delafterwards)
{
   FTPCommandStruct *str;
   QString last;

   if (Upload.ErrorCondition) {
     qWarning("CWebCamViewer::StartFTPUpload() Error condition was set");
     return;
   }
   if (!Upload.Commands.isEmpty()) {
     qWarning("CWebCamViewer::StartFTPUpload() Still commands in queue, uploading previous image?");
     if (delafterwards) {
       // remove local_name to prevent a bunch of lingering files...
       QFile::remove(local_name);
     }
     return;
   }

   if (Upload.pClient == NULL) {
     Upload.pClient = new CFTPClient();
     if (Upload.pClient == NULL) {
       qWarning("Failed to create FTP client process.");
       return;
     }
     connect(Upload.pClient, SIGNAL(StateChange(int, int, int, const QString &)), this, SLOT(FTPChangeState(int, int, int, const QString &)));
     connect(Upload.pClient, SIGNAL(ControlPortClosed()), this, SLOT(FTPClosed()));
     Upload.pClient->SetLogging(true);

     str = new FTPCommandStruct;
     str->Command = CFTPClient::cmdLogin;
     str->Param[0] = pVOptions->GetFTPUser();
     str->Param[1] = pVOptions->GetFTPPass();
     str->Param[2] = pVOptions->GetFTPServer();
     str->Param[3] = QString::number(pVOptions->GetFTPPort());
     Upload.Commands.append(str);

     str = new FTPCommandStruct;
     str->Command = CFTPClient::cmdSetType;
     str->Param[0] = "B";
     Upload.Commands.append(str);

     if (!pVOptions->GetFTPPath().isEmpty()) {
       str = new FTPCommandStruct;
       str->Command = CFTPClient::cmdChangeDir;
       str->Param[0] = pVOptions->GetFTPPath();
       Upload.Commands.append(str);
     }
     if (pVOptions->GetFTPPassive()) {
       str = new FTPCommandStruct;
       str->Command = CFTPClient::cmdPassive;
       Upload.Commands.append(str);
     }
   }

   str = new FTPCommandStruct;
   str->Command = CFTPClient::cmdUpload;
   str->Param[0] = local_name;
   str->Param[1] = remote_name;
   Upload.Commands.append(str);

   if (delafterwards)
   {
     str = new FTPCommandStruct;
     str->Command = CFTPClient::cmdDeleteLocalFile;
     str->Param[0] = local_name;
     Upload.Commands.append(str);
   }

   TriggerNextFTPCommand();
}

/**
  \brief Halft FTP transfer; clean up.

  Any lingering temporary local files are removed, and the FTP command queue
  is cleared (the current command is finished, though), and a 'destroy'
  command appended.
*/
void CWebCamViewer::StopFTP()
{
   FTPCommandStruct *str;
   uint u;
   TR_ENTER();
   // Look for cmdDeleteLocalFile, so we don't have any lingering files
   for (u = 0; u < Upload.Commands.count(); u++) {
     str = Upload.Commands.at(u);
     if (str->Command = CFTPClient::cmdDeleteLocalFile) {
       QFile::remove(str->Param[0]);
     }
   }
   Upload.Commands.clear();
   if (Upload.pClient != NULL) {
     str = new FTPCommandStruct;
     str->Command = CFTPClient::cmdDestroy;
     Upload.Commands.append(str);
     TriggerNextFTPCommand();
   }
   statusBar()->clear();
   TR_LEAVE();
}

void CWebCamViewer::TriggerNextFTPCommand()
{
   TR_ENTER();
   QTimer::singleShot(0, this, SLOT(NextFTPCommand()));
   TR_LEAVE();
}


void CWebCamViewer::StartSubProcess(const QString &command, const QString &filename, bool del_after_use)
{
#if QT_VERSION >= 0x30000
   if (m_ExternalCommand.Process.isRunning()) {
     qWarning("External command still busy, let it go...");
     if (del_after_use)
       if (!QFile::remove(filename))
         qWarning("Odd... cannot remove file for external command.");
     return;
   }
   m_ExternalCommand.FileName = filename;
   m_ExternalCommand.Delete = del_after_use;
   m_ExternalCommand.Process.clearArguments();
   m_ExternalCommand.Process.addArgument(command);
   m_ExternalCommand.Process.addArgument(filename);
   if (m_ExternalCommand.Process.start())
     qWarning("Failed to start external command...");
#endif
}


// private slots

void CWebCamViewer::ClickedVideoConfig()
{
   TR_ENTER();
   m_pSnapshotSettingsDlg->show();
   TR_LEAVE();
}

void CWebCamViewer::ClickedShowLastSnapshot()
{
   TR_ENTER();
   if (pLastSnapshot) {
     if (pButton[pbt_showsnap]->isOn())
       pLastSnapshot->show();
     else
       pLastSnapshot->hide();
     // Only enable RGB when we might need it (and then it's still a waste of CPU)
     if (pButton[pbt_showsnap]->isOn())
       pVideo->EnableRGB();
     else
       pVideo->DisableRGB();
     RecalcTotalViewSize();
   }
   TR_LEAVE();
}

void CWebCamViewer::ClickedTimeSnapDlg()
{
   TR_ENTER();
   if (pButton[pbt_timesnap]->isOn()) // was pushed down
   {
     int r;
     CTimeSnapDlg TimeSnapDlg(this, "timesnap dialog" + pVideo->GetIntfName(), true);

     r = pVOptions->GetSnapInterval();
     if (r > 60) {
       TimeSnapDlg.m_Interval->setValue(r / 60);
       TimeSnapDlg.m_Minutes->setChecked(true);
     }
     else {
       TimeSnapDlg.m_Interval->setValue(r);
       TimeSnapDlg.m_Seconds->setChecked(true);
     }

     if (TimeSnapDlg.exec() == QDialog::Accepted) {
       r = TimeSnapDlg.m_Interval->value();
       if (TimeSnapDlg.m_Minutes->isChecked()) {
         r *= 60; // convert to minutes
       }
       StartTimeSnap(r);
       pVOptions->SetSnapInterval(r);
     }
     else {
       pButton[pbt_timesnap]->setOn(false); // turn it off again
     }
   }
   else {
      StopTimeSnap();
   }
   TR_LEAVE();
}



void CWebCamViewer::ClickedSoundOnOff()
{
   TR_ENTER();
   if (pVideo == 0)
     return;
   pVideo->Mute(pButton[pbt_mute]->isOn());
   TR_LEAVE();
}



void CWebCamViewer::FTPChangeState(int command, int new_state, int result, const QString &server_msg)
{
   TR_ENTER();
   TR(2, "State (%d \"%s\", %d \"%s\", %d)", command, CFTPClient::CommandStr[command], new_state, CFTPClient::StateStr[new_state], result);

   switch(new_state) {
     case CFTPClient::stNOC:
       if (command != CFTPClient::cmdLogout) {
         Upload.ErrorCondition = true;
         // TODO: retry-button
         QMessageBox::warning(this, tr("FTP failure"), tr("FTP server closed connection"), QMessageBox::Ok, QMessageBox::NoButton);
         Upload.ErrorCondition = false;
       }
       StopFTP();
       break;

     case CFTPClient::stIdle:
       // success
       switch(command) {
         case CFTPClient::cmdUpload:
         case CFTPClient::cmdDownload:
           statusBar()->message(tr("Transfer completed in %1 seconds.").arg(Upload.StartTime.elapsed() / 1000.0, 3, 'f', 2));
           break;
       }
       TriggerNextFTPCommand();
       break;

     case CFTPClient::stWaitData:
       switch(command) {
         case CFTPClient::cmdUpload:
         case CFTPClient::cmdDownload:
           statusBar()->message(tr("Waiting for data connection."));
           break;
       }
       break;

     case CFTPClient::stTransfer:
       switch(command) {
         case CFTPClient::cmdUpload:
         case CFTPClient::cmdDownload:
           statusBar()->message(tr("Transfering file."));
           break;
       }
       break;

     case CFTPClient::stClosingData:
       switch(command) {
         case CFTPClient::cmdUpload:
         case CFTPClient::cmdDownload:
           statusBar()->message(tr("Closing data connection."));
           break;
       }
       break;

     case CFTPClient::stFailed:
       Upload.ErrorCondition = TRUE;
       switch(command) {
         case CFTPClient::cmdLogin:
           switch(result) {
             case 810:
               QMessageBox::warning(this,
                            tr("Name lookup"),
                            tr("Server name could not be resolved. Please check servername."), QMessageBox::Ok, QMessageBox::NoButton);
               break;

             case 811:
               QMessageBox::warning(this,
                            tr("Server connection"),
                            tr("Could not connect to server. Please check servername/IP address and FTP port number."), QMessageBox::Ok, QMessageBox::NoButton);
               break;

             default:
               QMessageBox::warning(this, tr("FTP login"), tr("Login failed. Please check username & password."), QMessageBox::Ok, QMessageBox::NoButton);
               break;
           }
           break;

         case CFTPClient::cmdChangeDir:
           QMessageBox::warning(this, tr("Changing directory"), tr("Failed to change to directory on FTP server.\nPlease check name and permissions on the server."), QMessageBox::Ok, QMessageBox::NoButton);
           break;

         case CFTPClient::cmdUpload:
           QMessageBox::warning(this, tr("Upload failed"), tr("Uploading of image failed. Server said:\n") + server_msg, QMessageBox::Ok, QMessageBox::NoButton);
           break;

         default:
           QMessageBox::warning(this, tr("Other error"), tr("Some other (unexpected) error occured (command = '%1').\nPlease see debug messages for more info.").arg(CFTPClient::CommandStr[command]), QMessageBox::Ok, QMessageBox::NoButton);
           break;
       }
       Upload.ErrorCondition = FALSE;
       StopFTP();
       break;

     case CFTPClient::stUnknown:
       qDebug("Action resulted in unknown result code");
       break;
   }
   TR_LEAVE();
}

void CWebCamViewer::NextFTPCommand()
{
   FTPCommandStruct *str;
   int PortNumber = 21; // Default FTP port
   bool PortNumberOk;

   if (Upload.Commands.isEmpty())
     return;

   str = Upload.Commands.first();
qDebug("CWebCamViewer::NextFTPCommand(): next_cmd = %d \"%s\"", str->Command, CFTPClient::CommandStr[str->Command]);

   if (Upload.pClient == NULL) {
     qDebug("Uhm, our FTP connections seems to be gone?");
     Upload.Commands.clear();
     return;
   }

   switch(str->Command) {
     case CFTPClient::cmdLogin:
       statusBar()->message(tr("Connecting to FTP server."));
       PortNumber = str->Param[3].toInt(&PortNumberOk);
       if (!PortNumberOk)
         PortNumber = 21;
       Upload.pClient->Connect(str->Param[0], str->Param[1], str->Param[2], PortNumber);
       break;

     case CFTPClient::cmdSetType:
       if (str->Param[0] == "B")
         Upload.pClient->SetTypeBinary();
       else
         Upload.pClient->SetTypeAscii();
       break;

     case CFTPClient::cmdChangeDir:
       Upload.pClient->ChangeDir(str->Param[0]);
       break;

     case CFTPClient::cmdUpload:
       Upload.StartTime.start();
       statusBar()->message(tr("Uploading..."));
       Upload.pClient->Upload(str->Param[0], str->Param[1]);
       break;

     case CFTPClient::cmdLogout:
       Upload.pClient->Logout();
       break;

     case CFTPClient::cmdListDir:
       Upload.pClient->ListDir();
       break;

     case CFTPClient::cmdPassive:
       Upload.pClient->SetPassive();
       TriggerNextFTPCommand();
       break;

     case CFTPClient::cmdDeleteLocalFile:
       QFile::remove(str->Param[0]);
       TriggerNextFTPCommand(); // This is not a real FTP command...
       break;

     case CFTPClient::cmdRename:
       Upload.pClient->Rename(str->Param[0], str->Param[1]);
       break;

     case CFTPClient::cmdDestroy:
       if (Upload.pClient != 0)
       {
         Upload.pClient->disconnect(this); // prevent loops
         delete Upload.pClient;
       }
       Upload.pClient = 0;
       break;

     default:
       qDebug("Unknown command.... bleh");
       StopFTP();
       break;
   }
   Upload.Commands.remove();
}

/**
  \brief Called when the remote FTP server closed the connection

  Clean up the FTP client on our side.
*/
void CWebCamViewer::FTPClosed()
{
  TR_ENTER();
  Upload.Commands.clear();
  if (Upload.pClient != 0)
  {
    Upload.pClient->disconnect(this); // prevent loops
    Upload.pClient->deleteLater(); // not now, since we are being called from that object!
  }
  Upload.pClient = 0;
  TR_LEAVE();
}


/* Called when an external command for an image has finished */
void CWebCamViewer::SubProcessDone()
{
#if QT_VERSION >= 0x30000
   QString status_msg;

   if (m_ExternalCommand.Delete) {
     if (!QFile::remove(m_ExternalCommand.FileName))
       qWarning("Could not remove temporary file after external command finished.");
   }
   if (m_ExternalCommand.Process.exitStatus()) {
     status_msg = tr("Command exited with status %1").arg( m_ExternalCommand.Process.exitStatus());
     statusBar()->message(status_msg);
   }
   else
     statusBar()->message(tr("External command successful."));
#endif
}

/* When the user clicks 'Ok' in the TimeSnap dialog, this function gets
   called and we will start the timed snapshots
 */
void CWebCamViewer::StartTimeSnap(int interval)
{
   m_SnapInterval = interval;
   if (m_SnapInterval < 1) {
     qWarning("Ignoring interval of %d seconds", m_SnapInterval);
     return;
    }
   m_SnapCounter = 0;
   pSnapTimer->start(1000);
   pVideo->EnableRGB();
}


void CWebCamViewer::StopTimeSnap()
{
   TR_ENTER();
   pSnapTimer->stop();
   pSnapLabel->setText("--:--");
   if (m_SnapInterval > 0) {
     m_SnapInterval = -1;
     m_SnapCounter = 0;
     pVideo->DisableRGB();
   }
   StopFTP();
   TR_LEAVE();
}

/**
  This function gets called once a second for a countdown to the next
  snapshot.
 */
void CWebCamViewer::TimeSnapTick()
{
   int rem;
   QString TimerText;

   if (m_SnapInterval < 0) {
     StopTimeSnap();
     return; // not active
   }
   m_SnapCounter++;

   if (m_SnapCounter >= m_SnapInterval) {
     TakeSnapshot();

     m_SnapCounter = 0;
   }
   // Update status bar, show remaining time
   rem = m_SnapInterval - m_SnapCounter;
   TimerText.sprintf("%02d:%02d", rem / 60 , rem % 60);
   pSnapLabel->setText(TimerText);
}


/**
  \brief Takes an image from the video stream, saves it in the snapshot imagebuffer, starts FTP upload
*/
void CWebCamViewer::TakeSnapshot()
{
   time_t now;
   struct tm *pTM = 0;
   bool stampimage, stampfile;
   bool savetodisk, ftptoserver;
   bool delafterftp, runcommand, delaftercmd;
   int sequence;

   TR_ENTER();
   QString SnapExtension, SnapBase, SnapFmt;
   QString savename, local_ftpname, remote_ftpname, cmdname, command;
   QString YourMessage;
   QFileInfo savefile;

   QImage DrawImage;
   QPainter p;
   QSize viewersize = pVideo->GetSize();
   QPixmap DrawArea(viewersize);

   stampimage = pVOptions->GetTimeInImage();
   stampfile = (pVOptions->GetMiddlePart() == CVideoOptions::Timestamp);
   savetodisk = pVOptions->GetSaveToDisk();
   ftptoserver = pVOptions->GetFTPToServer();
   runcommand = pVOptions->GetRunCommand();
   YourMessage = pVOptions->GetUserMessage();

qDebug("TakeSnapshot: stampimage = %c, stampfile %c, savetodisk = %c, ftptoserver = %c, runcommand = %c",
       stampimage ? 'T' : 'F',
       stampfile ? 'T' : 'F',
       savetodisk ? 'T' : 'F',
       ftptoserver ? 'T' : 'F',
       runcommand ? 'T' : 'F');

   if (stampimage || stampfile) {
     time(&now);
     pTM = localtime(&now);
   }

   if (stampimage || !YourMessage.isEmpty()) {
     char strbuf[80];
     QRect TextRect, DrawRect;

     DrawRect.setRect(5, 5, viewersize.width() - 10, viewersize.height() - 10);

     // prepare image for drawing
     DrawImage = pViewer->GetImage();
     DrawArea.convertFromImage(DrawImage, 0);
     p.begin(&DrawArea);
     p.setPen(pVOptions->GetTextColor());
     p.setFont(pVOptions->GetTextFont());

     // work from the bottom up
     if (stampimage)
     {
       /* Paint time & text in image */
       strftime(strbuf, 80, "%a, %d %b %Y %H:%M:%S %z %Z", pTM);
       p.drawText(DrawRect, AlignLeft | AlignBottom, strbuf, -1, &TextRect);
       DrawRect.setBottom(TextRect.top());
     }
     if (!YourMessage.isEmpty())
     {
       p.drawText(DrawRect, AlignLeft | AlignBottom, YourMessage, -1, &TextRect);
       DrawRect.setBottom(TextRect.top());
     }
     // end of drawing, convert back
     p.end();
     DrawImage = DrawArea.convertToImage();
   }
   else
     DrawImage = pViewer->GetImage().copy(); // Make deep copy of the current image

   // Store image in LastSnapshot
   pLastSnapshot->SetSize(viewersize);
   pLastSnapshot->SetImage(0, DrawImage);
   RecalcTotalViewSize();

   /* Create filename to save to disk */
   SnapFmt = pVOptions->GetFileFormat();
   SnapExtension = CCamStreamApp::FormatStrToExtension(SnapFmt);
   SnapBase = pVOptions->GetBaseName();
   command = pVOptions->GetCommand();

   switch (pVOptions->GetMiddlePart()) {
     case CVideoOptions::Overwrite: // overwrite
       savename.sprintf("%s.%s", (const char *)SnapBase, (const char *)SnapExtension);
       break;

     case CVideoOptions::Timestamp: // timestamp filename
       savename.sprintf("%s-%04d%02d%02d-%02d%02d%02d.%s",
                  (const char *)SnapBase,
                  1900 + pTM->tm_year, 1 + pTM->tm_mon, pTM->tm_mday,
                  pTM->tm_hour, pTM->tm_min, pTM->tm_sec,
                  (const char *)SnapExtension);
       break;

     case CVideoOptions::Sequence: // sequential numbering
       sequence = pVOptions->GetSequence();
       if (sequence > pVOptions->GetMaxSequence())
         sequence = 0;
       savename.sprintf("%s%03d.%s", SnapBase.ascii(), sequence, SnapExtension.ascii());
       pVOptions->SetSequence(++sequence);
       break;
   }

   /* extract filename, withouth paths */
   savefile.setFile(savename);
   remote_ftpname = savefile.fileName();

   delafterftp = false;
   delaftercmd = false;
   // In case we're saving to disk, we'll use that file for FTPing and external commands too
   if (savetodisk) {
     TR(2, "Saving image to %s", savename.ascii());
     if (!DrawImage.save(savename, SnapFmt)) {
       // XXX Failed. pop up message box, or set message in statusbar
       qWarning("Could not save image!");
       statusBar()->message(tr("Could not save image (%1)!").arg(savename));
       ftptoserver = runcommand = false;
     }
     else {
       local_ftpname = savename;
       cmdname = savename;
     }
   }
   else {
     if (ftptoserver) {
       delafterftp = true;
       local_ftpname = g_pCamApp->GetTempFileName();
       TR(2, "Saving temp FTP image to %s", local_ftpname.ascii());
       if (!DrawImage.save(local_ftpname, SnapFmt))
       {
         qWarning("Could not save temp image for FTPing!");
         statusBar()->message(tr("Could not save temp image (%1) for FTPing!").arg(local_ftpname));
         ftptoserver = false;
       }
     }
     if (runcommand) {
       delaftercmd = true;
       cmdname = g_pCamApp->GetTempFileName();
       TR(2, "Saving temp CMD image to %s", cmdname.ascii());
       if (!DrawImage.save(cmdname, SnapFmt))
       {
         qWarning("Could not save temp image for external command!");
         statusBar()->message(tr("Could not save temp image (%1) for external command!").arg(cmdname));
         runcommand = false;
       }
     }
   }

   if (ftptoserver)
     StartFTPUpload(local_ftpname, remote_ftpname, delafterftp);

   if (runcommand)
     StartSubProcess(command, cmdname, delaftercmd);
   TR_LEAVE();
}

void CWebCamViewer::CycleNextChannel()
{
   TR_ENTER();
   if (pVideo != 0) {
     pVideo->SelectNextTVChannel();
   }
   TR_LEAVE();
}

void CWebCamViewer::CyclePrevChannel()
{
   TR_ENTER();
   if (pVideo != 0) {
     pVideo->SelectPrevTVChannel();
   }
   TR_LEAVE();
}

void CWebCamViewer::NewChannelSelected()
{
   TR_ENTER();
   if (pVideo != 0) {
     TVChannel tvc = pVideo->GetCurrentTVChannel();
     setCaption(pVideo->GetIntfName() + " : " + tvc.Name);
   }
   TR_LEAVE();

}



// protected

/**
  \brief Calculate the total required viewing area, based on video image and 'last snapshot'
*/

void CWebCamViewer::RecalcTotalViewSize()
{
   QSize asize, osize;
   int w1 = 0, w2 = 0, h;

   TR_ENTER();
   if (pViewer && pLastSnapshot) {
     asize = pViewer->GetVisibleSize();
     w1 = asize.width();
     h = asize.height();

     if (pButton[pbt_showsnap]->isOn()) {
       asize = pLastSnapshot->GetVisibleSize();
       w2 = asize.width();
       if (asize.height() > h)
         h = asize.height();
       if (w2 == 0)
         w2 = w1;
     }

     asize.setWidth(w1 + w2);
     asize.setHeight(h);

     osize = centralWidget()->size();
     osize = size();
     if (asize != osize) { // Set new viewport only when necessary
       TR(2, "Resize viewport= %dx%d", asize.width(), asize.height());

       pViewer->move(0, 0);
       pLastSnapshot->move(w1, 0);

       centralWidget()->setFixedSize(asize);
       /* Bug or feature? At this point I cannot call adjustSize(), probably
          because we are in an event loop or something. So therefor we
          (ab)use a single shot timer to call adjustSize() for us, just
          after all current events have been processed (timeout = 0).
        */
       m_SizeTimer.start(0, true);
     }
   }
   TR_LEAVE();
}

// protected slots


/** Called from \ref CVideoDevice::Resized */
void CWebCamViewer::DeviceChangedSize(const QSize &new_size)
{
   TR_ENTER();
   TR(1, "New dimension = %dx%d", new_size.width(), new_size.height());
   RecalcTotalViewSize();
   TR_LEAVE();
}


void CWebCamViewer::DeviceError(int err_no)
{
   // Show pop-up box
   QString errstr;

   errstr.sprintf("The device experienced an error %d (%s). The window will be closed.", err_no, strerror(-err_no));
   QMessageBox::critical(this, "device error", errstr, QMessageBox::Ok, QMessageBox::NoButton);
   close(true); // End!
}


/**
  \brief Work around for adjustSize()

  Because sometimes adjustSize() doesn't have the desired (or a delayed)
  effect, we call it through a single-shot timer. Unfortunately adjustSize()
  isn't a slot, so this is just a wrapper function.
*/
void CWebCamViewer::CallAdjustSize()
{
   adjustSize();
}

/// public slots

void CWebCamViewer::showMaximized()
{
   qDebug("void CWebCamViewer::showMaximized()");
}
