/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <string.h>

#include "kozo-entry.h"
#include "kozo-field-private.h"
#include "kozo-pack.h"

struct _KozoField {
        int id;

        guint8 *data;
        guint32 length;

        gboolean data_allocated;
};

/* Tweakable parameters */

/* Amount of field structures per memory block */
#define BLOCK_SIZE 32

static GMutex *mem_lock;
static GMemChunk *mem_chunk;
static GTrashStack *trash_stack = NULL;

static void
dump_data (gint8 *data,
           int    len)
{
        int i;

        for (i = 0; i < len; i++) {
                if (g_ascii_isprint(data[i])) {
                        g_print ("%c ", data[i]);
                } else {
                        g_print ("%x ", data[i]);
                }
        }
        g_print ("\n");
}

static KozoField *
finish_new (KozoPackContext *c,
            int              field_id)
{
        c->dest = g_malloc (c->length);

        kozo_pack_finish (c);

        /* dump_data (c->dest, c->length); */
        return kozo_field_new (field_id, TRUE, c->dest, c->length);
}

KozoField *
kozo_field_new_string (int         field_id,
                       const char *value)
{
        KozoPackContext context;
        KozoField *ret;
        char *normalized;

        kozo_pack_context_init (&context, TRUE, 1, sizeof (char));
        kozo_pack_start (&context);

        if (value) {
            normalized = g_utf8_normalize (value, -1, G_NORMALIZE_DEFAULT);
        } else {
            normalized = g_strdup ("");
        }

        kozo_pack_string (&context, normalized);

        ret = finish_new (&context, field_id);

        g_free (normalized);

        return ret;
}

KozoField *
kozo_field_new_int (int    field_id,
                    gint32 value)
{
        KozoPackContext context;
        gint32 *val;

        kozo_pack_context_init (&context, TRUE, 1, sizeof (gint32));
        kozo_pack_start (&context);

        /* The reason this has to be an allocated space storing
           the value is because we need pass by reference, but integers
           are passed by value */
        val = g_newa (gint32, 1);
        *val = value;
        kozo_pack_int (&context, val);
        return finish_new (&context, field_id);
}

KozoField *
kozo_field_new_time (int    field_id,
                     time_t value)
{
        KozoPackContext context;
        time_t *val;

        kozo_pack_context_init (&context, TRUE, 1, sizeof (time_t));
        kozo_pack_start (&context);

        val = g_newa (time_t, 1);
        *val = value;
        kozo_pack_time (&context, val);

        return finish_new (&context, field_id);
}

KozoField *
kozo_field_new_string_list (int     field_id,
                            GSList *values)
{
        KozoPackContext context;
        GSList *l, *free_list = NULL;
        KozoField *ret;
        char *val;

        kozo_pack_context_init (&context, TRUE,
                                g_slist_length (values), sizeof (char));
        kozo_pack_start (&context);
        for (l = values; l; l = l->next) {
                if (l->data) {
                        val = g_utf8_normalize (l->data, -1,
                                                G_NORMALIZE_DEFAULT);
                } else {
                        val = g_strdup ("");
                }

                kozo_pack_string (&context, val);
                free_list = g_slist_prepend (free_list, val);
        }

        ret = finish_new (&context, field_id);

        while (free_list) {
                val = free_list->data;
                free_list = g_slist_delete_link (free_list, free_list);
                g_free (val);
        }

        return ret;
}

KozoField *
kozo_field_new_int_list (int     field_id,
                         GSList *values)
{
        KozoPackContext context;
        GSList *l;
        gint32 *val;

        kozo_pack_context_init (&context, TRUE,
                                g_slist_length (values), sizeof (gint32));
        kozo_pack_start (&context);

        for (l = values; l; l = l->next) {
                val = g_newa (gint32, 1);
                *val = GPOINTER_TO_INT (l->data);

                kozo_pack_int (&context, val);
        }

        return finish_new (&context, field_id);
}

KozoField *
kozo_field_new_time_list (int     field_id,
                          GSList *values)
{
        KozoPackContext context;
        GSList *l;
        time_t *val;

        kozo_pack_context_init (&context, TRUE,
                                g_slist_length (values), sizeof (time_t));
        kozo_pack_start (&context);

        for (l = values; l; l = l->next) {
                val = g_newa (time_t, 1);
                *val = GPOINTER_TO_INT (l->data);

                kozo_pack_time (&context, val);
        }

        return finish_new (&context, field_id);
}

int
kozo_field_get_id (KozoField  *field)
{
        return field->id;
}

const guint8 *
kozo_field_get_raw_data (KozoField *field,
                         guint32   *length)
{
        *length = field->length;
        return field->data;
}

guint32
kozo_field_get_value_count (KozoField *field)
{
        return *(guint32 *) field->data;
}

G_INLINE_FUNC guint32
kozo_field_get_value_offset (KozoField *field,
                             int        n)
{
        return *(guint32 *) (field->data + ((n + 1) * sizeof (guint32)));
}

G_INLINE_FUNC guint8 *
kozo_field_get_value (KozoField *field,
                      int        n)
{
        return field->data + kozo_field_get_value_offset (field, n);
}

const char *
kozo_field_get_value_string (KozoField *field,
                             int        n)
{
        return (const char *) kozo_field_get_value (field, n);
}

gint32
kozo_field_get_value_int (KozoField *field,
                          int        n)
{
        return *(gint32 *) kozo_field_get_value (field, n);
}

time_t
kozo_field_get_value_time (KozoField *field,
                           int        n)
{
        return *(time_t *) kozo_field_get_value (field, n);
}

KozoField *
kozo_field_new (int      field_id,
                gboolean data_allocated,
                guint8  *data,
                guint32  length)
{
        KozoField *field;

        g_mutex_lock (mem_lock);

        field = g_trash_stack_pop (&trash_stack);
        if (!field)
                field = g_chunk_new (KozoField, mem_chunk);

        g_mutex_unlock (mem_lock);

        field->id = field_id;
        field->data_allocated = data_allocated;
        field->data = data;
        field->length = length;

        return field;
}

void
kozo_field_free (KozoField *field)
{
        if (field->data_allocated)
                g_free (field->data);

        g_mutex_lock (mem_lock);

        g_trash_stack_push (&trash_stack, field);

        g_mutex_unlock (mem_lock);
}

void
kozo_field_list_free (GSList *field_list)
{
        KozoField *field;

        while (field_list) {
                field = field_list->data;
                kozo_field_free (field);
                field_list = g_slist_delete_link (field_list, field_list);
        }
}

void
kozo_field_init (void)
{
        mem_lock = g_mutex_new ();

        mem_chunk = g_mem_chunk_create (KozoField,
                                        BLOCK_SIZE,
                                        G_ALLOC_ONLY);
}

void
kozo_field_shutdown (void)
{
        g_mem_chunk_destroy (mem_chunk);

        g_mutex_free (mem_lock);
}
