/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Alex Launi <alex.launi@canonical.com>
 *
 */

using GLib;

namespace Unity.MusicLens {
  
  public class MusicStoreScopeProxy : SimpleScope
  {
    private MusicStoreCollection collection;

    public MusicStoreScopeProxy ()
    {
      base ();

      scope = new Unity.Scope ("/com/canonical/unity/scope/musicstore");
      scope.search_in_global = false;
      scope.activate_uri.connect (activate);

      base.initialize ();

      collection = new MusicStoreCollection ();

      try {
	scope.export ();
      } catch (GLib.IOError e) {
	stdout.printf (e.message);
      }
    }

    protected override int num_results_without_search { get { return 100; } }
    protected override int num_results_global_search { get { return 20; } }
    protected override int num_results_lens_search { get { return 50; } }

    public Unity.ActivationResponse activate (string uri)
    {
      /* launch the music store streaming client  or the webstore or whatevz */
	try {
	  AppInfo.launch_default_for_uri (uri, null);
	} catch (GLib.Error e) {
	  warning ("Failed to open uri %s. %s", uri, e.message);
	}
	return new Unity.ActivationResponse (Unity.HandledType.HIDE_DASH);  
    }

    public override async void perform_search (LensSearch search,
                                               SearchType search_type,
                                               owned List<FilterParser> filters,
                                               int max_results = -1,
                                               Cancellable? cancellable = null)
    {
      if (is_search_empty (search))
        return;
      
      try {
	debug ("model has %u rows before search", search.results_model.get_n_rows ());
	yield collection.search (search, search_type, (owned) filters, max_results, cancellable);
	debug ("model has %u rows after search", search.results_model.get_n_rows ());
      } catch (IOError e) {
	warning ("Failed to search for '%s': %s", search.search_string, e.message);
      }
    }

  }
}
