# Copyright (C) 2008-2011  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Test the authenticator."""

import unittest

from computerjanitord.authenticator import Authenticator

SUCCESS_PID = 801
FAILURE_PID = 999

AUTHENTICATED_USER = 'desktop-user'
IMPOSTER_USER = 'imposter'
EXPECTED_PRIVILEGE = 'com.ubuntu.computerjanitor.cleanpackages'
BOGUS_PRIVILEGE = 'com.example.evil-corp.killsystem'


class MockPolicyKit(object):
    """Mock the PolicyKit's CheckAuthorization() method."""

    def CheckAuthorization(self, subject, privilege, details, flags,
                           cancellation_id):
        """See `policykit.CheckAuthorization()`.

        :return: (is_authorized, is_challenge, details)
        """
        if privilege == BOGUS_PRIVILEGE:
            return False, False, ''
        assert isinstance(subject, tuple) and len(subject) == 2, (
            'subject is not a 2-tuple')
        assert subject[0] == 'unix-process', 'Badly formed subject'
        assert isinstance(subject[1], dict), 'Badly formed subject details'
        assert subject[1]['start-time'] == 0, 'subject missing start-time'
        if subject[1]['pid'] == SUCCESS_PID:
            return True, False, ''
        else:
            return False, False, ''


class TestableAuthenticator(Authenticator):
    """See `Authenticator`."""

    def _get_policykit_proxy(self):
        """See `Authenticator`."""
        return MockPolicyKit()

    def _get_sender_pid(self, connection, sender):
        """See `Authenticator`."""
        if sender == AUTHENTICATED_USER:
            return SUCCESS_PID
        else:
            return FAILURE_PID


class TestAuthenticator(unittest.TestCase):
    """Tests of the PolicyKit authenticator."""

    def setUp(self):
        """See `unittest.TestCase`."""
        self.authenticator = TestableAuthenticator()
        self.connection = object()

    def tearDown(self):
        """See `unittest.TestCase`."""

    def test_good_path(self):
        # Test for successful authentication.
        self.assertTrue(self.authenticator.authenticate(
            AUTHENTICATED_USER, self.connection, EXPECTED_PRIVILEGE))

    def test_bogus_privilege(self):
        # Test for bogus privilege fails.
        self.assertFalse(self.authenticator.authenticate(
            AUTHENTICATED_USER, self.connection, BOGUS_PRIVILEGE))

    def test_unauthorized(self):
        # Test for some imposter not being able to authenticate.
        self.assertFalse(self.authenticator.authenticate(
            IMPOSTER_USER, self.connection, EXPECTED_PRIVILEGE))


def test_suite():
    suite = unittest.TestSuite()
    suite.addTests(unittest.makeSuite(TestAuthenticator))
    return suite
