# Copyright (C) 2008-2011  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Maintaining package ignored state."""

from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'DEFAULT_STATE_FILE',
    'State',
    ]


import ConfigParser
import textwrap


DEFAULT_STATE_FILE = '/var/lib/computer-janitor/state.dat'


class State:
    """Maintain the state of cruft which should be ignored.

    The file's format is a `ConfigParser` style .ini file.  Each section is a
    cruft's `.get_name()` value and contains a single boolean setting
    `ignore`.  If true, then the cruft is ignored.

    For backward compatibility purposes, the setting `enabled` is also
    recognized (on read only).  If `enabled` is false then the cruft is
    ignored.
    """

    def __init__(self):
        self.ignore = set()

    def load(self, filename):
        """Load ignored state from a file.

        This reset any previously determined state and re-initializes it with
        the state stored in the file.

        :param filename: The file to load.
        :type filename: string
        """
        parser = ConfigParser.ConfigParser()
        parser.read(filename)
        # Reset the set of ignored packages.
        self.ignore = set()
        for cruft_name in parser.sections():
            # For backwards compatibility, recognize both the 'ignore' setting
            # and the 'enabled' setting.  We only write the former.
            try:
                ignore = parser.getboolean(cruft_name, 'ignore')
            except ConfigParser.NoOptionError:
                try:
                    ignore = not parser.getboolean(cruft_name, 'enabled')
                except ConfigParser.NoOptionError:
                    # No other settings are recognized.
                    ignore = False
            if ignore:
                self.ignore.add(cruft_name)

    def save(self, filename):
        """Save the ignored state to a file.

        Only the packages being ignored are stored to the file, and writing
        overwrites the previous contents of the file.

        :param filename: The file to load.
        :type filename: string
        """
        # It's easier just to write the .ini file directly instead of using
        # the ConfigParser interface.  This way we can guarantee sort order
        # and can automatically cull unignored packages from the file.
        with open(filename, 'w') as fp:
            for cruft_name in self.ignore:
                print >> fp, textwrap.dedent("""\
                [{0}]
                ignore: true
                """.format(cruft_name))
