/*
 * dvpump.cc -- DV Pump and Sundry Common Implementations
 * Copyright (C) 2002 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include "PlayListDVProvider.h"
#include "extensions.h"

/** PlayListDVProvider implementation - provides the DV frames in a playlist 
    via a threaded pump.
*/

PlayListDVProvider::PlayListDVProvider( ) : 
	PlayList( ), 
	speed( 1 ), 
	index( 0 )
{
	SetBlockable( true );
	pthread_mutex_init( &mutex, NULL );
}

PlayListDVProvider::PlayListDVProvider( PlayList &playlist ) : 
	PlayList( playlist ),
	speed( 1 ), 
	index( 0 )
{
	SetBlockable( true );
	pthread_mutex_init( &mutex, NULL );
}

PlayListDVProvider::~PlayListDVProvider( )
{
	pthread_mutex_destroy( &mutex );
}

void PlayListDVProvider::SetPlayList( PlayList &_playlist )
{
	pthread_mutex_lock( &mutex );
	CleanPlayList( );
	InsertPlayList( _playlist, 0 );
	index = 0;
	speed = 1;
	FlushOutput( );
	QueueInputFrame( );
	pthread_mutex_unlock( &mutex );
}

void PlayListDVProvider::SetSpeed( double _speed )
{
	pthread_mutex_lock( &mutex );
	FlushOutput( );
	speed = _speed;
	pthread_mutex_unlock( &mutex );
}

void PlayListDVProvider::SetPosition( double _index )
{
	pthread_mutex_lock( &mutex );
	FlushOutput( );
	QueueInputFrame( );
	index = _index;
	pthread_mutex_unlock( &mutex );
}

void PlayListDVProvider::SetSpeedAndPosition( double _speed, double _index )
{
	pthread_mutex_lock( &mutex );
	FlushOutput( );
	QueueInputFrame( );
	index = _index;
	speed = _speed;
	pthread_mutex_unlock( &mutex );
}

bool PlayListDVProvider::ReadFrame( Frame &frame )
{
	pthread_mutex_lock( &mutex );
	if ( index < 0 )
		index = 0;
	
	bool result = GetFrame( (int)index, frame );
	
	if ( result == false )
	{
		switch( action )
		{
			case loop_on_end:
				index = 0;
				result = GetFrame( (int)index, frame );
				break;
			case pause_on_end:
				index = GetNumFrames( ) - 1;
				result = GetFrame( (int)index, frame );
				break;
			case terminate_on_end:
				break;
		}
	}
	frame.playlist_position = (int)index;
	index += speed;
	pthread_mutex_unlock( &mutex );
	return result;
}

double PlayListDVProvider::GetSpeed( ) const
{
	return speed;
}

void PlayListDVProvider::SetEndAction( end_of_playlist _action )
{
	action = _action;
}

void PlayListDVProvider::ChangeScene( int position, int scene, bool relative )
{
	pthread_mutex_lock( &mutex );
	FlushOutput( );
	QueueInputFrame( );
	index = position;
	if ( relative && scene > 0 )
	{
		while( scene -- > 0 && index < GetNumFrames( ) )
			index = FindEndOfScene( (int)index ) + 1;
	}
	else if ( relative && scene <= 0 )
	{
		index = FindStartOfScene( (int)index );
		if ( ( position - index ) > 15 )
			scene ++;
		while( scene ++ < 0 && index < GetNumFrames( ) )
			index = FindStartOfScene( (int)index - 1 );
	}
	else if ( scene >= 0 )
	{
		index = 0;
		for ( ; scene > 0 && index < GetNumFrames( ); scene -- )
			index = FindEndOfScene( (int)index ) + 1;
	}
	pthread_mutex_unlock( &mutex );
}

