
/*
 * Copyright (C) 2007 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: utils_config.c 2345 2007-06-27 12:11:08Z mschwerin $
 *
 */

#include <config.h>

#include <stdlib.h>
#include <string.h>

#include "i18n.h"
#include "environment.h"
#include "logger.h"
#include "oxine.h"
#include "utils.h"

extern oxine_t *oxine;


inline int
config_lookup_entry (const char *key, xine_cfg_entry_t * entry)
{
    int result = xine_config_lookup_entry (oxine->xine, key, entry);
    if (strcmp (key, entry->key) != 0) {
        error (_("Failed to lookup entry '%s'!"), key);
    }
    return result;
}


inline void
config_update_entry (xine_cfg_entry_t * entry)
{
    xine_config_update_entry (oxine->xine, entry);
}


inline void
config_get_entry (const char *key, xine_cfg_entry_t * entry)
{
    int result = xine_config_lookup_entry (oxine->xine, key, entry);
    if ((result == 0) || (strcmp (key, entry->key) != 0)) {
        error (_("Failed to lookup entry '%s'!"), key);
    }
}


inline int
config_get_number (const char *key)
{
    xine_cfg_entry_t entry;
    config_get_entry (key, &entry);
    return entry.num_value;
}


inline int
config_set_number (const char *key, int value)
{
    xine_cfg_entry_t entry;
    config_get_entry (key, &entry);
    entry.num_value = value;
    config_update_entry (&entry);
    return entry.num_value;
}


inline const char *
config_get_string (const char *key)
{
    xine_cfg_entry_t entry;
    config_get_entry (key, &entry);
    return entry.str_value;
}


inline bool
config_get_bool (const char *key)
{
    xine_cfg_entry_t entry;
    config_get_entry (key, &entry);
    return entry.num_value ? true : false;
}


inline bool
config_set_bool (const char *key, bool value)
{
    xine_cfg_entry_t entry;
    config_get_entry (key, &entry);
    entry.num_value = value;
    config_update_entry (&entry);
    return entry.num_value ? true : false;
}


inline const char *
config_get_enum (const char *key)
{
    xine_cfg_entry_t entry;
    config_get_entry (key, &entry);
    return entry.enum_values[entry.num_value];
}


static void
config_changed_handler (void *user_data, xine_cfg_entry_t * entry)
{
    if (!entry || !entry->key) {
        return;
    }
    if (!oxine || !oxine->odk) {
        return;
    }

    debug ("configuration '%s' has changed", entry->key);

    oxine_event_t event;
    event.type = OXINE_EVENT_CONFIG_CHANGED;
    event.data.config.entry = entry;
    odk_oxine_event_send (oxine->odk, &event);
}


inline const char *
config_register_string (const char *key, const char *init_value,
                        const char *label)
{
    return xine_config_register_string (oxine->xine, key, init_value,
                                        label, label, 10,
                                        config_changed_handler, NULL);
}


inline int
config_register_number (const char *key, int init_value, const char *label)
{
    return xine_config_register_num (oxine->xine, key, init_value,
                                     label, label, 10,
                                     config_changed_handler, NULL);
}


inline int
config_register_range (const char *key, int init_value, int min, int max,
                       const char *label)
{
    return xine_config_register_range (oxine->xine, key, init_value, min, max,
                                       label, label, 10,
                                       config_changed_handler, NULL);
}


inline int
config_register_enum (const char *key, int init_value, char **values,
                      const char *label)
{
    return xine_config_register_enum (oxine->xine, key, init_value, values,
                                      label, label, 10,
                                      config_changed_handler, NULL);
}


inline bool
config_register_bool (const char *key, bool init_value, const char *label)
{
    return xine_config_register_bool (oxine->xine, key, init_value,
                                      label, label, 10,
                                      config_changed_handler, NULL);
}


inline void
config_save (void)
{
    xine_config_save (oxine->xine, get_file_config ());
}
