
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: menu_playlist.c 2603 2007-07-25 08:37:30Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <unistd.h>

#include "disc.h"
#include "download.h"
#include "environment.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "oxine.h"
#include "playlist.h"
#include "scheduler.h"
#include "utils.h"

#include "menu_base.h"
#include "menu_filelist.h"
#include "menu_main.h"
#include "menu_playback.h"
#include "menu_playlist.h"

extern oxine_t *oxine;

static l_list_t *prev_filelist = NULL;
static l_list_t *next_filelist = NULL;

static filelist_t *current_filelist = NULL;
static filelist_t *toplevel_filelist = NULL;

static otk_widget_t *button_up = NULL;
static otk_widget_t *button_prev = NULL;
static otk_widget_t *button_next = NULL;
static otk_widget_t *button_home = NULL;

static int thumbnail_job = 0;

#define FILELIST_BUTTONS_NUM 2
static otk_widget_t *filelist_menu_buttons[FILELIST_BUTTONS_NUM];
static otk_widget_t *filelist_menu_list = NULL;
static otk_widget_t *filelist_menu_window = NULL;
static otk_widget_t *filelist_menu_title = NULL;
#ifdef HAVE_OSD_IMAGE
static otk_widget_t *filelist_menu_image_large = NULL;
#endif

/*
 * ***************************************************************************
 * Some prototypes of methods declared in this file
 * ***************************************************************************
 */
static void filelist_menu_update_list (void);
static void filelist_menu_update_buttons (void);
static void filelist_menu_update_title (void);

static void filelist_menu_set_filelist (filelist_t * newlist);

#ifdef HAVE_OSD_IMAGE
static void filelist_menu_thumbnail_job (void *entry_cb_data);
#endif

static void filelist_focus_enter_cb (void *entry_cb_data);
static void filelist_focus_leave_cb (void *entry_cb_data);

static void filelist_select_cb (void *entry_cb_data);
static void filelist_activate_cb (void *entry_cb_data);
static void filelist_remove_cb (void *entry_cb_data);

static void playlist_save_cb (void *p);
static playitem_t *playlist_get_selected (void);
static fileitem_t *subtitle_get_selected (void);

static void show_subtitle_menu_cb (void *p);


static bool
current_is_toplevel (void)
{
    return (current_filelist == toplevel_filelist);
}


/*
 * ***************************************************************************
 * Methods that are part of the filelist GUI.
 * ***************************************************************************
 */
static void
filelist_menu_update_buttons (void)
{
    if (!is_current_menu (show_subtitle_menu_cb)) {
        return;
    }

    bool enabled;
    bool has_prev = (l_list_length (prev_filelist) > 0);
    bool has_next = (l_list_length (next_filelist) > 0);

    otk_widget_set_enabled (button_up, !current_is_toplevel ());
    otk_widget_set_enabled (button_home, !current_is_toplevel ());
    otk_widget_set_enabled (button_prev, has_prev);
    otk_widget_set_enabled (button_next, has_next);

    fileitem_t *selected = subtitle_get_selected ();
    enabled = (selected && is_file_allowed (selected->mrl,
                                            ALLOW_FILES_SUBTITLE));
    otk_widget_set_enabled (filelist_menu_buttons[0], enabled);
}


static void
filelist_menu_addto_list (fileitem_t * item)
{
    otk_widget_t *w;

    {
        w = otk_listentry_new (filelist_menu_list, item->title,
                               filelist_activate_cb, item,
                               filelist_select_cb, item,
                               filelist_remove_cb, item);
    }

    otk_widget_set_focus_callbacks (w, filelist_focus_enter_cb, item,
                                    filelist_focus_leave_cb, item);
}


static void
filelist_menu_update_list (void)
{
    if (!is_current_menu (show_subtitle_menu_cb)) {
        return;
    }

    /* If we're currently not in the toplevel filelist and the filelist does
     * NOT have a parent we're probably in a removable disc, that has been
     * removed (see disc.c). To rescue ourself of segmentation faults we jump
     * to the toplevel filelist and clear the prev and next lists. */
    if (!current_is_toplevel ()
        && (current_filelist->parent_list == NULL)) {
        filelist_menu_set_filelist (toplevel_filelist);
        l_list_clear (prev_filelist, NULL);
        l_list_clear (next_filelist, NULL);
    }

    otk_list_clear (filelist_menu_list);

    filelist_lock (current_filelist);

    /* next we add all the entries from the current filelist */
    {
        fileitem_t *item = filelist_first (current_filelist);
        while (item) {
            filelist_menu_addto_list (item);
            item = filelist_next (current_filelist, item);
        }
    }

    /* last we add all removable drives currently available */
#ifdef HAVE_HAL
    if (current_is_toplevel ()
        && (filelist_trylock (oxine->hal_volume_list) != EBUSY)) {
        fileitem_allowed_t allowed = toplevel_filelist->allowed_filetypes;
        fileitem_t *item = filelist_first (oxine->hal_volume_list);
        while (item) {
            bool add = false;
            /* We only allow audio CDs in the audio menu and DVDs and VCDs
             * in the video menu. */
            add |= is_file_allowed (item->mrl, allowed);
            /* Mountable discs (e.g. external HD, CDROM) are allowed in all
             * menus. */
            add |= (item->type == FILE_TYPE_MOUNTPOINT);
            /* We only add volumes. Empty drives are not show in this list
             * as they are not really of interest to a user. */
            add &= (item->volume_udi != NULL);

            if (add) {
                filelist_menu_addto_list (item);
            }
            item = filelist_next (oxine->hal_volume_list, item);
        }
        filelist_unlock (oxine->hal_volume_list);
    }
#endif

    otk_list_set_pos (filelist_menu_list, current_filelist->top_position);
    otk_list_set_focus (filelist_menu_list, current_filelist->cur_position);

    filelist_unlock (current_filelist);
}


static void
filelist_menu_update_title (void)
{
    if (!is_current_menu (show_subtitle_menu_cb)) {
        return;
    }

    if (current_filelist == toplevel_filelist) {
        otk_label_set_text (filelist_menu_title, _("Choose an entry..."));
    }
    else {
        char *tmp = ho_strdup (current_filelist->title);
        char *title = tmp;

        if (title[0] == '[') {
            title++;
        }
        if (title[strlen (title) - 1] == ']') {
            title[strlen (title) - 1] = '\0';
        }

        otk_label_set_text (filelist_menu_title, title);
        ho_free (tmp);
    }
}


static void
filelist_menu_set_filelist (filelist_t * newlist)
{
    filelist_t *prev = (filelist_t *) l_list_last (prev_filelist);
    filelist_t *next = (filelist_t *) l_list_last (next_filelist);

    /* We are returning to the directory on top of the prev stack. We 
     * remove it from the prev stack and add the current directory to the 
     * next stack. */
    if (newlist == prev) {
        l_list_remove (prev_filelist, newlist);
        l_list_append (next_filelist, current_filelist);
    }

    /* We are returning to the directory on top of the next stack. We 
     * remove it from the next stack and add the current directory to the 
     * prev stack. */
    else if (newlist == next) {
        l_list_remove (next_filelist, newlist);
        l_list_append (prev_filelist, current_filelist);
    }

    /* We are entering a directory that is neither on top of the prev stack
     * nor on top of the next stack. We clear the next stack and add the
     * current directory to the prev stack. */
    else {
        l_list_clear (next_filelist, NULL);
        l_list_append (prev_filelist, current_filelist);
    }


    /* We update the position in the list. */
    if (otk_list_get_length (filelist_menu_list) > 0) {
        current_filelist->top_position =
            otk_list_get_pos (filelist_menu_list);
        current_filelist->cur_position =
            otk_list_get_focus (filelist_menu_list);
    }
    filelist_ref_set (&current_filelist, newlist);

#ifdef HAVE_OSD_IMAGE
    /* We cancel any thumbnail job that might be running. */
    cancel_job (thumbnail_job);
    /* We schedule a new thumbnail job. */
    thumbnail_job = schedule_job (100, filelist_menu_thumbnail_job, NULL);
#endif

    filelist_menu_update_list ();
    filelist_menu_update_title ();
    filelist_menu_update_buttons ();

    show_user_interface (oxine);
}


static void
filelist_remove_cb (void *entry_cb_data)
{
}


/*
 * This is the callback for a click on an item in the list.
 */
static void
filelist_select_cb (void *entry_cb_data)
{
    filelist_menu_update_buttons ();
    show_user_interface (oxine);
}


static void
filelist_please_wait (void *p)
{
    otk_label_set_text (filelist_menu_title, _("Please wait..."));
    otk_widget_set_enabled (button_up, false);
    otk_widget_set_enabled (button_home, false);
    otk_widget_set_enabled (button_prev, false);
    otk_widget_set_enabled (button_next, false);
    otk_list_clear (filelist_menu_list);
}


/*
 * This is the callback for a doubleclick on an item in the list.
 */
static void
filelist_activate_cb (void *entry_cb_data)
{
    playitem_t *playitem = playlist_get_selected ();
    fileitem_t *fileitem = (fileitem_t *) entry_cb_data;

    current_filelist->top_position = otk_list_get_pos (filelist_menu_list);
    current_filelist->cur_position = otk_list_get_focus (filelist_menu_list);

    /* For lists that may take a while to load we set the title to 'Please
     * wait...' and clear the list, so that the user gets a feedback that
     * something is happening. So we don't see a flashing of the list when a
     * list does load quickly we wait a few milliseconds before clearing the
     * list. */
    int job_id = 0;
    switch (fileitem->type) {
    case FILE_TYPE_DIRECTORY:
    case FILE_TYPE_MOUNTPOINT:
    case FILE_TYPE_MEDIAMARKS:
        job_id = schedule_job (100, filelist_please_wait, NULL);
        break;
    default:
        break;
    }

    /* Make sure the fileitem is expanded. */
    filelist_expand (fileitem);
    cancel_job (job_id);

    switch (fileitem->type) {
    case FILE_TYPE_DIRECTORY:
    case FILE_TYPE_MOUNTPOINT:
    case FILE_TYPE_MEDIAMARKS:
        if (fileitem->child_list->wait_for_password) {
            /* Do nothing. */
        }
        else if (fileitem->child_list->error) {
            if (otk_list_get_length (filelist_menu_list) == 0) {
                filelist_menu_update_list ();
            }
            otk_label_set_text (filelist_menu_title,
                                fileitem->child_list->error);
        }
        else {
            filelist_menu_set_filelist (fileitem->child_list);
        }
        break;
    case FILE_TYPE_REGULAR:
        if (is_file_allowed (fileitem->mrl, ALLOW_FILES_SUBTITLE)) {
            ho_free (playitem->subtitle_mrl);
            playitem->subtitle_mrl = ho_strdup (fileitem->mrl);
            show_menu_playlist (oxine);
        }
        break;
    default:
        break;
    }
}


#ifdef HAVE_OSD_IMAGE
static void
filelist_menu_update_thumbnail (const char *thumbnail_mrl)
{
    char *mrl = NULL;
    if (thumbnail_mrl) {
        if (is_downloadable (thumbnail_mrl)) {
            mrl = download_to_cache (thumbnail_mrl, NULL, false);
        }
        else {
            mrl = ho_strdup (thumbnail_mrl);
        }

        if (!file_exists (mrl)) {
            ho_free (mrl);
        }
    }

    otk_image_set_mrl (filelist_menu_image_large, mrl);

    ho_free (mrl);
}


static void
filelist_menu_thumbnail_preload (fileitem_t * fileitem)
{
    if (!fileitem) {
        return;
    }

    const char *mrl = fileitem_get_thumbnail (fileitem);
    if (mrl) {
        int a = ODK_ALIGN_CENTER | ODK_ALIGN_VCENTER;
        int x = filelist_menu_image_large->x;
        int y = filelist_menu_image_large->y;
        int w = filelist_menu_image_large->w;
        int h = filelist_menu_image_large->h;

        odk_osd_preload_image (oxine->odk, mrl, x, y, w, h, true, 0, a);
    }
}


static void
filelist_menu_thumbnail_set (fileitem_t * fileitem)
{
    const char *mrl = fileitem_get_thumbnail (fileitem);
    if (!mrl) {
        mrl = filelist_get_thumbnail (current_filelist);
    }
    filelist_menu_update_thumbnail (mrl);

    /* We try and preload the thumbnail images of the next and of the
     * previous item in the list. */
    if (fileitem) {
        filelist_t *filelist = fileitem->parent_list;
        filelist_menu_thumbnail_preload (filelist_next (filelist, fileitem));
        filelist_menu_thumbnail_preload (filelist_prev (filelist, fileitem));
    }
}


static void
filelist_menu_thumbnail_job (void *p)
{
    fileitem_t *fileitem = (fileitem_t *) p;

    if (!is_current_menu (show_subtitle_menu_cb)) {
        oxine_event_t ev;
        ev.type = OXINE_EVENT_THUMBNAIL_UPDATE;
        ev.data.user_data = fileitem;
        odk_oxine_event_send (oxine->odk, &ev);
    }
}
#endif /* HAVE_OSD_IMAGE */


static void
filelist_focus_enter_cb (void *entry_cb_data)
{
#ifdef HAVE_OSD_IMAGE
    fileitem_t *item = (fileitem_t *) entry_cb_data;

    /* We cancel any thumbnail job that might be running. */
    cancel_job (thumbnail_job);
    /* We want to wait for 200ms before showing the thumbnail. This is a
     * good idea, because if the user scrolls down a list quickly, showing
     * the thumbnail at once would slow down scrolling. */
    thumbnail_job = schedule_job (300, filelist_menu_thumbnail_job, item);
#endif /* HAVE_OSD_IMAGE */
}


static void
filelist_focus_leave_cb (void *entry_cb_data)
{
#ifdef HAVE_OSD_IMAGE
    /* We cancel any thumbnail job that might be running. */
    cancel_job (thumbnail_job);
    /* If available we show the thumbnail of the current directory. */
    thumbnail_job = schedule_job (600, filelist_menu_thumbnail_job, NULL);

    filelist_menu_update_title ();
#endif /* HAVE_OSD_IMAGE */
}


static void
filelist_up_cb (void *p)
{
    if (current_filelist) {
        filelist_t *newlist = current_filelist->parent_list;
#ifdef HAVE_HAL
        if (current_filelist->parent_list == oxine->hal_volume_list) {
            newlist = toplevel_filelist;
        }
#endif
        filelist_menu_set_filelist (newlist);
    }
}


static void
filelist_home_cb (void *p)
{
    if (current_filelist != toplevel_filelist) {
        filelist_menu_set_filelist (toplevel_filelist);
    }
}


static void
filelist_prev_cb (void *p)
{
    filelist_t *last = (filelist_t *) l_list_last (prev_filelist);
    if (last) {
        filelist_menu_set_filelist (last);
    }
}


static void
filelist_next_cb (void *p)
{
    filelist_t *last = (filelist_t *) l_list_last (next_filelist);
    if (last) {
        filelist_menu_set_filelist (last);
    }
}


static void
filelist_menu_show_list (void)
{
    int x = odk_osd_get_width (oxine->odk) - 20 - 4 * 46 + 6;
    int y = 100;

    filelist_menu_title =
        otk_label_new (oxine->otk, 220, 117, x - 240,
                       OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER,
                       _("Choose an entry..."));
    otk_widget_set_font (filelist_menu_title, "sans", 32);

    button_prev =
        otk_vector_button_new (oxine->otk, x, y, 40, 35,
                               OSD_VECTOR_ARROW_LEFT, 20, 20,
                               filelist_prev_cb, oxine);
    x += 46;
    button_next =
        otk_vector_button_new (oxine->otk, x, y, 40, 35,
                               OSD_VECTOR_ARROW_RIGHT, 20, 20,
                               filelist_next_cb, oxine);
    x += 46;
    button_up =
        otk_vector_button_new (oxine->otk, x, y, 40, 35,
                               OSD_VECTOR_ARROW_UP, 20, 20,
                               filelist_up_cb, oxine);
    x += 46;
    button_home =
        otk_vector_button_new (oxine->otk, x, y, 40, 35,
                               OSD_VECTOR_HOME, 20, 20,
                               filelist_home_cb, oxine);

    int w = odk_osd_get_width (oxine->odk) - 240;
    int h = odk_osd_get_height (oxine->odk) - 160;
    filelist_menu_list =
        otk_list_new (oxine->otk, 220, 140, w, h, 30, 33, true, true,
                      OTK_LIST_SELECTION_SINGLE, playlist_get_selected ());
}


static void
filelist_menu_event_handler (void *p, oxine_event_t * event)
{
    if (!is_current_menu (show_subtitle_menu_cb)) {
        return;
    }

#ifdef HAVE_OSD_IMAGE
    if (event->type == OXINE_EVENT_THUMBNAIL_UPDATE) {
        fileitem_t *fileitem = (fileitem_t *) event->data.user_data;
        filelist_menu_thumbnail_set (fileitem);
    }
#endif

    if (event->type != OXINE_EVENT_KEY) {
        return;
    }

    switch (event->source.key) {
    case OXINE_KEY_HOME:
        filelist_home_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_BACK:
        if (current_filelist != toplevel_filelist) {
            filelist_up_cb (oxine);
        }
        else {
            show_menu_playlist (oxine);
        }
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_PREV:
        filelist_prev_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_NEXT:
        filelist_next_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    default:
        break;
    }
}


static fileitem_t **
filelist_get_selected_fileitems (int *num_selected)
{
    return (fileitem_t **) otk_list_get_selected (filelist_menu_list,
                                                  num_selected);
}


static fileitem_t *
subtitle_get_selected (void)
{
    int num_selected;
    fileitem_t **fileitems = filelist_get_selected_fileitems (&num_selected);
    if (!fileitems) {
        return NULL;
    }

    fileitem_t *res = fileitems[0];
    ho_free (fileitems);

    return res;
}


static void
playitem_add_subtitle (void *playitem_p)
{
    playitem_t *playitem = (playitem_t *) playitem_p;
    fileitem_t *selected = subtitle_get_selected ();

    if (selected && is_file_allowed (selected->mrl, ALLOW_FILES_SUBTITLE)) {
        ho_free (playitem->subtitle_mrl);
        playitem->subtitle_mrl = ho_strdup (selected->mrl);
        show_menu_playlist (oxine);
    }
}


static void
filelist_menu_show_gui (void)
{
    if (filelist_menu_window) {
        otk_set_current_window (oxine->otk, filelist_menu_window);

        current_filelist->top_position =
            otk_list_get_pos (filelist_menu_list);
        current_filelist->cur_position =
            otk_list_get_focus (filelist_menu_list);

        filelist_menu_update_list ();
        filelist_menu_update_buttons ();

        return;
    }
    odk_add_event_handler (oxine->odk, filelist_menu_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_NORMAL);

    filelist_menu_window = create_new_window (true, true);

    int h = 0;
    int x = 20;
    int y = 100;
    filelist_menu_buttons[0] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Add"),
                             playitem_add_subtitle, playlist_get_selected ());
    y += 40;
    filelist_menu_buttons[1] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Back"),
                             show_menu_playlist, oxine);
    y += 40;

#ifdef HAVE_OSD_IMAGE
    h = (odk_osd_get_height (oxine->odk) - 30 - y);
    y = y + 10 + (h / 2);
    filelist_menu_image_large =
        otk_image_new (oxine->otk, 110, y, 180, h, NULL,
                       ODK_ALIGN_CENTER | ODK_ALIGN_VCENTER, true);
#endif /* HAVE_OSD_IMAGE */

    filelist_menu_show_list ();
    filelist_menu_update_list ();
    filelist_menu_update_buttons ();
}


static void
show_subtitle_menu_cb (void *p)
{
    if (!playlist_get_selected ())
        return;

    if (!toplevel_filelist) {
        filelist_ref_set (&toplevel_filelist,
                          filelist_new (NULL, NULL, NULL,
                                        ALLOW_FILES_SUBTITLE));
        filelist_ref_set (&current_filelist, toplevel_filelist);
        {
            const char *mrl = get_dir_home ();
            char *title = create_title (mrl);
            filelist_add (toplevel_filelist, title, mrl, FILE_TYPE_DIRECTORY);
            ho_free (title);
        }
        {
            const char *mrl = "/";
            char *title = create_title (mrl);
            filelist_add (toplevel_filelist, title, mrl, FILE_TYPE_DIRECTORY);
            ho_free (title);
        }
        filelist_sort (toplevel_filelist, NULL);
    }

    if (!prev_filelist) {
        prev_filelist = l_list_new ();
    }
    if (!next_filelist) {
        next_filelist = l_list_new ();
    }

    filelist_menu_show_gui ();

    set_backto_menu (show_subtitle_menu_cb, NULL);
    set_current_menu (show_subtitle_menu_cb, NULL);

    show_user_interface (NULL);
    show_menu_background ("menu_subtitle.png");
}


/* 
 * *************************************************************** 
 * Playlist Menu Stuff
 * *************************************************************** 
 */
#define PLAYLIST_BUTTONS_NUM 10
static otk_widget_t *playlist_menu_buttons[PLAYLIST_BUTTONS_NUM];
static otk_widget_t *playlist_menu_list = NULL;
static otk_widget_t *playlist_menu_title = NULL;
static otk_widget_t *playlist_menu_time = NULL;
static otk_widget_t *playlist_menu_count = NULL;
static otk_widget_t *playlist_menu_window = NULL;

static otk_widget_t *playlist_menu_editbox = NULL;

static void playlist_menu_update_list (void);
static void playlist_menu_update_buttons (void);

static void playlist_select_cb (void *entry_cb_data);
static void playlist_activate_cb (void *entry_cb_data);
static void playlist_remove_cb (void *entry_cb_data);

static void
playlist_menu_update_buttons (void)
{
    if (!is_current_menu (show_menu_playlist)) {
        return;
    }

    bool enabled;
    playitem_t *selected = playlist_get_selected ();

    /* Save, clear and play playlist */
    enabled = (playlist_length (oxine->rw_playlist) > 0);
    otk_widget_set_enabled (playlist_menu_buttons[0], enabled);
    otk_widget_set_enabled (playlist_menu_buttons[3], enabled);
    otk_widget_set_enabled (playlist_menu_buttons[4], enabled);

    /* Remove item, move item up or down */
    enabled = (otk_list_get_selected_count (playlist_menu_list) > 0);
    otk_widget_set_enabled (playlist_menu_buttons[2], enabled);

    /* Add/ remove subtitle */
    enabled = selected && is_file_allowed (selected->mrl, ALLOW_FILES_VIDEO);
    otk_widget_set_enabled (playlist_menu_buttons[5], enabled);
    if (selected && (selected->subtitle_mrl != NULL)) {
        otk_button_set_text (playlist_menu_buttons[5], _("Delete subtitle"));
    }
    else {
        otk_button_set_text (playlist_menu_buttons[5], _("Add subtitle"));
    }

    /* Move item up */
    enabled = selected && (selected != playlist_first (oxine->rw_playlist));
    otk_widget_set_enabled (playlist_menu_buttons[6], enabled);

    /* Move item down */
    enabled = selected
        && (playlist_next (oxine->rw_playlist, selected) != NULL);
    otk_widget_set_enabled (playlist_menu_buttons[7], enabled);

    /* Current title */
    enabled = odk_current_is_playback_mode (oxine->odk);
    otk_widget_set_visible (playlist_menu_buttons[9], enabled);

    /* Focus the add button on empty playlist */
    if (playlist_length (oxine->rw_playlist) == 0) {
        otk_widget_set_focused (playlist_menu_buttons[1], true);
    }
}


static void
playlist_menu_update_list (void)
{
    if (!is_current_menu (show_menu_playlist)) {
        return;
    }

    playlist_lock (oxine->current_playlist);

    otk_list_clear (playlist_menu_list);

    playitem_t *cur = playlist_first (oxine->rw_playlist);
    while (cur) {
        char title[1024];
        if (cur->subtitle_mrl) {
            snprintf (title, 1024, "%s (+%s)", cur->title, _("Subtitle"));
        }
        else {
            snprintf (title, 1024, "%s", cur->title);
        }

        otk_widget_t *w = otk_listentry_new (playlist_menu_list, title,
                                             playlist_activate_cb, cur,
                                             playlist_select_cb, cur,
                                             playlist_remove_cb, cur);

        if (odk_current_is_playback_mode (oxine->odk)
            && (oxine->current_playlist == oxine->rw_playlist)
            && (cur == playlist_get_current (oxine->rw_playlist)))
            otk_widget_set_enabled (w, false);

        cur = playlist_next (oxine->rw_playlist, cur);
    }

    {
        int pb_length = playlist_get_playback_length (oxine->rw_playlist);
        int num_titles = playlist_length (oxine->rw_playlist);
        int days = pb_length / 86400;
        int hours = (pb_length - days * 86400) / 3600;
        int minutes = (pb_length - days * 86400) % 3600;
        minutes = ceil ((double) minutes / 60.0);

        char *time = NULL;
        if (pb_length == 0) {
            time = ho_strdup_printf ("%s: -", _("Duration"));
        }
        else if (days == 1) {
            time = ho_strdup_printf ("%s: %d %s, %02d:%02d %s", _("Duration"),
                                     days, _("day"), hours, minutes, _("h"));
        }
        else if (days > 1) {
            time = ho_strdup_printf ("%s: %d %s, %02d:%02d %s", _("Duration"),
                                     days, _("days"), hours, minutes, _("h"));
        }
        else if (hours > 0) {
            time = ho_strdup_printf ("%s: %02d:%02d %s", _("Duration"),
                                     hours, minutes, _("h"));
        }
        else {
            time = ho_strdup_printf ("%s: %d %s", _("Duration"),
                                     minutes, _("min"));
        }
        otk_label_set_text (playlist_menu_time, time);
        ho_free (time);

        char *count = NULL;
        if (num_titles == 0) {
            count = ho_strdup_printf ("%s: -", _("Number of titles"));
        }
        else {
            count = ho_strdup_printf ("%s: %d", _("Number of titles"),
                                      num_titles);
        }
        otk_label_set_text (playlist_menu_count, count);
        ho_free (count);
    }

    otk_list_set_pos (playlist_menu_list, oxine->rw_playlist->top_position);
    otk_list_set_focus (playlist_menu_list, oxine->rw_playlist->cur_position);

    playlist_unlock (oxine->current_playlist);
}


static void
playlist_activate_cb (void *playitem_p)
{
    playitem_t *playitem = (playitem_t *) playitem_p;

    playlist_play_item (oxine->rw_playlist, playitem);
}


static void
playlist_select_cb (void *playitem_p)
{
    playlist_menu_update_buttons ();
    show_user_interface (oxine);
}


static void
playlist_remove_cb (void *playitem_p)
{
    playitem_t *playitem = (playitem_t *) playitem_p;

    if (!(odk_current_is_playback_mode (oxine->odk)
          && (playitem == playlist_get_current (oxine->rw_playlist)))) {

        playlist_remove (oxine->rw_playlist, playitem);

        oxine->rw_playlist->top_position =
            otk_list_get_pos (playlist_menu_list);
        oxine->rw_playlist->cur_position =
            otk_list_get_focus (playlist_menu_list);

        playlist_menu_update_list ();
        playlist_menu_update_buttons ();
        show_user_interface (oxine);
    }
}


static void
playlist_clear_cb (void *p)
{
    /* If we're playing a track from the R/W playlist we stop playback before
     * clearing the list. */
    if (odk_current_is_playback_mode (oxine->odk)
        && (oxine->current_playlist == oxine->rw_playlist)) {
        odk_stop_stream (oxine->odk);
    }

    playlist_clear (oxine->rw_playlist);

    playlist_menu_update_list ();
    playlist_menu_update_buttons ();
    show_user_interface (oxine);
}


static playitem_t *
playlist_get_selected (void)
{
    int count;
    playitem_t *playitem = NULL;
    void **selected = otk_list_get_selected (playlist_menu_list, &count);

    if (count != 1) {
        return NULL;
    }

    playitem = (playitem_t *) selected[0];
    ho_free (selected);

    return playitem;
}


static int
playlist_get_selected_pos (void)
{
    int num_selected;

    int *positions = otk_list_get_selected_pos (playlist_menu_list,
                                                &num_selected);

    if (num_selected != 1)
        return -1;

    int pos = positions[0];
    ho_free (positions);
    return pos;
}


static void
playlist_remove_subtitle_cb (void *p)
{
    playitem_t *selected = playlist_get_selected ();
    if (selected && selected->subtitle_mrl) {
        int position = playlist_get_selected_pos ();
        oxine->rw_playlist->top_position =
            otk_list_get_pos (playlist_menu_list);

        ho_free (selected->subtitle_mrl);
        selected->subtitle_mrl = NULL;

        playlist_menu_update_list ();
        otk_list_set_selected (playlist_menu_list, position, true);

        playlist_menu_update_buttons ();
        otk_widget_set_focused (playlist_menu_buttons[5], true);

        show_user_interface (oxine);
    }
}


static void
playlist_subtitle_cb (void *p)
{
    playitem_t *selected = playlist_get_selected ();
    if (selected && selected->subtitle_mrl) {
        playlist_remove_subtitle_cb (oxine);
    }
    else {
        show_subtitle_menu_cb (oxine);
    }
}


static void
playlist_remove_selected_cb (void *p)
{
    playitem_t *selected = playlist_get_selected ();

    if (!selected)
        return;
    if (odk_current_is_playback_mode (oxine->odk)
        && (selected == playlist_get_current (oxine->rw_playlist)))
        return;

    int position = playlist_get_selected_pos ();
    oxine->rw_playlist->top_position = otk_list_get_pos (playlist_menu_list);

    playlist_remove (oxine->rw_playlist, selected);

    playlist_menu_update_list ();
    if (position >= playlist_length (oxine->rw_playlist)) {
        otk_list_set_selected (playlist_menu_list, position - 1, true);
    }
    else {
        otk_list_set_selected (playlist_menu_list, position, true);
    }

    playlist_menu_update_buttons ();
    otk_widget_set_focused (playlist_menu_buttons[2], true);

    show_user_interface (oxine);
}


static void
playlist_move_up_selected_cb (void *p)
{
    playitem_t *selected = playlist_get_selected ();
    if (selected) {
        int position = playlist_get_selected_pos () - 1;
        oxine->rw_playlist->top_position =
            otk_list_get_pos (playlist_menu_list);

        playlist_move_up (oxine->rw_playlist, selected);

        playlist_menu_update_list ();
        while (position < 0) {
            position++;
        }
        otk_list_set_selected (playlist_menu_list, position, true);

        playlist_menu_update_buttons ();
        otk_widget_set_focused (playlist_menu_buttons[6], true);

        show_user_interface (oxine);
    }
}


static void
playlist_move_down_selected_cb (void *p)
{
    playitem_t *selected = playlist_get_selected ();
    if (selected) {
        int position = playlist_get_selected_pos () + 1;
        oxine->rw_playlist->top_position =
            otk_list_get_pos (playlist_menu_list);

        playlist_move_down (oxine->rw_playlist, selected);

        playlist_menu_update_list ();
        while (position >= playlist_length (oxine->rw_playlist)) {
            position--;
        }
        otk_list_set_selected (playlist_menu_list, position, true);

        playlist_menu_update_buttons ();
        otk_widget_set_focused (playlist_menu_buttons[7], true);

        show_user_interface (oxine);
    }
}


static void
playlist_play_cb (void *p)
{
    playitem_t *current = playlist_get_current (oxine->rw_playlist);
    if (!current) {
        playlist_play_first (oxine->rw_playlist);
    }
    else {
        playlist_play_item (oxine->rw_playlist, current);
    }
}


static void
playlist_save_now_cb (void *p, char *title)
{
    if (!title || (strlen (title) == 0)) {
        ho_free (title);
        playlist_save_cb (oxine);
        return;
    }

    char time_str[640];
    time_t current = time (NULL);
    struct tm *brokentime = localtime (&current);
    strftime (time_str, 639, "%d.%m.%y, %H:%M", brokentime);

    char *mrl = ho_strdup_printf ("%s/%s %s.oxp",
                                  get_dir_oxine_playlists (),
                                  time_str, title);
    playlist_xml_save (oxine->rw_playlist, mrl);
    ho_free (mrl);
    ho_free (title);

    playlist_menu_editbox = NULL;
    show_menu_playlist (NULL);
}


static void
playlist_save_ok_cb (void *p)
{
    assert (playlist_menu_editbox);

    char *title = otk_editbox_get_text (playlist_menu_editbox);

    playlist_save_now_cb (oxine, title);
}


static void
playlist_save_cb (void *p)
{
    if (playlist_length (oxine->rw_playlist) > 0) {
        create_new_window (false, true);
        otk_border_new (oxine->otk, 100, 200, 600, 200);
        otk_label_new (oxine->otk, 120, 250, 560,
                       OTK_ALIGN_LEFT | OTK_ALIGN_BOTTOM,
                       _("Please enter a name for the playlist:"));

        playlist_menu_editbox =
            otk_editbox_new (oxine->otk, 120, 270, 560, 40, 100,
                             NULL, NULL, playlist_save_now_cb, oxine);
        otk_widget_set_focused (playlist_menu_editbox, true);

        if (playlist_length (oxine->rw_playlist) == 1) {
            playitem_t *item = playlist_first (oxine->rw_playlist);
            otk_editbox_set_text (playlist_menu_editbox, item->title);
        }

        otk_widget_t *b;
        b = otk_text_button_new (oxine->otk, 230, 330, 160, 40, _("OK"),
                                 playlist_save_ok_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        b = otk_text_button_new (oxine->otk, 410, 330, 160, 40, _("Cancel"),
                                 show_menu_playlist, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);

        show_user_interface (oxine);
    }
}


static void
playlist_menu_event_handler (void *p, oxine_event_t * event)
{
    if (!is_current_menu (show_menu_playlist)) {
        return;
    }
    if (event->type != OXINE_EVENT_KEY) {
        return;
    }

    switch (event->source.key) {
    case OXINE_KEY_BACK:
    case OXINE_KEY_INSERT:
        show_menu_filelist (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    case OXINE_KEY_SAVE:
        playlist_save_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    default:
        break;
    }
}


void
show_menu_playlist (void *p)
{
    if (playlist_menu_window) {
        otk_set_current_window (oxine->otk, playlist_menu_window);

        oxine->rw_playlist->top_position =
            otk_list_get_pos (playlist_menu_list);
        oxine->rw_playlist->cur_position =
            otk_list_get_focus (playlist_menu_list);

        goto out_show;
    }
    odk_add_event_handler (oxine->odk, playlist_menu_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_NORMAL);

    playlist_menu_window = create_new_window (true, true);

    int x = 20;
    int y = 100;
    playlist_menu_buttons[0] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Play"),
                             playlist_play_cb, oxine);
    y += 40;
    playlist_menu_buttons[1] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Add"),
                             show_menu_filelist, oxine);
    y += 40;
    playlist_menu_buttons[2] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Remove"),
                             playlist_remove_selected_cb, oxine);
    y += 40;
    playlist_menu_buttons[3] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Clear"),
                             playlist_clear_cb, oxine);
    y += 40;
    playlist_menu_buttons[4] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Save"),
                             playlist_save_cb, oxine);
    y += 50;
    playlist_menu_buttons[6] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Move up"),
                             playlist_move_up_selected_cb, oxine);
    y += 40;
    playlist_menu_buttons[7] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Move down"),
                             playlist_move_down_selected_cb, oxine);
    y += 50;
    playlist_menu_buttons[5] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Add subtitle"),
                             playlist_subtitle_cb, oxine);
    y += 50;
    playlist_menu_buttons[8] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Mainmenu"),
                             show_menu_main, oxine);
    y += 40;
    playlist_menu_buttons[9] =
        otk_text_button_new (oxine->otk, x, y, 180, 35, _("Current title"),
                             show_menu_playback, oxine);

    playlist_menu_list =
        otk_list_new (oxine->otk, 220, 140,
                      odk_osd_get_width (oxine->odk) - 240,
                      odk_osd_get_height (oxine->odk) - 190,
                      30, 33, true, true, OTK_LIST_SELECTION_SINGLE, oxine);

    playlist_menu_title =
        otk_label_new (oxine->otk, 220, 117,
                       odk_osd_get_width (oxine->odk) - 240,
                       OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER,
                       _("Current playlist..."));
    otk_widget_set_font (playlist_menu_title, "sans", 32);

    playlist_menu_count =
        otk_label_new (oxine->otk, 220, odk_osd_get_height (oxine->odk) - 20,
                       260, OTK_ALIGN_LEFT | OTK_ALIGN_BOTTOM, "");

    playlist_menu_time =
        otk_label_new (oxine->otk, odk_osd_get_width (oxine->odk) - 20,
                       odk_osd_get_height (oxine->odk) - 20, 300,
                       OTK_ALIGN_RIGHT | OTK_ALIGN_BOTTOM, "");

  out_show:
    if (odk_current_is_logo_mode (oxine->odk)) {
        set_playback_ended_menu (show_menu_playlist, NULL);
    }

    set_backto_menu (show_menu_playlist, NULL);
    set_current_menu (show_menu_playlist, NULL);

    playlist_menu_update_list ();
    playlist_menu_update_buttons ();

    show_user_interface (NULL);
    show_menu_background ("menu_playlist.png");
}


void
free_menu_playlist (void)
{
    /* We cancel any thumbnail job that might be running. */
    cancel_job (thumbnail_job);

    if (next_filelist) {
        l_list_free (next_filelist, NULL);
        next_filelist = NULL;
    }
    if (prev_filelist) {
        l_list_free (prev_filelist, NULL);
        prev_filelist = NULL;
    }

    filelist_ref_set (&current_filelist, NULL);
    filelist_ref_set (&toplevel_filelist, NULL);
}
