/*
  CLAW - a C++ Library Absolutely Wonderful

  CLAW is a free library without any particular aim but being useful to 
  anyone.

  Copyright (C) 2005-2008 Julien Jorge

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: julien_jorge@yahoo.fr
*/
/**
 * \file image.cpp
 * \brief Implementation of the claw::graphic::image class.
 * \author Julien Jorge
 */
#include <claw/image.hpp>
#include <claw/exception.hpp>
#include <claw/bitmap.hpp>
#include <claw/pcx.hpp>
#include <claw/targa.hpp>
#include <claw/xbm.hpp>

#ifdef CLAW_PNG_SUPPORT
/* The png.h file must be included before any other file that includes setjmp.h
   (as jpeg.hpp). */
#include <claw/png.hpp>
#endif // CLAW_PNG_SUPPORT

#ifdef CLAW_JPEG_SUPPORT
#include <claw/jpeg.hpp>
#endif // CLAW_JPEG_SUPPORT

#include <algorithm>

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on the first pixel.
 */
claw::graphic::image::scanline::iterator claw::graphic::image::scanline::begin()
{
  return super::begin();
} // image::scanline::begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get en iterator past the last pixel.
 */
claw::graphic::image::scanline::iterator claw::graphic::image::scanline::end()
{
  return super::end();
} // image::scanline::end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on constant data on the first pixel.
 */
claw::graphic::image::scanline::const_iterator
claw::graphic::image::scanline::begin() const
{
  return super::begin();
} // image::scanline::begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on constant data past the last pixel.
 */
claw::graphic::image::scanline::const_iterator
claw::graphic::image::scanline::end() const
{
  return super::end();
} // image::scanline::end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the length of the line.
 */
claw::graphic::image::scanline::size_type
claw::graphic::image::scanline::size() const
{
  return super::size();
} // image::scanline::size()




/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor. Creates an image without datas.
 * \post width() == height() == 0
 */
claw::graphic::image::image()
{

} // image::image() [default constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor. Creates an empty image.
 * \param w Image's width.
 * \param h Image's height.
 * \pre w > 0 and h > 0
 */
claw::graphic::image::image( unsigned int w, unsigned int h )
{
  set_size(w, h);
} // image::image() [constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor. Reads an image from an input stream.
 * \param f The stream to read from.
 */
claw::graphic::image::image( std::istream& f )
{
  load(f);
} // image::image() [constructor]

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
claw::graphic::image::~image()
{
  // nothing to do
} // image::~image()

/*----------------------------------------------------------------------------*/
/**
 * \brief Gets image's width.
 */
unsigned int claw::graphic::image::width() const
{
  if ( m_data.empty() )
    return 0;
  else
    return m_data[0].size(); 
} // image::width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Gets image's height.
 */
unsigned int claw::graphic::image::height() const
{
  return m_data.size();
} // image::height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator pointing on the first pixel.
 */
claw::graphic::image::iterator claw::graphic::image::begin()
{
  return iterator(*this);
} // image::begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator pointing just past the last pixel.
 */
claw::graphic::image::iterator claw::graphic::image::end()
{
  return iterator(*this, width(), height()-1);
} // image::end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator pointing on the first pixel.
 */
claw::graphic::image::const_iterator claw::graphic::image::begin() const
{
  return const_iterator(*this);
} // image::begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator pointing just past the last pixel.
 */
claw::graphic::image::const_iterator claw::graphic::image::end() const
{
  return const_iterator(*this, width(), height());
} // image::end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy an image on the current image.
 * \param that The image to copy.
 * \param pos The position of the top left corner.
 */
void claw::graphic::image::partial_copy
(const image& that, const claw::math::coordinate_2d<int>& pos )
{
  claw::math::rectangle<int> my_box(0, 0, width(), height());
  claw::math::rectangle<int> his_box(pos.x, pos.y, that.width(), that.height());

  if ( my_box.intersects( his_box ) )
    {
      claw::math::rectangle<int> intersection;
      unsigned int that_y = pos.y < 0 ? -pos.y : 0;
      unsigned int that_x = pos.x < 0 ? -pos.x : 0;

      intersection = my_box.intersection( his_box );

      for (int y=0; y!=intersection.height; ++y)
        {
          scanline::const_iterator first = that[y + that_y].begin() + that_x;
          scanline::const_iterator last = first + intersection.width;
          scanline::iterator dest = (*this)[y + intersection.position.y].begin()
            + intersection.position.x;

          std::copy( first, last, dest );
        }
    }
} // image::partial_copy()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the image upside down.
 */
void claw::graphic::image::flip()
{
  for (unsigned int y=0; y!=height()/2; ++y)
    std::swap( m_data[y], m_data[height()-y-1] );
} // image::flip()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a new size to the image.
 * \remark Image's data won't be lost. If a dimension is set larger than its
 *         current value, extra pixels won't be initialized.
 * \pre (w!=0) && (h!=0)
 */
void claw::graphic::image::set_size( unsigned int w, unsigned int h )
{
  if (w == 0)
    m_data.clear();
  else
    {
      m_data.resize(h);
  
      for (unsigned int y=0; y!=height(); ++y)
        m_data[y].resize(w);
    }
} // image::set_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the image from a stream.
 * \param f The stream to read from.
 */
void claw::graphic::image::load( std::istream& f )
{
  bool ok = false;

#ifdef CLAW_JPEG_SUPPORT
  if (!ok)
    try { jpeg::reader( *this, f ); ok = true; }
    catch( ... ) { }
#endif // CLAW_JPEG_SUPPORT

#ifdef CLAW_PNG_SUPPORT
  if (!ok)
    try { png::reader( *this, f ); ok = true; }
    catch( ... ) { }
#endif // CLAW_PNG_SUPPORT

  if (!ok)
    try { bitmap::reader( *this, f ); ok = true; }
    catch( ... ) { }

  if (!ok)
    try { targa::reader( *this, f ); ok = true; }
    catch( ... ) { }

  if (!ok)
    try { pcx::reader( *this, f ); ok = true; }
    catch( ... ) { }

  if (!ok)
    try { xbm::reader( *this, f ); ok = true; }
    catch( ... ) { }

  if (!ok)
    throw claw::bad_format( "image::load: file format isn't supported." );
} // image::load()
