package tests::UserSessionAnalyserTest;

use strict;

use base qw/Lire::Test::DlfAnalyserTestCase/;

use Lire::Utils qw/tempdir/;
use Lire::DlfStore;
use Lire::DlfAnalysers::UserSessionAnalyser;
use Lire::DlfAnalyserProcess;
use Lire::PluginManager;
use File::Basename;

sub schemasdir {
    return dirname( __FILE__ ) . "/../..";
}

sub create_analyser {
    return new Lire::DlfAnalysers::UserSessionAnalyser();
}

sub registration_file {
    return dirname( __FILE__ ) . "/../../www-user_session_init";
}

sub test_create_session {
    my $self = $_[0];

    my $analyser = $self->create_analyser();

    my $dlf = $analyser->create_session( { 'time' => 1000,
                                           'client_host' => 'myhost',
                                           'useragent' => 'myagent',
                                           'requested_page' => 'mypage'
                                         } );
    $self->assert_deep_equals( { 'session_id'    => '1000:myhost:myagent',
                                 'session_start' => 1000,
                                 'req_count' => 0,
                                 'page_count' => 0,
                                 'visit_number' => 1,
                                 'dlf_ids' => [],
                                 'completed' => 'no',
                                 'first_page' => 'None',
                                 'page_2' => 'None',
                                 'page_3' => 'None',
                                 'page_4' => 'None',
                                 'page_5' => 'None',
                                 'last_page' => 'None',
                               }, $dlf );
    $self->assert_num_equals( 1, $analyser->{'visits'}{'myhost:myagent'} );

    $dlf = $analyser->create_session( { 'time' => 1005,
                                        'client_host' => 'myhost',
                                        'useragent' => 'myagent',
                                        'requested_page' => 'anotherpage',
                                        'first_page' => 'None',
                                        'page_2' => 'None',
                                        'page_3' => 'None',
                                        'page_4' => 'None',
                                        'page_5' => 'None',
                                        'last_page' => 'None',
                                      } );
    $self->assert_deep_equals( { 'session_id'     => '1005:myhost:myagent',
                                 'session_start'  => 1005,
                                 'req_count'      => 0,
                                 'page_count'     => 0,
                                 'visit_number'   => 2,
                                 'completed' => 'no',
                                 'dlf_ids' => [],
                                 'first_page' => 'None',
                                 'page_2' => 'None',
                                 'page_3' => 'None',
                                 'page_4' => 'None',
                                 'page_5' => 'None',
                                 'last_page' => 'None',
                               }, $dlf );
    $self->assert_num_equals( 2, $analyser->{'visits'}{'myhost:myagent'} );
}

sub test_is_session_expired {
    my $self = $_[0];

    my $analyser = $self->create_analyser();
    $self->assert( $analyser->is_session_expired( { 'session_end' => 1000 },
                                                  { 'time' => 1000 + 7200 } ));
    $self->assert( ! $analyser->is_session_expired( { 'session_end' => 1000 },
                                                    { 'time' => 1000 + 900 } ) );
}

sub test_update_session {
    my $self = $_[0];

    my $analyser = $self->create_analyser();

    my $session = { 'session_start' => 1000,
                    'session_end' => 1000,
                    'session_length' => 0,
                    'req_count' => 0,
                    'page_count' => 0,
                    'dlf_ids' => [],
                    'first_page' => 'None',
                    'page_2' => 'None',
                    'page_3' => 'None',
                    'page_4' => 'None',
                    'page_5' => 'None',
                    'last_page' => 'None',
                  };
    $analyser->update_session( $session, { 'time' => 1000, 'dlf_id' => 1,
                                           'requested_page' => 'my_page' } );
    $self->assert_deep_equals( { 'session_start' => 1000,
                                 'session_end' => 1000,
                                 'session_length' => 0,
                                 'req_count' => 1,
                                 'page_count' => 1,
                                 'first_page' => 'my_page',
                                 'last_page' => 'my_page',
                                 'page_2' => 'None',
                                 'page_3' => 'None',
                                 'page_4' => 'None',
                                 'page_5' => 'None',
                                 'dlf_ids' => [ 1 ] }, $session );
    $analyser->update_session( $session, { 'time' => 1005, 'dlf_id' => 2,
                                           'requested_page' => 'my_page.gif'});
    $self->assert_deep_equals( { 'session_start' => 1000,
                                 'session_end' => 1005,
                                 'session_length' => 5,
                                 'req_count' => 2,
                                 'page_count' => 1,
                                 'first_page' => 'my_page',
                                 'last_page' => 'my_page',
                                 'page_2' => 'None',
                                 'page_3' => 'None',
                                 'page_4' => 'None',
                                 'page_5' => 'None',
                                 'dlf_ids' => [ 1, 2 ] }, $session );
    $analyser->update_session( $session, { 'time' => 1006, 'dlf_id' => 3,
                                           'requested_page' => 'index.html'});
    $self->assert_deep_equals( { 'session_start' => 1000,
                                 'session_end' => 1006,
                                 'session_length' => 6,
                                 'req_count' => 3,
                                 'page_count' => 2,
                                 'first_page' => 'my_page',
                                 'page_2' => 'index.html',
                                 'page_3' => 'None',
                                 'page_4' => 'None',
                                 'page_5' => 'None',
                                 'last_page' => 'index.html',
                                 'dlf_ids' => [ 1, 2, 3 ] }, $session );
}

sub test_end_session {
    my $self = $_[0];

    my $process = $self->create_mock_process();
    my $analyser = $process->analyser_instance();

    $analyser->end_session( $process,
                            { 'session_end' => 1000,
                              'completed' => 'no',
                              'dlf_ids' => [ 1, 2, 3 ],
                            },
                            { 'time' => 1004 } );
    $self->assert_num_equals( 1, $process->dlf_count() );
    $self->assert_deep_equals( [ { 'session_end' => 1000,
                                   'completed' => 'no',
                                   'dlf_ids' => [ 1, 2, 3 ] },
                                 1, 2, 3 ], $process->dlf()->[-1] );

    $analyser->end_session( $process,
                            { 'session_end' => 10000,
                              'completed' => 'no',
                              'dlf_ids' => [ 1, 2 ],
                            },
                            { 'time' => 10000 + 1900 } );
    $self->assert_num_equals( 2, $process->dlf_count() );
    $self->assert_deep_equals( [ { 'session_end' => 10000,
                                   'completed' => 'yes',
                                   'dlf_ids' => [ 1, 2 ] },
                                 1, 2 ], $process->dlf()->[-1] );
}

sub set_up_store {
    my $self = $_[0];

    my $tmpdir = tempdir( 'test_analyse_XXXXXXX', 'CLEANUP' => 1 );
    $self->{'cfg'}{'lr_schemas_path'} = 
      [ $self->schemasdir(), $self->schemasdir() . "/../all/schemas" ];
    $self->{'cfg'}{'_lr_config_spec'} = $self->lire_default_config_spec();

     $self->{'store'} = Lire::DlfStore->open( "$tmpdir/store", 1 );

    my $stream = $self->{'store'}->open_dlf_stream( 'www', 'w' );
    foreach my $dlf ( [ undef, '192.168.250.1', '/index.html', 'An agent' ],
                      [ 100_000, undef, '/index.html', 'An agent' ],
                      [ 100_000, '192.168.250.1', undef, 'An agent' ],
                      [ 100_000, '192.168.250.1', '/index.html', undef ],
                      [ 100_000, '192.168.250.1', '/index.html', 'An agent' ],
                      [ 100_001, '192.168.250.1', '/logo.gif', 'An agent' ],
                      [ 100_010, '192.168.250.1', '/page.html', 'An agent' ],
                      [ 103_000, '192.168.250.1', '/index.html', 'An agent' ],
                      [ 103_010, '192.168.250.1', '/a_page.html', 'Another agent' ], )
    {
        $stream->write_dlf( { 'time' => $dlf->[0],
                              'client_host' => $dlf->[1],
                              'requested_page' => $dlf->[2],
                              'useragent' => $dlf->[3] } );
    }
    $stream->close();

    return;
}

sub annotate_errors {
    my $self = $_[0];

    my $stream = $self->{'store'}->open_dlf_stream( 'lire_import_log', 'r' );
    while ( defined( my $dlf = $stream->read_dlf() ) ) {
        $self->annotate( $dlf->{'msg'} );
    }
    $stream->close();
}

sub test_analyse {
    my $self = $_[0];

    my $analyser = $self->create_analyser();
    $self->set_up_store();
    Lire::PluginManager->register_plugin( $analyser );
    my $process = new Lire::DlfAnalyserProcess( $self->{'store'},
                                                'www-user_session', {} );
    $process->run_analysis_job();
    $self->annotate_errors();

    my $stream = $self->{'store'}->open_dlf_stream( 'www-user_session', 'r' );
    $self->assert_num_equals( 3, $stream->nrecords() );

    my $dlf = [];
    while ( defined( my $r = $stream->read_dlf() ) ) {
        delete $r->{'dlf_id'};
        delete $r->{'dlf_source'};
        push @$dlf, $r;
    }
    $stream->close();

    my $e_dlf =
      [
       {
        'session_id' => '100000:192.168.250.1:An agent',
        'session_start' => 100_000,
        'session_end'   => 100_010,
        'session_length' => 10,
        'completed' => 'yes',
        'req_count' => 3,
        'page_count' => 2,
        'first_page' => '/index.html',
        'page_2' => '/page.html',
        'page_3' => 'None',
        'page_4' => 'None',
        'page_5' => 'None',
        'last_page' => '/page.html',
        'visit_number' => 1,
       },
       {
        'session_id' => '103000:192.168.250.1:An agent',
        'session_start' => 103_000,
        'session_end'   => 103_000,
        'session_length' => 0,
        'completed' => 'no',
        'req_count' => 1,
        'page_count' => 1,
        'first_page' => '/index.html',
        'page_2' => 'None',
        'page_3' => 'None',
        'page_4' => 'None',
        'page_5' => 'None',
        'last_page' => '/index.html',
        'visit_number' => 2,
       },
       {
        'session_id' => '103010:192.168.250.1:Another agent',
        'session_start' => 103_010,
        'session_end'   => 103_010,
        'session_length' => 0,
        'completed' => 'no',
        'req_count' => 1,
        'page_count' => 1,
        'first_page' => '/a_page.html',
        'page_2' => 'None',
        'page_3' => 'None',
        'page_4' => 'None',
        'page_5' => 'None',
        'last_page' => '/a_page.html',
        'visit_number' => 1,
       },
      ];
    $self->assert_deep_equals( $e_dlf, $dlf );
}

1;
