/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iomanip>

#include "lifeobase.hpp"
#include "diary.hpp"
#include "views.hpp"
#include "gui_tag.hpp"


using namespace LIFEO;


// TAG =========================================================================
TagView::TagView( void )
{
	try
	{
		Lifeobase::builder->get_widget( "vbox_tag", m_vbox );
		Lifeobase::builder->get_widget( "alignment_tag_toolbar", m_alignment_toolbar );
		Lifeobase::builder->get_widget( "togglebutton_tag_filter", m_button_filter );
		Lifeobase::builder->get_widget( "button_tag_dismiss", m_button_dismiss );
		Lifeobase::builder->get_widget( "entry_tag_name", m_entry_name );
		Lifeobase::builder->get_widget( "button_tag_name_apply", m_button_name_apply );
		//Lifeobase::builder->get_widget( "label_tag_period", m_label_period );
		Lifeobase::builder->get_widget( "combobox_tag_category", m_combobox_category );
		Lifeobase::builder->get_widget_derived( "drawingarea_tag_chart", m_widgetchart );

		m_liststore_categories = Gtk::ListStore::create( colrec_category );
	}
	catch( ... ) {}

	Tag::shower = this;

	m_combobox_category->set_model( m_liststore_categories );
	m_combobox_category->pack_start( colrec_category.name );
	m_combobox_category->set_row_separator_func(
			sigc::mem_fun( this, &TagView::is_row_separator ) );

	// SIGNALS
	m_button_filter->signal_toggled().connect(
			sigc::mem_fun( this, &TagView::handle_filter_toggled ) );
	m_button_dismiss->signal_clicked().connect(
			sigc::mem_fun( this, &TagView::dismiss_tag ) );

	m_button_name_apply->signal_clicked().connect(
			sigc::mem_fun( this, &TagView::handle_name_applied ) );
	m_entry_name->signal_changed().connect(
			sigc::mem_fun( this, &TagView::handle_name_changed ) );
	m_entry_name->signal_activate().connect(
			sigc::mem_fun( *m_button_name_apply, &Gtk::Button::clicked ) );

	m_combobox_category->signal_changed().connect(
			sigc::mem_fun( this, &TagView::handle_category_changed ) );
}

void
TagView::handle_filter_toggled( void )
{
	if( Lifeobase::m_internaloperation )
		return;

	Diary::d->set_filter_tag( m_button_filter->get_active() ? m_ptr2tag : NULL );
    Lifeobase::base->update_button_filter_tag();
    Lifeobase::base->update_button_remove_filters();
    Lifeobase::base->update_tag_icon( m_ptr2tag );
    Lifeobase::base->update_entry_list();
}

void
TagView::handle_name_changed( void )
{
	Glib::ustring name = m_entry_name->get_text();
	Tag *tag = Diary::d->get_tags()->get_tag( name );
	m_button_name_apply->set_sensitive( !( tag || name.empty() ) );
}

void
TagView::handle_name_applied( void )
{
	// a caution against calls by an entry activation event:
	if( ! m_button_name_apply->is_sensitive() )
		return;
	Glib::ustring name = m_entry_name->get_text();
	Diary::d->get_tags()->rename( m_ptr2tag, name );
	Lifeobase::base->m_label_gui_elem->set_label( name );

	m_button_name_apply->set_sensitive( false );

	Gtk::TreeRow row = * Lifeobase::base->get_element_row( m_ptr2tag );
	row[ TagPanel::colrec->name ] = Glib::Markup::escape_text( name );
}

void
TagView::handle_category_changed( void )
{
	if( Lifeobase::m_internaloperation )
		return;

	Gtk::TreeIter iter = m_combobox_category->get_active();
	if( iter )
	{
		Gtk::TreeRow row = *iter;
		if( row[ colrec_category.type ]  == TCT_NEW ) // create new
		{
			CategoryTags *category = Diary::d->create_tag_ctg();
			m_ptr2tag->set_category( category );
			update_combobox_categories();
			Lifeobase::base->update_tag_list();
			category->show();
		}
		else	// assign a category or reset
		{
			m_ptr2tag->set_category( row[ colrec_category.ptr ] );
			Lifeobase::base->update_tag_list();
		}
	}
}

void
TagView::update_combobox_categories( void )
{
	m_liststore_categories->clear();

	Gtk::TreeRow row = * m_liststore_categories->append();

	row[ colrec_category.name ] = _( "None" );
	row[ colrec_category.ptr ] = NULL;
	row[ colrec_category.type ] = TCT_NONE;

	// separator 1:
	row = * m_liststore_categories->append();
	row[ colrec_category.type ] = TCT_SEPARATOR;
	row[ colrec_category.ptr ] = ( CategoryTags* ) 0x1;

	for( PoolCategoriesTags::const_iterator iter = Diary::d->m_tag_categories.begin();
		 iter != Diary::d->m_tag_categories.end();
		 ++iter )
	{
		row = * m_liststore_categories->append();

		row[ colrec_category.name ] = iter->first;
		row[ colrec_category.ptr ] = iter->second;
		row[ colrec_category.type ] = TCT_CATEGORY;
	}

	// separator 2:
	if( Diary::d->m_tag_categories.size() > 0 )
	{
		row = * m_liststore_categories->append();
		row[ colrec_category.type ] = TCT_SEPARATOR;
		row[ colrec_category.ptr ] = ( CategoryTags* ) 0x1;
	}

	row = * m_liststore_categories->append();

	row[ colrec_category.name ] = _( "Create New..." );
	// a dummy value to differentiate it from "None":
	row[ colrec_category.ptr ] = ( CategoryTags* ) 0x1;
	row[ colrec_category.type ] = TCT_NEW;
}

void
TagView::show( Tag &tag )
{
	// do nothing is entry is already the current element:
	if( ! Lifeobase::base->change_current_element( &tag, m_vbox ) )
		return;

	m_ptr2tag = &tag;

	Lifeobase::m_internaloperation++;
	// HEADER
	Lifeobase::m_image_gui_elem->set( Lifeobase::icons->tag_32 );
	Lifeobase::base->set_header_texts(
			tag.m_name,
			Glib::ustring::compose( _( "%1 entries" ), tag.m_items.size() ) );
	Lifeobase::set_gelem_toolbar( m_alignment_toolbar );

	// BODY
	m_button_filter->set_active( Diary::d->get_filter_tag() == &tag );

	m_entry_name->set_text( tag.m_name );

	for( Gtk::TreeIter iter = m_liststore_categories->children().begin();
		 iter != m_liststore_categories->children().end();
		 ++iter )
	{
		if( ( *iter )[ colrec_category.ptr ] == tag.get_category() )
			m_combobox_category->set_active( iter );
	}

	// CHART DATA
	ChartPoints *points( new ChartPoints );
	Date yearmonth_last( 0 );
	int value( 0 );
	for( SetDiaryElements::reverse_iterator iter = tag.m_items.rbegin();
		 iter != tag.m_items.rend(); ++iter )
	{
		Entry *entry = dynamic_cast< Entry* >( *iter );
        if( entry->get_date().is_ordinal() )
            continue;
		const Date ym_entry( entry->get_date().get_yearmonth() );
		if( yearmonth_last == 0 )
			yearmonth_last = ym_entry;
		else
		if( yearmonth_last != ym_entry )
		{
			points->add( yearmonth_last, value );

			// fill empty months
			for( Date ym_zero = yearmonth_last; ; )
			{
				ym_zero.forward_month();
				if( ym_zero >= ym_entry )
					break;
				points->add( ym_zero, 0 );
			}

			yearmonth_last = ym_entry;
			value = 0;
		}
		value++;
	}
	points->add( yearmonth_last, value );

	m_widgetchart->set_points( points );

	Lifeobase::m_internaloperation--;
}

void
TagView::dismiss_tag( void )
{
	if( ! Lifeobase::base->confirm_dismiss_element() )
		return;

    Diary::d->show();

	Lifeobase::base->remove_element_from_history( m_ptr2tag );

	// clear filter if necessary:
	if( Diary::d->get_filter_tag() == m_ptr2tag )
	{
		Diary::d->set_filter_tag( NULL );
        Lifeobase::base->update_button_remove_filters();
        Lifeobase::base->update_entry_list();
	}
	Diary::d->dismiss_tag( m_ptr2tag );
	Lifeobase::base->update_tag_list();
}

bool
TagView::is_row_separator( const Glib::RefPtr< Gtk::TreeModel > &model,
							 const Gtk::TreeIter iter )
{
	Gtk::TreeRow row = * iter;
	return ( row[ colrec_category.type ] == TCT_SEPARATOR );
}

// TAG CATEGORY ====================================================================================
CategoryTagsView::CategoryTagsView( void )
{
	CategoryTags::shower = this;

	try
	{
		Lifeobase::builder->get_widget( "vbox_categorytags", m_vbox );
		Lifeobase::builder->get_widget( "alignment_categorytags_toolbar", m_alignment_toolbar );
		Lifeobase::builder->get_widget( "entry_categorytags_name", m_entry_name );
		Lifeobase::builder->get_widget( "button_categorytags_name_apply", m_button_name_apply );
		Lifeobase::builder->get_widget( "button_categorytags_dismiss", m_button_dismiss );
	}
	catch( ... ) {}

	m_entry_name->signal_changed().connect(
			sigc::mem_fun( this, &CategoryTagsView::handle_name_changed ) );

	m_entry_name->signal_activate().connect(
			sigc::mem_fun( *m_button_name_apply, &Gtk::Button::clicked ) );

	m_button_name_apply->signal_clicked().connect(
			sigc::mem_fun( this, &CategoryTagsView::handle_name_applied ) );

	m_button_dismiss->signal_clicked().connect(
			sigc::mem_fun( this, &CategoryTagsView::dismiss_category ) );
}

void
CategoryTagsView::show( CategoryTags &category )
{
	// do nothing is entry is already the current element:
	if( ! Lifeobase::base->change_current_element( &category, m_vbox ) )
		return;

	m_ptr2category = &category;

	// HEADER
	Lifeobase::m_image_gui_elem->set( Lifeobase::icons->tag_category_32 );
	Lifeobase::base->set_header_texts(
			category.m_name,
			Glib::ustring::compose( _( "%1 tags" ), category.size() ) );
	Lifeobase::set_gelem_toolbar( m_alignment_toolbar );

	// BODY
	m_entry_name->set_text( category.m_name );
}

void
CategoryTagsView::handle_name_changed( void )
{
	Glib::ustring name = m_entry_name->get_text();
	PoolCategoriesTags::const_iterator iter = Diary::d->m_tag_categories.find( name );
	m_button_name_apply->set_sensitive(
			!( iter != Diary::d->m_tag_categories.end() || name.empty() ) );
}

void
CategoryTagsView::handle_name_applied( void )
{
	Glib::ustring name = m_entry_name->get_text();
	Diary::d->m_tag_categories.rename_category( m_ptr2category, name );
	Lifeobase::base->m_label_gui_elem->set_label( name );
	Lifeobase::base->update_tag_list();
	Lifeobase::base->update_tag_categories();	// updates the combobox in TagView

	m_button_name_apply->set_sensitive( false );
}

void
CategoryTagsView::dismiss_category( void )
{
	if( ! Lifeobase::base->confirm_dismiss_element() )
		return;

    Diary::d->show();

	Lifeobase::base->remove_element_from_history( m_ptr2category );
	// TODO: offer merging with a category
	Diary::d->dismiss_tag_ctg( m_ptr2category );
	Lifeobase::base->update_tag_list();
	Lifeobase::base->update_tag_categories();
}

// CHAPTER =========================================================================================
ChapterView::ChapterView( void )
{
	Chapter::shower = this;

	Gtk::Button		*button_new_entry;

	try
	{
		Lifeobase::builder->get_widget( "vbox_chapter", m_vbox );
		Lifeobase::builder->get_widget( "alignment_chapter_toolbar", m_alignment_toolbar );
		Lifeobase::builder->get_widget( "button_chapter_dismiss", m_button_dismiss );
		Lifeobase::builder->get_widget( "entry_chapter_name", m_entry_name );
		Lifeobase::builder->get_widget( "button_chapter_name_apply", m_button_name_apply );

		Lifeobase::builder->get_widget( "label_chapter_period", m_label_period );
		Lifeobase::builder->get_widget( "hbox_chapter_period", m_hbox_period );
		Lifeobase::builder->get_widget( "entry_chapter_begins", m_entry_begins );
		Lifeobase::builder->get_widget( "button_chapter_date_apply", m_button_date_apply );
		Lifeobase::builder->get_widget( "label_chapter_ends", m_label_ends );

		Lifeobase::builder->get_widget( "frame_chapter_actions", m_frame_actions );
		Lifeobase::builder->get_widget( "button_chapter_new_entry", button_new_entry );

		//m_liststore_chapter_type = Gtk::ListStore::create( colrec_chapter_type );
	}
	catch( ... ) {}

	// TYPE
	//m_combo_type->set_model( m_liststore_chapter_type );
	//m_combo_type->pack_start( colrec_chapter_type.name );
	//Gtk::TreeRow row( * m_liststore_chapter_type->append() );
	//row[ colrec_chapter_type.name ] = _( "Time Based" );
	//row[ colrec_chapter_type.type ] = CT_TEMPORAL;
	//row = * m_liststore_chapter_type->append();
	//row[ colrec_chapter_type.name ] = _( "Time Independent" );
	//row[ colrec_chapter_type.type ] = CT_ORDINAL;

	
	// SIGNALS
	m_button_dismiss->signal_clicked().connect(
			sigc::mem_fun( this, &ChapterView::dismiss_chapter ) );

	m_entry_name->signal_changed().connect(
			sigc::mem_fun( *this, &ChapterView::handle_name_changed ) );
	m_entry_name->signal_activate().connect(
			sigc::mem_fun( *m_button_name_apply, &Gtk::Button::clicked ) );
	m_button_name_apply->signal_clicked().connect(
			sigc::mem_fun( *this, &ChapterView::handle_name_applied ) );

	m_entry_begins->signal_changed().connect(
			sigc::mem_fun( *this, &ChapterView::handle_date_changed ) );
	m_entry_begins->signal_activate().connect(
			sigc::mem_fun( *m_button_date_apply, &Gtk::Button::clicked ) );
	m_button_date_apply->signal_clicked().connect(
			sigc::mem_fun( *this, &ChapterView::handle_date_applied ) );

	button_new_entry->signal_clicked().connect(
			sigc::mem_fun( *this, &ChapterView::add_new_entry ) );
}

void
ChapterView::dismiss_chapter( void )
{
	if( ! Lifeobase::base->confirm_dismiss_element() )
		return;

    Diary::d->show();

	Lifeobase::base->remove_element_from_history( m_ptr2chapter );
	Diary::d->dismiss_chapter( m_ptr2chapter );
	Lifeobase::base->update_entry_list();
}

void
ChapterView::handle_name_changed( void )
{
	Glib::ustring name = m_entry_name->get_text();
	m_button_name_apply->set_sensitive( ! name.empty() &&
										name != m_ptr2chapter->m_name );
}

void
ChapterView::handle_name_applied( void )
{
	// a caution against calls by an entry activation event:
	if( ! m_button_name_apply->is_sensitive() )
		return;
	Glib::ustring name = m_entry_name->get_text();
	m_ptr2chapter->m_name = name;
	m_button_name_apply->set_sensitive( false );
	Lifeobase::base->m_label_gui_elem->set_label( name );
	Gtk::TreeRow row = * Lifeobase::base->get_element_row( m_ptr2chapter );
	row[ ListData::colrec->info ] = m_ptr2chapter->get_list_str();
}

void
ChapterView::handle_date_changed( void )
{
	try
	{
		Date date( m_entry_begins->get_text() );
		m_button_date_apply->set_sensitive( 
				! Diary::d->get_current_chapter_ctg()->get_chapter( date ) );
	}
	catch( LIFEO::Error &er )
	{
		m_button_date_apply->set_sensitive( false );
	}
}

void
ChapterView::handle_date_applied( void )
{
	// a caution against calls by an entry activation event:
	if( ! m_button_date_apply->is_sensitive() )
		return;
	if( Diary::d->get_current_chapter_ctg()->set_chapter_date(
			m_ptr2chapter, Date( m_entry_begins->get_text() ) ) )
	{
		m_button_date_apply->set_sensitive( false );
		Lifeobase::base->update_entry_list();
		update_labels();
	}
}

void
ChapterView::update_labels( void )
{
	if( m_ptr2chapter->m_date_begin.is_ordinal() )
	{
		Lifeobase::base->set_header_texts(
				Glib::ustring::compose( "%1. %2", m_ptr2chapter->get_date().format_string(),
									    m_ptr2chapter->m_name ),
				Glib::ustring::compose( _( "%1 entries" ), m_ptr2chapter->get_size() ) );
	}
	else
	{
		int timespan( 0 );

		m_entry_begins->set_text( m_ptr2chapter->m_date_begin.format_string() );

		CategoryChapters::iterator iter( Diary::d->get_current_chapter_ctg()->find(
												m_ptr2chapter->get_date().m_date ) );
		if( iter == Diary::d->get_current_chapter_ctg()->begin() )
			m_label_ends->set_text( _( "Unlimited" ) );
		else
		{
			--iter;
			m_label_ends->set_text( iter->second->m_date_begin.format_string() );
			Glib::Date gdate_begin( m_ptr2chapter->m_date_begin.get_glib() );
			Glib::Date gdate_end( iter->second->m_date_begin.get_glib() );
			timespan = gdate_begin.days_between( gdate_end );
		}

		Lifeobase::base->set_header_texts(
				m_ptr2chapter->m_name,
				Glib::ustring::compose(
						timespan ? _( "%1 entries in %2 days" ) : _( "%1 entries" ),
						m_ptr2chapter->get_size(), timespan ) );
	}
}

void
ChapterView::add_new_entry( void )
{
	Diary::d->create_entry( m_ptr2chapter->get_free_order().m_date );

	Lifeobase::base->update_entry_list();
	Lifeobase::base->update_calendar();
}

void
ChapterView::show( Chapter &chapter )
{
	// do nothing is entry is already the current element:
	if( ! Lifeobase::base->change_current_element( &chapter, m_vbox ) )
		return;

	m_ptr2chapter = &chapter;

	Lifeobase::m_image_gui_elem->set( chapter.get_icon_32() );
	Lifeobase::set_gelem_toolbar( m_alignment_toolbar );

	Lifeobase::m_internaloperation++;
	update_labels();
	m_entry_name->set_text( chapter.m_name );
	Lifeobase::m_internaloperation--;

	bool flag_temporal( ! chapter.m_date_begin.is_ordinal() );
	m_label_period->set_visible( flag_temporal );
	m_hbox_period->set_visible( flag_temporal );
	m_frame_actions->set_visible( ! flag_temporal );
}

// THEME ===========================================================================================
ThemeView::ThemeView( void )
{
	Theme::shower = this;

	try
	{
		Lifeobase::builder->get_widget( "vbox_theme", m_vbox );
		Lifeobase::builder->get_widget( "entry_theme_name", m_entry_name );
		Lifeobase::builder->get_widget( "button_theme_name_apply", m_button_name_apply );
		Lifeobase::builder->get_widget( "alignment_theme_toolbar", m_alignment_toolbar );
		Lifeobase::builder->get_widget( "fontbutton_theme", m_fontbutton );
		Lifeobase::builder->get_widget( "colorbutton_theme_bground", m_colorbutton_bground );
		Lifeobase::builder->get_widget( "colorbutton_theme_text", m_colorbutton_text );
		Lifeobase::builder->get_widget( "colorbutton_theme_heading", m_colorbutton_heading );
		Lifeobase::builder->get_widget( "colorbutton_theme_subheading", m_colorbutton_subheading );
		Lifeobase::builder->get_widget( "colorbutton_theme_highlight", m_colorbutton_highlight );
		Lifeobase::builder->get_widget( "eventbox_theme_preview", m_eventbox_preview );
		Lifeobase::builder->get_widget( "label_theme_preview", m_label_preview );
		Lifeobase::builder->get_widget( "button_theme_duplicate", m_button_duplicate );
		Lifeobase::builder->get_widget( "button_theme_dismiss", m_button_dismiss );
		Lifeobase::builder->get_widget( "button_theme_default", m_button_make_default );
	}
	catch( ... ) { }

	m_entry_name->signal_changed().connect(
			sigc::mem_fun( this, &ThemeView::handle_name_changed ) );

	m_entry_name->signal_activate().connect(
			sigc::mem_fun( *m_button_name_apply, &Gtk::Button::clicked ) );

	m_button_name_apply->signal_clicked().connect(
			sigc::mem_fun( this, &ThemeView::handle_name_applied ) );

	m_fontbutton->signal_font_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_font_changed ) );
	m_colorbutton_bground->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );
	m_colorbutton_text->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );
	m_colorbutton_heading->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );
	m_colorbutton_subheading->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );
	m_colorbutton_highlight->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );

	// TOOLBAR
	m_button_duplicate->signal_clicked().connect(
			sigc::mem_fun( this, &ThemeView::duplicate ) );
	m_button_dismiss->signal_clicked().connect(
			sigc::mem_fun( this, &ThemeView::dismiss_theme ) );
	m_button_make_default->signal_clicked().connect(
			sigc::mem_fun( this, &ThemeView::make_default ) );
}

void
ThemeView::show( Theme &theme )
{
	// do nothing is entry is already the current element:
	if( ! Lifeobase::base->change_current_element( &theme, m_vbox ) )
		return;

	m_ptr2theme = &theme;

	// HEADER
	Lifeobase::m_image_gui_elem->set( Lifeobase::icons->theme_32 );
	Lifeobase::base->set_header_texts(
			theme.get_name(), theme.is_system() ? _( "System theme" ) : _( "Custom theme" ) );
	Lifeobase::set_gelem_toolbar( m_alignment_toolbar );

	// BODY
	m_entry_name->set_text( theme.get_name() );
	m_fontbutton->set_font_name( theme.font.to_string() );
	m_colorbutton_bground->set_color( theme.color_base );
	m_colorbutton_text->set_color( theme.color_text );
	m_colorbutton_heading->set_color( theme.color_heading );
	m_colorbutton_subheading->set_color( theme.color_subheading );
	m_colorbutton_highlight->set_color( theme.color_highlight );

	update_preview();
	m_vbox->set_sensitive( ! theme.is_system() );
	m_button_dismiss->set_sensitive( ! theme.is_system() );
	m_button_make_default->set_visible( &theme != Diary::d->get_default_theme() );
}

void
ThemeView::handle_name_changed( void )
{
	Glib::ustring name = m_entry_name->get_text();
	PoolThemes::const_iterator iter = Diary::d->get_themes().find( name );
	m_button_name_apply->set_sensitive( !( iter != Diary::d->get_themes().end() || name.empty() ) );
}

void
ThemeView::handle_name_applied( void )
{
	Glib::ustring name = m_entry_name->get_text();
	Diary::d->m_themes.rename_theme( m_ptr2theme, name );
	Lifeobase::base->m_label_gui_elem->set_label( name );
	Lifeobase::base->update_theme_list();

	m_button_name_apply->set_sensitive( false );
}

void
ThemeView::handle_font_changed( void )
{
	m_ptr2theme->font = Pango::FontDescription( m_fontbutton->get_font_name() );
	update_preview();
}

void
ThemeView::handle_color_changed( void )
{
	m_ptr2theme->color_base = m_colorbutton_bground->get_color();
	m_ptr2theme->color_text = m_colorbutton_text->get_color();
	m_ptr2theme->color_heading = m_colorbutton_heading->get_color();
	m_ptr2theme->color_subheading = m_colorbutton_subheading->get_color();
	m_ptr2theme->color_highlight = m_colorbutton_highlight->get_color();
	update_preview();
}

void
ThemeView::update_preview( void )
{
    Glib::ustring str_preview( Glib::ustring::compose(
            "<span color='%1' size='xx-large' weight='bold'>Heading</span>\n"
            " <span color='%2'size='larger' weight='bold'>Subheading</span>\n"
            "Body text <span bgcolor='%3'>highlighted text</span>",
            convert_gdkcolor_to_html( m_ptr2theme->color_heading ),
            convert_gdkcolor_to_html( m_ptr2theme->color_subheading ),
            convert_gdkcolor_to_html( m_ptr2theme->color_highlight ) ) );
    m_label_preview->set_markup( str_preview );
    m_label_preview->modify_fg( Gtk::STATE_NORMAL, m_ptr2theme->color_text );
    m_label_preview->modify_fg( Gtk::STATE_INSENSITIVE, m_ptr2theme->color_text );
    m_eventbox_preview->modify_bg( Gtk::STATE_NORMAL, m_ptr2theme->color_base );
    m_eventbox_preview->modify_bg( Gtk::STATE_INSENSITIVE, m_ptr2theme->color_base );
    m_label_preview->modify_font( m_ptr2theme->font );
}

// creating new themes is done by duplicating existing ones
void
ThemeView::duplicate( void )
{
	Theme *new_theme = Diary::d->duplicate_theme( m_ptr2theme );
	Lifeobase::base->update_theme_list();
	new_theme->show();
}

void
ThemeView::make_default( void )
{
	Diary::d->set_default_theme( m_ptr2theme );
	Lifeobase::base->update_theme_list();
	m_button_make_default->set_visible( false );
}

void
ThemeView::dismiss_theme( void )
{
	if( ! Lifeobase::base->confirm_dismiss_element() )
		return;

    Diary::d->show();

	Lifeobase::base->remove_element_from_history( m_ptr2theme );

	Theme *theme = m_ptr2theme; // always be on the safe side
	ThemeSystem::get()->show();
	Diary::d->dismiss_theme( theme );
	Lifeobase::base->update_theme_list();
}

// DIARY ===========================================================================================
DiaryView::DiaryView( void )
:	m_vbox_diary( NULL ), m_dialog_password( NULL ), m_dialog_import( NULL ), m_dialog_export( NULL )
{
	Diary::shower = this;
	m_ptr2diary = Diary::d;	// for now no other diary is shown in the view

	Gtk::Button *button_import( NULL );
	Gtk::Button *button_export( NULL );
	Gtk::HBox   *hbox_chapter_ctg( NULL );
	try
	{
		Lifeobase::builder->get_widget( "vbox_diary", m_vbox_diary );
		Lifeobase::builder->get_widget( "alignment_diary_toolbar", m_alignment_toolbar );
		Lifeobase::builder->get_widget( "label_path", m_label_path );
		Lifeobase::builder->get_widget( "label_encryption", m_label_encryption );

		Lifeobase::builder->get_widget( "cmb_startup_type", m_combobox_startup_type );
		Lifeobase::builder->get_widget( "label_startup_elem", m_label_startup_elem );
		Lifeobase::builder->get_widget( "label_startup_elem_drop", m_label_startup_elem_drop );
		Lifeobase::builder->get_widget( "button_startup_elem_go", m_button_startup_elem_go );

//		Lifeobase::builder->get_widget( "combobox_chapter_ctg",
//				m_combobox_chapter_ctg );
		Lifeobase::builder->get_widget( "hbox_cur_chapter_ctg",
		        hbox_chapter_ctg );
		Lifeobase::builder->get_widget( "button_chapter_ctg_dismiss",
				m_button_chapter_ctg_dismiss );
		Lifeobase::builder->get_widget( "button_chapter_ctg_rename",
				m_button_chapter_ctg_rename );

        Lifeobase::builder->get_widget_derived( "drawingarea_diary_chart", m_widgetchart );

		Lifeobase::builder->get_widget( "button_import", button_import );
		Lifeobase::builder->get_widget( "button_export", button_export );
		Lifeobase::builder->get_widget( "button_password", m_button_password );

		Lifeobase::builder->get_widget( "button_log_out_wosaving",
				m_button_logoutwosaving );

		m_liststore_startup_type = Gtk::ListStore::create( colrec_startup_type );
		m_liststore_chapter_ctg = Gtk::ListStore::create( colrec_chapter_ctg );

		// temporary until we upgrade to gtkmm-3:
		m_combobox_chapter_ctg = Gtk::manage( new Gtk::ComboBoxEntry );
	}
	catch( ... ) { }

	// STARTUP
	m_combobox_startup_type->set_model( m_liststore_startup_type );
	m_combobox_startup_type->pack_start( colrec_startup_type.name );
	Gtk::TreeRow row( * m_liststore_startup_type->append() );
	row[ colrec_startup_type.name ] = _( "Show Most Current Entry" );
	row[ colrec_startup_type.type ] = HOME_CURRENT_ELEM;
	row = * m_liststore_startup_type->append();
	row[ colrec_startup_type.name ] = _( "Remember Last Item" );
	row[ colrec_startup_type.type ] = HOME_LAST_ELEM;
	row = * m_liststore_startup_type->append();
	row[ colrec_startup_type.name ] = _( "Always Show a Fixed Item" );
	row[ colrec_startup_type.type ] = DEID_MIN;

	m_label_startup_elem_drop->drag_dest_set( Lifeobase::base->drag_targets_entry );

	// CHAPTER CATEGORIES
	hbox_chapter_ctg->pack_start( *m_combobox_chapter_ctg );
	hbox_chapter_ctg->reorder_child( *m_combobox_chapter_ctg, 0 );
	m_combobox_chapter_ctg->show();
	m_combobox_chapter_ctg->set_model( m_liststore_chapter_ctg );
	m_combobox_chapter_ctg->set_text_column( colrec_chapter_ctg.name );
	m_combobox_chapter_ctg->set_row_separator_func(
			sigc::mem_fun( this, &DiaryView::is_row_separator ) );

	// SIGNALS
	m_button_password->signal_clicked().connect(
			sigc::mem_fun( *this, &DiaryView::start_dialog_password ) );

	m_combobox_startup_type->signal_changed().connect(
			sigc::mem_fun( this, &DiaryView::handle_startup_type_changed ) );
	m_label_startup_elem_drop->signal_drag_data_received().connect(
			sigc::mem_fun( this, &DiaryView::handle_startup_elem_dropped ) );
	m_button_startup_elem_go->signal_clicked().connect(
			sigc::mem_fun( *this, &DiaryView::go_to_startup_elem ) );

	m_combobox_chapter_ctg->signal_changed().connect(
			sigc::mem_fun( this, &DiaryView::handle_cur_chapter_ctg_changed ) );
	m_combobox_chapter_ctg->get_entry()->signal_activate().connect(
			sigc::mem_fun( this, &DiaryView::rename_cur_chapter_ctg ) );
	m_button_chapter_ctg_rename->signal_clicked().connect(
			sigc::mem_fun( this, &DiaryView::rename_cur_chapter_ctg ) );
	m_button_chapter_ctg_dismiss->signal_clicked().connect(
			sigc::mem_fun( this, &DiaryView::dismiss_cur_chapter_ctg ) );

	button_import->signal_clicked().connect(
			sigc::mem_fun( *this, &DiaryView::start_dialog_import ) );
	button_export->signal_clicked().connect(
			sigc::mem_fun( *this, &DiaryView::start_dialog_export ) );

	m_button_logoutwosaving->signal_clicked().connect(
			sigc::bind( sigc::mem_fun( *Lifeobase::base, &Lifeobase::logout ), false ) );
}

DiaryView::~DiaryView( void )
{
	Diary::shower = NULL;
}

void
DiaryView::handle_login( void )
{
	Lifeobase::m_internaloperation++;

	update_combobox_chapter_ctg();
	if( Diary::d->m_startup_elem < HOME_FIXED_ELEM )
		m_combobox_startup_type->set_active( Diary::d->m_startup_elem - 1 );
	else
		m_combobox_startup_type->set_active( 2 );

	Lifeobase::m_internaloperation--;
}

void
DiaryView::show( Diary &diary )
{
	// do nothing is entry is already the current element:
	if( ! Lifeobase::base->change_current_element( &diary, m_vbox_diary ) )
		return;

	// not necessary for now as there is only one diary to show in the view:
	// m_ptr2diary = &diary;

	// HEADER
	int timespan( 0 );
	if( ! diary.m_entries.empty() )
	{
		for( Entryiter iter = diary.m_entries.begin();
			 iter != diary.m_entries.end();
			 ++iter )
		{
			if( ! iter->second->get_date().is_ordinal() )
			{
				Glib::Date gdate_begin( diary.m_entries.rbegin()->second->get_date().get_glib() );
				Glib::Date gdate_end( iter->second->get_date().get_glib() );
				timespan = gdate_begin.days_between( gdate_end );
				break;
			}
		}
	}

	Lifeobase::m_image_gui_elem->set( Lifeobase::icons->diary_32 );
	Lifeobase::base->set_header_texts( diary.get_name(),
									   Glib::ustring::compose(
												_( "%1 entries spanning %2 days" ),
												diary.m_entries.size(),
												timespan ) );
	Lifeobase::set_gelem_toolbar( m_alignment_toolbar );

	// BODY
	m_button_password->set_label( diary.m_passphrase.empty() ? _( "Encrypt..." ) :
												_( "Change Password..." ) );
	m_label_encryption->set_label( diary.m_passphrase.empty() ? _( "Unencrypted" ) :
												_( "Encrypted" ) );

    // CHART DATA
    ChartPoints *points( new ChartPoints );
    Date yearmonth_last( 0 );
    int value( 0 );
    for( PoolEntries::reverse_iterator iter = diary.m_entries.rbegin();
         iter != diary.m_entries.rend(); ++iter )
    {
        Entry *entry( iter->second );
        if( entry->get_date().is_ordinal() )
            continue;
        const Date ym_entry( entry->get_date().get_yearmonth() );
        if( yearmonth_last == 0 )
            yearmonth_last = ym_entry;
        else
        if( yearmonth_last != ym_entry )
        {
            points->add( yearmonth_last, value );

            // fill empty months
            for( Date ym_zero = yearmonth_last; ; )
            {
                ym_zero.forward_month();
                if( ym_zero >= ym_entry )
                    break;
                points->add( ym_zero, 0 );
            }

            yearmonth_last = ym_entry;
            value = 0;
        }
        value++;
    }
    points->add( yearmonth_last, value );

    m_widgetchart->set_points( points );

#if GTK_CHECK_VERSION(2,18,0)
	m_label_path->set_markup( Glib::ustring::compose( "<a href=\"file://%1\">%2</a>",
			Glib::path_get_dirname( diary.m_path ), diary.m_path ) );
#else
	m_label_path->set_label( m_path );
#endif
}

void
DiaryView::start_dialog_password( void )
{
	try
	{
		m_dialog_password = new DialogPassword(	m_ptr2diary->m_passphrase,
				m_ptr2diary->m_passphrase.size() > 0 ?
		// TRANSLATORS: this is the title of the password dialog
						_( "Change Password" ) :
		// TRANSLATORS: this is the title of the password dialog
						_( "Add Password" ) );
	}
	catch( ... )
	{
		// TODO: throw
		return;
	}

	m_dialog_password->run();

	if( m_dialog_password->is_passphrase_set() )
	// TODO: if( m_dialog_password->run() )
	{
		if( m_ptr2diary->set_passphrase( m_dialog_password->get_passphrase() ) )
		{
			m_label_encryption->set_label( _( "Encrypted" ) );
			m_button_password->set_label( _( "Change Password..." ) );
		}
	}

	delete m_dialog_password;
	m_dialog_password = NULL;
}

void
DiaryView::start_dialog_import( void )
{
    if( m_dialog_import == NULL )
        Lifeobase::builder->get_widget_derived( "dialog_import", m_dialog_import );

    int response = m_dialog_import->run();
    m_dialog_import->hide();

    if( response == RESPONSE_GO )
    {
        Lifeobase::base->update_entry_list();
        Lifeobase::base->update_tag_list();
        Lifeobase::base->update_theme_list();
    }
}

void
DiaryView::start_dialog_export( void )
{
    if( m_dialog_export == NULL )
        Lifeobase::builder->get_widget_derived( "dialog_export", m_dialog_export );

    m_dialog_export->run();
    m_dialog_export->hide();
}

//void
//DiaryView::open_diary_folder( void )
//{
//	std::string uri = "file://" + Glib::path_get_dirname( m_path );
//	GError *err = NULL;
//	gtk_show_uri( NULL, uri.c_str(), GDK_CURRENT_TIME, &err);
//}

void
DiaryView::handle_startup_type_changed( void )
{
	if( ! Lifeobase::m_internaloperation )
	{
		Gtk::TreeRow row( * m_combobox_startup_type->get_active() );
		Diary::d->m_startup_elem = row[ colrec_startup_type.type ];
	}
	bool flag_show_fixed_item( Diary::d->m_startup_elem > HOME_FIXED_ELEM );
	m_label_startup_elem->set_visible( flag_show_fixed_item );
	m_label_startup_elem_drop->set_visible( flag_show_fixed_item );
	m_button_startup_elem_go->set_visible( flag_show_fixed_item );
	if( flag_show_fixed_item )
		m_label_startup_elem_drop->set_label( Diary::d->get_startup_elem()->get_name() );
}

void
DiaryView::handle_startup_elem_dropped(
				const Glib::RefPtr< Gdk::DragContext > &context,
				int x, int y,
				const Gtk::SelectionData &selection_data,
				uint info,
				guint time )
{
	context->drag_finish( true, false, time );

	Glib::ustring name = selection_data.get_data_as_string();

	if( name.empty() )	// should never happen
		return;

	switch( info )
	{
		case DRAG_TARGET_ENTRY_INFO:
		{
			DiaryElement *element = Lifeobase::base->get_dragged_elem();
			if( element != NULL )
			{
				m_label_startup_elem_drop->set_label( element->get_name() );
				Diary::d->m_startup_elem = element->get_id();
			}
			break;
		}
		default:
			break;
	}
}

void
DiaryView::go_to_startup_elem( void )
{
	DiaryElement *element( Diary::d->get_startup_elem() );
	if( element != NULL && element != Diary::d )
		element->show();
}

void
DiaryView::update_combobox_chapter_ctg( void )
{
	m_liststore_chapter_ctg->clear();

	Gtk::TreeRow row;

	for( PoolCategoriesChapters::const_iterator iter =
				m_ptr2diary->m_chapter_categories.begin();
		 iter != m_ptr2diary->m_chapter_categories.end();
		 ++iter )
	{
		row = * m_liststore_chapter_ctg->append();

		CategoryChapters *category = iter->second;

		row[ colrec_chapter_ctg.name ] = category->get_name();
		row[ colrec_chapter_ctg.ptr ] = category;
		row[ colrec_chapter_ctg.type ] = CCCT_CATEGORY;

		// setting current item:
		if( m_ptr2diary->m_ptr2chapter_ctg_cur == category )
			m_combobox_chapter_ctg->set_active( row );
	}

	m_button_chapter_ctg_rename->set_visible( false );
	m_button_chapter_ctg_dismiss->set_visible( true );
	m_button_chapter_ctg_dismiss->set_sensitive( m_ptr2diary->m_chapter_categories.size() > 1 );

	// separator 2:
	if( m_ptr2diary->m_chapter_categories.size() > 0 )
	{
		row = * m_liststore_chapter_ctg->append();
		row[ colrec_chapter_ctg.type ] = CCCT_SEPARATOR;
	}

	row = * m_liststore_chapter_ctg->append();

	row[ colrec_chapter_ctg.name ] = _( "Create New..." );
	row[ colrec_chapter_ctg.ptr ] = NULL;
	row[ colrec_chapter_ctg.type ] = CCCT_NEW;
}

void
DiaryView::handle_cur_chapter_ctg_changed( void )
{
	if( Lifeobase::m_internaloperation ) return;

	Gtk::TreeIter iter = m_combobox_chapter_ctg->get_active();
	bool flag_renamed = ! ( iter ), flag_existing_ctg( false );

	if( flag_renamed )
	{
		Glib::ustring name = m_combobox_chapter_ctg->get_active_text();
		bool flag_name_usable = m_ptr2diary->m_chapter_categories.count( name ) < 1;
		m_button_chapter_ctg_rename->set_sensitive( flag_name_usable );
	}
	else
	{
		Gtk::TreeRow row = *iter;
		switch( row[ colrec_chapter_ctg.type ] )
		{
			case CCCT_NEW: // create new
				m_ptr2diary->m_ptr2chapter_ctg_cur = m_ptr2diary->create_chapter_ctg();
				update_combobox_chapter_ctg();
				flag_existing_ctg = true;
				break;
			case CCCT_CATEGORY:
				flag_existing_ctg = true;
				m_ptr2diary->m_ptr2chapter_ctg_cur = row[ colrec_chapter_ctg.ptr ];
				Lifeobase::base->update_entry_list();
				break;
			default:	// for separators
				break;
		}

		m_button_chapter_ctg_dismiss->set_sensitive( flag_existing_ctg );
	}

	m_button_chapter_ctg_rename->set_visible( flag_renamed );
	m_button_chapter_ctg_dismiss->set_visible( ! flag_renamed );
}

void
DiaryView::rename_cur_chapter_ctg( void )
{
	Glib::ustring name = m_combobox_chapter_ctg->get_active_text();
	if( name == m_ptr2diary->m_ptr2chapter_ctg_cur->get_name() )
		return;
	m_ptr2diary->rename_chapter_ctg( m_ptr2diary->m_ptr2chapter_ctg_cur, name );
	update_combobox_chapter_ctg();
}

void
DiaryView::dismiss_cur_chapter_ctg( void )
{
	if( ! Lifeobase::base->confirm_dismiss_element() )
		return;

    Diary::d->show();

	m_ptr2diary->dismiss_chapter_ctg( m_ptr2diary->m_ptr2chapter_ctg_cur );
	update_combobox_chapter_ctg();
}

bool
DiaryView::is_row_separator( const Glib::RefPtr< Gtk::TreeModel > &model,
							 const Gtk::TreeIter iter )
{
	Gtk::TreeRow row = * iter;
	return ( row[ colrec_chapter_ctg.type ] == CCCT_SEPARATOR );
}
