// 
// Copyright (c) 2006-2012, Benjamin Kaufmann
// 
// This file is part of Clasp. See http://www.cs.uni-potsdam.de/clasp/ 
// 
// Clasp is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// Clasp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with Clasp; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
#ifndef CLASP_SOLVER_STRATEGIES_H_INCLUDED
#define CLASP_SOLVER_STRATEGIES_H_INCLUDED
#ifdef _MSC_VER
#pragma once
#endif

#include <clasp/constraint.h>
#include <clasp/util/misc_types.h>

/*!
 * \file 
 * Contains strategies used to configure search of a Solver.
 */
namespace Clasp {

//! Implements clasp's configurable schedule-strategies.
/*!
 * clasp currently supports five different strategies:
 *  - fixed-interval schedule with length n
 *  - geometric schedule  : length = n1 * n2^k   (k >= 0)
 *  - arithmetic schedule : length = n1 + (n2*k) (k >= 0)
 *  - inner-outer schedule: geometric/arithmetic schedule that is repeated once outer bound is reached.
 *                          Then, outer = outer {*,+} n2
 *  - luby's schedule: see: Luby et al. "Optimal speedup of las vegas algorithms."
 *  .
 */
struct ScheduleStrategy {
public:
	enum Type { geometric_schedule = 0, arithmetic_schedule = 1, luby_schedule = 2, null_schedule = 3 }; 
	
	ScheduleStrategy(Type t = geometric_schedule, uint32 b = 100, double g = 1.5, uint64 o = 0)
		: grow(g)
		, outer(o)
		, base(b)
		, type(t)
		, idx(0) {}
	
	//! Create luby's sequence with unit-length unit.
	static ScheduleStrategy luby(uint32 unit)                                { return ScheduleStrategy(luby_schedule, unit, 0, 0);  }
	//! Create geometric sequence base * (grow^k) with optional outer limit.
	static ScheduleStrategy geom(uint32 base, double grow, uint64 outer = 0) { return ScheduleStrategy(geometric_schedule, base, grow, outer);  }
	//! Create arithmetic sequence base + (add*k) with optional outer limit.
	static ScheduleStrategy arith(uint32 base, double add, uint64 outer = 0) { return ScheduleStrategy(arithmetic_schedule, base, add, outer);  }
	static ScheduleStrategy none()                                           { return ScheduleStrategy(geometric_schedule, 0); }
	static ScheduleStrategy null()                                           { return ScheduleStrategy(null_schedule, 0); }

	uint64 current() const;
	bool   disabled()const { return base == 0; }
	bool   ignore()  const { return base == 0 && type == null_schedule; }
	void   reset()         { idx  = 0;         }
	uint64 next();
	
	double grow;
	uint64 outer;
	uint32 base : 30;
	uint32 type :  2;
	uint32 idx;
};

uint32 lubyR(uint32 idx);
double growR(uint32 idx, double g);
double addR(uint32 idx, double a);
inline uint32 log2(uint32 x) {
	uint32 ln = 0;
	if (x & 0xFFFF0000u) { x >>= 16; ln |= 16; }
	if (x & 0xFF00u    ) { x >>=  8; ln |=  8; }
	if (x & 0xF0u      ) { x >>=  4; ln |=  4; }
	if (x & 0xCu       ) { x >>=  2; ln |=  2; }
	if (x & 0x2u       ) { x >>=  1; ln |=  1; }
	return ln;
}

//! Reduce strategy used during solving.
/*!
 * A reduce strategy mainly consists of an algorithm and a scoring scheme
 * for measuring "activity" of learnt constraints.
 */
struct ReduceStrategy {
	//! Reduction algorithm to use during solving.
	enum Algorithm {
		reduce_linear   = 0, /*!< Linear algorithm from clasp-1.3.x. */
		reduce_in_place = 1, /*!< Reorder learnt db and remove inactive constraints. */
		reduce_stable   = 2  /*!< Similar to reduce_in_place but without changing the order of learnt db. */
	};
	//! Score to measure "activity" of learnt constraints.
	enum Score {
		score_act  = 0, /*!< Activity only: how often constraint is used during conflict analysis. */
		score_lbd  = 1, /*!< Use literal block distance as activity. */
		score_both = 2  /*!< Use activity and lbd together. */
	};
	ReduceStrategy() : glue(0), score(0), algo(0), noGlue(0) {}
	static int    compare(Score sc, const Clasp::LearntConstraint::Activity& lhs, const Clasp::LearntConstraint::Activity& rhs) {
		int fs = 0;
		if      (sc == score_act) { fs = ((int)lhs.scoreAct()) - ((int)rhs.scoreAct()); }
		else if (sc == score_lbd) { fs = ((int)lhs.scoreLbd()) - ((int)rhs.scoreLbd()); }
		return fs != 0 ? fs : ((int)lhs.scoreBoth()) - ((int)rhs.scoreBoth()); 
	}
	static uint32 asScore(Score sc, const Clasp::LearntConstraint::Activity& act) {
		if (sc == score_act)  { return act.scoreAct(); }
		if (sc == score_lbd)  { return act.scoreLbd(); }
		/*  sc == score_both*/{ return act.scoreBoth();}
	}
	uint32 glue   : 27; /*!< Don't remove nogoods with lbd <= glue.    */
	uint32 score  :  2; /*!< One of Score.                             */
	uint32 algo   :  2; /*!< One of Algorithm.                         */
	uint32 noGlue :  1; /*!< Do not count glue clauses in limit        */
};

class DecisionHeuristic;

//! Parameter-Object for configuring a solver.
struct SolverStrategies {
	typedef std::auto_ptr<DecisionHeuristic>  Heuristic;
	//! Clasp's two general search strategies
	enum SearchStrategy {
		use_learning = 0, /*!< Analyze conflicts and learn First-1-UIP-clause */
		no_learning  = 1  /*!< Don't analyze conflicts - chronological backtracking */
	};
	//! Antecedents to consider during conflict clause minimization.
	enum CflMinAntes {
		no_antes              = 0,  /*!< Don't minimize first-uip-clauses. */
		binary_antes          = 2,  /*!< Consider only binary antecedents.         */
		binary_ternary_antes  = 6,  /*!< Consider binary- and ternary antecedents. */
		all_antes             = 7   /*!< Consider all antecedents.                 */
	};
	enum HeuOpts {
		heu_use_static_sign = 1,
		heu_use_model       = 2,
		heu_init_dynamic    = 4
	};
	//! Creates a default-initialized object.
	SolverStrategies();
	Heuristic heuristic;           /*!< Active decision heuristic. */
	RNG       rng;                 /*!< RNG used during search.    */
	uint32    compress;            /*!< If > 0, enable compression for learnt clauses of size > compress. */
	uint32    saveProgress : 17;   /*!< Enable progress saving if > 0. */
	uint32    cflMinAntes  :  3;   /*!< Antecedents to look at during conflict clause minimization. */
	uint32    heuOpts      :  3;   /*!< Set of HeuOpts. */
	uint32    reverseArcs  :  2;   /*!< Use "reverse-arcs" during learning if > 0. */
	uint32    otfs         :  2;   /*!< Enable "on-the-fly" subsumption if > 0. */
	uint32    updateLbd    :  2;   /*!< Update lbds of antecedents during conflict analysis. */
	uint32    strRecursive :  1;   /*!< If 1, use more expensive recursive nogood minimization. */
	uint32    randomWatches:  1;   /*!< Initialize watches randomly in clauses. */
	uint32    search       :  1;   /*!< Current search strategy. */
};

}
#endif
