#include "cp_types.h"
#include "cp_proto.h"

/* Write circle packing p to an open file fp. Return 0 on error.
If basic combinatoric data is to be included, use key "NODECOUNT:";
if only ancilliary data is to be included, use "CHECKCOUNT:".
(Note: specialized routines may be necessary: eg., when aims are
specified, only the non-default ones are included.)

When 'apend' is set, just add specified data (without NODECOUNT
or CHECKCOUNT) to existing file before 'END' (assume file is
open in "r+" read/write mode, positioned already, and will be closed 
by the calling routine; don't allow 0001 bit to be set in act).

Data to write specified in 'act' with bit-code as follows: 

   basic combinatoric info:     
     1: 00001     nodecount, a/b/c, flowers, packname
   (else (if not append), CHECKCOUNT: nodecount)

   default standard:
     2: 00002     geometry
     3: 00004     non-default inv_dist & aims
     4: 00010     radii

   optional:
     5: 00020     centers
     6: 00040     angle sums

     7: 00100     vertex_map (if it exists)
     8: 00200     lists of verts/faces/edges (non default)
     9: 00400     colors (non default)

     10: 01000    nonzero vertex plot_flags
     11: 02000    edge-pairing Mobius transformations

Note: standard write would be act= 0017, max would be act=0777.
*/

extern char buf[];

int writepack(FILE *fp,struct p_data *p,int act,int append)
{
  int i,j,k,n,flag,colorflag,jj,digits;
  double angle;
  char format_buf[32];
  struct K_data *pK_ptr;
  struct R_data *pR_ptr;
  struct Vertlist *trace;
  struct Edgelist *etrace;
  
  if (!fp) return 0;
  pK_ptr=p->packK_ptr;
  pR_ptr=p->packR_ptr;
  if (append) act &= 07776; /* append mode, can't have 0001 bit set */
  else if (act & 0001) /* new pack basic comb info */
    fprintf(fp,"NODECOUNT:   %d\n\n",p->nodecount);
  else fprintf(fp,"CHECKCOUNT:   %d\n\n",p->nodecount); /* partial data */
  if ((act & 0002) || (act & 0010) || (act & 0020))
    /* geometry (needed if radii or centers given) */
    {
      if (p->hes<0) sprintf(buf,"hyperbolic\n");
      else if (p->hes>0) sprintf(buf,"spherical\n");
      else sprintf(buf,"euclidean\n");
      fprintf(fp,"GEOMETRY:   %s\n",buf);
    }
  if (act & 0001) 
    {
      fprintf(fp,"ALPHA/BETA/GAMMA:   %d  %d  %d\n\n",
	      p->alpha,p->beta,p->gamma);

      buf[0]='\0';
      sscanf(p->file_name,"%s",buf);
      stripsp(buf);
      if (strlen(buf)>0)
	fprintf(fp,"PACKNAME: %s\n\n",buf);

      fprintf(fp,"FLOWERS: \n");
      for (n=1;n<=p->nodecount;n++)
	{
	  fprintf(fp,"\n%d %d   ",n,pK_ptr[n].num);
	  for (i=0;i<=pK_ptr[n].num;i++) 
	    fprintf(fp,"%d ",pK_ptr[n].flower[i]);
	}
      fprintf(fp,"\n\n");
    }
  if (act & 0004) /* inv_dist? aims? (non-default only) */
    {
      if (p->overlap_status)
	{
	  fprintf(fp,"INV_DISTANCES:\n");
	  for (i=1;i<=p->nodecount;i++)
	    for (j=0;j<(pK_ptr[i].num+pK_ptr[i].bdry_flag);j++)
	      if (i<(k=pK_ptr[i].flower[j])
		  && fabs((angle=pK_ptr[i].overlaps[j])-1.0)>okerr)
		fprintf(fp,"\n%d %d  %.6e ",
			i,k,angle);
	  fprintf(fp,"\n  (done)\n\n");
	}
      flag=0;
      for (i=1;i<=p->nodecount;i++)
	{
	  if (pK_ptr[i].bdry_flag 
	      && pR_ptr[i].aim>=0.0) flag++;
	  else if ( !pK_ptr[i].bdry_flag 
		    && fabs(pR_ptr[i].aim-2.0*M_PI)>(10.0)*toler ) flag++;
	}
      if (flag) /* at least one interior aim out of tolerance */
	{
	  jj=j=0;
	  for (i=1;i<=p->nodecount && jj==0;i++) 
	    {
	      if ( (pK_ptr[i].bdry_flag && pR_ptr[i].aim>=0) 
		   || (!pK_ptr[i].bdry_flag 
		       && (pR_ptr[i].aim<(2.0*M_PI+okerr)
			   || pR_ptr[i].aim>(2.0*M_PI-okerr))) )
		jj++;
	    }
	  if (jj>0) /* at least one non-default aim */
	    {
	      fprintf(fp,"ANGLE_AIMS:\n");
	      for (i=1;i<=p->nodecount;i++)
		if ( (pK_ptr[i].bdry_flag && pR_ptr[i].aim>=0) 
		     || (!pK_ptr[i].bdry_flag 
			 && (pR_ptr[i].aim<(2.0*M_PI-okerr)
			     || pR_ptr[i].aim>(2.0*M_PI+okerr))) )
		  {
		    fprintf(fp," %d % .10e  ",i,pR_ptr[i].aim);
		    j++;
		    if ((j % 3)==0) fprintf(fp,"\n");
		  }
	    }
	  fprintf(fp,"\n  (done)\n\n");
	}
    }
  if (act & 0010) /* radii? */
    {
      digits=1;
      while ((pow(10,(double)digits))*toler < 0.1 
	     && digits<MAX_ACCUR) digits++;
      fprintf(fp,"RADII: \n");
      for (i=1;i<=p->nodecount;i++)
	{
	  sprintf(format_buf,"%% .%de ",digits);
	  fprintf(fp,format_buf,radius(p,i));
	  if ((i % 4)==0) fprintf(fp,"\n");
	}
      fprintf(fp,"\n\n");
    }
  if (act & 0020) /* centers? (often easier to recompute) */
    {
      fprintf(fp,"CENTERS:\n");
      for (i=1;i<=p->nodecount;i++)
	{
	  fprintf(fp," % .6e % .6e\t",pR_ptr[i].center.re,
		  pR_ptr[i].center.im);
	  if ((i % 2)==0) fprintf(fp,"\n");
	}
      fprintf(fp,"\n\n");
    }
  if (act & 0040) /* angle sums? (often easier to recompute) */
    {
      fprintf(fp,"ANGLESUMS: \n");
      for (i=1;i<=p->nodecount;i++)
	{
	  fprintf(fp," %.6e\t",pR_ptr[i].curv);
	  if ((i % 5)==0) fprintf(fp,"\n");
	}
      fprintf(fp,"\n\n");
    }
  if (act & 01000) /* nonzero plot_flags */
    {
      fprintf(fp,"CIRCLE_PLOT_FLAGS: \n");
      for (i=1;i<=p->nodecount;i++)
	if (pK_ptr[i].plot_flag)
	  fprintf(fp,"%d  %d\n",i,pK_ptr[i].plot_flag);
      fprintf(fp," (done)\n\n");
    }
  if (act & 02000) /* edge-pairing Mobius */
    {
      i=0;
      while ((i<3*MAX_COMPONENTS) && p->edge_pair[i+1].edge) i++;
      if (i) /* yes, mobius exist */
	fprintf(fp,"EDGE_PAIRING MOBIUS: %d \n\n",i);
      i=1;
      while ((i<3*MAX_COMPONENTS) && p->edge_pair[i].edge)
	{
	  output_mobius(fp,p->edge_pair[i].mob);
	  i++;
	}
    }
  if (act & 0400) /* any non default colors? */
    {
      colorflag=0;
      for (i=1;i<=p->nodecount && colorflag==0;i++) 
	if (pK_ptr[i].color>0 
	    && pK_ptr[i].color != FG_COLOR) colorflag++;
      if (colorflag) /* found some non-default colors */
	{
	  fprintf(fp,"CIRCLE_COLORS:\n");
	  j=0;
	  for (i=1;i<=p->nodecount;i++)
	    {
	      if (pK_ptr[i].color!=FG_COLOR)
		{
		  fprintf(fp," %d %d  ",
			  i,pK_ptr[i].color);
		  j++;
		}
	      if ((j % 10)==0) fprintf(fp,"\n");
	    }
	  fprintf(fp,"\n  (done)\n\n");
	}
      colorflag=0;
      for (i=1;i<=p->facecount && colorflag;i++) 
	if (p->faces[i].color != FG_COLOR) colorflag++;
      if (colorflag) /* found some non-default colors */
	{
	  fprintf(fp,"TRI_COLORS:\n");
	  j=0;
	  for (i=1;i<=p->facecount;i++)
	    {
	      if (p->faces[i].color>0 
		  && p->faces[i].color!=FG_COLOR)
		{
		  fprintf(fp," %d %d %d %d  ",
			  p->faces[i].vert[0],
			  p->faces[i].vert[1],
			  p->faces[i].vert[2],
			  p->faces[i].color);
		  j++;
		}
	      if ((j % 5)==0) fprintf(fp,"\n");
	    }
	  fprintf(fp,"\n (done)\n\n");
	}
    }
  if (act & 0200) /* print lists? (verts/faces/edges) */
    {
      if (p->vlist)
	{
	  fprintf(fp,"VERT_LIST:\n");
	  trace=p->vlist;
	  while (trace)
	    {
	      fprintf(fp," %d\n",trace->v);
	      trace=trace->next;
	    }
	  fprintf(fp," (done)\n\n");
	}
      if (p->flist)
	{
	  fprintf(fp,"FACE_LIST:\n");
	  trace=p->flist;
	  while (trace)
	    {
	      fprintf(fp," %d\n",trace->v);
	      trace=trace->next;
	    }
	  fprintf(fp," (done)\n\n");
	}
      if (p->elist)
	{
	  fprintf(fp,"EDGE_LIST:\n");
	  etrace=p->elist;
	  while (etrace)
	    {
	      fprintf(fp," %d %d\n",etrace->v,etrace->w);
	      etrace=etrace->next;
	    }
	}
    }
  if ((act & 0100) && (etrace=p->vertex_map)) /* vertex_map */
    {
      fprintf(fp,"VERTEX_MAP:\n");
      while (etrace)
	{
	  fprintf(fp,"%d  %d\n",etrace->v,etrace->w);
	  etrace=etrace->next;
	}
      fprintf(fp,"   (done)\n\n");
    }      
  if (!append) fprintf(fp,"END\n");
  fflush(fp);
  return 1;
} /* writepack */
