#include "cp_types.h"
#include "cp_proto.h"

/* Drawing flag scheme: flags indicate drawing instructions
   for objects -- circles/lines/faces. Bits set as follows:
      1  ->  draw object?
      2  ->  fill? (4 and 16 imply 2, also)
      4  ->  off = foreground, on = background 
             (applies to interior, not border, overriden by bit 16)
      8  ->  border color? (off = foreground, on = recorded color)
      16 ->  interior color? (default set by bit 4, on --> recorded color)
      32 ->  display label?

Eg.  flag=3: filled object, in foreground
     flag=9: open object, border in (recorded) color
             (for 'edge', this gives colored edge)
     flag=19: filled in color, border in foreground
     flag=27: filled, border and interior in color
     flag=15: filled with background, border in color
     flag=32: label only (face or circle)

Normally, flag for each type of object; often, when passed on to
subroutine, you may need to pass color code too:
Eg. (cflag, ecol, ccol) for circle flag, border color code, int color 
code. */

int alist[300];

int draw_any_circle(struct p_data *p,int n,int cflag,
		    int ccol,int col,int show)
/* draws circle in approp geometry; plot_flag indicates 
circles 'reliably' placed (see, e.g. fix -l). */
{
        int front;
	double r;
	complex ctr;

	if (n<1 || n > p->nodecount || !p->packK_ptr[n].plot_flag) return 0;
	ctr=p->packR_ptr[n].center;
	r=p->packR_ptr[n].rad;
	if (cflag & 1)
	  {
	    if (p->hes > 0) 
	      s_circle(p->screen,ss_view(p->screen,ctr,1,&front),
		       r,cflag,ccol,col,show);
	    else
	      {
		if (p->hes < 0) h_to_e_data(ctr,r,&ctr,&r);
		circle(p->screen,ctr,r,cflag,ccol,col,show);
	      }
	  }
	if (cflag & 32) draw_cir_number(p,n,n,show);
	return 1;
} /* draw_any_circle */
	
int draw_any_circle_data(struct s_data *q,int hes,double r,complex ctr,
			 int cflag,int ccol,int col,int label,int show)
/* like draw_any_circle, except data r and ctr given directly, allow for
label. */
{
  int front;
  double stretch;

  if (hes < 0) 
    {
      h_to_e_data(ctr,r,&ctr,&r);
      if (cflag & 1) circle(q,ctr,r,cflag,ccol,col,show);
      if (cflag & 32 && r<0) /* horocycle */
	{
	  stretch=1.0+
	    (q->box.rx-q->box.lx)*15/
	    (q->pix_box.rx-q->pix_box.lx); 
	  /* shift label to miss unit circle */
	  ctr.re=ctr.re*stretch;
	  ctr.im=ctr.im*stretch;
	}
    }
  else if (hes > 0)
    {
      ctr=ss_view(q,ctr,1,&front);
      if (cflag & 1 && !front)
	return (s_circle(q,ctr,r,cflag,ccol,col,show));
      if (cflag & 1)
	s_circle(q,ctr,r,cflag,ccol,col,show);
      if (cflag & 32) ctr=s_pt_to_visual_plane(ctr);
    }
  else if (cflag & 1) circle(q,ctr,r,cflag,ccol,col,show);

  if (cflag & 32) shownumber(q,label,ctr,show);

  return 1;
} /* draw_any_circle_data */

int draw_any_dot(struct p_data *p,int v,double diam,int cf,
		 int shape_flag,int show)
/* draw shape at center of circle. diam gives size, cf = color 
flag (use circle color), shapes (so far) are fill/open dot or cross. */
{
  int dum_int;
  double rad;
  complex ctr;

  if (p->hes>0) /* sphere */
    {
      ctr=ss_view(p->screen,p->packR_ptr[v].center,1,&dum_int);
      if (cos(ctr.re)<=0) return 0; /* not on front */
      ctr.re=sin(ctr.im)*sin(ctr.re);
      ctr.im=cos(ctr.im);
    }
  else if (p->hes < 0) 
    h_to_e_data(p->packR_ptr[v].center,
            p->packR_ptr[v].rad,&ctr,&rad);
  else ctr=p->packR_ptr[v].center;
  if (!cir_ck(ctr,diam,p->screen->box)) return 0; /* not on screen */
  if (cf)
    return circle(p->screen,ctr,diam/2,1,p->packK_ptr[v].color,
		  p->packK_ptr[v].color,show);
  return circle(p->screen,ctr,diam/2,1,FG_COLOR,FG_COLOR,show);
} /* draw_any_dot */

int draw_circles(struct p_data *p,int cflag,int show)
/* draw whole pack. standard flag */
{
	int i,ccol=FG_COLOR,col=FG_COLOR;

	for (i=1;i<=p->nodecount;i++)
	  {
	    if (cflag & 16) col=p->packK_ptr[i].color;
	    if (cflag & 8)
	      {
		ccol=p->packK_ptr[i].color;
		if (ccol>=bc_min && ccol<=bc_max) ccol+=bc_offset;
	      }
	    draw_any_circle(p,i,cflag,ccol,col,show);
	  }
	return 1;
} /* draw_circles */

int circle(struct s_data *q,complex ctr,double r,int cflag,
	   int ccol,int col,int show)
{
  int diam,nx,ny;
  complex normpt;

  ctr.re -= r;ctr.im +=r;
  r_to_pix(ctr,&normpt,q->pix_box,q->box);
  diam=(int) 2*r*(q->pix_box.rx)/(q->box.rx-q->box.lx)+.5;
  if (diam > 20000  /* too big or rectangle misses pix_box */
      || (nx=(int)normpt.re)>q->pix_box.rx
      || nx+diam<0 
      || ((ny=(int)normpt.im)+diam)<0  
      || ny > q->pix_box.ry) return 0;
  if (diam<=0) diam=1;
  if (cflag & 2) 
    {
      if (!(cflag & 16))
	{
	  col=FG_COLOR;
	  if (cflag & 4) ccol=BG_COLOR;
	}
      FillArc(q->xpm,nx,ny,diam,diam,0,360*64,colors[col],1);
    }
  if (!(cflag & 8)) ccol=FG_COLOR;
  DrawArc(q->xpm,nx,ny,diam,diam,0,360*64,colors[ccol]);
  
  if (show) refresh_canvas(q);
  return 1;
} /* circle */

int polyarc(struct s_data *q,double rad,complex ctr,double a1,double a2)
     /* puts points in 'alist' for arc of circle
from angle a1 counterclockwise to a2 on screen q. Uses matrix
approximation to d.e. (suggested by Ohannes):  given pt (wr,wi) on
circle, step as proportion of 2pi, then next counterclkws pt is
(zr,zi)=A*(wr,wi), matrix A. Return number of pts in list. */
{
  int i,n;
  double ac,step,ss,a11,a12,a21,a22,zr,zi,wr,wi;
  complex pt,normpt;

  while ((ac=a2-a1)<0) a2 +=2.0*M_PI; /* make sure a2>a1 */
  n=ac/(2.0*M_PI)*10;
  n=n*10; /* n will be 0,10,20, etc. */
  if (n<10) n=10; /* tailor number of steps to proportion of circle */

  step=ac/n;
  ss=step*step/4;
  a11=(1-ss)/(1+ss);
  a12= (-step)/(1+ss);
  a21= (-a12);
  a22=a11; /* set up matrix A */

  zr=rad*cos(a1); zi=rad*sin(a1);
  n++;
  for (i=0;i<n;i++)
  {
    wr=zr;
    wi=zi;
    pt.re=wr+ctr.re;
    pt.im=wi+ctr.im;
    r_to_pix(pt,&normpt,q->pix_box,q->box);
    alist[i*2]=normpt.re;
    alist[i*2+1]=normpt.im;
    zr=a11*wr+a12*wi;
    zi=a21*wr+a22*wi;
  } 
  return (n);
} /* polyarc */

int unitcircle(struct s_data *q)
/* scales data for unit circle, screen q, and calls for circle */
{
  complex zero;
  
  zero.re=0;zero.im=0;
  return (circle(q,zero,1.0,0,FG_COLOR,0,1)); /* open circle */
} /* unitcircle */


