/* RePack.c 		Started version numbers 1/2000 	*/

/* This is a repack engine to be run separately or called from 
CirclePack; based on "remote_shell.c". Incorporates Chuck Collins 
modifications to packing algorithms. Also used as testbed for
algorithm changes before incorporation in CirclePack. Compile
with CirclePack libraries

Call standalone with

	"RePack <packname> -a d -c m -r n -f"

where d = number of digits of accuracy (d <= 15), m = limit on total 
repack cycles, n = cycles between status reports; Results are stored
in file <packname>p. The f flag directs the program to use the 
iterative (versus superstep) routines (as is automatic if the packing 
has specified inversive distances). 

Call from CirclePack with 

	"exec RePack p -a d -c m -r n -f", 

where p = pack number. All of d, m, and n are optional. The routine 
computes only the new radii, saving them with d places of accuracy. 
*/
 
/* ----- user specified variables set before compiling -------- */

#undef _KERNEL
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/resource.h>
#include "cp_types.h"
#include "cp_proto.h"

double toler=TOLER,okerr=OKERR,m_toler=M_TOLER;
int **face_org;

int cmdmode=0;
char emsgbuf[BUFSIZE],buf[BUFSIZE],*msgbuf; /* utility buffers for strings. */
double cycles=2000;	/* default number of repack cycles */
int localcycles=1000;	/* default interval between status reports */
int iterates=2;		/* parameter of current packing routines */
int digits=6;           /* digits of accurracy */
struct rusage RU;
int elapse_hr,elapse_min,elapse_sec;
double e_time;
long timein,elapsetime;
FILE *fp;

/* procedure prototypes */
int pack_Rewrite(int q);
int report_status(double cycount);
int emsg();
int msg();
int repack_activity_msg(char *msg);


/* -------- main routine ----------- */

int main(int argc,char *argv[])
{
  int retval,i,j,old_flag=0,read_key=0017,int_count=0;
  int p=0,limit_flag=0,stop_flag=0,nextcount;
  double cycount=0.0;
  char filename[256],inmsg[256];
  fd_set rfds;
  struct timeval tv;

		/* ---------- initialization --------------- */

  if (argc < 2)
    {
      printf("RePack usage: RePack <packname> -[acrf] \n");
      printf("   a=digits of accuracy\n");
      printf("   c=number of cycles through the packing\n");
      printf("   r=number of cycles between status reports\n");
      printf("   f=optional: means use old riffle, not superstep\n");
      exit(0);
    }
  strcpy(emsgbuf,"ERROR: ");
  msgbuf=emsgbuf+7;
  if ((packdata=(struct p_data *)
       malloc(sizeof(struct p_data)))==NULL)
    {
      sprintf(msgbuf,"RePack exited: insufficient memory.");
      emsg();
      exit(0);
    }
  packdata->overlap_status=0;
  packdata->faces=(f_data *)NULL;	

		/* -------- interpret arguments --------- */

  if (strlen(argv[1])==1) p=atoi(argv[1]); 	
                /* pack no. of input data from parent */
  else	/* must be filename; read/write data in file */
    {
      filename[0]='\0';
      strncpy(filename,argv[1],255);
      cmdmode=1;
    }
  i=2;
  while (i< argc) /* get other command line info */
    {
      if (!strcmp(argv[i],"-a"))
	{
	  if (sscanf(argv[++i],"%d",&digits) && digits>2)
	    {
	      if (digits>15) digits=15;
	      toler=0.1;
	      for (j=1;j<=digits;j++) toler *=0.1;
	      if (okerr>(.01)*toler) okerr=(.01)*toler;
	    }
	}
      else if (!strcmp(argv[i],"-c"))
	{
	  if (sscanf(argv[++i],"%lf",&cycles) ) 
	    { 
	      if (localcycles>cycles) localcycles=cycles;
	      limit_flag=1;
	    }
	}
      else if (!strcmp(argv[i],"-r"))
	{if (sscanf(argv[++i],"%d",&localcycles) ) ;}
      else if (!strcmp(argv[i],"-f")) 
	old_flag=1;
      i++;
    }

  if (!cmdmode) /* running remotely under CirclePack */
    {
		/* --- if called from CirclePack, lock packing there ---- */

      if (!cmdmode && !r_lock_set(p)) 
	{
	  sprintf(msgbuf,"RePack: lock failed.");
	  emsg();
	}
      
		/* -------- read data phase ---------- */

      if ( (packdata->sizelimit=r_pack_read(packdata,p,0017))<=0 
	   || (packdata->packK_ptr=
	       (struct K_data *)malloc((packdata->sizelimit+1)
		   *sizeof(struct K_data)))==NULL
	   || (packdata->packR_ptr=
	       (struct R_data *)malloc((packdata->sizelimit+1)
		   *sizeof(struct R_data)))==NULL
	   || !(read_key=readpack(stdin,packdata)) )
	{
	  sprintf(msgbuf,"RePack: read from parent failed.");
	  emsg();
	  exit(1);
	}
    }
  else /* running stand-alone; read from file */
    {
      printf("\n\n ================== \n"
	     "RePack: stand-alone computation of radii "
	     "for circle packing.\n");
      for (i=0;i<argc;i++) printf("%s ",argv[i]);
      printf("\n\n");
      printf("   Type 'stop' or 'exit' to intervene, saving "
	     "the results.\n\n");

      sprintf(msgbuf,"Reading %s.",filename);
      msg();
      if ((fp=fopen(filename,"r"))!=NULL) 
	{
	  packdata->sizelimit=r_get_pack_size(fp,packdata);
	  fclose(fp);
	}
      if (!packdata->sizelimit || (fp=fopen(filename,"r"))==NULL
	  || (packdata->packK_ptr=
	      (struct K_data *)malloc((packdata->sizelimit+1)
		 *sizeof(struct K_data)))==NULL
	  || (packdata->packR_ptr=
	      (struct R_data *)malloc((packdata->sizelimit+1)
		 *sizeof(struct R_data)))==NULL
	  || !(read_key=readpack(fp,packdata)) )
	{
	  sprintf(msgbuf,"RePack: read of file %s failed.",filename);
	  emsg();
	  if (fp) fclose(fp);
	  exit(2);
	}
      if (fp) fclose(fp);
      sprintf(msgbuf,
	"Reading %s is complete:\n       nodecount = %d, geometry = ",
	      filename,packdata->nodecount);
      if (packdata->hes<0) strcat(msgbuf,"hyperbolic.\n");
      else strcat(msgbuf,"euclidean.\n");
      msg();
      if (old_flag)
	{
	  sprintf(msgbuf,"User specified 'iterative' RePacking routine.");
	  msg();    
	}
    }

  fillcurves(packdata);

/* ======================= manipulations =================== */

  for (i=1;i<=packdata->nodecount;i++) /* set bdry flags, count int. */
    {
      if (packdata->packK_ptr[i].flower[0]==
	  packdata->packK_ptr[i].flower[packdata->packK_ptr[i].num])
	{
	  packdata->packK_ptr[i].bdry_flag=0;
	  int_count++;
	}
      else packdata->packK_ptr[i].bdry_flag=1;
    }
  if (old_flag) /* adjust method of counting passes */
    {
      cycles *= int_count;
      localcycles *= int_count;
    }
  getrusage(RUSAGE_SELF,&RU);
  timein=RU.ru_utime.tv_sec;
  while ( !stop_flag && (!limit_flag || cycount<cycles) )
    {
      if (packdata->hes < 0) /* hyperbolic */
	{
/*	  if (cycount<40)
	    nextcount=old_h_riffle(packdata,localcycles);
	  else nextcount=h_riffle(packdata,localcycles);
*/
	  if (packdata->overlap_status || old_flag) 
	    nextcount=old_h_riffle(packdata,localcycles);
	  else nextcount=h_riffle(packdata,localcycles);
	  if (nextcount==0) stop_flag=1;
	}
      else if (packdata->hes==0) /* euclidean */
	{
	  if (packdata->overlap_status || old_flag) 
	    nextcount=old_e_riffle(packdata,localcycles);
	  else nextcount=e_riffle(packdata,localcycles);
	  if (nextcount==0) stop_flag=1;
	}
      else if (packdata->hes>0) /* spherical, no routine */
	{
	  sprintf(msgbuf,
	    "RePack: no packing algorithm available in spherical geometry.");
	  emsg();
	  exit(3);
	}
      cycount += (double) nextcount;
      if (nextcount < localcycles) break;
      report_status(cycount);

      /* check for user input on stdin */
      FD_ZERO(&rfds);
      FD_SET(0, &rfds);
      tv.tv_sec = 0;
      tv.tv_usec = 0;
      retval = select(1, &rfds, NULL, NULL, &tv); /* check stdin */
      if (retval && scanf("%s",inmsg))
	{
	  if (sscanf(inmsg,"%s",buf) 
		   && (!strncmp(buf,"quit",4) || !strncmp(buf,"stop",4)
		       || !strncmp(buf,"exit",4)))
	    {
	      sprintf(msgbuf,"User interrupt detected; saving results");
	      msg();
	      stop_flag=1;
	    }
        } 
    } /* end of while */
  report_status(cycount);
  if (limit_flag && cycount >= cycles)
    sprintf(msgbuf,
     "RePack stopped due to specified cycle limits:\n"
	    "final count = %f  ",
      cycount);
  else sprintf(msgbuf,
    "\nRePack completed or interrupted:\n"
	       "final count = %f  "
    ,cycount);
  msg();

/* ===================== end of manipulations =============== */

  if (!cmdmode) /* send data to parent. */
    {

		/* -------- write data phase ---------------- */

      if (!pack_Rewrite(p)) /* send back to p only radii */
	{
	  sprintf(msgbuf,"RePack: write to parent failed.");
	  emsg();
	}

		/* -------- unlock phase ---------------- */

      if (!r_unlock(p))
	{
	  sprintf(msgbuf,"RePack: unlock failed.");
	  emsg();
	}
    }
  else /* store data in file */
    {
      strcat(filename,"p"); /* add extra p to filename */
      if ((fp=fopen(filename,"w"))==NULL 
	  || !writepack(fp,packdata,((read_key & 0717) | 0013),0) )
	/* note: don't write angle sums or centers, do write basic
	   comb data and radii, also write other things (such as
	   vertex_map) which were read in with the original packing */
	{
	  sprintf(msgbuf,"RePack: failed to write data to %s.",filename);
	  emsg();
	  fclose(fp);
	  exit(4);
	}
      sprintf(msgbuf,"RePack results are stored in file %s.\n",filename);
      msg();
      if (fp) fclose(fp);
    }

/* -------- exit ------ */

  sprintf(msgbuf,"RePack: finished successfully.");
  msg();
  exit(8);

} /* end of main */


/*=================== repacking routines =======================*/

int pack_Rewrite(int q)
/* ask parent to accept packing into its pack q 
and await acknowledge. */
{
	if (packdata->nodecount<=0)
		return 0;
	sprintf(buf,"sending pack %d %d\n",q,packdata->nodecount);
	fputs(buf,stdout);
	fflush(stdout);
	*buf='\n';
	while (*buf=='\n') fgets(buf,1024,stdin); 
	sscanf(buf,"%s",msgbuf);
	if (strcmp(msgbuf,"ready")!=0) return 0; 
		/* can add more analysis of return message */
	if (!writepack(stdout,packdata,0010,0)) return 0; 
	*buf='\n';
	while (*buf=='\n') fgets(buf,1024,stdin); 
	sscanf(buf,"%s",msgbuf);
	if (strncmp(msgbuf,"failed",6)==0) return 0;
		/* can add more analysis of return message */
	return 1;
} /* pack_Rewrite */

int report_status(double cycount)
/* print out status of packing comps.
fix?? need to check incompats for eucl case. Need report
on inv_dist errors for inv_dist cases. */
{
  int incompat,i,flag=0,aimcount=0;
  double accum=0.0,worst=0.0,hold,curv;

  getrusage(RUSAGE_SELF,&RU);
  elapsetime=RU.ru_utime.tv_sec-timein;	
  e_time=(double)elapsetime;
  elapse_hr=(int)floor((double)(e_time/3600.0));
  elapse_min=(int)floor((double)(e_time-elapse_hr*3600.0)/60.0);
  elapse_sec=(int)(e_time-elapse_hr*3600.0-elapse_min*60.0);
  sprintf(msgbuf,"\nElapsed time:  %02d:%02d:%02d (hh:mm:ss),  count = %f",
	  elapse_hr,elapse_min,elapse_sec,cycount);
  msg();
/* error feedback */
  fillcurves(packdata);
  incompat=0;
  for (i=1;i<=packdata->nodecount;i++)
    if (packdata->packR_ptr[i].aim >= 0)
      {
	if (packdata->hes<0)
	  {
	    h_anglesum_overlap(packdata,i,
	      packdata->packR_ptr[i].rad,&curv,&flag);
	    if (flag) incompat++;
	    else
	      {
		hold=fabs(curv-packdata->packR_ptr[i].aim);
		accum += hold;
		worst=(hold>worst) ? hold : worst;
		aimcount++;
	      }
	  }
	else
	  {
	    hold=fabs(packdata->packR_ptr[i].curv-
		      packdata->packR_ptr[i].aim);
	    accum += hold;
	    worst = (hold > worst) ? hold : worst;
	    aimcount++;
	  }
      }
  sprintf(msgbuf,
    "\n     Worst curv error, %d vertices: % .15e"
	  "\n     Accumulated curv error:        % .15e\n",
	  aimcount,worst,accum);
  msg();
  if (incompat)
    {
      sprintf(msgbuf,"%d vertices had incompatibilities.\n",incompat);
      msg();
    }
  return 1;
} /* report_status */

int emsg()
{
  if (!cmdmode)
    {
      stripsp(msgbuf);
      r_emsg(msgbuf);
    }
  else printf(emsgbuf);
  return 1;
}

int msg()
{
  if (!cmdmode) 
    {
      stripsp(msgbuf);
      r_msg(msgbuf);
    }
  else {printf(msgbuf);fsync((int)stdout);}
  return 1;
}

int repack_activity_msg(char *msg)
{
  return 1; /* dummy call */
}
