/* PlugPack.c   Routines to recursively decompose complexes for
repacking. 

Idea: measure verts of (simply connected) packing K up to n (depth) 
generations from bdry(K); store the remaining 'n-islands', 
L1,..,Lj, as separate packings; continue in K to n+m generations 
from bdry(K) (m the number of overlap generations), cut out and 
plug the (n+m)-islands left, each with a single ideal vertex, to 
get K'. Now, max pack K', use the resulting boundary radii to 
pack L1,..,Lj, then blend the Li and K' to get packing for original K. 

Recursive: set up whole thing to act recursively so same method 
can be applied to the islands at each stage. Store some sort of
pack_nest structure. Store packings in separate files, along with 
pointers to children and parent; each island has vertex_map 
vis-a-vis its parent. 

Construction of tree structure and packing should be separate
issues. (Should vertex_map be to parent or its plugged version?)

User can choose depth to which to pack, perhaps, eg, to blow up 
some local area; should be able to construct and manipulate a 
"plugged" complex Kt associated with any subtree t, so if 
T is the full structure tree, then KT = original K.

Pack a tree t from root out, since one needs bdry values from
parent before packing its islands.

NAMING CONVENTIONS: 
  Reading in <packname>.p and desired 'depth': 
  * store associated plugged packing as <packname>.plug
  * store j-th island as <packname>_j.p
  * add this 'pack_tree' info to the end of <packname>.plug:

  PACK_PLUG_TREE: 
    PARENT: packname
    GENERATION_DEPTH: d
    OVERLAP: k
    PLUG_COUNT: m
    CHILD_COUNT: n
*/
/* TODO:

 * fix all error messages and exit strategies
 * add commands to *.cmd file for better display of results.

 */

#include <time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/resource.h>
#include "cp_types.h"
#include "cp_proto.h"
int **face_org;
double okerr=OKERR,toler=TOLER;

struct K_tree
{
  char name[NAME_MAX];      /* parent packing name */
  int child_count;          /* number of islands */
  struct K_tree *parent;    /* pointer to parent */
  struct K_tree **child;    /* ptrs to children,  */
};

struct TreeList
{
  struct K_tree *tree;      /* ptr to K_tree */
  struct TreeList *next;    /* linked list */
};

struct K_data *pK;   /* utility pointers for packdata */
struct R_data *pR;

FILE *fpcmd;

/* variables needed for various libraries */
int iterates=2;
int num_plot=40;
Mobius Mob;  /* initialize with Mob=ident_mob(); */
unsigned long int rand_next;

char emsgbuf[BUFSIZE],next[BUFSIZE],buf[BUFSIZE],*msgbuf;
char scn[128];

/* procedure prototypes */
int plugs_islands(struct p_data *p_top,char *top_name,int depth,
		  int target_size,int over_lap,struct K_tree *tree_ptr);
int set_scn(char *name);
int msg();
int emsg();
int repack_activity_msg(char *msg);


/* ==================================================================== 
   main routine
 ==================================================================== */

main(int argc,char *argv[])
{
  int i,depth=-1,target_size=-1,over_lap=1,recur_aim=1;
  int recur_count=1,count;
  int child_count=1,child=1,abandon_count=0;
  char packname[NAME_MAX],scriptname[NAME_MAX];
  FILE *fp;
  struct p_data *p_child;
  struct K_data *pK_child;
  struct K_tree *tree_ptr;
  struct TreeList *new_islands=NULL,*old_islands=NULL;
  struct TreeList *trace_new,*trace_old;

  packname[0]='\0';
  strcpy(emsgbuf,"ERROR: ");
  msgbuf=emsgbuf+7;

  if (argc < 4)
    {
      printf("PlugPack usage: PlugPack -[dotr] <packname> \n");
      printf("   d=depth (in generations) from outer bdry to island\n");
      printf("   o=overlap (in generations) of parent with islands\n");
      printf("   t=optional target size for plugged packings\n");
      printf("   r=optional max recursions, default 1\n");
      exit(1);
    }

  printf("PlugPack started.\n");
  if (!(packdata=(struct p_data *)calloc((size_t)1,sizeof(struct p_data)))
      || !(p_child=(struct p_data *)calloc((size_t)1,sizeof(struct p_data)))
      || !alloc_pack_space(packdata,5000,0)
      || !alloc_pack_space(p_child,5000,0))
    exit(5);

  pK=packdata->packK_ptr;
  pK_child=p_child->packK_ptr;
  i=1;

  /* read original complex, depth of islands for plugging, overlap 
     of islands with the plugged parent, how many recursion
     levels to pursue, etc; original packing needs to be simply 
     connected -- islands should then automatically be simply
     connected. */
  while (i<argc)
    {
      if (!(strncmp(argv[i],"-d",2))) 
	depth=atoi(argv[1+i++]);       /* depth */
      else if (!(strncmp(argv[i],"-o",2)))
	{ 
	  if ((over_lap=atoi(argv[1+i++]))<1)
	    over_lap=1;                /* overlap generations */
	}
      else if (!(strncmp(argv[i],"-t",2)))
	target_size=atoi(argv[1+i++]); /* target size for islands */
      else if (!(strncmp(argv[i],"-r",2))) 
	{
	  if ((recur_aim=atoi(argv[1+i++]))<1)
	    recur_aim=1;               /* max recursions, default = 1 */
	}
      else strncpy(packname,argv[i],255);
      i++;
    }
  if (target_size<0 && depth<0) depth=5; /* default */

  /* open file for CirclePack script */

  strcpy(scriptname,packname);
  strcat(scriptname,".cmd");
  if (!(fpcmd=fopen(scriptname,"w"))) exit(14);

  fprintf(fpcmd,"Script file for plug-island decomposition of a packing"
	  " generated by PlugPack\n"
	  "   Specified depth for islands = %d,\n"
	  "   Number of generations of overlap = %d,\n"
	  "   Target size of plugged packings = %d,\n"
	  "   Recursion level requested = %d\n\n",
	  depth,over_lap,target_size,recur_aim);

  fprintf(fpcmd,"\t[]:=act 0;set_accur .000000000001;set_dir;\n\n");

  fprintf(fpcmd,"Top level packing\n\n"
	  "\t[top]:=read %s;set_plot_flags 0 a;disp -w -u;\n",
	  packname);

  fprintf(fpcmd,"\nFollowing are the various plugged, terminal, and "
	  "abandoned islands.\n\n");
  /* use scn to keep track of labels for script cmd lines */
  sprintf(scn,"0"); 

  /* Want to recursively process the islands, starting with the original
     pack. Need list of current islands (old_islands), and as we process
     these, create the list of new islands (new_islands) to be visited
     at the next recursion level (up to max level requested). Note that
     there are other ways to run this, eg., depth first construction
     might be more appropriate for homing in on specific region. */

  /* create list of islands to process */
  new_islands=trace_new=(struct TreeList *)
    calloc((size_t)1,sizeof(struct TreeList));

  /* start with original pack */
  tree_ptr=(struct K_tree *)calloc((size_t)1,sizeof(struct K_tree));
  strcpy(tree_ptr->name,packname);
  old_islands=trace_old=(struct TreeList *)
    calloc((size_t)1,sizeof(struct TreeList));
  old_islands->tree=tree_ptr;
  new_islands=trace_new=(struct TreeList *)
    calloc((size_t)1,sizeof(struct TreeList));
  while(old_islands && recur_count<=recur_aim) 
    {
      if (!(fp=fopen(old_islands->tree->name,"r"))) goto BOMB;
      if (!(readpack(fp,packdata))) {fclose(fp);goto BOMB;}
      if (packdata->euler!=1 || packdata->genus) /* not a topological disc */
	{fclose(fp);goto BOMB;}
      fclose(fp);
      printf("read packing %s, nodecount %d.\n",
	     old_islands->tree->name,packdata->nodecount);
      /* while islands found? create K_trees, add to new_islands. */
      set_scn(old_islands->tree->name);
      if (plugs_islands(packdata,old_islands->tree->name,
			 depth,target_size,over_lap,old_islands->tree))
	{
	  for (i=1;i<=old_islands->tree->child_count;i++)
	    {
	      trace_new=trace_new->next=(struct TreeList *)
		calloc((size_t)1,sizeof(struct TreeList));
	      trace_new->tree=(old_islands->tree->child)[i];
	    }
	  fprintf(fpcmd,"\t[%s]:=act 1;read %slug;\t# (plugged island)\n",
		  scn,old_islands->tree->name);
	}
      else fprintf(fpcmd,"\t[%s]:=act 1;read %s;\t# (terminal island)\n",
		   scn,old_islands->tree->name);
      free(old_islands->tree);
      old_islands->tree=NULL;
      trace_old=old_islands;
      old_islands=old_islands->next;
      free(trace_old);
      if (!old_islands && new_islands->next) /* done going through 
						old list and have new? */
	{
	  fprintf(fpcmd,"\nMoving to level %d.\n\n",recur_count);
	  old_islands=new_islands->next; /* first entry of new_islands 
					always empty */
	  free(new_islands);
	  new_islands=trace_new=(struct TreeList *)
	    calloc((size_t)1,sizeof(struct TreeList));
	  if (old_islands) recur_count++; /* more to do? */
	}
      
    } /* end of while */
  sprintf(buf,"\nPlugPack completed %d recursions successfully, "
	  "depth=%d, overlap=%d.\n",recur_count,depth,over_lap);
  printf(buf);
  fprintf(fpcmd,buf);
  if (old_islands) 
    {
      fprintf(fpcmd,"However, the following islands were abandoned "
	      "(not checked for plugging).\n");
      trace_old=old_islands;
      old_islands=old_islands->next;
      set_scn(trace_old->tree->name);
      fprintf(fpcmd,"\t[%s]:=act 1;read %s;\t# (abandoned island)\n",
	      scn,trace_old->tree->name);
      free(trace_old);
      abandon_count++;
    }
  while (new_islands) /* clean up */
    {
      trace_new=new_islands;
      new_islands=new_islands->next;
      free(trace_new);
    }
 WRAPUP:
  fprintf(fpcmd,"\n-------------------- tools "
	  "-----------------------\n\n"
	  "  [m}:= put_data -p1 -q0 -t -r a;put_data -p1 -q0 -t -z a;[f];\n"
	  "     # move top level plugged data\n"
	  "  [g}:= get_data -p1 -q0 -t -r b;\n"
	  "     # get bdry data for island\n"
	  "  [b}:= blend -p1 -q0 0 %d;[f];\n"
	  "     # blend and set plot_flags\n" 
	  "  [f}:= map -p1 a;set_plot_flags -p0 1 Vlist;\n"
	  "     # set plot_flags\n"
	  "  [p}:= exec RePack 1 -a 10 -c 2000 -r 500;\n"
	  "  [r}:= set_plot_flags -p0 0 a\n"
	  "     # send for remote repacking\n(end)\n",
	  over_lap);
	  
  fclose(fpcmd);
  if (abandon_count)
    {
      sprintf(buf,"However, %d islands were abandoned "
	      "(not checked for plugging).",abandon_count);
      printf(buf);
    }
  exit(1);

 BOMB:
  fclose(fpcmd);
  printf("\nBombed somehow.\n");
  exit(0);

} /* main */

/* ============================================================== */

int plugs_islands(struct p_data *p_top,char *top_name,int depth,
		  int target_size,int over_lap,struct K_tree *tree_ptr)
/* Here's where the work is done, various files generated, names
derived from top_name, K_tree structures created and filled, etc.; 
this will be called recursively, as desired. (The only non-recursive
aspect is that the vertex_maps generated point all the way back to
the original packing, not to the parent at this stage. This is for 
convenience with reconstruction.) Return 0 on error. */
 {
  int i,j,k,m,w,v=0,outer_seed=0,err_val=0,island_depth,max_depth;
  int hits,outer,island_count=0,plug_count=0,total=0,mark_count,min_gen;
  int *top_map=NULL,*newold=NULL;
  char newname[NAME_MAX],root_name[NAME_MAX],*endptr;
  FILE *fp=NULL;
  struct Vertlist *seed_list=NULL;
  struct Edgelist *trace;
  struct p_data *p=NULL;
  struct K_data *pK_top,*pK_ptr;
  struct K_tree *child_ptr=NULL;

  pK_top=p_top->packK_ptr;

  /* get root_name */
  i=strcspn(top_name,".");
  strncpy(root_name,top_name,i);
  root_name[i]='\0';

  /* ------------------------------ Create plugged packing ------ */

  /* mark generations in p_top from outer boundary, labeling util_flags.
     See how many vertices are picked up when through generation mark_count
     and get min_gen, the depth required to get target_size. 
     After copying p_top to p, have to decide depths to go to mark
     poison verts. */

  mark_count= ((2*over_lap)>(depth+over_lap)) ? 
	       2*over_lap : depth+over_lap;
  sprintf(buf,"b");
  if (!(seed_list=Node_link_parse(p_top,buf,&endptr,&hits,
	      &Vlist,&Elist,&Flist,&region,pathlist,pathlength))
      || !(max_depth=incr_generations(p_top,&seed_list,&mark_count,
				      &min_gen,target_size))
      || !(p=(struct p_data *)calloc((size_t)1,sizeof(struct p_data)))
      || !copy_pack(p_top,p))
    {
      if (seed_list) vert_free(&seed_list);
      if (p) free_p_data(&p);
      return 0;
    }
  pK_ptr=p->packK_ptr;
  
  /* now for the decisions and marking of poisons. */
  if (target_size>0 && depth>0)
    island_depth=((min_gen-over_lap) >= depth) ?
      (min_gen-over_lap) : depth;
  else if (target_size > 0)
    island_depth=min_gen-over_lap;
  else island_depth=depth; /* depth should be set if target_size isn't */

  if (max_depth<(island_depth+over_lap)) /* not worth plugging */
    {
      free_p_data(&p);
      return 0;
    }

  /* for efficiency, put vertex_map in array */
  if (trace=p_top->vertex_map) 
    {
      top_map=(int *)calloc((size_t)(p_top->nodecount+1),sizeof(int));
      while (trace)
	{
	  top_map[trace->v]=trace->w;
	  trace=trace->next;
	}
    }

  /* need to set util_flags of p according to depth, then remove 
     isolated or dendritic chains of poison verts; the latter may 
     take more than one pass. */

  for (i=1;i<=p->nodecount;i++) 
    if (pK_top[i].util_flag>=(island_depth+over_lap)) 
      pK_ptr[i].util_flag=-1;
  hits=1;
  while (hits)
    {
      hits=0;
      for (i=1;i<=p->nodecount;i++)
	if (pK_ptr[i].util_flag==-1)
	  {
	    k=0;
	    for (j=0;j<=pK_ptr[i].num;j++) 
	      k+= (pK_ptr[pK_ptr[i].flower[j]].util_flag==-1);
	    if (k<2) /* isolated or single poison neighbor */
	      {
		pK_ptr[i].util_flag=0;
		hits++;
	      }
	  }
    }

  /* find a seed vert for plugged packing */
  i=1;
  while (i<=p->nodecount)
    {
      if (!pK_ptr[i].bdry_flag && pK_ptr[i].util_flag > 0)
	{
	  outer_seed=i;
	  i=p->nodecount+1;
	}
      else i++;
    }

  /* util_flags of -1 interpreted as poison in call to cookie */
  sprintf(buf,"-v %d",outer_seed);
  if (!outer_seed || !(cookie_cutter(p,buf,&newold))) /* no islands? */
    {
      tree_ptr->child_count=0;
      if (newold) free(newold);
      if (p) free_p_data(&p);
      return 0;
    }
  if (!complex_count(p,1) || p->num_bdry_comp<=1) exit(7);
  pK_ptr=p->packK_ptr;

  /* repeatedly plug punctured complex until it has only one bdry
     component (the outer bdry) and write the resulting plugged packing */

  /* need to be able to pick out the original outer bdry. */
  plug_count=p->num_bdry_comp-1;
  while (p->num_bdry_comp>1)
    {
      hits=0;
      outer=0;
      do
	{
	  outer++;
	  w=p->bdry_starts[outer];
	  if (pK_top[newold[w]].bdry_flag) hits=1;
	} while (outer<p->num_bdry_comp && !hits);
      if (!hits) exit(10);

      /* plug either bdry 1 (if outer!=1) or 2 */
      if (!ideal_bdry_node(p,p->bdry_starts[1+(outer==1)])) exit(15);
      complex_count(p,0);
    } /* end of while: should have plugged all islands */
  set_aim_default(p); 
  set_default_overlaps(p);
  
  /* create vertex_map to refer to original packing (if p_top
     has vertex_map) or to parent. */
  if (top_map && newold) 
    {
      p->vertex_map=trace=(struct Edgelist *)
	calloc((size_t)1,sizeof(struct Edgelist));
      for (i=1;i<=p->nodecount;i++)
	if (w=top_map[newold[i]])
	  {
	    trace=trace->next=(struct Edgelist *)
	      calloc((size_t)1,sizeof(struct Edgelist));
	    trace->v=i;
	    trace->w=w;
	  }
      trace=p->vertex_map;
      p->vertex_map=p->vertex_map->next;
      free(trace);
    }
  else if (newold)
    {
       p->vertex_map=trace=(struct Edgelist *)
	calloc((size_t)1,sizeof(struct Edgelist));
      for (i=1;i<=p->nodecount;i++)
	if (w=newold[i])
	  {
	    trace=trace->next=(struct Edgelist *)
	      calloc((size_t)1,sizeof(struct Edgelist));
	    trace->v=i;
	    trace->w=w;
	  }
      trace=p->vertex_map;
      p->vertex_map=p->vertex_map->next;
      free(trace);
    }

  /* save resulting packing (tree structure, basic data, and 
     vertex_map (to original top packing) using extension ".plug" */
  strcpy(newname,root_name);
  strcat(newname,".plug");
  strcat(p->file_name,newname);
  /* fixup: should set radii, depends on p->hes */
  if (!(fp=fopen(newname,"w")) || !writepack(fp,p,0103,0))
    exit(21);
  fclose(fp);
  sprintf(msgbuf,"  writing %s, %d circles.\n",
	  newname,p->nodecount);
  msg();

  /* successively identify n-islands and define vertex_map (via
     the parent) to the original top packing; look for vert 
     of generation island_count+over_lap and use it as seed to cut out 
     island. (Keep track of verts taken care of by setting p_top 
     mark.) fixup: what to avoid islands with few verts --- have 
     to do some work to identify them. */

  for (i=1;i<=p_top->nodecount;i++) pK_top[i].mark=0;
  island_count=0;
  do
    {
      hits=0;
      copy_pack(p_top,p); 
      pK_ptr=p->packK_ptr;
      /* make those within island_depth of bdry into poison */
      for (i=1;i<=p_top->nodecount;i++) 
	{
	  if (pK_top[i].util_flag<=island_depth) 
	    pK_ptr[i].util_flag=-1;
	  else pK_ptr[i].util_flag=0;
	}

      /* find next seed using p_top info */
      for (i=1;i<=p_top->nodecount;i++)
	if (pK_top[i].util_flag==(island_depth+over_lap) 
	    && !pK_top[i].mark)
	  {
	    island_count++;
	    sprintf(buf,"_%d.p",island_count);
	    strcpy(newname,root_name);
	    strcat(newname,buf);
	    strcpy(p->file_name,newname);
	    sprintf(buf,"-v %d",i);
	    if (!cookie_cutter(p,buf,&newold)) exit(24);
	    /* have to create new pack's vertex_map via p_top's to
	       get back to original packing (unless p_top IS the original,
	       in which case it wouldn't have a vertex_map). */
	    if (top_map && newold)
	      {
		p->vertex_map=trace=(struct Edgelist *)
		  calloc((size_t)1,sizeof(struct Edgelist));
		for (i=1;i<=p->nodecount;i++)
		  if (w=top_map[newold[i]])
		    {
		      trace=trace->next=(struct Edgelist *)
			calloc((size_t)1,sizeof(struct Edgelist));
		      trace->v=i;
		      trace->w=w;
		    }
		trace=p->vertex_map;
		p->vertex_map=p->vertex_map->next;
		free(trace);
	      }
	    else if (newold)
	      {
		p->vertex_map=trace=(struct Edgelist *)
		  calloc((size_t)1,sizeof(struct Edgelist));
		for (i=1;i<=p->nodecount;i++)
		  if (w=newold[i])
		    {
		      trace=trace->next=(struct Edgelist *)
			calloc((size_t)1,sizeof(struct Edgelist));
		      trace->v=i;
		      trace->w=w;
		    }
		trace=p->vertex_map;
		p->vertex_map=p->vertex_map->next;
		free(trace);
	      }

	    if (!(fp=fopen(newname,"w")) || !writepack(fp,p,0103,0))
	      exit(25);
	    fclose(fp);
	    sprintf(msgbuf,"  writing %s, %d circles.\n",
		    newname,p->nodecount);
	    msg();

	    /* set mark of p_top */
	    if (newold)
	      for (i=1;i<=p->nodecount;i++)
		if (w=newold[i]) pK_top[w].mark=1;

	    hits=1;
	    i=p_top->nodecount+1; /* to leave for-loop */
	  }
    } while (hits); /* end of do */
  
  /* build K_tree structures for islands */

  tree_ptr->child_count=island_count;
  tree_ptr->child=(struct K_tree **)
    calloc((size_t)(island_count+1),sizeof (struct K_tree *));
  for (i=1;i<=island_count;i++)
    {
      (tree_ptr->child)[i]=child_ptr=(struct K_tree *)
	calloc((size_t)1,sizeof(struct K_tree));
      sprintf(buf,"_%d.p",i);
      strcpy(child_ptr->name,root_name);
      strcat(child_ptr->name,buf);
    }
  /* add tree info at end of .plug pack */
  strcpy(newname,root_name);
  strcat(newname,".plug");
  if (!(fp=fopen(newname,"a"))) exit(29);
  fprintf(fp,"\nPACK_PLUG_TREE:\n"
	  "   PARENT: %s\n"
	  "   GENERATION_DEPTH: %d\n"
	  "   OVERLAP: %d\n"
	  "   PLUG_COUNT: %d\n"
	  "   CHILD_COUNT: %d\n\n",
	  root_name,depth,over_lap,plug_count,island_count);
  fclose(fp);

  if (newold) free(newold);
  if (top_map) free(top_map);
  if (p) free_p_data(&p);
  return island_count;

 ERROR_RETURNS:
  switch (err_val)
    {
    case 5:
      {
	sprintf(msgbuf,"");
	break;
      }
    case 10:
      {
	sprintf(msgbuf,"");
	break;
      }
    case 15:
      {
	sprintf(msgbuf,"");
	break;
      }
    case 20:
      {
	sprintf(msgbuf,"");
	break;
      }
    } /* end of switch */
  emsg();
  if (p) free_p_data(&p);
  return 0;
} /* plugs_islands */

int set_scn(char *name) 
/* get script cmd-line label from pack name: 
chars between first _ and first . */
{
  int i=0;
  char *ptr=NULL;

  if ((ptr=strpbrk(name,"_")))
    {
      while (ptr[i]!='.' && ptr[i]!='\0')
	  scn[i]=ptr[i++];
      scn[i]='\0';
    }
  else 
    {
      scn[0]='0';
      scn[1]='\0';
    }
  return 1;
} /* set_scn */

int msg()
/* print message stored in 'msgbuf' */
{
  printf(msgbuf);
}

int emsg()
/* print error message stored in 'msgbuf' */
{
  printf(emsgbuf);
}

int repack_activity_msg(char *msg)
{
  printf(msg);
}
