/*
 * Copyright 2009, Google Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * NaCl Safety Macro Definitions
 */
#ifndef NATIVE_CLIENT_SRC_INCLUDE_NACL_MACROS_H_
#define NATIVE_CLIENT_SRC_INCLUDE_NACL_MACROS_H_ 1

#include <stdlib.h>

/*****************************************************************************
 * Safety macros                                                             *
 *****************************************************************************/

#define NACL_ARRAY_SIZE_UNSAFE(arr) ((sizeof arr)/sizeof arr[0])

/*
 * ASSERT_IS_ARRAY(arr) generates a somewhat opaque compile-time
 * error if arr is a non-array pointer.  This protects against
 * situations where one writes code like:
 *
 * foo.h:  struct Foo { char buffer[BUFFERSIZE]; size_t sofar; ... };
 *
 * foo.c:  got = read(d, fp->buffer + fp->sofar, sizeof fp->buffer - fp->sofar);
 *         if (-1 == got) { ... }
 *         fp->sofar += got;
 *
 *         for (ix = 0; ix < sizeof arr/sizeof arr[0]; ++ix) { ... }
 *
 * and have it break and create a security problem when somebody later
 * changes Foo to dynamically allocate buffer, viz,
 *
 * foo.h:  struct Foo { char *buffer; size_t sofar; ... };
 *
 * and now sizeof fp->buffer is 4 or 8, with size_t (type of sizeof)
 * being unsigned, when fp->sofar is larger than 4 or 8, getting an
 * enormous maximum read size being used.  Such bugs can remain
 * undiscovered when conforming implementations of protocol engines
 * are used where the actual amount sent is small and would never
 * cause a buffer overflow, but an adversarial implementation would be
 * able to clobber the heap.  The solution is to write:
 *
 * foo.c:  NACL_ASSERT_IS_ARRAY(fp->buffer);
 *         got = read(d, fp->buffer + fp->sofar, sizeof fp->buffer - fp->sofar);
 *         if (-1 == got) { ... }
 *         fp->sofar += got;
 *
 *         for (ix = 0; ix < NACL_ARRAY_SIZE(arr); ++ix) { ... }
 *
 * and when foo.h is modified, it will generate a compile-time error
 * alerting the engineer makin the change that the read code will need to
 * be modified.
 *
 * NB: The -pedantic flag is REQUIRED for the C version to catch the
 *     error.  No special warning flags are required for the C++
 *     version to work.
 */

#ifdef __cplusplus
/*
 * C++ version is taken from chrome's basictypes.h, and renamed to
 * avoid collision in case of multiple includes.  NACL_ARRAY_SIZE
 * relies on template matching failure if the argument is not an
 * array.
 */
template <typename T, size_t N>
char (&NaClArraySizeHelper(T (&array)[N]))[N];

#ifndef _MSC_VER
template <typename T, size_t N>
char (&NaClArraySizeHelper(const T (&array)[N]))[N];
#endif  /* _MSC_VER */

#define NACL_ARRAY_SIZE(array) (sizeof(NaClArraySizeHelper(array)))

/*
 * Dead code elimination will get rid of this if there are no
 * compile-time errors generated by ARRAY_SIZE.
 */
#define NACL_ASSERT_IS_ARRAY(array)                     \
  do {                                                  \
    char __array__[NACL_ARRAY_SIZE(array)];             \
    if (0 == sizeof __array__) {                        \
      abort();                                          \
    }                                                   \
  } while (0)

#else  /* __cplusplus */

/*
 * The C version uses the fact that __builtin_types_compatible_p can
 * be used to discriminate between T * and T *const.  (Note that this
 * difference is not a top-level qualifier difference as mentioned in
 * the gcc info node; that would apply to T * versus T const *.)  In
 * the assertion statement version (NACL_ASSERT_IS_ARRAY), we use this
 * to allocate an array, and ISO C forbids a zero-sized array.  In the
 * expression version (ARRAY_SIZE), we assign to a global void * --
 * assigning a zero is fine, but assigning a 1 results in a warning
 * that making a pointer from an integer is verboten.  When ARRAY_SIZE
 * is used in a loop control context, e.g.,
 *
 * for (ix = 0; ix < ARRAY_SIZE(arr); ++ix) { ... }
 *
 * with -O the optimizer recognizes that the store can be moved out of
 * the loop, so the performance impact should be minimal.
 */
# if __GNUC__
#  define NACL_ASSERT_IS_ARRAY(arr)                           \
  do {                                                        \
    char __is_array__[!__builtin_types_compatible_p(          \
        __typeof__(&arr[0]),                                  \
        __typeof__(arr))];                                    \
    /* dead code, but gets rid of unused-variable warnings */ \
    if (0 == sizeof __is_array__) {                           \
      abort();                                                \
    }                                                         \
  } while (0)

static inline void *NaClArrayCheckHelper(void *arg) {
  /*
   * Doing runtime checks is not really necessary -- this code is in
   * fact unreachable code that gets optimized out when used with the
   * NACL_ARRAY_SIZE definition below.
   *
   * The runtime check is only useful when the build system is using
   * the inappropriate flags (e.g., missing -pedantic -Werror or
   * -pedantic-error), in which case instead of a compile-time error,
   * we'd get a runtime error.
   */
  if (NULL != arg) {
    abort();
  }
  return arg;
}

#  define NACL_ARRAY_SIZE(arr)                                         \
  (NaClArrayCheckHelper(                                               \
      __builtin_types_compatible_p(__typeof__(&arr[0]),                \
                                   __typeof__(arr))),                  \
  NACL_ARRAY_SIZE_UNSAFE(arr))
# else  /* __GNUC__ */

/*
 * Not gcc.  So far, we only compile NaCl under gcc and visual studio,
 * but if/when a new compiler is introduced that's capable of doing
 * compile-time checking (or we figure out how to do it w/ visual
 * studio), check for those compilers here, and enable the
 * corresponding compile-failure tests in
 * src/trusted/service_runtime/build.scons.
 */

#  define NACL_ASSERT_IS_ARRAY(arr)
#  define NACL_ARRAY_SIZE(arr) NACL_ARRAY_SIZE_UNSAFE(arr)
# endif  /* __GNUC__ */
#endif  /* __cplusplus */

/*
 * NACL_ASSERT_IS_POINTER(arr) generates a somewhat opaque compile-time
 * error if lvalue is not a pointer lvalue but is instead an actual
 * array (which is a T * const object).  This is complementary to
 * NACL_ASSERT_IS_ARRAY.
 */
#define NACL_ASSERT_IS_POINTER(ptr) do { if (0) { ++ptr; } } while (0)

/*
 * NACL_ASSERT_SAME_SIZE(t1, t2) verifies that the two types have the same size
 * (as reported by sizeof).  When the check fails it generates a somewhat
 * opaque warning, mitigated by the variable's name.
 *
 * Examples:
 *   NACL_ASSERT_SAME_SIZE(void *, char *);  // Likely to succeed!
 *   NACL_ASSERT_SAME_SIZE(char, long);  // Unlikely to succeed
 */
#define NACL_ASSERT_SAME_SIZE(t1, t2) \
  do { char tested_types_are_not_the_same_size[sizeof(t1) == sizeof(t2)]; \
       (void) tested_types_are_not_the_same_size; } while (0)


/*****************************************************************************
 * MAX/MIN macros for integral types                                         *
 ****************************************************************************/

/*
 * For NACL_MAX_VAL, T must be a type where u ## T is the unsigned
 * version of the type.
 *
 * These macros rely on -1 being signed extended to the width of T (or
 * u ## T), and on two's complement representation of integers.
 *
 * Generally, stdint.h's INT16_MAX etc can be used, but these are
 * useful for macros that take a type parameter and need the max or
 * min value for the type, since then the macro would not have to also take
 * the max or min value as additional parameter(s).
 */
#define NACL_UMAX_VAL(T)  ((T) -1)
#define NACL_MAX_VAL(T)   ((T) (((u ## T) -1) >> 1))
#define NACL_UMIN_VAL(T)  ((T) 0)
#define NACL_MIN_VAL(T)   ((T) ~NACL_MAX_VAL(T))


/*****************************************************************************
 * Readability macros                                                        *
 ****************************************************************************/

#define NACL_NANOS_PER_MICRO          1000
#define NACL_100_NANOS_PER_MILLI      (10 * 1000)
#define NACL_NANOS_PER_MILLI          (1000 * 1000)
#define NACL_MICROS_PER_UNIT          (1000 * 1000)
#define NACL_MILLIS_PER_UNIT          1000
#define NACL_UNIT_CONVERT_ROUND(v, m) (((v) + (m) - 1)/(m))

/*****************************************************************************
 * C++ coding convention macros                                              *
 ****************************************************************************/

#ifdef __cplusplus
/*
 * A macro to disallow the copy constructor and operator= functions
 * This should be used in the private: declarations for a class
 */
#define NACL_DISALLOW_COPY_AND_ASSIGN(TypeName) \
    TypeName(const TypeName&);                  \
    void operator=(const TypeName&)

/* A macro to use in place of unimplemented sections of code */
#define NACL_UNIMPLEMENTED()                                       \
    fprintf(stderr, "%s:%d: unimplemented\n", __FILE__, __LINE__); \
    exit(1);


// nacl_bit_cast<Dest,Source> is a template function that implements the
// equivalent of "*reinterpret_cast<Dest*>(&source)".  We need this in
// very low-level functions like the protobuf library and fast math
// support.
//
//   float f = 3.14159265358979;
//   int i = nacl_bit_cast<int32>(f);
//   // i = 0x40490fdb
//
// The classical address-casting method is:
//
//   // WRONG
//   float f = 3.14159265358979;            // WRONG
//   int i = * reinterpret_cast<int*>(&f);  // WRONG
//
// The address-casting method actually produces undefined behavior
// according to ISO C++ specification section 3.10 -15 -.  Roughly, this
// section says: if an object in memory has one type, and a program
// accesses it with a different type, then the result is undefined
// behavior for most values of "different type".
//
// This is true for any cast syntax, either *(int*)&f or
// *reinterpret_cast<int*>(&f).  And it is particularly true for
// conversions betweeen integral lvalues and floating-point lvalues.
//
// The purpose of 3.10 -15- is to allow optimizing compilers to assume
// that expressions with different types refer to different memory.  gcc
// 4.0.1 has an optimizer that takes advantage of this.  So a
// non-conforming program quietly produces wildly incorrect output.
//
// The problem is not the use of reinterpret_cast.  The problem is type
// punning: holding an object in memory of one type and reading its bits
// back using a different type.
//
// The C++ standard is more subtle and complex than this, but that
// is the basic idea.
//
// Anyways ...
//
// nacl_bit_cast<> calls memcpy() which is blessed by the standard,
// especially by the example in section 3.9 .  Also, of course,
// nacl_bit_cast<> wraps up the nasty logic in one place.
//
// Fortunately memcpy() is very fast.  In optimized mode, with a
// constant size, gcc 2.95.3, gcc 4.0.1, and msvc 7.1 produce inline
// code with the minimal amount of data movement.  On a 32-bit system,
// memcpy(d,s,4) compiles to one load and one store, and memcpy(d,s,8)
// compiles to two loads and two stores.
//
// I tested this code with gcc 2.95.3, gcc 4.0.1, icc 8.1, and msvc 7.1.
//
// WARNING: if Dest or Source is a non-POD type, the result of the memcpy
// is likely to surprise you.

template <class Dest, class Source>
inline Dest nacl_bit_cast(const Source& source) {
  // Compile time assertion: sizeof(Dest) == sizeof(Source)
  // A compile error here means your Dest and Source have different sizes.
  typedef char VerifySizesAreEqual[sizeof(Dest) == sizeof(Source) ? 1 : -1];

  Dest dest;
  memcpy(&dest, &source, sizeof(dest));
  return dest;
}

#endif  /* __cplusplus */

#endif  /* NATIVE_CLIENT_SRC_INCLUDE_NACL_MACROS_H_ */
